/*
 * Copyright (C) 1993-2004 Robert & Jeremy Lain
 * See AUTHORS file for a full list of contributors.
 *
 * $Id: saildef.h,v 1.60 2005/05/03 21:05:50 sailcuter Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef SAILDEF_H
#define SAILDEF_H

#include <qstring.h>
#include "sailmould.h"
#include "geocpp/vect_geom.h"
#include "geocpp/subspace.h"

// forward declarations
class CSail;
class CSeam;

// enumerated types
enum enumSailType { MAINSAIL, JIB, WING };
enum enumSailCut { CROSS, TWIST, HORIZONTAL, VERTICAL, RADIAL, MITRE };


/** This class holds the parameters that define a sail.
 *
 * @see CSailWorker, CSail
 */
class CSailDef
{
public:
    CSailDef();
    ~CSailDef();

    CSailDef & operator=( const CSailDef & );

    // member variables
    /** sail ID name */
    QString sailID;
    /** type of cut */
    enumSailCut sailCut;
    /** type of sail */
    enumSailType sailType;
    /* boat data */
    real LOA, foreI, foreJ, tackX, tackY;
    /* sides of the sail */
    real luffL, rake, gaffDeg, gaffL, footL, leechL;
    /* shape of sides */
    real luffR,  gaffR,  leechR,  footR;
    int  luffRP, gaffRP, leechRP, footRP;
    /** cloth width */
    real clothW;
    /** seam width */
    real seamW, leechHemW, hemsW;

    /** sail twist */
    real twistDeg;
    /** sail sheeting angle */
    real sheetDeg;

    /** wing diedral angle */
    real dihedralDeg;

    /** number of radial sections */
    unsigned int nbSections;
    /** number of radial gores */
    unsigned int nbGores;
    /** number of luff gores */
    unsigned int nbLuffGores;

    /** sail mould */
    CSailMould mould;
};


/** This class does all the sail-related calculations. It is used to create
 * the sail from its definition.
 *
 * @see CSailDef, CSail
 */
class CSailWorker : public CSailDef
{
public:
    CSailWorker(const CSailDef &s);
    ~CSailWorker()
    {}
    ;

    real Area();
    real Diagonal();
    real IRCwidth( const real &h );
    real SailW( const real &h );
    real LeechLength( const real &h );
    real LuffLength( const real &h );
    real SailLP( );

    CSail makeSail();
    CSail makeSail( CSail &flatsail, CSail &dispsail );

protected:
    // layout of sail's panels
    CSail Layout0( CSail &flatsail, CSail &dispsail ); // CROSS
    CSail LayoutTwist( CSail &flatsail, CSail &dispsail ); // TWIST
    CSail LayoutVertical( CSail &flatsail, CSail &dispsail ); // VERTICAL
    CSail LayoutRadial( CSail &flatsail, CSail &dispsail ); // RADIAL
    CSail LayoutMitre( CSail &flatsail, CSail &dispsail ); // MITRE
    CSail LayoutTriRadial( CSail &flatsail, CSail &dispsail ); // TRIRADIAL
    CSail LayoutWing( CSail &flatsail, CSail &dispsail ); // WING

    // intersections with edges
    CPoint3d FwdIntersect( const CPoint3d &p1 );
    CPoint3d AftIntersect( const CPoint3d &p1 );
    CPoint3d FootIntersect( const CPoint3d &pt1, const CVector3d &v1 );
    CPoint3d GaffIntersect( const CPoint3d &pt1, const CVector3d &v1 );
    CPoint3d LeechIntersect( const CPoint3d &pt1, const CVector3d &v1 );
    CPoint3d LuffIntersect( const CPoint3d &pt1, const CVector3d &v1 );
    CPoint3d MitreIntersect( const CPoint3d &pt1, const CVector3d &v1 );
    // depth at any point
    CPoint3d Zpoint( const CPoint3d &p1 );
    // corners of the sail
    CPoint3d tack, head, peak, clew;
    // mitre intersection with luff
    CPoint3d mitrePt;
    // define useful edges vectors
    CVector3d footV, gaffV, leechV, luffV, mitreV;
    // and their unitary perpendicular vectors
    CVector3d footVP, gaffVP, leechVP, luffVP;
    // define useful (3d) straight lines
    CSubSpace foot, gaff, leech, luff, mitre;
};

#endif
