/*
 * Copyright (C) 1993-2005 Robert & Jeremy Lain
 * See AUTHORS file for a full list of contributors.
 *
 * $Id: formmain.cpp,v 1.138 2005/12/06 17:44:30 jeremy_laine Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
  #include "config.h"
#endif

#include "formmain.h"
#include "formdef.h"
#include "formhelp.h"
#include "formmould.h"
#include "formrig.h"

#include "sailcutqt.h"
#include "saildoc.h"
#include "sailpainter.h"
#include "sailreader-xml.h"
#include "sailworker.h"
#include "sailwriter-carlson.h"
#include "sailwriter-dxf.h"
#include "sailwriter-hand.h"
#include "sailwriter-txt.h"
#include "sailwriter-xml.h"
#include "sailviewer-panel.h"

#include "../icons/sailcut.xpm"

#include <QDir>
#include <QLayout>
#include <QMenuBar>
#include <QStatusBar>
#include <QTabWidget>
#include <QPainter>
#include <QPrintDialog>
#include <QPrinter>


/**
 * Constructs Sailcut CAD's main window.
 *
 * @param pref the user's preferences
 */
CFormMain::CFormMain(CSailApp *myApp, QWidget *parent)
        : QMainWindow(parent), app(myApp), prefs(&myApp->prefs)
{
    setMinimumSize( QSize( 300, 220 ) );

    // locate Handbook
    app->loadTranslation(prefs->language);
    handbook = app->findHandbook(prefs->language);
    qDebug("handbook : %s", (const char*)handbook.toLocal8Bit());

    // create status bar
    statusbar = new QStatusBar(this);
    setStatusBar(statusbar);

    // create menu bar
    setupMenuBar();

    // create main widget
    setupMainWidget();

    // set language
    languageChange();

    // set icon
    setWindowIcon( QPixmap( (const char **)sailcut_xpm ) );

    // resize to prefered size
    resize( QSize(prefs->sailWindowWidth,prefs->sailWindowHeight).expandedTo(minimumSizeHint()) );
}


/**
 * This event is received when the user closes the dialog.
 */
void CFormMain::closeEvent(QCloseEvent *e)
{
    prefs->sailWindowHeight = height();
    prefs->sailWindowWidth = width();
    QMainWindow::closeEvent(e);
}


/**
 * We received a keypress, we pass it down to the visible tab.
 */
void CFormMain::keyPressEvent ( QKeyEvent * e )
{
    panel[tabs->currentIndex()]->keyPressEvent(e);
}


/**
 * This method is called whenever we read or write a sail definition.
 * It updates the Most Recently Used files list correspondingly.
 *
 * @param event The action performed on the file (read, write)
 * @param file The file that was accessed
 */
void CFormMain::fileAccess(QString event, QString file)
{
    statusbar->showMessage(event);
    prefs->mruSaildef.touchEntry(file);
    makeMenuMru();
}


/**
 * Sets the strings of the subwidgets using the current
 * language.
 */
void CFormMain::languageChange()
{
    setWindowTitle( "Sailcut CAD" );

    menuFile->setTitle( tr("&File") );

    actionNew->setText( tr("&New") );
    actionOpen->setText( tr("&Open") );
    menuRecent->setTitle( tr("Open &recent") );
    actionSave->setText( tr("&Save") );
    actionSaveAs->setText( tr("Save &As") );

    // print submenu
    menuPrint->setTitle( tr("&Print") );
    actionPrintData->setText( tr("data") );
    actionPrintDwg->setText( tr("drawing") );
    actionPrintDev->setText( tr("development") );

    // export 3d submenu
    menuExport3d->setTitle( tr("E&xport 3D sail") );
    actionExport3dDXF->setText( tr("to &DXF") );
    actionExport3dTXT->setText( tr("to &TXT sail") );
    actionExport3dXML->setText( tr("to &XML sail") );

    // export flat submenu
    menuExportFlat->setTitle( tr("Export &development") );
    actionExportFlatCarlson->setText( tr("to &Carlson plotter") );
    actionExportFlatDXF->setText( tr("to &DXF") );
    //actionExportFlatDXFBlocks->setText( tr("to &DXF-BLOCKS") );
    actionExportFlatHand->setText( tr("to &Hand-plotting format") );
    actionExportFlatTXT->setText( tr("to &TXT sail") );
    actionExportFlatXML->setText( tr("to &XML sail") );

    actionQuit->setText( tr("&Quit") );


    // View menu

    menuView->setTitle( tr("&View") );
    actionViewDef->setText( tr("&Dimensions") );
    actionViewMould->setText( tr("&Mould") );
    actionViewPatch->setText( tr("&Patches") );
    actionViewRig->setText( tr("&Rig") );

    // language submenu
    menuLanguage->setTitle( tr("Language") );


    // Help menu
    menuHelp->setTitle( tr("&Help") );
    actionHandbook->setText( tr("Sailcut &Handbook") );
    actionAboutQt->setText( tr("About &Qt") );
    actionAbout->setText( tr("About &Sailcut") );

    // send changeLanguage to the tabs
    for (unsigned int i = 0; i < panel.size(); i++)
        panel[i]->languageChange();

#ifdef HAVE_GL
    tabs->setTabText(0, tr("shaded view"));
    tabs->setTabText(1, tr("wireframe view"));
    tabs->setTabText(2, tr("development"));
#else
    tabs->setTabText(0, tr("wireframe view"));
    tabs->setTabText(1, tr("development"));
#endif

}


/**
 * Creates the "Open Recent" menu from the Most Recently Used files list.
 */
void CFormMain::makeMenuMru()
{
    menuRecent->clear();

    for ( unsigned int i = 0; i < prefs->mruSaildef.size(); i++)
    {
        menuRecent->addAction( prefs->mruSaildef[i], this, SLOT( slotOpenRecent() ) )->setData(i);
    }
}


/**
 * Replaces the current sail definition.
 *
 * @param newdef
 */
void CFormMain::setSailDef(const CSailDef newdef)
{
    saildef = newdef;
    sail = CSailWorker(saildef).makeSail(flatsail,dispsail);
    panel[0]->setSail(sail);
#ifdef HAVE_GL

    panel[1]->setSail(sail);
    panel[2]->setSail(dispsail);
#else

    panel[1]->setSail(dispsail);
#endif
}


/**
 * Creates the menu bar
 */
void CFormMain::setupMenuBar()
{
    // File menu
    menuFile = menuBar()->addMenu("");
    actionNew = menuFile->addAction("", this, SLOT( slotNew() ) );
    actionOpen = menuFile->addAction("", this, SLOT( slotOpen() ) );

    menuRecent = menuFile->addMenu("");

    menuFile->addSeparator();

    // print submenu
    menuPrint = menuFile->addMenu("");
    actionPrintData = menuPrint->addAction("", this, SLOT( slotPrintData() ));
    actionPrintDwg = menuPrint->addAction("", this, SLOT( slotPrintDwg() ));
    actionPrintDev = menuPrint->addAction("", this, SLOT( slotPrintDev() ));

    menuFile->addSeparator();


    actionSave = menuFile->addAction("", this, SLOT( slotSave() ) );
    actionSaveAs = menuFile->addAction("", this, SLOT( slotSaveAs() ) );

    // export 3d submenu
    menuExport3d = menuFile->addMenu("");
    actionExport3dDXF = menuExport3d->addAction("", this, SLOT( slotExportDXF() ) );
    actionExport3dTXT = menuExport3d->addAction("", this, SLOT( slotExportTXT() ) );
    actionExport3dXML = menuExport3d->addAction("", this, SLOT( slotExportXML() ) );

    // export flat submenu
    menuExportFlat = menuFile->addMenu("");
    actionExportFlatCarlson = menuExportFlat->addAction("", this, SLOT( slotExportFlatCarlson() ) );
    actionExportFlatDXF = menuExportFlat->addAction("", this, SLOT( slotExportFlatDXF() ) );
    //actionExportFlatDXFBlocks = menuExportFlat->addAction("", this, SLOT( slotExportFlatDXFBlocks() ) );
    actionExportFlatHand = menuExportFlat->addAction("", this, SLOT( slotExportFlatHand() ) );
    actionExportFlatTXT = menuExportFlat->addAction("", this, SLOT( slotExportFlatTXT() ) );
    actionExportFlatXML = menuExportFlat->addAction("", this, SLOT( slotExportFlatXML() ) );

    menuFile->addSeparator();

    actionQuit = menuFile->addAction( "", this, SLOT( close() ) );

    // View menu

    menuView = menuBar()->addMenu("");
    actionViewDef = menuView->addAction( "", this, SLOT( slotDef() ) );
    actionViewMould = menuView->addAction( "", this, SLOT ( slotMould() ) );
    // TODO : enable the following action when the patch viewer is ready
    actionViewPatch = menuView->addAction( "" );
    actionViewPatch->setEnabled(false);
    actionViewRig = menuView->addAction( "", this, SLOT ( slotRig() ) );

    menuView->addSeparator();

    // language submenu
    menuLanguage = menuView->addMenu("");

    // language text is not to be translated
    menuLanguage->addAction( "English", this, SLOT( slotLanguage() ) )->setData("en");
    menuLanguage->addAction( "Franais", this, SLOT( slotLanguage() ) )->setData("fr");
    menuLanguage->addAction( "Nederlands", this, SLOT( slotLanguage() ) )->setData("nl");
    menuLanguage->addAction( "Deutsch", this, SLOT( slotLanguage() ) )->setData("de");
    menuLanguage->addAction( "Italiano", this, SLOT( slotLanguage() ) )->setData("it");
    menuLanguage->addAction( "Norsk", this, SLOT( slotLanguage() ) )->setData("no");
    menuLanguage->addAction( "Dansk", this, SLOT( slotLanguage() ) )->setData("dk");
//    menuLanguage->addAction( "Svenska", this, SLOT( slotLanguage() ) )->setData("sv");
    menuLanguage->addAction( "Portugus", this, SLOT( slotLanguage() ) )->setData("pt");
//    menuLanguage->addAction( "Espaol", this, SLOT( slotLanguage() ) )->setData("es");
    menuLanguage->addAction( "Russian", this, SLOT( slotLanguage() ) )->setData("ru");


    // Help menu
    menuHelp = menuBar()->addMenu("");

    actionHandbook = menuHelp->addAction( "", this, SLOT( slotHandbook() ) );
    actionAboutQt = menuHelp->addAction( "", this, SLOT( slotAboutQt() ) );
    actionAbout = menuHelp->addAction( "", this, SLOT( slotAbout() ) );

}


/**
 * Creates the main widget
 */
void CFormMain::setupMainWidget()
{
    tabs = new QTabWidget(this);

    CSailViewerPanel *tmp;
#ifdef HAVE_GL
    tmp = new CSailViewerPanel(0, SHADED, true);
    panel.push_back(tmp);
#endif

    tmp = new CSailViewerPanel(0, WIREFRAME, true);
    panel.push_back(tmp);
    tmp = new CSailViewerPanel(0, WIREFRAME, false);
    panel.push_back(tmp);

    for (unsigned int i = 0 ; i < panel.size(); i++)
    {
        tabs->addTab(panel[i],"");
    }

    setCentralWidget(tabs);
}


/**
 * Opens a given sail file
 */
void CFormMain::show(const QString newname)
{
    // load preferences
    makeMenuMru();

    // load or create sail
    CSailDef newdef;
    filename = newname;
    if ( !filename.isNull() )
    {
        newdef = CSailDefXmlReader("saildef").read(filename);
        fileAccess(tr("loaded '%1'").arg(filename),filename);
    }
    else
    {
        statusbar->showMessage( tr("created new sail") );
    }
    setSailDef(newdef);
    QMainWindow::show();
}


/**
 * Displays the "About Sailcut" dialog box
 */
void CFormMain::slotAbout()
{
    QMessageBox::about( this, tr("About Sailcut"),

      "<h2>Sailcut CAD"
#ifdef VERSION
      " "VERSION
#endif
      "</h2>"
      "<p>Sailcut is a software for designing boat sails<br/>"
      "(C) 1993-2005 Robert & Jeremy Lain.</p>"

      "<p>For more information, visit the project's page at <a href=\"http://sailcut.sourceforge.net/\">http://sailcut.sourceforge.net/</a>.</p>"

      "<p>Sailcut is a trademark registered by Robert Lain.</p>"

      "<table border=\"1\"><tr><td><small>"
      "This program is free software; you can redistribute it and/or modify "
      "it under the terms of the GNU General Public License as published by "
      "the Free Software Foundation; either version 2 of the License, or "
      "(at your option) any later version.<br/><br/>"

      "This program is distributed in the hope that it will be useful, "
      "but WITHOUT ANY WARRANTY; without even the implied warranty of "
      "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the "
      "GNU General Public License for more details.<br/><br/>"

      "You should have received a copy of the GNU General Public License "
      "along with this program; if not, write to the Free Software "
      "Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA"
      "</small></td></tr></table>"
  );
}


/**
 * Displays the "About Qt" dialog box
 */
void CFormMain::slotAboutQt()
{
    QMessageBox::aboutQt( this );
}



/**
 * Displays the sail CFormDef sail definition dialog.
 */
void CFormMain::slotDef()
{  // we pass the CFormDef a pointer to a copy of the sail definition so
    // that it can update it if necessary
    CSailDef saildefcopy = saildef;

    if ( CFormDef(this , &saildefcopy).exec() )
    {
        // we returned from the dialog with an 'OK',
        setSailDef(saildefcopy);
    }
}


/**
 * Export the 3D sail to a DXF file
 */
void CFormMain::slotExportDXF()
{
    CSailDxfWriter3d(sail).writeDialog();
}


/**
 * Exports the 3D sail to a TXT file.
 */
void CFormMain::slotExportTXT()
{
    CSailTxtWriter(sail).writeDialog();
}


/**
 * Exports the 3D sail to an XML file.
 */
void CFormMain::slotExportXML()
{
    CSailXmlWriter(sail, "sail").writeDialog();
}


/**
 * Exports the flat sail to a Carlson plotter file
 */
void CFormMain::slotExportFlatCarlson()
{
    CSailCarlsonWriter(flatsail).writeDialog();
}


/**
 * Exports the flat sail with panels staggered as displayed to a DXF file
  */
void CFormMain::slotExportFlatDXF()
{
    CSailDxfWriter2d(dispsail).writeDialog();
}


/**
 * Exports the flat sail with panels superimposed to a DXF file with blocks
  */
void CFormMain::slotExportFlatDXFBlocks()
{
    CSailDxfWriter2d(flatsail).writeDialog();
}


/**
 * Exports the flat sail to a "hand" sail file.
 */
void CFormMain::slotExportFlatHand()
{
    CSailWriterHand(flatsail).writeDialog();
}


/**
 * Exports the flat sail to a TXT sail file.
 */
void CFormMain::slotExportFlatTXT()
{
    CSailTxtWriter(flatsail).writeDialog();
}


/**
 * Exports the flat sail to an XML sail file
 */
void CFormMain::slotExportFlatXML()
{
    CSailXmlWriter(flatsail,"sail").writeDialog();
}


/**
 * Display the Sailcut handbook.
 */
void CFormMain::slotHandbook()
{
    //qDebug("handbook : %s", (const char*)handbook.toLocal8Bit());
    if ( !handbook.isEmpty() )
    {
        CFormHelp(this , prefs , handbook).exec();
    }
}


/**
 * Switches the current language.
 */
void CFormMain::slotLanguage()
{
    QString locale;

    QAction *a = qobject_cast<QAction *>(sender());
    if ( !a ) return;

    locale = a->data().toString();

    prefs->language = locale;
    app->loadTranslation(locale);

    // try to locate handbook
    handbook = app->findHandbook(locale);

    languageChange();
}


/**
 * Displays the CFormMould sail mould definition dialog.
 */
void CFormMain::slotMould()
{  // we pass the CFormMould a pointer to a copy of the sail mould so
    // that it can update it if necessary
    CSailDef saildefcopy = saildef;

    if ( CFormMould(this , &saildefcopy.mould).exec() )
    {
        // we returned from the dialog with an 'OK'
        setSailDef(saildefcopy);
    }
}


/**
 * Creates a new sail
 */
void CFormMain::slotNew()
{
    filename = "";
    setSailDef(CSailDef());
    statusbar->showMessage( tr("created new sail") );
}


/**
 * Displays a dialog box to open an XML sail definition.
 */
void CFormMain::slotOpen()
{
    CSailDef newdef;
    QString newname = CSailDefXmlReader("saildef").readDialog(newdef,filename);
    if ( !newname.isNull() )
    {
        filename = newname;
        fileAccess(tr("loaded '%1'").arg(filename), filename);
        setSailDef(newdef);
    }
}


/**
 * Opens a recently used sail definition.
 */
void CFormMain::slotOpenRecent()
{
    // retrieve the index of the MRU entry
    QAction *a = qobject_cast<QAction *>(sender());
    if ( !a ) return;
    int index = a->data().toInt();

    filename = prefs->mruSaildef[index];
    try
    {
        setSailDef(CSailDefXmlReader("saildef").read(filename));
        fileAccess(tr("loaded '%1'").arg(filename), filename);
    }
    catch (CException e)
    {
        prefs->mruSaildef.removeEntry(filename);
        makeMenuMru();
        statusbar->showMessage( tr("error loading '%1'").arg(filename) );
    }
}


/**
 * Print the current sail data.
 */
void CFormMain::slotPrintData()
{  // try printing
    try
    { // Epson printer scale is 72 pixel/inch
        QPrinter myprinter;
        myprinter.setOrientation(QPrinter::Portrait);
        myprinter.setFullPage(FALSE);

        QPrintDialog printDialog(&myprinter, this);
        if ( printDialog.exec() == QDialog::Accepted )
        {
            QPainter painter(&myprinter);

            //double aspect = (double)myprinter.widthMM() / (double)myprinter.heightMM();
            unsigned int dpix = myprinter.logicalDpiX();
            //unsigned int dpiy = myprinter.logicalDpiY();
            real scale = dpix/72;

            unsigned int fontsz1 = 10;
            painter.setFont(QFont("times", fontsz1 ));
            QString text1=" ", text2=" ", text3=" ", text4=" ";
            unsigned int x=0, x1=0, x2=0, x3=0, x4=0, y=0, w=0, h=0;
            x1 = int(scale*72);  // first position at 1 inch
            x2 = x1 + int(scale*22*fontsz1);
            x3 = x2 + int(scale*12*fontsz1);
            x4 = x3 + int(scale*12*fontsz1); // fourth position
            h = int(scale*1.5*fontsz1); // line spacing

            // text of page header
            y = int(1+scale*(2*fontsz1));
            x = int(x1*0.6);
            painter.drawText(x,y, tr("Sailcut data sheet"));

            // text of sail identification
            text2 = saildef.sailID;
            if ( text2.length() < 1 )
                text2="*";
            painter.drawText(x2,y, text2);

            // draw boxe around header
            h = int(scale*(2*fontsz1));
            w = int(scale*(20*fontsz1));
            x = int(x1*0.6-scale*(0.5*fontsz1));
            y = int(1+scale*(0.5*fontsz1));
            painter.drawRect(x,y, w,h);
            // draw boxe around sail ID
            w = int(scale*(36*fontsz1));
            x = int(x2- scale*(0.5*fontsz1));
            painter.drawRect(x,y, w,h);

            // initialise printing font and position
            y = y + 3*h;

            // sail cut and type
            text1 = tr("Sail type");
            switch (saildef.sailType )
            {
            case MAINSAIL:
                text2 = tr("Mainsail");
                break;
            case JIB:
                text2 = tr("Jib");
                break;
            case WING:
                text2 = tr("Wing")+" @ " + QString::number(saildef.dihedralDeg) + tr("deg");
                break;
            }
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            y = y + h;

            text1 = tr("Sail layout");
            switch ( saildef.sailCut )
            {
            case CROSS:
                text2 = tr("Cross Cut");
                break;
            case HORIZONTAL:
                text2 = tr("Horizontal Cut");
                break;
            case RADIAL:
                text2 = tr("Radial Cut");
                break;
            case TWIST:
                text2 = tr("Twist Foot Cut");
                break;
            case VERTICAL:
                text2 = tr("Vertical Cut");
                break;
            case MITRE:
                text2 = tr("Mitre Cut");
                break;
            }
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            if ( saildef.sailCut == RADIAL )
            {
                text1  = QString::number(saildef.nbSections);
                text1  = text1 + tr(" sections,   ");
                text2 = QString::number(saildef.nbGores);
                text1  = text1 +text2 + tr(" head gores,   ");
                text2 = QString::number(saildef.nbLuffGores);
                text1  = text1 +text2 + tr(" luff gores.");
                painter.drawText(x3,y, text1);
            }
            y = y + h;

            painter.drawText(int(x1*0.6),y, tr("Rig "));
            y = y + h;

            //* boat data
            text1 = tr("Boat LOA");
            text2 = QString::number(saildef.LOA);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Fore triangle hoist I");
            text2 = QString::number(saildef.foreI);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Fore triangle base J");
            text2 = QString::number(saildef.foreJ);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Tack position X");
            text2 = QString::number(saildef.tackX);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Tack height Y");
            text2 = QString::number(saildef.tackY);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            painter.drawText(int(x1/2),y, tr("Sail dimensions"));
            y = y + h;

            // sides of the sail
            text1 = tr("Luff length");
            text2 = QString::number(saildef.luffL);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Mast/Luff rake");
            text2 = QString::number(saildef.rake);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Gaff angle wrt luff");
            text2 = QString::number(saildef.gaffDeg);
            text3 = "deg.";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Gaff length");
            text2 = QString::number(saildef.gaffL);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Foot length");
            text2 = QString::number(saildef.footL);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Leech length");
            text2 = QString::number(saildef.leechL);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            painter.drawText(int(x1/2), y, tr("Shape of edges"));
            y = y + h;

            // shape of sides
            text1 = tr("Luff round");
            text2 = QString::number(saildef.luffR);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Gaff round");
            text2 = QString::number(saildef.gaffR);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Leech round");
            text2 = QString::number(saildef.leechR);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Foot round");
            text2 = QString::number(saildef.footR);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Luff round position");
            text2 = QString::number(saildef.footRP);
            text3 = "%";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Gaff round position");
            text2 = QString::number(saildef.gaffRP);
            text3 = "%";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Leech round position");
            text2 = QString::number(saildef.leechRP);
            text3 = "%";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            text1 = tr("Foot round position");
            text2 = QString::number(saildef.footRP);
            text3 = "%";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            // sail setting
            painter.drawText(int(x1/2), y, tr("Sail settings"));
            y = y + h;

            // twist
            text1 = tr("Twist angle");
            text2 = QString::number(saildef.twistDeg);
            text3 = tr("deg");
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            // sheeting
            text1 = tr("Sheeting angle");
            text2 = QString::number(saildef.sheetDeg);
            text3 = tr("deg");
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y + h;

            painter.drawText(int(x1/2), y, tr("Cloth seams and hems"));
            y = y + h;

            // cloth width, seam and hems width
            text1 = tr("Cloth width");
            text2 = QString::number(saildef.clothW);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y+h;

            text1 = tr("Seams width");
            text2 = QString::number(saildef.seamW);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y+h;

            text1 = tr("Leech hem width");
            text2 = QString::number(saildef.leechHemW);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y+h;

            text1 = tr("Other hem width");
            text2 = QString::number(saildef.hemsW);
            text3 = "mm";
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            y = y+h;

            painter.drawText(int(x1/2), y, tr("Sail mould"));
            y = y + h;
            x = int(scale*fontsz1);
            text2 = tr("Luff factor");
            text3 = tr("Depth");
            text4 = tr("Leech factor");
            painter.drawText(x2-2*x,y, text2);
            painter.drawText(x3-x,y, text3);
            painter.drawText(x4-2*x,y, text4);
            y = y + h;

            text1 = tr("Top profile");
            text2 = QString::number( saildef.mould.profile[2].getLuff() );
            text3 = QString::number( saildef.mould.profile[2].getDepth()*100 )+ "%";
            text4 = QString::number( saildef.mould.profile[2].getLeech() *50);
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            painter.drawText(x4,y, text4);
            y = y + h;

            text1 = tr("Mid profile at h = ") +QString::number( saildef.mould.vertpos )+"%";
            text2 = QString::number( saildef.mould.profile[1].getLuff() );
            text3 = QString::number( saildef.mould.profile[1].getDepth()*100 )+"%";
            text4 = QString::number( saildef.mould.profile[1].getLeech() *50);
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            painter.drawText(x4,y, text4);
            y = y + h;

            text1 = tr("Bottom profile");
            text2 = QString::number( saildef.mould.profile[0].getLuff() );
            text3 = QString::number( saildef.mould.profile[0].getDepth()*100 )+"%";
            text4 = QString::number( saildef.mould.profile[0].getLeech() *50);
            painter.drawText(x1,y, text1);
            painter.drawText(x2,y, text2);
            painter.drawText(x3,y, text3);
            painter.drawText(x4,y, text4);

            // now draw rectangle at end of text
            x = int(scale*fontsz1);
            y = y + h;
            w = x4 - x2 + 3*x;
            h = int(scale*2);
            painter.drawRect (x2-x,y, w,h);
            ///
        }
    }
    catch (CException e)
    {
        QMessageBox::information(this, tr("error"), tr("There was a data printing error"));
    }
}


/**
 * Print the current developed sail drawings one panel per page
 * with coordinates of key points for handplotting.
 */
void CFormMain::slotPrintDev()
{
    try
    {  // try printing
        QPrinter myprinter;
        // set landscape printing
        myprinter.setOrientation(QPrinter::Landscape);
        myprinter.setFullPage(FALSE);

        QPrintDialog printDialog(&myprinter, this);
        if ( printDialog.exec() == QDialog::Accepted )
        {
            CSailPainter painter(&myprinter);
            int x=1, y=1, x1=1, y1=1;
            unsigned int npt;

            real xo=0, yo=0, xp=0, yp=0, dx=0, dy=0;

            // set the viewport
            CSailDisp disp;
            disp.setSail(flatsail);
            QRect vRect = painter.viewport();
            disp.setVRect(vRect.width(), vRect.height());
            disp.setZoom(0.8);

            //double aspect = (double)myprinter.widthMM() / (double)myprinter.heightMM();
            unsigned int height = myprinter.heightMM();
            real scale = height/50;

            QString text1, text2;
            unsigned int fontsz1 = int(10*scale);
            painter.setFont(QFont ("times", fontsz1));

            // print the panels out one by one
            for (unsigned int i = 0; i < flatsail.panel.size(); i++)
            {
                if ( i > 0 )
                {
                    myprinter.newPage();
                }
                // set coordinate system to match the logical viewport
                painter.setWindow(disp.getLRect());

                CPanel currentPanel = disp.getSail().panel[i];
                CRect3d rp = currentPanel.boundingRect();
                painter.draw(currentPanel);

                // mark corners of cloth rectangle
                painter.setPen(Qt::green);
                /*
                xo = rp.max.x();
                yo = rp.max.y();
                painter.drawLine(int(xo),-int(yo+.5*fontsz1),int(xo),-int(yo-.5*fontsz1));
                painter.drawLine(int(xo+.5*fontsz1),-int(yo),int(xo-.5*fontsz1),-int(yo));
                */
                xo = rp.min.x();
                yo = rp.min.y();
                painter.drawLine(int(xo),-int(yo+.5*fontsz1),int(xo),-int(yo-.5*fontsz1));
                painter.drawLine(int(xo+.5*fontsz1),-int(yo),int(xo-fontsz1),-int(yo));
                x = int( xo - 4*fontsz1);
                y = -int(yo + fontsz1);
                text2 = "X=0";
                painter.drawText(x, y, text2);
                y = y +int(1.5*fontsz1);
                text2 = "Y=0";
                painter.drawText(x, y, text2);

                // print the current panel number
                x = int(currentPanel.centroid().x());
                y = -int(currentPanel.centroid().y());
                text2 = QString::number( i );
                painter.drawText(x, y, text2);

                // add coordinates of inner lines
                painter.setPen(Qt::blue);
                // top fwd corners
                npt = 0;
                x = int(currentPanel.top.point[npt].x() );
                y = -int(currentPanel.top.point[npt].y() );
                x1 = int(x -fontsz1 );
                y1 = int(y - 3* fontsz1);
                painter.drawLine(x,y, x1,y1);
                x1 = x1 - int(5*fontsz1);
                y1 = y1 - int(2* fontsz1);
                xp = .1* int(10 * flatsail.panel[i].top.point[0].x() ) ;
                yp = .1 * int(10* flatsail.panel[i].top.point[0].y() );
                text1 = "X=" + QString::number(xp);
                painter.drawText(x1,y1, text1);
                y1 = y1 +int(1.5*fontsz1);
                text1 = "Y=" + QString::number(yp);
                painter.drawText(x1,y1, text1);

                // top  middle
                npt = int ( (currentPanel.top.nbpoints() -1) /2 );
                if ( CVector3d(flatsail.panel[i].top.point[npt]-flatsail.panel[i].top.point[0]).norm() > 5 )
                {
                    x = int(currentPanel.top.point[npt].x() );
                    y = -int(currentPanel.top.point[npt].y() );
                    x1 = x -int(fontsz1);
                    y1 = y -int(6* fontsz1);
                    painter.drawLine(x,y, x1,y1);
                    x1 = x1- int(4* fontsz1);
                    y1 = y1 -int(2* fontsz1);
                    dx = .1 * int(10 * CVector3d( flatsail.panel[i].top.point[npt] - flatsail.panel[i].top.point[0] ) * CVector3d( flatsail.panel[i].top.point[flatsail.panel[i].top.nbpoints() -1] - flatsail.panel[i].top.point[0] ).unit() );
                    text1 = "dX=" + QString::number( dx );
                    painter.drawText(x1, y1, text1);
                    y1 = y1 +int(1.5*fontsz1);
                    dy = .1 * int(10 * Distance3d(flatsail.panel[i].top.point[npt] , flatsail.panel[i].top.point[0] , flatsail.panel[i].top.point[flatsail.panel[i].top.nbpoints()-1] ) );
                    if ( (dy<.0001) && (dy>-.0001) )  dy = 0;
                    text1 = "dY=" + QString::number(dy );
                    painter.drawText(x1,y1, text1);
                }
                // top aft corner
                npt = currentPanel.top.nbpoints() -1;
                if ( CVector3d(flatsail.panel[i].top.point[npt]-flatsail.panel[i].top.point[0]).norm() > 5 )
                {
                    x = int(currentPanel.top.point[npt].x() );
                    y = -int(currentPanel.top.point[npt].y() );
                    x1 = x + int( fontsz1 );
                    y1 = y - int(3* fontsz1);
                    painter.drawLine(x,y, x1,y1);
                    x1 = x1 -int(3* fontsz1);
                    y1 = y1 -int(2* fontsz1);
                    xp = .1 * int(10 * flatsail.panel[i].top.point[npt].x() );
                    yp = .1 * int(10 * flatsail.panel[i].top.point[npt].y() );
                    text1 = "X=" + QString::number(xp);
                    painter.drawText(x1, y1, text1);
                    y1 = y1 +int(1.5* fontsz1);
                    text1 = "Y=" + QString::number(yp);
                    painter.drawText(x1,y1, text1);
                }
                // right middle
                npt = (currentPanel.right.nbpoints() -1)/2;
                x = int(currentPanel.right.point[npt].x() );
                y = -int(currentPanel.right.point[npt].y() );

                x1 = x +int(2* fontsz1 );
                y1 = y -int( fontsz1);
                painter.drawLine(x,y, x1,y1);
                x1 = x1 -int( fontsz1);
                y1 = y1 -int(2* fontsz1);
                xp = .1 * int(10 * flatsail.panel[i].right.point[npt].x() );
                yp = .1 * int(10 * flatsail.panel[i].right.point[npt].y() );
                text1 = "X=" + QString::number(xp);
                painter.drawText(x1,y1, text1);
                y1 = y1 +int(1.5* fontsz1);
                text1 = "Y=" + QString::number(yp);
                painter.drawText(x1,y1, text1);

                // bottom left corner
                if ( CVector3d(flatsail.panel[i].top.point[0]-flatsail.panel[i].bottom.point[0]).norm() > 5 )
                {
                    npt = 0;
                    x = int(currentPanel.bottom.point[npt].x() );
                    y = -int(currentPanel.bottom.point[npt].y() );
                    x1 = x - int( fontsz1 );
                    y1 = y + int(3* fontsz1);
                    painter.drawLine(x,y, x1,y1);
                    x1 = x1 - int(3* fontsz1);
                    y1 = y1+ int(1.5* fontsz1);
                    xp = .1 * int(10 * flatsail.panel[i].bottom.point[npt].x() );
                    yp = .1 * int(10 * flatsail.panel[i].bottom.point[npt].y() );
                    text1 = "X=" + QString::number(xp);
                    painter.drawText(x1,y1, text1);
                    y1 = y1+ int(1.5* fontsz1);
                    text1 = "Y=" + QString::number(yp);
                    painter.drawText(x1,y1, text1);
                    // mid left
                    npt = (currentPanel.left.nbpoints() -1)/2;
                    x = int(currentPanel.left.point[npt].x() );
                    y = -int(currentPanel.left.point[npt].y() );
                    x1 = x - int(2*fontsz1 );
                    y1 = y - int( fontsz1);
                    painter.drawLine(x,y, x1,y1);
                    x1 = x1 - int(5* fontsz1);
                    y1 = y1 - int(2* fontsz1);
                    xp = .1 * int(10 * flatsail.panel[i].left.point[npt].x() );
                    yp = .1 * int(10 * flatsail.panel[i].left.point[npt].y() );
                    text1 = "X=" + QString::number(xp);
                    painter.drawText(x1,y1, text1);
                    y1 = y1+ int(1.5* fontsz1);
                    text1 = "Y=" + QString::number(yp);
                    painter.drawText(x1,y1, text1);
                    if ( i == flatsail.panel.size() )
                    {
                        npt = 1+(currentPanel.left.nbpoints() -1)/2;
                        x = int(currentPanel.left.point[npt].x() );
                        y = -int(currentPanel.left.point[npt].y() );
                        x1 = x - int(2*fontsz1 );
                        y1 = y - int( fontsz1);
                        painter.drawLine(x,y, x1,y1);
                        x1 = x1 - int(5* fontsz1);
                        y1 = y1 - int(2* fontsz1);
                        xp = .1 * int(10 * flatsail.panel[i].left.point[npt].x() );
                        yp = .1 * int(10 * flatsail.panel[i].left.point[npt].y() );
                        text1 = "X=" + QString::number(xp);
                        painter.drawText(x1,y1, text1);
                        y1 = y1+ int(1.5* fontsz1);
                        text1 = "Y=" + QString::number(yp);
                        painter.drawText(x1,y1, text1);
                    }
                }
                // bottom intermediate fwd
                npt = int ( (currentPanel.bottom.nbpoints() -1) /5 );
                x = int(currentPanel.bottom.point[npt].x() );
                y = -int(currentPanel.bottom.point[npt].y() );
                x1 = x - int( fontsz1);
                y1 = y + int(3* fontsz1);
                painter.drawLine(x,y, x1,y1);
                x1 = x1 - int(2* fontsz1);
                y1 = y1 + int(1.5* fontsz1);
                dx = .1 * int(10 * CVector3d(  flatsail.panel[i].bottom.point[npt] -  flatsail.panel[i].bottom.point[0] ) * CVector3d(  flatsail.panel[i].bottom.point[ flatsail.panel[i].bottom.nbpoints() -1] -  flatsail.panel[i].bottom.point[0] ).unit() );
                text1 = "dX=" + QString::number( dx );
                painter.drawText(x1, y1, text1);
                y1 = y1 +int(1.5*fontsz1);
                dy = .1 * int(10 * Distance3d( flatsail.panel[i].bottom.point[npt] ,  flatsail.panel[i].bottom.point[0] ,  flatsail.panel[i].bottom.point[ flatsail.panel[i].bottom.nbpoints()-1] ) );
                if ( (dy<.01) && (dy>-.01) )  dy = 0;
                text1 = "dY=" + QString::number(dy );
                painter.drawText(x1,y1, text1);
                // bottom intermediate middle
                npt = int ( (currentPanel.bottom.nbpoints() -1) /2 );
                x = int(currentPanel.bottom.point[npt].x() );
                y = -int(currentPanel.bottom.point[npt].y() );
                x1 = x + int( fontsz1 );
                y1 = y - int(3* fontsz1);
                painter.drawLine(x,y, x1,y1);
                x1 = x1 - int(3* fontsz1);
                y1 = y1 - int(2* fontsz1);
                dx = .1 * int(10 * CVector3d(  flatsail.panel[i].bottom.point[npt] -  flatsail.panel[i].bottom.point[0] ) * CVector3d(  flatsail.panel[i].bottom.point[ flatsail.panel[i].bottom.nbpoints() -1] -  flatsail.panel[i].bottom.point[0] ).unit() );
                text1 = "dX=" + QString::number( dx );
                painter.drawText(x1, y1, text1);
                y1 = y1 +int(1.5*fontsz1);
                dy = .1 * int(10 * Distance3d( flatsail.panel[i].bottom.point[npt] ,  flatsail.panel[i].bottom.point[0] ,  flatsail.panel[i].bottom.point[ flatsail.panel[i].bottom.nbpoints()-1] ) );
                if ( (dy<.01) && (dy>-.01) )  dy = 0;
                text1 = "dY=" + QString::number(dy );
                painter.drawText(x1,y1, text1);
                // bottom intermediate aft
                npt = int ( (currentPanel.bottom.nbpoints() -1) *4/5 );
                x = int(currentPanel.bottom.point[npt].x() );
                y = -int(currentPanel.bottom.point[npt].y() );
                x1 = x - int( fontsz1 );
                y1 = y + int( 3* fontsz1);
                painter.drawLine(x,y, x1,y1);
                x1 = x1-int(4* fontsz1);
                y1 = y1 + int(1.5* fontsz1);
                dx = .1 * int(10 * CVector3d(  flatsail.panel[i].bottom.point[npt] -  flatsail.panel[i].bottom.point[0] ) * CVector3d(  flatsail.panel[i].bottom.point[ flatsail.panel[i].bottom.nbpoints() -1] -  flatsail.panel[i].bottom.point[0] ).unit() );
                text1 = "dX=" + QString::number( dx );
                painter.drawText(x1, y1, text1);
                y1 = y1 +int(1.5*fontsz1);
                dy = .1 * int(10 * Distance3d( flatsail.panel[i].bottom.point[npt] ,  flatsail.panel[i].bottom.point[0] ,  flatsail.panel[i].bottom.point[ flatsail.panel[i].bottom.nbpoints()-1] ) );
                if ( (dy<.01) && (dy>-.01) )  dy = 0;
                text1 = "dY=" + QString::number(dy );
                painter.drawText(x1,y1, text1);
                // bottom aft corner
                npt = currentPanel.bottom.nbpoints() -1;
                x = int(currentPanel.bottom.point[npt].x() );
                y = -int(currentPanel.bottom.point[npt].y() );
                x1 = x +int(fontsz1);
                y1 = y +int(2* fontsz1);
                painter.drawLine(x,y, x1,y1);
                x1 = x1-int(2*fontsz1);
                y1 = y1+int(1.5*fontsz1);
                xp = .1 * int(10 * flatsail.panel[i].bottom.point[npt].x() ) ;
                yp = .1 * int(10 * flatsail.panel[i].bottom.point[npt].y() );
                text1 = "X=" + QString::number(xp);
                painter.drawText(x1, y1, text1);
                y1 = y1+ int(1.5*fontsz1);
                text1 = "Y=" + QString::number(yp);
                painter.drawText(x1,y1, text1);

                // reset pen color
                painter.setPen(Qt::black);

            }
            ///
        }
    }
    catch (CException e)
    {
        QMessageBox::information(this, tr("error"), tr("There was a development printing error"));
    }
}


/**
 * Print the current sail drawing.
 */
void CFormMain::slotPrintDwg()
{  // try printing
    try
    {
        QPrinter myprinter;
        // set landscape printing
        myprinter.setOrientation(QPrinter::Portrait);
        myprinter.setFullPage(FALSE);

        QPrintDialog printDialog(&myprinter, this);
        if ( printDialog.exec() == QDialog::Accepted )
        {
            CSailPainter painter(&myprinter);

            // set the viewport
            CSailDisp disp;
            disp.setSail(sail);
            QRect vRect = painter.viewport();
            disp.setVRect(vRect.width(), vRect.height());
            disp.setZoom(0.80);

            // set coordinate system to match the logical viewport
            painter.setWindow(disp.getLRect());

            //double aspect = (double)myprinter.widthMM() / (double)myprinter.heightMM();
            unsigned int height = myprinter.heightMM();
            real scale = height/50;

            QString text1, text2;
            unsigned int fontsz1 = int(11*scale);
            int  x=1, y=1;
            painter.setFont(QFont ("times", fontsz1));

            // print the panels one by one
            for (unsigned int i = 0; i < flatsail.panel.size(); i++)
            {
               CPanel currentPanel = disp.getSail().panel[i];
               CPoint3d panelCenter= currentPanel.centroid();
                x = int(panelCenter.x());
                y = -int(panelCenter.y());

                painter.draw(currentPanel);
                text2 = QString::number( i );
                painter.drawText( x, y, text2);
            }
            ///
        }
    }
    catch (CException e)
    {
        QMessageBox::information(this, tr("error"), tr("There was a drawing printing error"));
    }
}


/**
 * Display the rig dialog
 */
void CFormMain::slotRig()
{
    CFormRig *frmRig = new CFormRig(app, this);
    frmRig->show();
}


/**
 * Saves the current sail definition to an XML file.
 */
void CFormMain::slotSave()
{
    if ( filename.isEmpty() )
    {
        slotSaveAs();
        return;
    }

    // try writing to file, catch exception
    try
    {
        CSailDefXmlWriter(saildef , "saildef").write(filename);
        fileAccess(tr("wrote '%1'").arg(filename),filename);
    }
    catch (CException e)
    {
        QMessageBox::information( this, tr("error"), tr("There was an error writing to the selected file") );
    }
}


/**
 * Opens a dialog to select the XML to which the sail definition should be saved.
 */
void CFormMain::slotSaveAs()
{
    QString newname = CSailDefXmlWriter(saildef , "saildef").writeDialog(filename);

    if ( !newname.isNull() )
    {
        filename = newname;
        fileAccess(tr("wrote '%1'").arg(filename), filename);
    }
}

