/* SAMHAIN file system integrity testing                                   */
/* Copyright (C) 2000 Rainer Wichmann                                      */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include "config_xor.h"

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>

#include "samhain.h"
#include "sh_error.h"
#include "sh_utils.h"
#include "sh_tiger.h"

#if defined(HAVE_MLOCK) && !defined(HAVE_BROKEN_MLOCK)
#include <sys/mman.h>
#endif


#undef  FIL__
#define FIL__  _("sh_err_log.c")

#undef  FIX_XML
#define FIX_XML 1 

#define MYSIGLEN (2*KEY_LEN + 32)

typedef struct _sh_log_buf {
  char   signature[KEY_LEN+1];
  char   timestamp[KEY_LEN+1];
#ifdef SH_USE_XML
  char   sig[MYSIGLEN];
#endif
  char * msg;
} sh_sh_log_buf;

extern struct  _errFlags  errFlags;

#define CHK_KEY 0
#define CHK_FIL 1
#define CHK_NON 2

int get_key_from_file(char * path, char * keyid, char * key)
{
  SL_TICKET  fd;
  char * buf;
  char * bufc;

  if (path[strlen(path)-1] == '\n')
    path[strlen(path)-1] = '\0';

  /* open the file, then check it 
   */
  if ( SL_ISERROR(fd = sl_open_read (path, SL_NOPRIV)))
    {
      fprintf(stderr, _("Could not open file <%s>\n"), path);
      _exit (EXIT_FAILURE);
    }

  buf     = SH_ALLOC( (unsigned long)(SH_BUFSIZE+1));
  bufc    = SH_ALLOC( (unsigned long)(SH_MAXBUF+1));

  while (1)
    {
      buf[0]  = '\0';
      bufc[0] = '\0';

      /* find start of next key
       */
      while (0 != sl_strncmp(buf, _("-----BEGIN LOGKEY-----"),
			     sizeof("-----BEGIN LOGKEY-----")-1)) 
	{
	  (void) sh_unix_getline (fd, buf, SH_BUFSIZE);
	  if (buf[0] == '\0')
	    {
	      /* End of file reached, return. 
	       */
	      fflush(stdout);
	      sl_close(fd);
	      return -1; 
	    }
	}

      /* read key
       */
      (void) sh_unix_getline (fd, buf, SH_BUFSIZE);

      if (0 == sl_strncmp(keyid, &buf[KEY_LEN], strlen(keyid)))
	{
	  sl_strlcpy(key, buf, KEY_LEN+1);
	  sl_close(fd);
	  return 0;
	}
    }
	  
  /* not found
   */
  sl_close(fd);
  return -1;
}

static int just_list = S_FALSE;

int sh_error_logverify_mod (char * s)
{
  just_list = S_TRUE;
  if (s)      /* compiler warning (unused var) fix */
    return 0;
  else
    return 0;
} 

int sh_error_logverify (char * s)
{
  SL_TICKET fd;
  int len;
  int status;
  int count =  0;
  int start = -1;
  char * buf;
  char * bufc;
#ifdef SH_USE_XML
  char * ptr;
  int fixed_xml = S_TRUE;
  char c_start;
#endif
  char signature[64];
  char key[KEY_LEN+1];
  char path[KEY_LEN+1];
  char timestamp[64];
  char c_cont;
  int  chk_mode = CHK_KEY;

  sh_error_logoff();

  if (s == NULL || sl_strlen(s) >= PATH_MAX)
    {
      fprintf(stderr, _("FAIL: msg=\"Invalid input\", path=\"%s\"\n"), s);
      _exit (EXIT_FAILURE);
    }

  /* Open the file, then check it. 
   */
  if (sl_is_suid())
    {
      fprintf(stderr, _("Cannot open file %s in suid mode\n"), s);
      _exit (EXIT_FAILURE);
    }

  if ( SL_ISERROR(fd = sl_open_read (s, SL_NOPRIV)) )
    {
      fprintf(stderr, 
	      _("FAIL: msg=\"File not accessible\", path=\"%s\"\n"), s);
      _exit (EXIT_FAILURE);
    }

  /* Find space value.
   */
  c_cont  = ' ';
#ifdef SH_STEALTH
  c_cont ^= XOR_CODE;
#endif

#ifdef SH_USE_XML
  c_start  = '<';
#ifdef SH_STEALTH
  c_start ^= XOR_CODE;
#endif
#endif

  buf  = (char *) SH_ALLOC( 2*SH_BUFSIZE+1 );
  bufc = (char *) SH_ALLOC( 2*SH_BUFSIZE+1 );

  while (1) 
    {
      /* get the log message
       */
      if (sh_unix_getline (fd, buf, (2*SH_BUFSIZE)) < 0) 
	break;

      len = (int) sl_strlen(buf);

#ifdef SH_USE_XML
#ifdef SH_STEALTH
      if (0 == sl_strncmp (buf, N_("<trail>"), 7)) 
#else
      if (0 == sl_strncmp (buf, _("<trail>"),  7)) 
#endif
#else 
#ifdef SH_STEALTH
      if (0 == sl_strncmp (buf, N_("[SOF]"), 5)) 
#else
      if (0 == sl_strncmp (buf, _("[SOF]"),  5)) 
#endif
#endif
	{
	  if (just_list == S_TRUE)
	    {
#ifdef SH_STEALTH
	      sh_do_decode (buf, sl_strlen(buf));
#endif
	      fprintf (stdout, _("%s\n"), buf);
	    }

	  /* Found start of audit trail, read first line. 
	   */
	  start = 1;
	  do {
	    if ( sh_unix_getline (fd, buf, (2*SH_BUFSIZE)) < 0)
	      break;
	  } while (buf[0] == '\0' || buf[0] == '\n');
	  len = (int) sl_strlen(buf);

	  if (just_list == S_TRUE)
	    {
#ifdef SH_STEALTH
	      if (buf[0] != '\n') 
		sh_do_decode (buf, sl_strlen(buf));
#endif
	      fprintf (stdout, _("%s\n"), buf);
	      start = 0;
	    }

	  ++count;
	}
      else if (buf[0] == '\n'
#ifdef SH_USE_XML
	       ||
#ifdef SH_STEALTH
	       0 == sl_strncmp(buf, N_("</trail>"), 7)
#else
	       0 == sl_strncmp(buf,  _("</trail>"), 7)
#endif
#endif
	       )
	{
	  if (just_list == S_TRUE)
	    {
#ifdef SH_STEALTH
	      if (buf[0] != '\n') 
		sh_do_decode (buf, sl_strlen(buf));
#endif
	      fprintf (stdout, _("%s\n"), buf);
	    }

	  /* A newline.
	   */
	  ++count;
	  continue;
	}
      else if (start == 0)
	{
	  /* We are inside an audit trail. 
	   */
	  ++count;
	  if (just_list == S_TRUE)
	    {
#ifdef SH_STEALTH
	      sh_do_decode (buf, sl_strlen(buf));
#endif
	      fprintf (stdout, _("%s\n"), buf);
	      continue;
	    }
	}
      else
	{
	  /* No start-of-file found yet. 
	   */
	  continue;
	}

      if (just_list == S_TRUE)
	continue;

      /* Check for a continuation line.
       */
      while (1 == 1)
	{
	  do {
	    if ( sh_unix_getline (fd, bufc, (2*SH_BUFSIZE)) < 0)
	      break;
	  } while (bufc[0] == '\0' || bufc[0] == '\n');
	  ++count;
	  if (bufc[0] == c_cont) 
	    {
	      /* A continuation line. Add the newline. 
	       */
	      sl_strlcat(buf, "\n", 2*SH_BUFSIZE+1);
	      ++len;
	      sl_strlcat(buf, bufc, 2*SH_BUFSIZE+1);
	      len += (int) sl_strlen(bufc);
	    }
	  else
	    {
	      /* No continuation line. Use it as signature. 
	       * A48014C05604EF7C9472330E85453E704024943E556163C2
	       */
#ifdef SH_USE_XML
#ifdef SH_STEALTH
	      if (bufc[0] == c_start) /* FIX XML */
#else
	      if (bufc[0] == c_start)
#endif
		{
		  sl_strlcpy(signature, &bufc[5], KEY_LEN+1);
		  fixed_xml = S_TRUE;
		}
	      else
		{
		  sl_strlcpy(signature, &bufc[4], KEY_LEN+1);
		  fixed_xml = S_FALSE;
		}
	      if (sl_strlen(bufc) > (KEY_LEN+18))
		{
#ifdef SH_STEALTH
		  if (bufc[0] == c_start) /* FIX XML */
#else
		  if (bufc[0] == c_start)
#endif
		    sl_strlcpy(timestamp, &bufc[KEY_LEN+5], 64);
		  else
		    sl_strlcpy(timestamp, &bufc[KEY_LEN+4], 64);
#ifdef SH_STEALTH
		  ptr = strchr(timestamp, c_start);
#else
		  ptr = strchr(timestamp, c_start);
#endif
		  if (ptr) *ptr = '\0';
		}
	      break;
#else
	      sl_strlcpy(signature, bufc, KEY_LEN+1);
	      if (sl_strlen(bufc) > KEY_LEN)
		sl_strlcpy(timestamp, &bufc[KEY_LEN], 64);
	      break;
#endif
	    }
	}
      
      /* Get starting key from command line. 
       */    
      if (start == 1) 
	{
	  
	  /* Get the timestamp.
	   */
	  
#ifdef SH_STEALTH
	  sh_do_decode (timestamp, sl_strlen(timestamp));
#endif
	  key[0] = '\0';
	  
	findKey:
	  
	  if (chk_mode != CHK_FIL)
	    {
	      /* Ask for the key.
	       */
	      chk_mode = CHK_KEY;
	      fprintf(stdout, _("\nNew audit trail (%s), enter key|keyfile: "),
		      timestamp);
	      key[0] = '\0';
	      
	      while (sl_strlen(key) < KEY_LEN ) 
		{ 
		  if (key[0] != '\n' && key[0] != '\0')
		    fprintf(stdout, _("New audit trail, enter key: "));
		  else if (key[0] == '\n')
		    {
		      sl_strlcpy(key, sh_tiger_hash(NULL, TIGER_DATA, 0), 
				 KEY_LEN+1);
		      chk_mode = CHK_NON;
		      break;
		    }
		  fflush(stdout); 
		  key[0] = '\0';
		  fgets(key, KEY_LEN+1, stdin);
		  if (key[0] == '/')
		    {
		      chk_mode = CHK_FIL;
		      sl_strlcpy(path, key, KEY_LEN+1); 
		      break;
		    }
		}
	    }
	  /* we now have either a key (chk_mode == CHK_NON|CHK_KEY)
	   * or a file (chk_mode == CHK_FIL)
	   */
	  if (chk_mode == CHK_FIL)
	    {
	      fprintf(stdout, _("\nAudit trail (%s), searching file %s\n"), 
		      timestamp, path);
	      if (-1 == get_key_from_file(path, timestamp, key))
		{
		  chk_mode = CHK_KEY;
		  fprintf(stdout, _("Key not found in file\n"));
		  goto findKey;
		}
	    }
	  
	  
	  sh_util_encode(key, buf, 1, 'B');
	  start = 0;
	} 
      else
	{ 
	  /* Iterate the key.
	   */
	  sl_strlcpy (key, 
		      sh_tiger_hash (key, TIGER_DATA, KEY_LEN), 
		      KEY_LEN+1);
	}
      
      sl_strlcat ( buf, key, 2*SH_BUFSIZE + 1);
      
#ifdef SH_STEALTH
      sh_do_decode (signature, sl_strlen(signature));
#endif
      
      status = sl_strncmp (signature, 
			   sh_tiger_hash (buf, TIGER_DATA, sl_strlen(buf)),
			   KEY_LEN);
      
      buf[len] = '\0';    /* do not print out the key */
#ifdef SH_STEALTH
      sh_do_decode (buf, sl_strlen(buf));
#endif
      
      if (status != 0) 
	{
#ifdef SH_USE_XML
	  if (chk_mode == CHK_NON)
	    {
	      if (fixed_xml == S_FALSE)
		fprintf (stdout, _("XFAIL: line=%05d %s/log>\n"), 
			 count-1, buf);
	      else
		fprintf (stdout, _("XFAIL: line=%05d %s</log>\n"), 
			 count-1, buf);
	    }
	  else
	    {
	      if (fixed_xml == S_FALSE)
		fprintf (stdout, _("FAIL:  line=%05d %s/log>\n"), 
			 count-1, buf);
	      else
		fprintf (stdout, _("FAIL:  line=%05d %s</log>\n"), 
			 count-1, buf);
	    }
#else
	  if (chk_mode == CHK_NON)
	    fprintf (stdout, _("XFAIL: line=%5d %s\n"), count-1, buf);
	  else
	    fprintf (stdout, _("FAIL:  line=%5d %s\n"), count-1, buf);
#endif
	}
      else
	{
#ifdef SH_USE_XML 
	  if (fixed_xml == S_FALSE)
	    fprintf (stdout, _("PASS:  line=%05d %s/log>\n"),  count-1, buf);
	  else
	    fprintf (stdout, _("PASS:  line=%05d %s</log>\n"), count-1, buf);
#else
	  fprintf (stdout, _("PASS:  line=%5d %s\n"), count-1, buf);
#endif    
	}
    }

  /* Cleanup and exit.
   */
  sl_close (fd);
  SH_FREE  (buf);
  SH_FREE  (bufc);
  fflush   (stdout);
  _exit    (EXIT_SUCCESS);

  /* Make compilers happy. 
   */
  return 0; 
}

/********************************************************************
 *
 *  Runtime code
 *
 ********************************************************************/

int sh_log_open (char * inet_peer, 
                 char * logfile, int * service_failure, SL_TICKET * fildesc)
{
  SL_TICKET            fd = -1;
  long int             status;
  char               * tmp = NULL;
  uid_t                uid;
  int                  i;
  char               * lockfile = NULL;

  /* open/create the file, then check it 
   */

  if (  0 !=  (status = tf_trust_check (logfile, SL_YESPRIV))
        && (*service_failure) == 0)
    {
      tmp  = sh_util_safe_name (logfile);
      sh_error_handle ((-1), FIL__, __LINE__, status, MSG_E_TRUST,
                      (long) sh.effective.uid, tmp);
    }

  if (status == 0)
    {
      fd = sl_open_write (logfile, SL_YESPRIV);
      if (SL_ISERROR(fd))
        {
	  tmp  = sh_util_safe_name (logfile);
	  sl_get_euid(&uid);
          if ((*service_failure) == 0)
            sh_error_handle ((-1), FIL__, __LINE__, status, MSG_E_ACCESS,
                             (long) uid, tmp);
          status = -1;
        }
    }


  if (status == 0 && inet_peer == NULL )
    {
      status = sh_unix_write_lock_file(logfile);
      if (status < 0)
        {
	  tmp  = sh_util_safe_name (logfile);
	  i        = 6 + sl_strlen(tmp);
	  lockfile = SH_ALLOC(i);
	  sl_strlcpy(lockfile,        tmp, i);
	  sl_strlcat(lockfile, _(".lock"), i);
          sl_get_euid(&uid);
          if ((*service_failure) == 0)
            sh_error_handle ((-1), FIL__, __LINE__, status, MSG_LOCKED,
                             (long) uid, tmp, lockfile);
          status = -1;
	  SH_FREE(lockfile);
          sl_close(fd);
        }
    }

  if (status == 0)
    {
      status = SL_ISERROR(sl_forward(fd) ); 
      if (status < 0)
        {
	  tmp  = sh_util_safe_name (logfile);
          sl_get_euid(&uid);
          if ((*service_failure) == 0)
            sh_error_handle ((-1), FIL__, __LINE__, status, MSG_E_ACCESS,
                             (long) uid, tmp);
          status = -1;
          sl_close(fd);
        }
    }
  
  if (status < 0)
    {
      if ((*service_failure) == 0) {
        sh_error_handle ((-1), FIL__, __LINE__, status, MSG_SRV_FAIL,
                         _("logfile"), tmp);
        (*service_failure) = 1;
      }
      SH_FREE(tmp);
      return -1;
    }

  *fildesc         = fd;
  *service_failure = 0;
  return 0;
}

typedef struct lfstc {
  char          * logfile;
  int             service_failure;
  int             log_start;
  char            sigkey_old[KEY_LEN+1];
  char            sigkey_new[KEY_LEN+1];
  char            crypt[KEY_LEN+1]; 
  struct  lfstc * next;
} open_logfile;

static open_logfile * logfile_list = NULL;

static int flag_sep_log = S_FALSE;

#ifdef SH_WITH_SERVER
int set_flag_sep_log (char * str)
{
  return sh_util_flagval(str, &flag_sep_log);
}
#endif

/*
 *   --- Log error message to log file. ---
 */
int  sh_log_file (char *errmsg, char * inet_peer)
{
  int                  store1;
  int                  store2;
  int                  store3;
  int                  store4;
  int                  store5;
  int                  store6;
  int                  store7;
  int                  store8;

  SL_TICKET            fd = -1;
  long int             status;
  struct _sh_log_buf   log_msg;

  char                 logfile[SH_PATHBUF+SH_MINIBUF+2];
  open_logfile       * current = logfile_list;  
  open_logfile       * next    = NULL;
  char               * sigkey_new;
  char               * sigkey_old;
  char               * crypt;

  SL_ENTER(_("sh_log_file"));

  if (errFlags.HaveLog == BAD)  /* paranoia */ 
    SL_RETURN((-1), _("sh_log_file"));

#ifdef SH_USE_XML
  if (NULL == errmsg)
    {
      while (current != NULL)
        {
	  /* don't write second EOF mark
	   */
	  if (current->log_start != S_TRUE)
	    {
	      /* Don't use inet_peer == NULL, userwise a lock file will
	       * be created.
	       */
	      sh_log_open ("\0", 
			   current->logfile, &(current->service_failure), &fd);
          
#ifdef SH_STEALTH
	      sl_write_line (fd, N_("</trail>"), 7);
	      sl_write (fd, "\n", 1);
	      sl_sync(fd);
#else
	      sl_write_line (fd, _("</trail>\n"),  8);
	      sl_sync(fd);
#endif
	      sl_close(fd);
	      /* sh_unix_rm_lock_file (current->logfile); */
	    }
	  next    = current->next;
	  SH_FREE(current->logfile);
	  SH_FREE(current);
	  current = next;
	}
      logfile_list = NULL;
      SL_RETURN( 0, _("sh_log_file"));
    }
#else
  if (NULL == errmsg)
    {
      while (current != NULL)
        {
	  /* sh_unix_rm_lock_file (current->logfile); */
	  next    = current->next;
          SH_FREE(current->logfile);
          SH_FREE(current);
          current = next;
        }
      logfile_list = NULL;
      SL_RETURN( 0, _("sh_log_file"));
    }
#endif

  sl_strlcpy (logfile, sh.srvlog.name, sizeof(logfile));
  if (inet_peer != NULL && flag_sep_log == S_TRUE)
    {
      sl_strlcat (logfile, ".",       sizeof(logfile));
      sl_strlcat (logfile, inet_peer, sizeof(logfile));
    }

  if (sh.flag.log_start == S_TRUE)
    {
      while (current != NULL)
        {
          current->log_start = S_TRUE;
          current = current->next;
        }
      sh.flag.log_start    = S_FALSE;
      current = logfile_list;
    }

  while (current != NULL)
    {
      if (strcmp(logfile, current->logfile) == 0)
        break;
      current = current->next;
    }

  if (current == NULL)
    {
      current                  = SH_ALLOC(sizeof(open_logfile));
      current->logfile         = SH_ALLOC(strlen(logfile) + 1);
      sl_strlcpy(current->logfile, logfile, strlen(logfile) + 1);
      current->service_failure = 0;
      current->log_start       = S_TRUE;
      memset(current->sigkey_old, '\0', KEY_LEN+1);
      memset(current->sigkey_new, '\0', KEY_LEN+1);
      memset(current->crypt,      '\0', KEY_LEN+1);
      current->next            = logfile_list;
      logfile_list             = current;
    }

  if (0 != sh_log_open (inet_peer, current->logfile, 
                        &(current->service_failure), &fd))
    {
      SL_RETURN ((-1), _("sh_log_file"));
    }


  /* --- Allocate storage and mlock it. ---
   */

  status      = sl_strlen (errmsg);
  log_msg.msg = (char *) SH_ALLOC ( 2*KEY_LEN + status + 32); 

#if defined(HAVE_MLOCK) && !defined(HAVE_BROKEN_MLOCK)
  if (skey->mlock_failed == SL_FALSE) 
    {
      sl_set_suid ();
      if ( (-1) == sh_unix_mlock( log_msg.msg, 2*KEY_LEN + status + 32 ) ) 
	{
	  sl_unset_suid ();
	  skey->mlock_failed = SL_TRUE;
#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE)
	  sh_error_handle ((-1), FIL__, __LINE__, EPERM, MSG_MLOCK); 
#endif
	}
      else
	sl_unset_suid ();
    }
#else
  if (skey->mlock_failed == SL_FALSE) 
    {
      skey->mlock_failed = SL_TRUE;
#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE)
      sh_error_handle ((-1), FIL__, __LINE__, EPERM, MSG_MLOCK);
#endif
    }
#endif

  /* --- Write the start marker. --- 
   */

  if (current->log_start == S_TRUE) 
    {
#ifdef SH_USE_XML
#ifdef SH_STEALTH
      sl_write (fd, "\n", 1);
      sl_write_line (fd, N_("<trail>"), 7);
      sl_sync(fd);
#else
      sl_write_line (fd, _("\n<trail>"),  8);
      sl_sync(fd);
#endif
#else
#ifdef SH_STEALTH
      sl_write (fd, "\n", 1);
      sl_write_line (fd, N_("[SOF]"), 5);
      sl_sync(fd);
#else
      sl_write_line (fd, _("\n[SOF]"),  6);
      sl_sync(fd);
#endif
#endif
    }

  /* reserve KEY_LEN chars at end for key 
   */
  sl_strlcpy (log_msg.msg, errmsg, status+1 );


#ifdef SH_USE_XML
  /* cut the trailing "/>"
   */
  if (log_msg.msg[status-2] == '/')
    {
#ifdef FIX_XML
      log_msg.msg[status-2] = ' '; /* ' ' FIX XML */
      log_msg.msg[status-1] = '>'; /* '>' FIX XML */
#else
      log_msg.msg[status-2] = '>'; /* ' ' FIX XML */
      log_msg.msg[status-1] = '<'; /* '>' FIX XML */
#endif
      log_msg.msg[status]   = '\0';
    }
  else if (status >= 6 && log_msg.msg[status-5] == '/' && 
	   log_msg.msg[status-6] == '<')
    {
#ifdef FIX_XML
      log_msg.msg[status-6]   = '\0';
      status -= 6;
#else
      log_msg.msg[status-5]   = '\0';
      status -= 5;
#endif
    }
#endif


#ifdef SH_STEALTH
  sh_do_encode (log_msg.msg, status);
#endif

  if (flag_sep_log == S_TRUE && inet_peer != NULL)
    {
      sigkey_old = current->sigkey_old;
      sigkey_new = current->sigkey_new;
      crypt      = current->crypt;
    }
  else
    {
      sigkey_old = skey->sigkey_old;
      sigkey_new = skey->sigkey_new;
      crypt      = skey->crypt;
    }

  /* write the signature 
   */
  if (current->log_start == S_TRUE) 
    {
      if (sh.real.user[0] == '\0') 
	(void) sh_unix_getUser();

      /* Initialize the key.
       */
      sh_util_keyinit(sigkey_old, KEY_LEN+1);

      /* Hash the key to make sure it has the correct format.
       */
      sl_strlcpy(sigkey_new, 
		 sh_tiger_hash (sigkey_old, TIGER_DATA, KEY_LEN), 
		 KEY_LEN+1);

      /* Copy it to 'crypt' for encryption.
       */
      sl_strlcpy(crypt, sigkey_new, KEY_LEN+1);

      /* Use message and compiled-in key to encrypt.
       */
      BREAKEXIT(sh_util_encode);
      sh_util_encode(crypt, log_msg.msg, 0, 'B');

      /* Send out the key.
       */
      sl_strlcpy(log_msg.timestamp, sh_unix_time(0), KEY_LEN+1); 

      store1               = errFlags.loglevel;
      store2               = errFlags.sysloglevel;
      store3               = errFlags.printlevel;
      store4               = errFlags.exportlevel;
      store5               = errFlags.maillevel;
      store6               = errFlags.externallevel;
      store7               = errFlags.databaselevel;
      store8               = errFlags.preludelevel;

      /* mail the key
       */
      errFlags.loglevel       = SH_ERR_NOT;
      errFlags.sysloglevel    = SH_ERR_NOT;
      errFlags.printlevel     = SH_ERR_NOT;
      errFlags.exportlevel    = SH_ERR_NOT;
      errFlags.externallevel  = SH_ERR_NOT;
      errFlags.databaselevel  = SH_ERR_NOT;
      errFlags.preludelevel   = SH_ERR_NOT;

      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_START_KEY_MAIL,
		       sh.prg_name, crypt, 
		       crypt, log_msg.timestamp);

      /* send to other allowed channels
       */
      errFlags.maillevel      = SH_ERR_NOT;
      /* errFlags.printlevel     = store3; */
      errFlags.exportlevel    = store4;
      errFlags.externallevel  = store6;
      errFlags.databaselevel  = store7;
      errFlags.preludelevel   = store8;

      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_START_KEY,
		       sh.prg_name, crypt);

      /* Cleanup.
       */
      errFlags.loglevel       = store1;
      errFlags.sysloglevel    = store2;
      errFlags.printlevel     = store3;
      errFlags.exportlevel    = store4;
      errFlags.maillevel      = store5;
      errFlags.externallevel  = store6;
      errFlags.databaselevel  = store7;


      memset (crypt, '\0', KEY_LEN);
      sh.flag.log_start    = S_FALSE;  
      current->log_start   = S_FALSE;
    } 
  else 
    {
      log_msg.timestamp[0] = '\0';
      sl_strlcpy (sigkey_new, 
		  sh_tiger_hash (sigkey_old, TIGER_DATA, KEY_LEN),
		  KEY_LEN+1);
    }

  /* --- Sign the message with the signature key. ---
   */

  sl_strlcat (log_msg.msg, sigkey_new, status + KEY_LEN + 2);
  
  sl_strlcpy (log_msg.signature,
	      sh_tiger_hash (log_msg.msg, TIGER_DATA, status + KEY_LEN), 
	      KEY_LEN+1);
  sl_strlcpy (sigkey_old, sigkey_new, KEY_LEN+1); 

#ifdef SH_USE_XML
  if (log_msg.timestamp[0] != '\0')
    sprintf(log_msg.sig,                            /* known to fit  */
#ifdef FIX_XML
	    _("\n<sig>%s%s</sig></log>\n"),          /* <sig> FIX XML */
#else
	    _("\nsig>%s%s</sig></log>\n"),          /* <sig> FIX XML */
#endif
	    log_msg.signature, log_msg.timestamp);
  else
    sprintf(log_msg.sig,                            /* known to fit  */
#ifdef FIX_XML
	    _("\n<sig>%s</sig></log>\n"),            /* <sig> FIX XML */
#else
	    _("\nsig>%s</sig></log>\n"),            /* <sig> FIX XML */
#endif
	    log_msg.signature);
#ifdef SH_STEALTH
  /* don't encode the line breaks (0 + last char)
   */
  sh_do_encode (&log_msg.sig[1], (sl_strlen(log_msg.sig)-2) );
#endif
#else
#ifdef SH_STEALTH
  sh_do_encode (log_msg.signature, KEY_LEN);
  sh_do_encode (log_msg.timestamp, sl_strlen(log_msg.timestamp));
#endif
#endif
  
#ifdef SH_USE_XML
  log_msg.msg[status] = '\0';
  sl_strlcat (log_msg.msg,   log_msg.sig, status + 2*KEY_LEN + 32);
#ifdef SH_STEALTH
  if (NULL != sl_strstr(log_msg.msg, N_("EXIT")) &&
      NULL == sl_strstr(log_msg.msg, N_("remote_host")))
    {
      sl_strlcat (log_msg.msg,  N_("</trail>"), status + 2*KEY_LEN + 32); 
#else
  if (NULL != sl_strstr(log_msg.msg,  _("msg=\"EXIT\"")) &&
      NULL == sl_strstr(log_msg.msg,  _("remote_host")))
    {
      sl_strlcat (log_msg.msg,   _("</trail>"), status + 2*KEY_LEN + 32); 
#endif
      
      sl_strlcat (log_msg.msg,   _("\n"), status + 2*KEY_LEN + 32); 
      current->log_start = S_TRUE;
    }
#else
  log_msg.msg[status] = '\0';
  sl_strlcat (log_msg.msg,              "\n", status + KEY_LEN + 2);
  sl_strlcat (log_msg.msg, log_msg.signature, status + KEY_LEN + 2);
  if (log_msg.timestamp[0] != '\0')
    sl_strlcat (log_msg.msg, log_msg.timestamp, status + 2*KEY_LEN + 2);
  sl_strlcat (log_msg.msg,              "\n", status + 2*KEY_LEN + 3);
#endif
  
  /* --- Write out the record. ---
   */
  sl_write (fd, log_msg.msg, strlen(log_msg.msg));
  sl_sync  (fd);
  sl_close (fd);

  /* --- Clean up and free record. ---
   */
  memset (log_msg.msg,       '\0', status + 2*KEY_LEN + 32);
  memset (log_msg.signature, '\0', KEY_LEN);
  sh_unix_munlock (log_msg.msg,  status + 2*KEY_LEN + 32);
  SH_FREE(log_msg.msg);

  SL_RETURN (0, _("sh_log_file"));
}

