
#include "config_xor.h"


#ifdef HAVE_LIBPRELUDE
#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <sys/time.h>

#include <libprelude/list.h>
#include <libprelude/idmef-tree.h>
#include <libprelude/idmef-tree-func.h>
#include <libprelude/prelude-io.h>
#include <libprelude/prelude-message.h>
#include <libprelude/prelude-message-buffered.h>
#include <libprelude/idmef-msg-send.h>
#include <libprelude/idmef-message-id.h>
#include <libprelude/prelude-message-id.h>
#include <libprelude/sensor.h>
#include <sys/utsname.h>

/* 
 * includes for samhain-specific functions (sl_strstr, sh_error_handle)
 */
#include "slib.h"
#include "sh_cat.h"
#include "sh_error_min.h"
#include "sh_prelude.h"

/*
 * _() macros are samhain specific; they are used to replace string
 * constants at runtime. This is part of the samhain stealth mode
 * (fill string constants with encoded strings, decode at runtime).
 */
#define FIL__  _("sh_prelude.c")


/*
 * Libprelude documentation is non-existent. There is nothing
 * except 'Function X exists and takes arguments of type Y, Z, ..'.
 * Apparently the existing libprelude sensor patches are written
 * by libprelude developers themself.
 *
 * Some observations:
 * 
 * (a) libprelude makes ample use of static variables. Most ..._new()
 *     functions will just initialize an internal static variable 
 *     rather than allocating any memory. It seems that libprelude
 *     is not thread-safe (although this is irrelevant for samhain).
 *
 * (b) The X_string_set_Y() functions are macros that just set a pointer
 *     to the string provided by the caller. It is thus not safe to
 *     modify this string.
 *
 * (c) libprelude causes gross pollution of the namespace with typedefs.
 *
 * (d) Apparently sensors may supply options to libprelude, but as there is 
 *     no documentation, samhain makes no use of this.
 *
 * (e) It seems that the third argument of idmef_msg_send() is unused.
 *
 * (f) It would be nice to send data with heartbeats, and this seems in fact
 *     possible, but as it is undocumented, samhain makes no use of it.
 *
 * This file is based on the snort patch by Krzysztof Zaraska 
 * <kzaraska@student.uci.agh.edu.pl> and the systrace patch by Laurent Oudot.
 */

static char               programname[64];
static idmef_heartbeat_t  heartbeat;
static prelude_msgbuf_t * hb_msgbuf; 
static struct utsname   * uname_data = NULL;

static char               hostname[256];
			  
static char               model[64];
static char               class[64];
static char               version[8];
static char               manufacturer[64];
			  
static char               name[16];
static char               url[32];
			  
static char               meaning[64];
static char               description[128];

#if 0
static void SendHeartbeat(void *data) 
{
  struct timeval tv;
  
  gettimeofday(&tv, NULL);
  heartbeat.create_time.sec = tv.tv_sec;
  heartbeat.create_time.usec = tv.tv_usec;
  
  /*
   * we could use additional data to send stats.
   */
  prelude_msgbuf_set_header(hb_msgbuf, PRELUDE_MSG_IDMEF, 0);
  idmef_send_heartbeat(hb_msgbuf, &heartbeat);
  prelude_msgbuf_mark_end(hb_msgbuf);

  /* avoid compiler warning
   */
  if (data) return; else return;
}
#endif

static int sh_prelude_init ()
{
  int ret = -1;

  uname_data = malloc(sizeof(struct utsname));
  if (!uname_data) {
    return -1; 
  }
  ret = uname(uname_data);
  if (ret < 0) {
    uname_data = NULL;
    return -1;
  }

  /* ------- LibPrelude Init ------- 
   */
  strncpy(programname, _("Samhain"), 64);
  programname[63] = '\0';

  if ( prelude_sensor_init(programname, NULL, 0, NULL) < 0)
    {
      sh_error_handle((-1), FIL__, __LINE__, -1, MSG_E_SUBGEN,
		      _("Failed to initialize Prelude"), 
		      _("sh_prelude_init"));
      return -1;
    }

  strncpy(model, _("Samhain"), 64);
  model[63] = '\0';
  strncpy(class, _("Samhain Host Intrusion Detection System"), 64);
  class[63] = '\0';
  strncpy(version, VERSION, 8); 
  version[7] = '\0';
  strncpy(manufacturer, _("Samhain by Rainer Wichmann"), 64);
  manufacturer[63] = '\0';

  /*
  if (sh.host.name[0] != '\0') {
    strncpy(hostname, sh.host.name, 256); hostname[255] = '\0';
  } else {
    gethostname (hostname, 256); hostname[255] = '\0';
  }
  */

  gethostname (hostname, 256); hostname[255] = '\0';

  strncpy (name, _("Samhain HIDS"), 16);
  name[15] = '\0';
  strncpy (url, _("http://www.la-samhna.de/samhain/"), 32);
  url[31] = '\0';

  strncpy (meaning, _("Message generated by Samhain"), 64);
  meaning[63] = '\0';

  /* analyzer information */
  idmef_string_set (&heartbeat.analyzer.model,   model);
  idmef_string_set (&heartbeat.analyzer.class,   class);
  idmef_string_set (&heartbeat.analyzer.version, version);
  
  /* analyzer address */
  idmef_analyzer_node_new(&heartbeat.analyzer);
  idmef_string_set (&heartbeat.analyzer.node->name, hostname);
  
  /* analyzer type */
  idmef_string_set(&heartbeat.analyzer.ostype,    uname_data->sysname);
  idmef_string_set(&heartbeat.analyzer.osversion, uname_data->release);

  
  INIT_LIST_HEAD(&heartbeat.additional_data_list);
  
  hb_msgbuf = prelude_msgbuf_new(0);
  if (!hb_msgbuf) {
    return -1;
  }

  /* prelude_heartbeat_register_cb(&SendHeartbeat, NULL); */
  return 1;
}



int sh_prelude_alert (int priority, int sh_class, const char * message)
{
  static int                initialized = 0;
  struct timeval            tv;
  char                      meaning[64];

  idmef_alert_t           * alert;
  idmef_message_t         * idmef;
  prelude_msgbuf_t        * msgbuf;
  idmef_classification_t  * classification;        
  idmef_target_t          * target;
  idmef_node_t            * node;
  idmef_source_t          * source;
  idmef_assessment_t      * assessment;
  idmef_additional_data_t * data;
  
  if (initialized == 0)
    {
      /* initialize
       */
      initialized = sh_prelude_init();
    }
  if (initialized == -1)
    {
      /* init failed
       */
      sh_error_handle((-1), FIL__, __LINE__, -1, MSG_E_SUBGEN,
		      _("Problem with prelude-ids support: init failed"), 
		      _("sh_prelude_alert"));
      return -1; 
    }
  if (sh_class == STAMP)
    {
      gettimeofday(&tv, NULL);
      heartbeat.create_time.sec  = tv.tv_sec;
      heartbeat.create_time.usec = tv.tv_usec;
  
      /*
       * we could use additional data to send stats.
       */
      prelude_msgbuf_set_header(hb_msgbuf, PRELUDE_MSG_IDMEF, 0);
      idmef_send_heartbeat(hb_msgbuf, &heartbeat);
      prelude_msgbuf_mark_end(hb_msgbuf);
      return 0;
    }

  /* This function serves to initialize a message structure.
   * The returned idmef_message_t structure is a static variable
   * declared in idmef_message_new().
   */
  idmef = idmef_message_new();
  if ( ! idmef )
    goto err;
        
  /* 'alert' is a static variable that gets initialized and
   * associated with the idmef_message_t idmef_alert_t member. 
   * -> no new memory allocated; not signal-safe or thread-safe.
   */
  idmef_alert_new(idmef);
  alert = idmef->message.alert;
      
  /* Set the 'detect time'. idmef_alert_new() will already set the
   * 'create time', whatever the difference is supposed to be.
   */
  gettimeofday(&tv, NULL);
  idmef_alert_detect_time_new(alert);
  alert->detect_time->sec  = tv.tv_sec;
  alert->detect_time->usec = tv.tv_usec;
       
  /* ------- Analyzer. -------
   * 
   * This apparently is supposed to provide some information
   * about the sensor to the server (what sensor process ? where ?).
   *
   * idmef_string_set (x, y) is a macro that will make x
   * a pointer to y. Therefore the caller must guarantee that y will
   * never be overwritten until the alert is sent.
   * With the samhain _() macros, this means we must copy to another
   * storage region.
   *
   * N.B.: with constant strings, you can use idmef_string_set_constant() 
   * instead.
   */
  idmef_string_set (&alert->analyzer.model,        model);
  idmef_string_set (&alert->analyzer.class,        class);
  idmef_string_set (&alert->analyzer.version,      version);
  idmef_string_set (&alert->analyzer.manufacturer, manufacturer);

  /* Here we add some information on the host OS.
   */
  idmef_string_set (&alert->analyzer.ostype,    uname_data->sysname);
  idmef_string_set (&alert->analyzer.osversion, uname_data->release);

  /* ------- Analyzer / Process ------- 
   *
   * Fill in minimal info about the process. Apparently one could also
   * supply things like path, argv, env (?).
   */
  idmef_analyzer_process_new (&alert->analyzer);
  alert->analyzer.process->pid = getpid();

  /* ------- Analyzer / Node ------- 
   *
   * Provide the name of this node, i.e. host.
   */
  idmef_analyzer_node_new (&alert->analyzer);
  idmef_string_set (&alert->analyzer.node->name, hostname);



  /* ------- Classification -------
   *
   * Apparently 'classification' provides details about the sensor
   * program.
   *
   * For reasons unbeknown to me (did not care to investigate),
   * this function does allocate memory, instead of using a static variable. 
   *
   */
  classification = idmef_alert_classification_new(alert);
  if ( ! classification )
    goto err;


  idmef_string_set (&classification->name, name);
  idmef_string_set (&classification->url,  url);

  classification->origin = vendor_specific;
         
              

  /* ------- Target -------
   *
   * Purpose ?
   *
   * Allocates memory.
   *
   */
  target = idmef_alert_target_new(alert);
  if ( ! target )
    goto err;
  node = idmef_target_node_new(target);
  if ( ! node )
    goto err;

  /* ------- Source -------
   *
   * Purpose ?
   *
   * Allocates memory.
   *
   */
  source = idmef_alert_source_new(alert);
  if ( ! source )
    goto err;
  node = idmef_source_node_new(source);
  if ( ! node )
    goto err;

  /* ------- Impact ------- 
   */
  idmef_alert_assessment_new(alert);
  assessment = alert->assessment;
  idmef_assessment_impact_new(assessment);

  if ((priority == SH_ERR_SEVERE) || (priority == SH_ERR_FATAL))
    {
      assessment->impact->severity   = impact_high;
    }
  else if ((priority == SH_ERR_ALL) || (priority == SH_ERR_INFO) ||
      (priority == SH_ERR_NOTICE))
    {
      assessment->impact->severity   = impact_low;
    }
  else
    {
      assessment->impact->severity   = impact_medium;
    }

  if (NULL != sl_strstr(message, _("POLICY")))
    {
      if (NULL != sl_strstr(message, _("POLICY KERNEL")))
	{
	  assessment->impact->severity   = impact_high;
	  assessment->impact->completion = succeeded;
	  assessment->impact->type       = other;
	  strncpy(description, 
		  _("Kernel modification detected by Samhain."),
		  128);
	  description[127] = '\0';
	}
      else
	{
	  assessment->impact->severity   = impact_high;
	  assessment->impact->completion = succeeded;
	  assessment->impact->type       = file;
	  strncpy(description, 
		  _("File system modification detected by Samhain."),
		  128);
	  description[127] = '\0';
	}
    }
  else
    {
      if ( ((NULL != sl_strstr(message, _("Login"))) ||
	    (NULL != sl_strstr(message, _("Multiple login"))) ||
	    (NULL != sl_strstr(message, _("Logout")))) &&
	   (NULL == sl_strstr(message, _("Checking"))))
	{
	  assessment->impact->completion = succeeded;
	  assessment->impact->type       = user;
	  strncpy(description, 
		  _("Login/logout detected by Samhain."),
		  128);
	  description[127] = '\0';
	}
      else
	{
	  /* assessment->impact->severity   = impact_low; */
	  assessment->impact->completion = succeeded;
	  assessment->impact->type       = other;
	  strncpy(description, 
		  _("Message by Samhain."),
		  128);
	  description[127] = '\0';
	}
    }
  idmef_string_set (&assessment->impact->description, description); 
  idmef_assessment_confidence_new(assessment);
  assessment->confidence->rating = high;

  /* ------- Additional Data ------- 
   * 
   * Here we supply the log message.
   *
   */
  data = idmef_alert_additional_data_new(alert);
  if ( ! data )
    goto err;

  data->type = string;
  idmef_string_set (&data->meaning, meaning);
  idmef_additional_data_set_data (data, string, message, strlen(message) + 1);
  
  /* ------- Send ------- 
   *
   * Finally, the preparated message is sent.
   */      
  msgbuf = prelude_msgbuf_new(0);
  if ( ! msgbuf )
    goto err;
  
  idmef_msg_send(msgbuf, idmef, PRELUDE_MSG_PRIORITY_HIGH);

  /* Cleanup
   */
  idmef_message_free(idmef);
  prelude_msgbuf_close(msgbuf);
  
  return 0;
        
 err:
  /* Cleanup
   */
  idmef_message_free(idmef);
  sh_error_handle((-1), FIL__, __LINE__, -1, MSG_E_SUBGEN,
		  _("Problem with IDMEF for prelude-ids support: alert lost"), 
		  _("sh_prelude_alert"));
  return -1;

}

/* HAVE_LIBPRELUDE */
#endif
