#!/usr/bin/env ruby
#
# Samizdat front page
#
#   Copyright (c) 2002-2005  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'samizdat/engine'

# messages that are related to any focus (and are not comments or old
# versions), ordered chronologically by date of relation to a focus (so that
# when message is edited, it doesn't flow up)
#
# in case of multiple related focuses, the one related most recently is used
#
def collect_features(skip, showhidden=false, limit=config['limit']['features'])
  hidden = " AND ?hidden = 'false'" unless showhidden
  list = cache.fetch_or_add(%{features/#{skip}/#{showhidden}/#{limit}}) do
    rdf.select_all( %{
SELECT ?msg
WHERE (rdf::predicate ?stmt dc::relation)
      (rdf::subject ?stmt ?msg)
      (rdf::object ?stmt ?focus)
      (dc::date ?stmt ?date)
      (s::inReplyTo ?msg ?parent)
      (dct::isVersionOf ?msg ?current)
      (s::rating ?stmt ?rating)
      (s::hidden ?msg ?hidden)
LITERAL ?rating >= :threshold AND
        ?parent IS NULL AND ?current IS NULL #{hidden}
GROUP BY ?msg
ORDER BY max(?date) DESC}, limit, limit * skip,
      { :threshold => config['limit']['features_threshold'] }
    ).collect {|m,| m }   # unwrap DBI::Row
  end
  list.collect {|msg| yield msg }
end

# messages that are not comments or older version, sorted chronologically
#
def collect_updates(skip, showhidden=false)
  hidden = " AND ?hidden = 'false'" unless showhidden
  list = cache.fetch_or_add(%{updates/#{skip}/#{showhidden}}) do
    rdf.select_all( %{
SELECT ?msg
WHERE (dc::date ?msg ?date)
      (s::inReplyTo ?msg ?parent)
      (dct::isVersionOf ?msg ?current)
      (s::hidden ?msg ?hidden)
LITERAL ?parent IS NULL AND ?current IS NULL #{hidden}
ORDER BY ?date DESC}, limit_page, limit_page * skip
    ).collect {|m,| m }   # unwrap DBI::Row
  end
  list.collect {|msg| yield msg }
end

# do Apache's job and include files in static headers
#
def gsub_ssi_file(static)
  static.gsub(/<!--#include\s+file="([^"]+)"\s*-->/i) do
    name = $1
    name = RequestSingleton.instance.env['DOCUMENT_ROOT'] + uri_prefix + '/' + name unless '/' == name[0]
    File.open(name.untaint) {|f| f.read }   # config files are always trusted
  end
end

# read from config and translate static header or footer
#
def static_text(session, name)
  text = config['site'][name]
  text = (
    text[session.language] or
    text[config['locale']['languages'][0]] or
    text.to_a[0][1]
  ) if text.kind_of? Hash
  gsub_ssi_file(text)
end

# RSS feed of features or updates
#
def feed_rss(session)
  session.options.update('type' => 'application/xml')
  feed = session['feed']
  feed = 'features' unless 'updates' == feed
  feed_title = ('features' == feed)? _('Features') : _('Recent Updates')

  require 'rss/maker'
  RSS::Maker.make("1.0") do |maker|
    maker.channel.about = session.base + "?format=rss&amp;feed=#{feed}"
    maker.channel.title = config['site']['name'] + ':' + feed_title
    maker.channel.description = static_text(session, 'header')
    maker.channel.link = session.base

    if config['site']['icon']
      maker.image.title = config['site']['name']
      maker.image.url = config['site']['icon']
    end

    if 'features' == feed
      collect_features(0, false, limit_page) do |msg|
        Message.cached(msg).rss(maker, session)
      end
    else
      collect_updates(0) do |msg|
        Message.cached(msg).rss(maker, session)
      end
    end
  end
end

# wrap _page_ in Template#page
#
def front_page(template, page, options={})
  template.page(config['site']['name'], page, options.merge({:front_page => true}))
end

request do |session|
  next feed_rss(session).to_s if 'rss' == session['format']

  t = session.template
  skip, skip_feature = session.params %w[skip skip_feature]
  skip = skip.to_i
  skip_feature = skip_feature.to_i

  # provide rss links
  if 0 == skip and 0 == skip_feature
    rss_features = '?format=rss&amp;feed=features'
    rss_updates = '?format=rss&amp;feed=updates'
    rss = { _('Features') => rss_features, _('Recent Updates') => rss_updates }
  end

  # try to run from cache
  cache_key = 'index/' + uri_prefix + '/' + session.accept_language.join(':') +
    %{/#{skip}/#{skip_feature}/#{'yes' != session.cookie('nostatic')}/#{session.showhidden?}}
  page = cache[cache_key] and next front_page(t, page, {:rss => rss})

  full_front_page = (0 == skip and 0 == skip_feature)
  render_features = (full_front_page or 0 == skip)
  render_updates = (full_front_page or 0 == skip_feature)

  if full_front_page   # render focuses
    focuses_title = _('Top Focuses')
    focuses =
      Focus.collect_focuses {|focus, usage|
        %{<div class="focus">} + t.resource_href(focus,
          Resource.new(session, focus).render(:title) +
            %{ (#{usage})}) + %{</div>\n}
      }.join
  end

  if render_features
    features_title = _('Features') +
      (skip_feature > 0 ? sprintf(_(', page %s'), skip_feature + 1) : '')
    features = collect_features(skip_feature, session.showhidden?) {|msg|
      Resource.new(session, msg).render(:short) }
    features = features.join + %{<div class="foot">} + t.nav_rss(rss_features) +
      t.nav(features.size < config['limit']['features'],
      skip_feature + 1, 'index.rb?', 'skip_feature') + "</div>\n"
  end

  if render_updates
    updates_title = _('Recent Updates') +
      (skip > 0? sprintf(_(', page %s'), skip + 1) : '')
    updates = collect_updates(skip, session.showhidden?) {|msg|
      Resource.new(session, msg).render }
    updates = t.list(updates,
      t.nav_rss(rss_updates) + t.nav(updates.size, skip + 1))
  end

  page =
    if full_front_page
%{<table>
  <thead>
    <tr><th>#{focuses_title}</th><th>#{features_title}</th><th>#{updates_title}</th></tr>
  </thead>
  <tr>
    <td class="focuses">#{focuses}</td>
    <td class="features" rowspan="3">#{features}</td>
    <td class="updates" rowspan="3">#{updates}</td>
  </tr>
  <tr><td class="links-head">}+_('Links')+'</td></tr>
  <tr><td class="links">
    <div class="focus"><a href="query.rb?run&amp;query='+CGI.escape('SELECT ?resource WHERE (dc::date ?resource ?date) (s::inReplyTo ?resource ?parent) LITERAL ?parent IS NOT NULL ORDER BY ?date DESC')+'">'+_('All Replies')+'</a></div>
    <div class="focus"><a href="foci.rb">'+_('All Focuses (verbose)')+'</a></div>
    <div class="focus"><a href="moderation.rb">'+_('Moderation Log')+'</a></div>
  </td></tr>
</table>'
    elsif render_features
      t.box(features_title, features)
    elsif render_updates
      t.box(updates_title, updates)
    end

  # static header and footer
  if full_front_page and 'yes' != session.cookie('nostatic')
    header = static_text(session, 'header')
    footer = static_text(session, 'footer')
    page = %{<div id="front-header">#{header}</div>\n} + page if
      header and header =~ /[^\s]/
    page += %{<div id="front-footer">#{footer}</div>\n} if
      footer and footer =~ /[^\s]/
  end

  # update cache
  cache[cache_key] = page

  front_page(t, page, {:rss => rss})
end
