#!/usr/bin/env ruby
#
# Samizdat resource display
#
#   Copyright (c) 2002-2005  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'samizdat/engine'

# messages that are related to this focus
#
def collect_related(id, skip, showhidden=false)
  hidden = " AND ?hidden = 'false'" unless showhidden
  list = cache.fetch_or_add(%{#{id}/related/#{skip}/#{showhidden}}) do
    rdf.select_all( %{
SELECT ?msg
WHERE (rdf::subject ?stmt ?msg)
      (rdf::predicate ?stmt dc::relation)
      (rdf::object ?stmt #{id})
      (s::inReplyTo ?msg ?parent)
      (dct::isVersionOf ?msg ?current)
      (s::rating ?stmt ?rating)
      (dc::date ?msg ?date)
      (s::hidden ?msg ?hidden)
LITERAL ?rating > 0 AND ?parent IS NULL AND ?current IS NULL #{hidden}
ORDER BY ?date DESC}, limit_page, limit_page * skip
    ).collect {|m,| m }   # unwrap DBI::Row
  end
  list.collect {|msg| yield msg }
end

# RSS feed of related resource
#
def feed_rss(session, id)
  session.options.update('type' => 'application/xml')

  require 'rss/maker'
  RSS::Maker.make("1.0") do |maker|
    maker.channel.about = session.base + %{resource.rb?id=#{id}&amp;format=rss}
    title = config['site']['name'] + ' / ' +
      Focus.focus_title(Resource.new(session, id).render(:title))
    maker.channel.title = title
    maker.channel.description = title
    maker.channel.link = session.base + id.to_s

    if config['site']['icon']
      maker.image.title = title
      maker.image.url = config['site']['icon']
    end

    collect_related(id, 0) do |msg|
      Message.cached(msg).rss(maker, session)
    end
  end
end

request do |session|

  # RewriteRule ^(/samizdat-base)/([0-9]+)$ $1/resource.rb?id=$2 [PT]
  #
  id, = session.params %w[id]
  id.nil? and raise ResourceNotFoundError,
    sprintf(_("Incorrect resource id '%s'"), id)

  next feed_rss(session, id).to_s if 'rss' == session['format']

  # render resource
  t = session.template
  resource = Resource.new(session, id)
  r = resource.render(session.has_key?('focus')? :short : :full)
  title = r[:head]
  title += ' (' + _(r[:type]) + ')' if 'Message' != r[:type] and
    [ _('Uriref'), _('Literal'), _('Member'), _('Statement'), _('Vote'),
      _('Item')   # rgettext hack
    ].include?(_(r[:type]))
  body = session.has_key?('skip')? r[:messages] : r.to_s

  # vote on focus rating
  if session.has_key?('focus') or session.has_key?('focus_id')
    session.access('vote') or raise AuthError,
      sprintf(_('Your current access level (%s) does not allow to vote'),
        _(session.role))

    focus, focus_id, rating = session.params %w[focus focus_id rating]
    focus = nil if 'select' == focus
    if focus_id = Resource.validate_id(focus_id)
      # manual entry overrides selection
      focus = focus_id
    end
    focus = Focus.new(session, focus, resource) if focus

    if focus.kind_of?(Focus) and rating   # commit vote
      # rating is validated by Focus#rating=
      db.transaction {|db| focus.rating = rating }
      session.redirect(id.to_s)
    else   # display vote form
      vote_form = t.form( 'resource.rb',
        *t.focus_fields(id, focus) +
        [ [:br], [:submit, nil, _('Submit')] ]
      )
      next t.page(_('Vote') + ': ' + r[:head],
        [ [_('Vote'), vote_form], [title, body] ])
    end
  end

  page = []

  # related resources
  if not session.has_key?('skip')
    skip, = session.params %w[skip_related]
    skip = skip.to_i
    resources = collect_related(id, skip, session.showhidden?) {|msg,|
      Resource.new(session, msg).render(:short)[:desc]
    }
    if resources.size > 0
      rss_link = %{resource.rb?id=#{id}&amp;format=rss}
      rss = { Focus.focus_title(r[:head]) => rss_link }
      page = [ [ Focus.focus_title(r[:head]) +
          (skip > 0? sprintf(_(', page %s'), skip + 1) : ''),
        t.list( resources,
          t.nav_rss(rss_link) +
          t.nav(resources.size, skip + 1,
            "resource.rb?id=#{id}&amp;", 'skip_related') )
      ] ]
    end
  end

  t.page(r[:head], page << [ title, body ], {:rss => rss})
end
