# Samizdat HTML helpers for messages
#
#   Copyright (c) 2002-2009  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 3 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'samizdat/helpers/application_helper'

module MessageHelper
  include ApplicationHelper

  # render link to full message
  #
  def full_message_href(id)
    %{<p><a href="#{id}">}+_('See the full message')+'</a></p>'
  end

  # list supplied focuses in a straight line
  #
  def focus_line(id, focuses)
    %{<a href="#{id}#focuses">}+_('related to')+'</a>: ' +
    focuses.sort_by {|f| -f.sort_index }.collect {|focus|
      resource_href(focus.id, focus.name)
    }.join(', ') if focuses.size > 0
  end

  # inline formats that need a link to view source
  #
  SOURCE_FORMAT = {
    'text/textile' => 'textile',
    'text/html' => 'html'
  }

  # render _message_ info line (except focuses)
  #
  def message_info(message, mode)
    creator = message.creator.id.nil? ?   # no link if published by guest
      _('guest') :
      %{<a href="#{message.creator.id}">#{escape_title(message.creator.full_name)}</a>}
    date = format_date(message.date)

    if :full == mode
      parent = %{<a href="#{message.parent}">} +
        _('parent message') + '</a>' if message.parent
      version_of = %{<a href="#{message.version_of}">} + _('current version') +
        '</a>' if message.version_of
      history = %{<a href="history/#{message.id}">} + _('history') +
        '</a>' if message.nversions.to_i > 0
      if format = SOURCE_FORMAT[message.content.format]
        source = %{<a href="message/#{message.id}/source" title="} +
          _('view source') + %{">#{format}</a>}
      end
    end

    replies = %{<a href="#{message.id}#replies">} + _('replies') +
      '</a>:&nbsp;' + message.nreplies.to_s if message.nreplies.to_i > 0

    translations = _('translation') + ': ' +
      (message.translations + [[message.lang, message.id, 3]]).sort_by {|l,m,r|
        -r.to_f
      }.collect {|l,m,r|
        %{<a href="#{m}">#{l}</a>}
      }.join(' ') if message.translations.to_a.size > 0

    focuses = focus_line(message.id, message.focuses.collect {|f|
      Focus.new(@request, f, message.id)
    })

    hidden = _('hidden') if message.hidden?

    moderation_log = %{<a href="moderation/#{message.id}">} + _('moderation log') + '</a>' if
      message.id.kind_of?(Integer) and
      Moderation.find(message.id, { :message => true }).size > 0

    [ sprintf(_('by&nbsp;%s on&nbsp;%s'), creator, date.to_s),
      hidden, moderation_log, parent, version_of, history, source, replies, translations, focuses
    ].compact.join(",\n ")
  end

  # render _message_ content, _mode_ can be :short or :full
  #
  def message_content(message, mode)
    return '' if :list == mode or (message.desc and :short == mode)

    content = message.content

    if content.cacheable?
      if :full == mode
        content.html_full
      else
        short = content.html_short
        if short
          content.html_short + full_message_href(message.id)
        else
          content.html_full
        end
      end
    else
      content.render(@request, mode)
    end
  end

  def message_button(id, action, label, css=:action)
    %{<a class="#{css}" href="message/#{id}/#{action}">#{label}</a>\n}
  end

  # render buttons for _message_ in :page mode
  #
  def message_buttons(message)
    buttons = ''

    if not message.version_of   # no buttons for old versions
      if @member.allowed_to?('post')
        unless message.is_translation?
          # don't allow to post replies to translations
          # todo: don't allow to post replies to focuses
          buttons << message_button(message.id, 'reply', _('Reply'))
        end

        open =
          if @session.member.nil?   # guest
            false
          elsif message.creator.id == @session.member   # creator
            true
          else   # everyone else
            message.open
          end
        if open
          buttons << message_button(message.id, 'edit', _('Edit'))
        end
      end

      if @request.moderate?
        buttons <<
          message_button(
            message.id,
            message.hidden? ? 'unhide' : 'hide',
            message.hidden? ? _('UNHIDE') :  _('HIDE'),
            :moderator_action
          ) <<
          message_button(message.id, 'reparent', _('REPARENT'), :moderator_action) <<
          message_button(message.id, 'takeover', _('TAKE OVER'), :moderator_action)
      end
    end

    request_status = Moderation.request_status(message.id)

    if @request.moderate?
      # old versions can still be replaced
      buttons << message_button(message.id, 'replace', _('REPLACE'), :moderator_action)

      if :requested == request_status
        buttons << message_button(message.id, 'acknowledge', _('ACKNOWLEDGE'), :moderator_action)
      end

    elsif @member.allowed_to?('post') and :none == request_status
      # moderation can be requested for any message version
      buttons << message_button(message.id, 'request_moderation', _('Request Moderation'))
    end

    ('' == buttons) ? '' : %{<div class="foot">#{buttons}</div>\n}
  end

  # render full message
  #
  # _message_ is an instance of Message class
  #
  # _mode_ can be :short or :full
  #
  def message(message, mode)
    info = message_info(message, mode)

    translation = (:full == mode) ?
      message :
      message.select_translation(@request.accept_language)

    if :short == mode   # title is already in the page head in :full mode
      title = translation.content.title
      title = Focus.focus_title(title) if message.nrelated > 0
      title = CGI.escapeHTML(limit_string(title))
      title = %{<div class="title">#{resource_href(message.id, title)}</div>\n}
    end

    if translation.desc and translation.desc != translation.id
      # use description of translation when applicable
      desc = message_content(Message.cached(translation.desc), :short).to_s
    end
    if desc and :short == mode
      content = desc + full_message_href(message.id)
    else
      content = ''
      if desc   # prepend content with description box
        content << box(resource_href(message.desc, _('Description')), desc, 'desc')
      end
      content << message_content(translation, mode).to_s
    end

%{<div class="message" id="id#{message.id}">
#{title}<div class="info">#{info}</div>
<div class="content">#{ hide_message(content, message.hidden?) }</div>
</div>\n}
  end

  # wrap message in a hidden-message div if it is hidden
  #
  def hide_message(message, hidden=false)
    if hidden
      message.gsub!(/<(a|img)\s+([^>]+?\s+?)?(href|src)([^>]*)>/i,
        '<\1 \2title\4>')   # cripple URLs and images
      %{<div class="hidden-message">\n#{message}</div>\n}
    else
      message
    end
  end
end
