 /* sane - Scanner Access Now Easy.
   Copyright (C) 2003 Johannes Hub (JohannesHub@foni.net)

   This file was initially copied from the hp3300 backend.
   This file is part of the SANE package.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

   As a special exception, the authors of SANE give permission for
   additional uses of the libraries contained in this release of SANE.

   The exception is that, if you link a SANE library with other files
   to produce an exutable, this does not by itself cause the
   resulting executable to be covered by the GNU General Public
   License.  Your use of that executable is in no way restricted on
   account of linking the SANE library code into it.

   This exception does not, however, invalidate any other reasons why
   the executable file might be covered by the GNU General Public
   License.

   If you submit changes to SANE to the maintainers to be included in
   a subsequent release, you agree by submitting the changes that
   those changes may be distributed with this exception intact.

   If you write modifications of your own for SANE, it is your choice
   whether to permit this exception to apply to your modifications.
   If you do not wish that, delete this exception notice.
*/

/*
    Concept for a backend for scanners based on the RT8891 chipset,
    such as HP4400C, HP4470C.
    Parts of this source were inspired by other backends.

    History:

    Version 0.17g 18.04.04 10.alpha, little fixes
    Version 0.17d 05.04.04 10.alpha, little fixes
    Version 0.17b 30.03.04 10.alpha, little fixes and libusb implemented
    Version 0.17a 22.03.04 10.alpha, little fixes
    Version 0.13 13/11/22 4. alpha
    Version 0.12 13/10/03 third alpha, Backend name changed to HP_RTS
    Version 0.11 30/08/03 second alpha
    Version 0.10 19/07/03 first alpha
*/

/*
    Provides a simple interface to read and write data from USB scanner
    over the scanner module.
*/


#ifdef LINUX_USB_SUPPORT


#include <unistd.h>    /* read, open, write */
#include <fcntl.h>     /* open */
#include <stdio.h>     /* printf */
#include <stdlib.h>
#include <errno.h>     /* better error reports */
#include <sys/ioctl.h>

#include "hp_rts_xfer.h"

/* From /usr/src/linux/driver/usb/scanner.h for Linux >= v2.4.13 */
#define SCANNER_IOCTL_VENDOR _IOR('U', 0x20, int)
#define SCANNER_IOCTL_PRODUCT _IOR('U', 0x21, int)
#define SCANNER_IOCTL_CTRLMSG _IOWR('U', 0x22, devrequest)
/* Older (unofficial) IOCTL numbers for Linux < v2.4.13 */
#define SCANNER_IOCTL_VENDOR_OLD _IOR('u', 0xa0, int)
#define SCANNER_IOCTL_PRODUCT_OLD _IOR('u', 0xa1, int)


/************************************************************************
  USB transfer routines implemented by using /dev/usb/scanner
************************************************************************/
SANE_Int send_cmd( SANE_Int, SANE_Byte *, SANE_Int);
SANE_Int read_cmd( SANE_Int, SANE_Byte *, SANE_Int);
SANE_Int read_usb( SANE_Int device, SANE_Byte *data,SANE_Int size);
SANE_Int write_usb(SANE_Int device, SANE_Byte *data,SANE_Int size);

/****************************************************************************/
static void
kernel_get_vendor_product (SANE_Int fd, SANE_Int *vendorID, SANE_Int *productID)
/****************************************************************************/
{
  /* read the vendor and product IDs via the IOCTLs */
  if (ioctl (fd, SCANNER_IOCTL_VENDOR, vendorID) == -1)
    {
      if (ioctl (fd, SCANNER_IOCTL_VENDOR_OLD, vendorID) == -1)
        DBG(DBG_ERR, "kernel_get_vendor_product: ioctl (vendor) "
                "of device %d failed: %s\n", fd, strerror (errno));
    }
  if (ioctl (fd, SCANNER_IOCTL_PRODUCT, productID) == -1)
    {
      if (ioctl (fd, SCANNER_IOCTL_PRODUCT_OLD, productID) == -1)
         DBG(DBG_ERR,"sanei_usb_get_vendor_product: ioctl (product) "
                "of device %d failed: %s\n", fd, strerror (errno));
    }
}

/****************************************************************************/
SANE_Int
send_cmd(SANE_Int fp, SANE_Byte * cmd, SANE_Int length) {
/****************************************************************************/
  SANE_Int result;

  if((result = write(fp, cmd, length)) != length) {
     DBG(DBG_USB,"Write warning: %d bytes requested, %d written\n",length,result);
  } else if (result < 0) {
     DBG(DBG_ERR,"send_cmd: failure\n");
     exit (1);
  }
  return (result);
}

/****************************************************************************/
SANE_Int
read_cmd(SANE_Int fp, SANE_Byte * response, SANE_Int length) {
/****************************************************************************/
  return read(fp, response, length);
}

/****************************************************************************/
SANE_Int
read_usb( SANE_Int device,SANE_Byte *data,SANE_Int size)
/****************************************************************************/
{
  SANE_Int read;
#ifdef USBDEBUG
  SANE_Byte *tmp=data;
  SANE_Int tmp_size=size;
  fprintf(stderr,"transfer type=bulk size=%d dir=IN\n",size);
#endif
  read = read_cmd(device, data, size);
  if ( read !=size ){
        DBG(DBG_ERR, "problem doing read; only %d read from %d\n", (SANE_Int)read, (SANE_Int)size );
        return(-1);
      }
  else
  {
#ifdef USBDEBUG
    while(tmp_size--) fprintf(stderr,"%02hhx ",*tmp++);
    fprintf(stderr,"\n");
#endif
    return(0);
  }
}

/****************************************************************************/
SANE_Int
write_usb(SANE_Int device,SANE_Byte *data,SANE_Int size)
/****************************************************************************/
{
    SANE_Int write;
#ifdef USBDEBUG
    SANE_Byte *tmp=data;
    SANE_Int tmp_size=size;
    fprintf(stderr,"transfer type=bulk size=%d dir=OUT\n",size);
    while(tmp_size--) fprintf(stderr,"%02hhx ",*tmp++);
    fprintf(stderr,"\n");
#endif
    write = send_cmd(device, data, size);
    if ( write != size ) {
      DBG(DBG_ERR,"problem doing write\n");
      return(-1);
    } else
      return(write);
}

/****************************************************************************/
static SANE_Int
LnxUsbInit(TFnReportDevice *pfnReportDevice)
/****************************************************************************/
{
  SANE_Int iVendor, iProduct, i;
  SANE_Int fd;
  TScannerModel *pModel;
  SANE_Char    szDeviceName[32];

  /* open scanner's device file */

  for (i = 0; i < 16; i++) {
    sprintf(szDeviceName, "/dev/usb/scanner%d", i);
    DBG(DBG_USB,"LnxUsbInit: Investigate %s\n", szDeviceName);

    fd = open(szDeviceName, O_RDWR);
    if (fd == -1) {
      continue;
    }
    iVendor = -1;
    iProduct = -1;
    kernel_get_vendor_product (fd, &iVendor, &iProduct);
    close(fd);
    DBG(DBG_USB,"VendorId 0x%04X, ProductId 0x%04X\n", iVendor, iProduct);
    if (Hp_rts_MatchUsbDevice(iVendor, iProduct, &pModel)) {
      /* report device back */
      pfnReportDevice(pModel, szDeviceName);
      return(SANE_TRUE);
    }
  }
  return(SANE_FALSE);
}


/****************************************************************************/
static SANE_Int
LnxUsbOpen(SANE_Char *pszDeviceName)
/****************************************************************************/
{
  SANE_Int fd;

  DBG(DBG_USB,"LnxUsbOpen: Opening %s ", pszDeviceName);

  if((fd = open(pszDeviceName, O_RDWR)) < 0) {
    DBG(DBG_USB,"Unable to open scanner device\n");
    return(-1);
  }
  return fd;
}


/****************************************************************************/
static void
LnxUsbExit(SANE_Int fd)
/****************************************************************************/
{
  /* close usb device */
  if (fd != -1) {
    close(fd);
  }
}


/****************************************************************************/
static SANE_Int
LnxUsbWriteBulk(SANE_Int fd, SANE_Byte reg, SANE_Byte *pabData,
                             SANE_Int iSize,SANE_Int wHeader)
/****************************************************************************/
{
  SANE_Byte init[]={0x88,0x00,0x00,0x01};
  SANE_Byte *request, *tst;
  SANE_Int returncode;

  returncode = -1;
  if (fd < 0) {
    return(returncode);
  };
  if (wHeader){
    init[1] = reg;
    init[2] = (iSize >> 8) & 0xFF; /*split count over these bytes.*/
    init[3] = (iSize) & 0xFF;

    request=malloc(iSize + 4);     /* Compose a request string. */
    memcpy(request,init,4);        /* stick them together */
    memcpy(request+4,pabData,iSize);

    tst=request;
    if (write_usb( fd, (SANE_Byte *)request, iSize+4 )!=0){
      returncode = 0;}
    free(request);                  /* clean up */
  }else
  {
    if (!wHeader) returncode = 0;
    if ( returncode == 0 ){
      if ((write_usb( fd, (SANE_Byte *)pabData, iSize))==0){
      returncode = -1;}
    }
  }
  return(returncode);
}


/****************************************************************************/
static SANE_Int
LnxUsbReadBulk(SANE_Int fd, SANE_Byte reg, SANE_Byte *pabData,
                            SANE_Int iSize,SANE_Int wWrite)
/****************************************************************************/
{ /* Reads count registers into pabData. */
  SANE_Byte init[]={0x80,0x00,0x00,0x01};
  SANE_Int returncode;

  returncode = -1;

  if (fd < 0) {
    return(returncode);
  }
  if (wWrite){
    init[1] = reg;
    init[2] = (iSize >> 8) & 0xFF; /* split count over these bytes. */
    init[3] = (iSize) & 0xFF;
    if (write_usb( fd, (SANE_Byte *)init, 4 )!=0)
      returncode = 0;
    else
      return -1;
  }
#if 0
  printf("have to read %d\n",iSize);
#endif
  if (read_usb( fd, (SANE_Byte *)pabData, iSize)==0)
    returncode = -1;
  return(0);
}


/****************************************************************************/
static SANE_Int
LnxUsbReadReg(SANE_Int fd, SANE_Byte bReg, SANE_Byte *pabData)
/****************************************************************************/
{
  return(LnxUsbReadBulk(fd, bReg, pabData, 1,SANE_TRUE));
}

/****************************************************************************/
static SANE_Int
LnxUsbWriteReg(SANE_Int fd, SANE_Byte bReg, SANE_Byte bData)
/****************************************************************************/
{
  return(LnxUsbWriteBulk(fd, bReg, &bData, 1, SANE_TRUE));
}

/****************************************************************************/
static SANE_Int
LibUsbReadBulk_all_regs(SANE_Int iHandle, SANE_Byte *pabData)
/****************************************************************************/
{
  SANE_Byte init[]={0x80,0,0x00,0xf4};
  if (iHandle < 0) {
    return(SANE_FALSE);
  };
  if     (LnxUsbWriteBulk(iHandle, 0, init, 4, SANE_FALSE)==0){
    if   (LnxUsbReadBulk(iHandle, 0, pabData, 192,SANE_FALSE)==0)
      if (LnxUsbReadBulk(iHandle, 0, pabData+192, 52,SANE_FALSE)==0)
         return(0);
   }
 return(-1);
}

/****************************************************************************/
XferModule LnxUsbDev = {
/****************************************************************************/
  "Linux scanner kernel module",
  LnxUsbInit,
  LnxUsbOpen,
  LnxUsbExit,
  LnxUsbWriteReg,
  LnxUsbReadReg,
  LnxUsbWriteBulk,
  LnxUsbReadBulk,
  LibUsbReadBulk_all_regs
};

#endif /* LINUX_USB_SUPPORT */
