 /* sane - Scanner Access Now Easy.
   Copyright (C) 2003 Johannes Hub (JohannesHub@foni.net)

   This file was initially copied from the hp3300 backend.
   This file is part of the SANE package.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

   As a special exception, the authors of SANE give permission for
   additional uses of the libraries contained in this release of SANE.

   The exception is that, if you link a SANE library with other files
   to produce an exutable, this does not by itself cause the
   resulting executable to be covered by the GNU General Public
   License.  Your use of that executable is in no way restricted on
   account of linking the SANE library code into it.

   This exception does not, however, invalidate any other reasons why
   the executable file might be covered by the GNU General Public
   License.

   If you submit changes to SANE to the maintainers to be included in
   a subsequent release, you agree by submitting the changes that
   those changes may be distributed with this exception intact.

   If you write modifications of your own for SANE, it is your choice
   whether to permit this exception to apply to your modifications.
   If you do not wish that, delete this exception notice.
*/

/*
    Concept for a backend for scanners based on the RTS88xx chipset,
    such as HP4400C, HP4470C.
    Parts of this source were inspired by other backends.

    History:

    Version 0.17g 18.04.04 10.alpha, little fixes
    Version 0.17d 05.04.04 10.alpha, little fixes
    Version 0.17c 31.03.04 10.alpha, little fixes
    Version 0.17b 30.03.04 10.alpha, little fixes and libusb implemented
    Version 0.17a 22.03.04 10.alpha, little fixes
    Version 0.17  09.03.04 9. alpha, HP3500 included
    Version 0.16  06.02.04 8. alpha, wait counting on LCD
    Version 0.15a 29.01.04 7. alpha, CCD switch moved to config file
    Version 0.15  11.01.04 6. alpha, a second CCD implemented
    Version 0.14  21/11/22 5. alpha
    Version 0.13a 21.11.04 4. alpha, an little fix included
    Version 0.13  13.11.04 4. alpha
    Version 0.12  22.10.03 third alpha, Backend name changed to HP_RTS88xx
    Version 0.11  30.08.03 second alpha
    Version 0.10  19.07.03 first alpha
*/

/*
    Provides a simple interface to read and write data from the scanner,
    without any knowledge whether it's a parallel or USB scanner

    enable DEDUG output:
    use a nomal shell (konsole) and type
    export SANE_DEBUG_HP_RTS88XX=32
    export XSANE_DEBUG=100
    xsane
*/

#include <unistd.h>    /* read, open, write */
#include <fcntl.h>     /* open */
#include <stdio.h>     /* printf */
#include <errno.h>     /* better error reports */
#include <string.h>    /* better error reports */
#include <stdlib.h>    /* malloc(), free() on FreeBSD */

#include "hp_rts_xfer.h"

#define MAX_XFER_MODULES 1 /* we have only one XCVR module on time */

static XferModule *_aXferModules[MAX_XFER_MODULES];
static SANE_Int _iNrXferModules;
static SANE_Int _fDeviceFound;
static SANE_Char _szDeviceName[64];
static TScannerModel *_pModel;


/* list of supported models */
TScannerModel ScannerModels[] = {

  {"Hewlett-Packard", "UnknownModel",   0x3F0, 0x000, eUnknownModel},
  {"Hewlett-Packard", "ScanJet 3500C",  0x3F0, 0x2205, eHp3500c},
#ifdef DEBUG_HP3500
  {"Hewlett-Packard", "ScanJet 3530C",  0x3F0, 0x805, eHp3530c},
#else
  {"Hewlett-Packard", "ScanJet 3530C",  0x3F0, 0x2005, eHp3530c},
#endif
/*{"Hewlett-Packard", "ScanJet 3570C",  0x3F0, 0x2005, eHp3570c}, is the same as 3500*/
  {"Hewlett-Packard", "ScanJet 4400C",  0x3F0, 0x705, eHp4400c},
  {"Hewlett-Packard", "ScanJet 4470C",  0x3F0, 0x805, eHp4470c},
/* last entry all zeros */
  {0, 0, 0, 0, 0}
};


/*#ifdef DEBUG*/
/****************************************************************************/
/* utility function to show a hexdump of a buffer */
void DumpHex(SANE_Byte *pabData, SANE_Int iLen, SANE_Int iWidth, SANE_Int withLf)
/****************************************************************************/
{
  SANE_Int i;

  printf("    ");
  for (i = 0; i < iWidth ; i++) {
    printf(" %02X", i);
  }
  for (i = 0; i < iLen; i++) {
    if ((i % iWidth) == 0) {
      if (withLf) printf("\n");
      printf("%04X", i);
    }
    printf(" %02X", pabData[i]);
  }
  if (withLf) printf("\n");
}
/*#endif*/

/****************************************************************************
  Hp_rts_MatchUsbDevice
  ==============
    Matches a given USB vendor and product id against a list of
    supported scanners.

  IN  iVendor   USB vendor ID
      iProduct  USB product ID
  OUT *ppModel  Pointer to TScannerModel structure

  Returns SANE_TRUE if a matching USB scanner was found
*/
SANE_Bool Hp_rts_MatchUsbDevice(SANE_Int iVendor, SANE_Int iProduct, TScannerModel **ppModel)
/****************************************************************************/
{
  SANE_Int model;
  TScannerModel *pModels = ScannerModels;

  model = 0;
  DBG(DBG_MSG,"Matching USB device 0x%04X-0x%04X ... ", iVendor, iProduct);
  while (pModels->pszName != NULL) {
    if ((pModels->iVendor == iVendor) && (pModels->iProduct == iProduct)) {
      DBG(DBG_MSG,"found %s %s\n", pModels->pszVendor, pModels->pszName);
      *ppModel = pModels;
      return SANE_TRUE;
    }
    /* next model to match */
    pModels++;
    model++;
  }
  DBG(DBG_MSG, "nothing found\n");
  return SANE_FALSE;
}



/************************************************************************
  Public functions
************************************************************************/


/****************************************************************************
  Hp_rts_XferRegisterModule
  =========================
    installs a transfer method
  IN    pXferModule   Module to registers
  Returns a negative number when an error occurred. */
SANE_Int Hp_rts_XferRegisterModule(XferModule *pXferModule)
/****************************************************************************/
{
  DBG(DBG_MSG, "Registering transfer method %d: %s\n", _iNrXferModules, pXferModule->pszName);

  if (_iNrXferModules >= MAX_XFER_MODULES) {
    DBG(DBG_ERR, "Max. number of installed transfer methods reached\n");
    return -1;
  }
  _aXferModules[_iNrXferModules++] = pXferModule;

  return 0;
}


/****************************************************************************
  _ReportDevice
  =============
    Is called by a transfer module to report a device that can be used to
    access a compatible scanner.

  Returns a value > 0 on success.
*/
static SANE_Int _ReportDevice(TScannerModel *pModel, SANE_Char *pszName)
/****************************************************************************/
{
  DBG(DBG_MSG, "_ReportDevice '%s'\n", pszName);

  if (_fDeviceFound) {
    DBG(DBG_MSG, "Skipping '%s'\n", pszName);
    return -1;
  }

  _pModel = pModel;
  strcpy(_szDeviceName, pszName);

  _fDeviceFound = SANE_TRUE;
  return 1;
}


/****************************************************************************
  Hp_rts_XferInit
  ===============
    Probes all registered modules to see if it can be used to access a
    compatible scanner. The first module that supports the scanner is
    used.

  Returns the handle to a TXferDev structure which was used to find the
  scanner, returns 0 or negative otherwise.
*/
SANE_Int Hp_rts_XferInit(EScannerModel *peModel)
/****************************************************************************/
{
  TXferDev *pDev;
  XferModule *pMod;
  SANE_Int i, iHandle;

  for (i = 0; i < _iNrXferModules; i++) {
    pMod = _aXferModules[i];

    DBG(DBG_MSG, "Probing transfer method %d...\n", i);

    _fDeviceFound = SANE_FALSE;
    iHandle = pMod->mfnInit(_ReportDevice);
    if (_fDeviceFound) {
      /* try to open the device */
      iHandle = pMod->mfnOpen(_szDeviceName);
      if (iHandle >= 0) {
        pDev = malloc(sizeof(TXferDev));

        *peModel = _pModel->eModel;
        pDev->iXferDev = pMod;
        pDev->iHandle = iHandle;
        return (int)pDev;
    }
  }
 }

  DBG(DBG_ERR, "No useable transfer method found\n");
  return -1;
}


/****************************************************************************/
SANE_Int Hp_rts_XferExit(SANE_Int iHandle)
/****************************************************************************/
{
  TXferDev *pDev;

  if (iHandle <= 0) {
    return(-1);
  }
  pDev = (TXferDev *)iHandle;

  /* deinit the driver */
  if (pDev->iHandle >= 0) {
    pDev->iXferDev->mfnExit(pDev->iHandle);
    free((void *)iHandle);
    return (0);
  }
  return (-1);
}


/****************************************************************************/
SANE_Int Hp_rts_RegWrite(SANE_Int iHandle, SANE_Byte bReg, SANE_Byte bData)
/****************************************************************************/
{
  TXferDev *pDev;

  if (iHandle <= 0) {
    return (-1);
  }
  pDev = (TXferDev *)iHandle;

  return (pDev->iXferDev->mfnWriteReg(pDev->iHandle, bReg, bData));
}


/****************************************************************************/
SANE_Int Hp_rts_RegRead(SANE_Int iHandle, SANE_Byte bReg, SANE_Byte *pbData)
/****************************************************************************/
{
  TXferDev *pDev;

  if (iHandle <= 0) {
    return -1;
  }
  pDev = (TXferDev *)iHandle;

  return( pDev->iXferDev->mfnReadReg (pDev->iHandle, bReg,  pbData) );
}


/****************************************************************************/
SANE_Int Hp_rts_BulkWrite (SANE_Int iHandle, SANE_Byte reg, SANE_Byte *pabBuf,
                       SANE_Int iSize, SANE_Int wHeader)
/****************************************************************************/
{
  TXferDev *pDev;

  if (iHandle <= 0) {
    return -1;
  }
  pDev = (TXferDev *)iHandle;

  return(pDev->iXferDev->mfnWriteBulk((int) pDev->iHandle,reg,pabBuf,iSize,wHeader));
}


/****************************************************************************/
SANE_Int Hp_rts_BulkRead(SANE_Int iHandle, SANE_Byte reg, SANE_Byte *pabBuf,
                     SANE_Int iSize, SANE_Int wWrite)
/****************************************************************************/
{
  TXferDev *pDev;

  if (iHandle <= 0) {
    return -1;
  }
  pDev = (TXferDev *)iHandle;

  return(pDev->iXferDev->mfnReadBulk(pDev->iHandle,reg,pabBuf,iSize,wWrite));
}


/****************************************************************************/
SANE_Int Hp_rts_BulkReadall(SANE_Int iHandle, SANE_Byte *pabData)
/****************************************************************************/
{
  TXferDev *pDev;

  if (iHandle <= 0) {
    return -1;
  }
  pDev = (TXferDev *)iHandle;

  return(pDev->iXferDev->mfnBulkReadall(pDev->iHandle, pabData));
}

/*************************************************************************
  Returns SANE_TRUE if a known chipset was found. */
static SANE_Bool Hp_rts_ProbeRegisters(THWParams *pHWParams)
/*************************************************************************/
{
  SANE_Byte bData1;
  SANE_Int iHandle;

  iHandle = pHWParams->iXferHandle;
  bData1 = 0;
  DBG(DBG_ERR, "iHandle = %d, Probing scanner...\n",(int)iHandle);
  Hp_rts_RegRead( iHandle, 0x00, &bData1); /* read register 0x00 */
  if ((bData1 == 0xE5) || (bData1 == 0xF5)) {
     return SANE_STATUS_GOOD;
  }
  else
  {
    DBG(DBG_ERR, "Hp_rts_ProbeRegisters: No valid scanner with RTS88xx chipset found!\n");
    Hp_rts_XferExit(iHandle);
    return SANE_STATUS_IO_ERROR ;
  }
}

/****************************************************************************/
SANE_Int Hp_rts_Set_double_reg(SANE_Int iHandle, SANE_Byte reg,
                           SANE_Byte c1, SANE_Byte c2 )
/****************************************************************************/
{
  SANE_Byte regs[5];

  regs[0] = c1;
  regs[1] = c2;
  return(Hp_rts_BulkWrite (iHandle, reg, regs, 2,SANE_TRUE));
}

/****************************************************************************/
SANE_Int Hp_rts_Read_double_reg(SANE_Int iHandle, SANE_Byte reg, SANE_Byte *pbData )
/****************************************************************************/
{
  return(Hp_rts_BulkRead(iHandle,reg,pbData,2,SANE_TRUE));
}

/****************************************************************************/
SANE_Byte read_reg(SANE_Int iHandle,SANE_Byte reg){
/****************************************************************************/
  SANE_Byte bData1;

  Hp_rts_RegRead( iHandle, reg, &bData1);
  return(bData1);
}

/****************************************************************************
void send_moving_seq(SANE_Int iHandle) {
****************************************************************************
  Hp_rts_RegWrite(iHandle,MOVE_START_STOP,2);
  Hp_rts_RegWrite(iHandle,MOVE_START_STOP,2);
  Hp_rts_RegWrite(iHandle,MOVE_START_STOP,0);
  Hp_rts_RegWrite(iHandle,MOVE_START_STOP,0);
  Hp_rts_RegWrite(iHandle,MOVE_START_STOP,8);
  Hp_rts_RegWrite(iHandle,MOVE_START_STOP,8);
}*/

/****************************************************************************/
/*Return the number of bytes ready for collection                           */
SANE_Word Hp_rts_data_ready(SANE_Int iHandle, SANE_Word *size){
/****************************************************************************/
  static SANE_Byte command[]={0x90,0x00,0x00,0x03};
  SANE_Byte data[5];

  /* check if the scanner is moving*/
  if (SANE_TRUE /*read_reg(iHandle,MOVE_START_STOP) & 0x08*/)
  {
    Hp_rts_BulkWrite(iHandle,0x00,command,4,SANE_FALSE);
    Hp_rts_BulkRead(iHandle,0x00,data,3,SANE_FALSE);
    *size = data[0];
    *size = *size + (data[1]<<8);
    *size = *size + (data[2]<<16);
    if(*size > 0xe500) *size=0xe500;
    if (*size > 0){ return(SANE_TRUE); } else return(SANE_FALSE);
  }else return(SANE_FALSE);
}


/****************************************************************************/
SANE_Int Hp_rts_read_data(SANE_Int iHandle, SANE_Word size,SANE_Byte *data) {
/****************************************************************************/
  static SANE_Byte write_command[]={0x91,0x00,0x00,0x00};

  write_command[3]=size & 0xff;
  write_command[2]=(size>>8) & 0xff;
  write_command[1]=(size>>16) & 0xff;

  Hp_rts_BulkWrite(iHandle,0x00,write_command,4,SANE_FALSE);
  Hp_rts_BulkRead(iHandle,0x00,data,size,SANE_FALSE);
  return(0);
}

/****************************************************************************/
SANE_Int
Hp_rts_is_moving( SANE_Int iHandle )
/****************************************************************************/
{
	SANE_Byte	r;

	if ( Hp_rts_RegRead(iHandle,REG_MOVE_CONTROL_TEST, &r) < 0)
		/*Hp35x0c_read_register_immediate( iHandle, , 1, &r) < 0)*/
		return -1;
	if (r == 0x08)
		return 1;
	return 0;
}

/****************************************************************************/
SANE_Int
Hp_rts_start_moving( SANE_Int iHandle )
/****************************************************************************/
{
	if (Hp_rts_RegWrite(iHandle,REG_MOVE_CONTROL_TEST, 2) < 0 ||
	    Hp_rts_RegWrite(iHandle,REG_MOVE_CONTROL_TEST, 2) < 0 ||
	    Hp_rts_RegWrite(iHandle,REG_MOVE_CONTROL_TEST, 0) < 0 ||
	    Hp_rts_RegWrite(iHandle,REG_MOVE_CONTROL_TEST, 0) < 0 ||
	    Hp_rts_RegWrite(iHandle,REG_MOVE_CONTROL_TEST, 8) < 0 ||
	    Hp_rts_RegWrite(iHandle,REG_MOVE_CONTROL_TEST, 8) < 0)
		return -1;
	return 0;
}

/****************************************************************************/
SANE_Int
Hp_rts_stop_moving( SANE_Int iHandle )
/****************************************************************************/
{
	if (Hp_rts_RegWrite(iHandle,REG_MOVE_CONTROL_TEST, 2) < 0 ||
	    Hp_rts_RegWrite(iHandle,REG_MOVE_CONTROL_TEST, 2) < 0 ||
	    Hp_rts_RegWrite(iHandle,REG_MOVE_CONTROL_TEST, 0) < 0 ||
	    Hp_rts_RegWrite(iHandle,REG_MOVE_CONTROL_TEST, 0) < 0)
		return 0;
  else
		return 1;
}


/****************************************************************************/
/* Write a gamma table */
void
WriteGammaCalibTable (SANE_Int iHandle, const int *pabGammaR, const int *pabGammaG,
		      const int *pabGammaB)
/****************************************************************************/
{
  SANE_Byte cmd[3];
  SANE_Byte *buffer;
  SANE_Int i, j;

  iHandle = iHandle;

  /* Setup dummy gamma correction table */
  buffer = malloc (2 * 65536);

  cmd[0] = 2;
  cmd[1] = 0;
  cmd[2] = 0;

  for (i = 0; i < 3; i++)
    {
      const int *ptr = (i == 0) ? pabGammaR :
	(i == 1) ? pabGammaG : pabGammaB;

      for (j = 0; j < 65536; j++)	/* Truncate integers to shorts */
	buffer[j] = ptr[j];

/*      hp5400_bulk_command_write (iHandle, 0x2A01 + i, cmd, 3, 2 * 65536,
				 65536, (void *) buffer);*/
    }
  free (buffer);

  return;
}

/****************************************************************************/
void
SetDefaultGamma (SANE_Int iHandle)
/****************************************************************************/
{
  SANE_Int *buffer = malloc (sizeof (SANE_Int) * 65536);
  SANE_Int i;

  for (i = 0; i < 65336; i++)
    buffer[i] = i;

  WriteGammaCalibTable (iHandle, buffer, buffer, buffer);
}

/****************************************************************************/
SANE_Int
Hp_rts_Check_Scanning (SANE_Int iHandle)
/****************************************************************************/
{
	SANE_Int i;
	SANE_Word size;

	DBG(DBG_MSG,"Hp_rts_Check_Scanning: wait for moving and check it\n");
	i = 0;
	while( !(Hp_rts_is_moving(iHandle)) && ( i < 10 )){
		usleep(10000);
		i++;
		};
	if ( i >= 10 ){
		DBG(DBG_MSG,"Hp_rts_Check_Scanning: Time over, no move : %d!\n",i);
		/*stop it and make sure it stopped!!! */
		Hp_rts_stop_moving(iHandle);
		return(SANE_FALSE);
	}
	/* the scanner is now scanning. Check it */
	DBG(DBG_MSG,"Hp_rts_Check_Scanning: the scanner is now scanning. Check it\n");
	i = 0;
	while( (Hp_rts_data_ready(iHandle,&size) == SANE_FALSE) && ( i < 15 )){
		usleep(10000);
		i++;
	}
	if ( i >= 15 ){
		DBG(DBG_MSG,"Hp_rts_Check_Scanning: Time over, no data : %d!\n",i);
		/*stop it and make sure it stopped!!! */
		Hp_rts_stop_moving(iHandle);
		return(SANE_FALSE);
	}
	return(SANE_TRUE);
}

/****************************************************************************/
SANE_Int
Hp_rts_Read_Sram (SANE_Int iHandle, SANE_Byte *pabData,
                            SANE_Int iSize)
/****************************************************************************/
{
	SANE_Byte init[]={0x81,0x00,0x08,0x18};

	if (iHandle < 0) {
		return(-1);
	}
	init[2]=(SANE_Byte)(iSize>>8); /*split count over these bytes.*/
	init[3]=(SANE_Byte)iSize;

	if (Hp_rts_BulkWrite(iHandle,0x00,/*(SANE_Byte *)*/init,4,SANE_FALSE) == 0){
#if 0
		DBG(DBG_MSG,"Read SRAM, READ NOW %d bytes\n",iSize);
		if (iSize >= 0x800){
			DBG(DBG_MSG,"Read SRAM, READ the first %d bytes\n",2048);
			if (Hp_rts_BulkRead(iHandle,0x00,(SANE_Byte *)pabData, 2048,SANE_FALSE) == 0){
				DBG(DBG_MSG,"Read SRAM, READ the left %d bytes\n",iSize-2048);
				return (Hp_rts_BulkRead(iHandle,0x00,(SANE_Byte *)pabData, iSize-2048,SANE_FALSE));
			}
		}
		else{
#endif
			return (Hp_rts_BulkRead(iHandle,0x00,(SANE_Byte *)pabData, iSize,SANE_FALSE));
#if 0
		}
#endif
	}
	return(-1);
}

/****************************************************************************/
static SANE_Int
Hp_rts_set_value_lsbfirst(	SANE_Byte	*regs,
			SANE_Int		firstreg,
			SANE_Int		totalregs,
			SANE_Word	value)
/****************************************************************************/
{
	while (totalregs--)
	{
		regs[firstreg++] = value & 0xff;
		value >>= 8;
	}
	return 0;
}

/****************************************************************************/
SANE_Int
Hp_rts_set_noscan_distance(			SANE_Byte	*regs,
					SANE_Word	value)
/****************************************************************************/
{
	return Hp_rts_set_value_lsbfirst(regs, 0x60, 2, value);
}

/****************************************************************************/
SANE_Int
Hp_rts_set_total_distance(			SANE_Byte	*regs,
					SANE_Word	value)
/****************************************************************************/
{
	return Hp_rts_set_value_lsbfirst(regs, 0x62, 2, value);
}

/****************************************************************************/
SANE_Int
Hp_rts_set_scanline_end(			SANE_Byte	*regs,
					SANE_Word	value)
/****************************************************************************/
{
	return Hp_rts_set_value_lsbfirst(regs, 0x6c, 2, value);
}

/****************************************************************************/
SANE_Int
Hp_rts_set_scanline_start(			SANE_Byte	*regs,
					SANE_Word	value)
/****************************************************************************/
{
	return Hp_rts_set_value_lsbfirst	(regs, 0x66, 2, value);
}


