// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#ifndef __framergb_h__
#define __framergb_h__

#include "framebase.h"
#include "colorscalergb.h"

// Frame

class FrameRGB : public FrameBase {
 protected:
  int channel;                  // current channel
  int view[3];                  // visible channels
  CoordSystem rgbSystem;        // alignment coordinate system
  Matrix rgb[3];                // rgb matrix

  FitsImage* fits[3];           // list of fits images
  FitsImage* cfits[3];          // current fits
  int fitsCount[3];             // count of fits images

  float bias[3];                // current colormap bias
  float contrast[3];            // current colormap contrast

  int colorCount;               // number of dynamic colors
  unsigned char* colorCells;    // current color values
  ColorScaleRGB* colorScale[3]; // current color scale

  FrScale frScale[3];           // scale parameters
  DisplayMode displayMode[3];   // display mode
  MosaicType mosaicType[3];
  CoordSystem mosaicSystem[3];

  Vector WCScdelt[3];
  double WCSrot[3];
  Orientation WCSorientation[3];
  BBox WCSbb[3];                // WCS bbox in IMAGE coords
  Matrix WCSmatrix[3];

  Contour* contour[3];

  int isRGBCube;                // only way to tell if rgb cube loaded
                                // and not a data cube

 private:
  void loadRGBCube(MemType, const char*, FitsImage*);
  void loadRGBImage(MemType, const char*, FitsImage*);
  void loadRGBFinish();

  void align();
  void rgbAlignWCS(int,int);

  void reset();
  void setCurrentFits(FitsImage*);
  void setBinCursor(const Vector&);
  void updateBin(const Matrix&);

  BBox imageBBox();

  int doRender() 
    {return ((fits[0]&&view[0]) || (fits[1]&&view[1]) || (fits[2]&&view[2]));}
  void unloadFits();
  void updateMatrices();
  void updateMagnifierMatrices(Matrix&);
  int isIIS() {return 0;}

  int hasKeyFits() {return (keyFits && *keyFits);}

  void getInfoSingle(const Vector&, InternalSystem,
		     Matrix& (FitsImage::*)(), Matrix& (FitsImage::*)(),
		     SkyFrame, SkyFormat, char*);
  void getInfoMosaic(const Vector&, InternalSystem,
		     Matrix& (FitsImage::*)(), 
		     Matrix& (FitsImage::*)(),
		     SkyFrame, SkyFormat, char*);

 protected:
  void updateColorScale();
  int validColorScale() 
    {return colorScale[0] && colorScale[1] && colorScale[2];}
  void unloadAllFits();
  void updateColorCells(unsigned char*, int);

 public:
  FrameRGB(Tcl_Interp*, Tk_Canvas, Tk_Item*);
  virtual ~FrameRGB();

  int hasFitsCube();

  // Fits Commands

  void getFitsSizeCmd();
  void getFitsSizeCmd(CoordSystem, SkyFormat);
  void zoomToFitCmd(double =1);
  void sliceCmd(int);

  // Colormap Commands

  virtual void colormapCmd(float, float, float, float, float, float, 
			   unsigned char*, int) =0;
  virtual void colormapMotionCmd(float, float, float, float, float, float,
				 unsigned char*, int) =0;
  virtual void colormapEndCmd(float, float, float, float, float, float,
			      unsigned char*, int) =0;
  void getColormapCmd();
  void getTypeCmd();

  // Pan Zoom Rotate Orient Commands

  void getWCSZoomCmd(CoordSystem, Precision);
  void wcsZoomCmd(CoordSystem, double);

  // Load Commands

  void loadRGBCubeAllocCmd(const char*);
  void loadRGBCubeAllocGZCmd(const char*);
  void loadRGBCubeChannelCmd(const char*, const char*);
  void loadRGBCubeMMapCmd(const char*, LoadMethod);
  void loadRGBCubeSMMapCmd(const char*, const char*, LoadMethod);
  void loadRGBCubeMMapIncrCmd(const char*, LoadMethod);
  void loadRGBCubeShareCmd(ShmType, int, const char*, LoadMethod);
  void loadRGBCubeSShareCmd(ShmType, int, int, const char*, LoadMethod);
  void loadRGBCubeSocketCmd(int, const char*);
  void loadRGBCubeSocketGZCmd(int, const char*);
  void loadRGBCubeVarCmd(const char*, const char*, LoadMethod);

  void loadRGBCubeSlaveCmd(const char*, FitsFile*);

  void loadRGBImageAllocCmd(const char*);
  void loadRGBImageAllocGZCmd(const char*);
  void loadRGBImageChannelCmd(const char*, const char*);
  void loadRGBImageMMapCmd(const char*, LoadMethod);
  void loadRGBImageMMapIncrCmd(const char*, LoadMethod);
  void loadRGBImageShareCmd(ShmType, int, const char*, LoadMethod);
  void loadRGBImageSocketCmd(int, const char*);
  void loadRGBImageSocketGZCmd(int, const char*);
  void loadRGBImageVarCmd(const char*, const char*, LoadMethod);

  void loadArrRGBCubeAllocCmd(const char*);
  void loadArrRGBCubeAllocGZCmd(const char*);
  void loadArrRGBCubeChannelCmd(const char*, const char*);
  void loadArrRGBCubeMMapCmd(const char*);
  void loadArrRGBCubeMMapIncrCmd(const char*);
  void loadArrRGBCubeShareCmd(ShmType, int, const char*);
  void loadArrRGBCubeSocketCmd(int, const char*);
  void loadArrRGBCubeSocketGZCmd(int, const char*);
  void loadArrRGBCubeVarCmd(const char*, const char*);

  // RGB Commands

  void getRGBChannelCmd();
  void setRGBChannelCmd(const char*);
  void getRGBViewCmd();
  void setRGBViewCmd(int, int, int);
  void getRGBSystemCmd();
  void setRGBSystemCmd(CoordSystem);

  // IIS Commands

  void iisCmd(int, int) {}
  void iisEraseCmd() {}
  void iisGetCmd(char*, int, int, int, int) {}
  void iisSetCmd(const char*, int, int, int, int) {}
  void iisWCSCmd(const Matrix&, const Vector&, int) {}
};

#endif
