#  Copyright (C) 1999-2005
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc ButtonMarker {which x y} {
    global marker
    global template

    # if nothing is loaded, abort
    if {![$which has fits]} {
	return
    }

    # see if we are on a handle
    set h [$which get marker handle $x $y]
    set id [lindex $h 0]
    set marker(handle) [lindex $h 1]

    if {$marker(handle)} {
	$which marker $id edit begin $marker(handle)
	set marker(motion) beginEdit
	return
    }

    # else, see if we are on a segment of a polygon
    set h [$which get marker polygon segment $x $y]
    set id [lindex $h 0]
    set segment [lindex $h 1]
    if {$segment} {
	$which marker $id create polygon vertex $segment $x $y
	$which marker $id edit begin $marker(handle)
	set marker(handle) [expr 4+$segment+1]
	set marker(motion) beginEdit
	return
    }

    # else, see if we are on a marker
    if {[$which get marker id $x $y]} {
	$which marker select only $x $y
	$which marker move begin $x $y
	set marker(motion) beginMove
	UpdateMarkerMenu
	return
    }

    # see if any markers are selected
    if {[$which get marker select number]>0} {
	$which marker unselect all
	set marker(motion) none
	return
    }

    # else, create a marker
    set marker(handle) 0
    set marker(motion) none

    switch -- $marker(shape) {
	circle -
	annulus -
	panda -
	ellipse -
	ellipseannulus -
	epanda -
	box -
	boxannulus -
	bpanda -
	polygon -
	line -
	vector -
	text -
	ruler -
	compass -
	projection -
	"circle point" -
	"box point" -
	"diamond point" -
	"cross point" -
	"x point" -
	"arrow point" -
	"boxcircle point" {MarkerCreateShape $which $x $y}
	default {
	    set fn "template/$template($marker(shape))"
	    set ch [open $fn r]

	    global vardata
	    set vardata [read $ch]
	    close $ch

	    $which marker create template $x $y var vardata
	    #$which marker create template $x $y "\{$fn\}"
	}
    }
}

proc ShiftMarker {which x y} {
    global marker

    # if nothing is loaded, abort
    if {![$which has fits]} {
	return
    }

    # see if we are on a handle
    set h [$which get marker handle $x $y]
    set id [lindex $h 0]
    set marker(handle) [lindex $h 1]

    if {$marker(handle)} {
	$which marker $id rotate begin
	set marker(motion) beginRotate
	return
    }

    # else, see if we are on a marker
    if {[$which marker select toggle $x $y]} {
	UpdateMarkerMenu
	$which marker move begin $x $y
	set marker(motion) beginMove
	return
    }

    # else, start a region select
    $which region select begin $x $y
    set marker(motion) shiftregion
}

proc ControlMarker {which x y} {
    global marker
    
    # if nothing is loaded, abort

    if {![$which has fits]} {
	return
    }

    # we need this cause MotionMarker maybe called, 
    # and we don't want it
    set marker(motion) none
    set marker(handle) -1

    set id [$which get marker id $x $y]
    if {$id} {
	# are we on a selected annulus?
	if {[$which get marker select $x $y] == $id} {
	    switch -- [$which get marker $id type] {
		annulus {
		    set marker(handle) \
			[$which marker $id create annulus radius $x $y]
		    $which marker $id edit begin $marker(handle)
		    set marker(motion) edit
		}
		panda {
		    set marker(handle) \
			[$which marker $id create panda radius $x $y]
		    $which marker $id edit begin $marker(handle)
		    set marker(motion) edit
		}
		ellipseannulus {
		    set marker(handle) \
			[$which marker $id create ellipseannulus radius $x $y]
		    $which marker $id edit begin $marker(handle)
		    set marker(motion) edit
		}
		epanda {
		    set marker(handle) \
			[$which marker $id create epanda radius $x $y]
		    $which marker $id edit begin $marker(handle)
		    set marker(motion) edit
		}
		boxannulus {
		    set marker(handle) \
			[$which marker $id create boxannulus radius $x $y]
		    $which marker $id edit begin $marker(handle)
		    set marker(motion) edit
		}
		bpanda {
		    set marker(handle) \
			[$which marker $id create bpanda radius $x $y]
		    $which marker $id edit begin $marker(handle)
		    set marker(motion) edit
		}
	    }
	}
    }
}

proc ControlShiftMarker {which x y} {
    global marker
    
    # if nothing is loaded, abort
    if {![$which has fits]} {
	return
    }

    # we need this cause MotionMarker maybe called, 
    # and we don't want it
    set marker(motion) none
    set marker(handle) -1

    set id [$which get marker id $x $y]
    if {$id} {
	# are we on a selected annulus?
	if {[$which get marker select $x $y] == $id} {
	    switch -- [$which get marker $id type] {
		panda {
		    set marker(handle) \
			[$which marker $id create panda angle $x $y]
		    $which marker $id edit begin $marker(handle)
		    set marker(motion) edit
		}
		epanda {
		    set marker(handle) \
			[$which marker $id create epanda angle $x $y]
		    $which marker $id edit begin $marker(handle)
		    set marker(motion) edit
		}
		bpanda {
		    set marker(handle) \
			[$which marker $id create bpanda angle $x $y]
		    $which marker $id edit begin $marker(handle)
		    set marker(motion) edit
		}
	    }
	}
    }
}

proc MotionMarker {which x y} {
    global marker

    # if nothing is loaded, abort
    if {![$which has fits]} {
	return
    }

    switch -- $marker(motion) {
	none {}

	beginCreate -
	create {
	    $which marker edit motion $x $y $marker(handle)
	    set marker(motion) create
	}

	beginMove -
	move {
	    $which marker move motion $x $y
	    set marker(motion) move
	}

	beginEdit -
	edit {
	    $which marker edit motion $x $y $marker(handle)
	    set marker(motion) edit
	}

	beginRotate -
	rotate {
	    $which marker rotate motion $x $y $marker(handle)
	    set marker(motion) rotate
	}

	region -
	shiftregion {$which region select motion $x $y}
    }
}

proc ReleaseMarker {which x y} {
    global marker

    # if nothing is loaded, abort
    if {![$which has fits]} {
	return
    }

    switch -- $marker(motion) {
	none {}
	beginCreate {
	    # the user has just clicked, so resize to make visible or delete
	    # assumes marker(id) from create
	    $which marker edit end
	    MarkerDefault $which

	    unset marker(createx)
	    unset marker(createy)
	    unset marker(id)
	}
	create {
	    $which marker edit end

	    # determine if this is an accident and just create the default
	    set diffx [expr $x-$marker(createx)]
	    set diffy [expr $y-$marker(createy)]
	    if {[expr sqrt($diffx*$diffx + $diffy*$diffy)]<2} {
		MarkerDefault $which
	    }

	    # special callbacks
	    switch -- [$which get marker $marker(id) type] {
		projection {ProjectionPlotInit $marker(id)}
	    }

	    unset marker(createx)
	    unset marker(createy)
	    unset marker(id)
	}
	beginMove -
	beginRotate {}
	beginEdit {}
	move {$which marker move end}
	edit {$which marker edit end}
	rotate {$which marker rotate end}
	region {$which region select end}
	shiftregion {$which region select shift end}
    }

    unset marker(motion)
    unset marker(handle)
}

proc DoubleMarker {which x y} {
    # if nothing is loaded, abort
    if {![$which has fits]} {
	return
    }

    set id [$which get marker id $x $y]

    if {$id} {
	if [$which get marker $id PROPERTY SELECT] {
	    MarkerDialog $which $id
	}
    }
}

proc CursorMarker {which x y handleCursor overCursor} {
    global ds9

    # if nothing is loaded, abort

    if {![$which has fits]} {
	return
    }

    # are we over any selected marker handles?
    # remember, handles are outside of a marker

    set h [$which get marker handle $x $y]
    set id [lindex $h 0]
    set handle [lindex $h 1]
    if {$handle} {
	if {$handle < 5} {
	    # edit/rotate handle
	    SetCursor $handleCursor
	} else { 
	    # polygon/annulus vertex
	    SetCursor dotbox
	}
	return
    }

    # else, see if we are on a segement of a polygon

    set h [$which get marker polygon segment $x $y]
    if {[lindex $h 0]} {
	SetCursor draped_box
	return
    }

    # are we over a marker?

    set id [$which get marker select $x $y]
    if {$id} {
	# are we on a selected annulus and control key down?
	switch -- [$which get marker $id type] {
	    annulus -
	    panda -
	    ellipseannulus -
	    epanda -
	    boxannulus -
	    bpanda {SetCursor $overCursor}
	    default {SetCursor fleur}
	}
	return
    }

    # else, set no cursor

    SetCursor {}
}

proc MarkerCreateShape {which x y} {
    global marker

    set cmd "$which marker create $marker(shape) $x $y"
    switch -- $marker(shape) {
	circle {append cmd " 0"}
	annulus {append cmd " .001 .002 $marker(annulus,annuli)"}
	panda {append cmd " $marker(panda,ang1) $marker(panda,ang2) $marker(panda,angnum) .001 .002 $marker(panda,annuli)"}
	ellipse {append cmd " 0 0"}
	ellipseannulus {append cmd " .001 .001 .002 $marker(ellipseannulus,annuli)"}
	epanda {append cmd " $marker(epanda,ang1) $marker(epanda,ang2) $marker(epanda,angnum) .001 .001 .002 $marker(epanda,annuli)"}
	box {append cmd " 0 0"}
	boxannulus {append cmd " .002 .002 .004 $marker(boxannulus,annuli)"}
	bpanda {append cmd " $marker(bpanda,ang1) $marker(bpanda,ang2) $marker(bpanda,angnum) .001 .001 .002 $marker(bpanda,annuli)"}
	polygon {append cmd " .001 .001"}
	line {append cmd " $x $y"}
	vector {append cmd " $x $y"}
	text {
	    set txt "Region"
	    set r [EntryDialog "Text Region" "Enter Text:" 40 txt]
	    if {$r == 1 && $txt != {}} {
		append cmd " 0 text = \{\"$txt\"\}"
	    } else {
		return
	    }
	}
	ruler {append cmd " $x $y $marker(dialog,system) $marker(dialog,sky)"}
	compass {append cmd " 15 $marker(dialog,system) $marker(dialog,sky) "}
	projection {append cmd " $x $y $marker(projection,thick)"}
    }
    append cmd " color = $marker(color)"
    append cmd " width = $marker(width)"
    append cmd " font = \{\"$marker(font) $marker(font,size) $marker(font,style)\"\}"
    append cmd " edit = $marker(edit)"
    append cmd " move = $marker(move)"
    append cmd " rotate = $marker(rotate)"
    append cmd " delete = $marker(delete)"
    append cmd " fixed = $marker(fixed)"
    append cmd " include = $marker(include)"
    append cmd " source = $marker(source)"

    $which marker unselect all

    set marker(id) [eval $cmd]
    set marker(motion) beginCreate
    set marker(createx) $x
    set marker(createy) $y

    switch -- $marker(shape) {
	circle -
	annulus -
	panda -
	ellipse -
	ellipseannulus -
	epanda -
	box -
	boxannulus -
	bpanda -
	compass -
	polygon {
	    set marker(handle) 1
	    $which marker $marker(id) edit begin $marker(handle)
	}
	line -
	vector -
	ruler -
	projection {
	    set marker(handle) 2
	    $which marker $marker(id) edit begin $marker(handle)
	}
    }
}

proc MarkerDefault {which} {
    global marker
    global current

    # scale the default size to take into account the current
    set z1 double([lindex $current(zoom) 0])
    set z2 double([lindex $current(zoom) 1])
    if {$z1>$z2} {
	set zz $z1
    } else {
	set zz $z2
    }

    switch -- [$which get marker $marker(id) type] {
	circle {
	    $which marker $marker(id) circle radius \
		[expr ($marker(circle,radius)/$zz)] \
		image degrees
	}
	annulus {
	    $which marker $marker(id) annulus radius \
		[expr ($marker(annulus,inner)/$zz)] \
		[expr ($marker(annulus,outer)/$zz)] \
		$marker(annulus,annuli) image degrees
	}
	panda {
	    $which marker $marker(id) panda edit \
		$marker(panda,ang1) $marker(panda,ang2) $marker(panda,angnum) \
		[expr ($marker(panda,inner)/$zz)] \
		[expr ($marker(panda,outer)/$zz)] \
		$marker(panda,annuli) image
	}
	ellipse {
	    $which marker $marker(id) ellipse radius \
		[expr ($marker(ellipse,radius1)/$z1)] \
		[expr ($marker(ellipse,radius2)/$z2)] \
		image degrees
	}
	ellipseannulus {
	    $which marker $marker(id) ellipseannulus radius \
	      [expr ($marker(ellipseannulus,radius1)/$z1)] \
	      [expr ($marker(ellipseannulus,radius2)/$z2)] \
	      [expr ($marker(ellipseannulus,radius3)/$z1)] \
		$marker(ellipseannulus,annuli) image
	}
	epanda {
	    $which marker $marker(id) epanda edit \
		$marker(epanda,ang1) $marker(epanda,ang2) \
		$marker(epanda,angnum) \
		[expr ($marker(epanda,radius1)/$z1)] \
		[expr ($marker(epanda,radius2)/$z2)] \
		[expr ($marker(epanda,radius3)/$z1)] \
		$marker(epanda,annuli) image
	}
	box {
	    $which marker $marker(id) box radius \
		[expr ($marker(box,radius1)/$z1)] \
		[expr ($marker(box,radius2)/$z2)] \
		image degrees
	}
	boxannulus {
	    $which marker $marker(id) boxannulus radius \
		[expr ($marker(boxannulus,radius1)/$z1)] \
		[expr ($marker(boxannulus,radius2)/$z2)] \
		[expr ($marker(boxannulus,radius3)/$z1)] \
		$marker(boxannulus,annuli) image
	}
	bpanda {
	    $which marker $marker(id) bpanda edit \
		$marker(bpanda,ang1) $marker(bpanda,ang2) \
		$marker(bpanda,angnum) \
		[expr ($marker(bpanda,radius1)/$z1)] \
		[expr ($marker(bpanda,radius2)/$z2)] \
		[expr ($marker(bpanda,radius3)/$z1)] \
		$marker(bpanda,annuli) image
	}
	compass {
	    $which marker $marker(id) compass radius $marker(compass,radius) \
		image degrees
	}
	polygon {
	    $which marker $marker(id) polygon reset \
		[expr ($marker(polygon,width)/$z1)] \
		[expr ($marker(polygon,height)/$z2)] \
		image degrees
	}
	line -
	vector -
	ruler -
	projection {$which marker $marker(id) delete}
    }
}

proc MarkerDelete {which x y} {
    # if nothing is loaded, abort
    if {![$which has fits]} {
	return
    }

    # see if we are on a polygon
    set h [$which get marker handle $x $y]
    set id [lindex $h 0]
    set handle [lindex $h 1]

    set t [$which get marker $id type]
    switch -- $t {
	polygon -
	annulus -
	panda -
	ellipseannulus -
	epanda -
	boxannulus -
	bpanda {
	    if {$handle > 4} {
		switch -- $t {
		    polygon {$which marker $id delete polygon vertex $handle}
		    annulus {$which marker $id delete annulus $handle}
		    panda {$which marker $id delete panda $handle}
		    ellipseannulus {$which marker $id delete \
					ellipseannulus $handle}
		    epanda {$which marker $id delete epanda $handle}
		    boxannulus {$which marker $id delete boxannulus $handle}
		    bpanda {$which marker $id delete bpanda $handle}
		}
	    } else {
		# delete polygon
		$which marker delete
		UpdateGroupDialog
	    }
	}
	default {
	    # delete marker
	    $which marker delete
	    UpdateGroupDialog
	}
    }
}

proc MarkerShow {} {
    global current
    global marker

    if {$current(frame) != {}} {
	$current(frame) marker show $marker(show)
    }
}

proc MarkerPreserve {} {
    global current
    global marker

    if {$current(frame) != {}} {
	$current(frame) marker preserve $marker(preserve)
    }
}

proc MarkerFront {} {
    global current
    
    if {$current(frame) != ""} {
	$current(frame) marker move front
    }
}

proc MarkerBack {} {
    global current
    
    if {$current(frame) != ""} {
	$current(frame) marker move back
	$current(frame) marker unselect all
    }
}

proc MarkerSelectAll {} {
    global current
    
    if {$current(frame) != ""} {
	$current(frame) marker select all
    }
}

proc MarkerUnselectAll {} {
    global current
    
    if {$current(frame) != ""} {
	$current(frame) marker unselect all
    }
}

proc MarkerSelectInvert {} {
    global current
    
    if {$current(frame) != ""} {
	$current(frame) marker select toggle
    }
}

proc MarkerDeleteSelect {} {
    global current
    
    if {$current(frame) != ""} {
	$current(frame) marker delete
	UpdateGroupDialog
    }
}

proc MarkerDeleteAllMenu {} {
    global current
    
    if {[tk_messageBox -type okcancel -default cancel \
	     -message "Delete All Regions?" -icon question] == "ok"} {
	MarkerDeleteAll
    }
}

proc MarkerDeleteAll {} {
    global current
    
    if {$current(frame) != ""} {
	$current(frame) marker delete all
	UpdateGroupDialog
    }
}

proc MarkerColor {} {
    global current
    global marker
    
    if {$current(frame) != ""} {
	$current(frame) marker color $marker(color)
    }
}

proc MarkerWidth {} {
    global current
    global marker
    
    if {$current(frame) != ""} {
	$current(frame) marker width $marker(width)
    }
}

proc MarkerProp {prop} {
    global current
    global marker

    if {$current(frame) != ""} {
	$current(frame) marker property $prop $marker($prop)
    }
}

proc MarkerFont {} {
    global current
    global marker

    if {$current(frame) != ""} {
	$current(frame) marker font \
	    \"$marker(font) $marker(font,size) $marker(font,style)\"
    }
}

proc MarkerOpen {} {
    MarkerLoad [OpenFileDialog markerfbox]
}

proc MarkerLoad {filename} {
    global current
    global marker
    global message

    if {$current(frame) != ""} {
	if {$filename != {}} {
	    SetWatchCursor

	    # determine if its a fits file
	    # first, strip the filename
	    if {![regexp -nocase {(.*)(\[.*\])} $filename foo base ext]} {
		set base $filename
		set ext {}
	    }
	    set fd [open $base]
	    if {$fd != ""} {
		set l [read $fd 9]
		close $fd
	    }

	    if {$l == "SIMPLE  ="} {
		# its a fits file

		# see if we need to add an extension
		if {$ext == ""} {
		    set filename "$base\[REGION\]"
		}

		# open it
		if [catch {$current(frame) marker load fits \
			       "\{$filename\}" $marker(color) $marker(width) \
			       "\{$marker(font) $marker(font,size) $marker(font,style)\}"}] {
		    Error "$message(error,marker,file)"
		}
	    } else {
		# no, its ascii
		switch -- $marker(format) {
		    xy {
			if [catch {$current(frame) marker load \
				       $marker(format) "\{$filename\}" \
				       $marker(system) $marker(sky)}] {
			    Error "$message(error,marker,file)"
			}
		    }
		    default {
			if [catch {$current(frame) marker load \
				       $marker(format) "\{$filename\}"}] {
			    Error "$message(error,marker,file)"
			}
		    }
		}
	    }
	    UnsetWatchCursor
	    UpdateGroupDialog
	}
    }
}

proc MarkerSave {} {
    global current
    global marker

    if {$current(frame) != {}} {
	SetWatchCursor
	set filename [SaveFileDialog markerfbox]

	if {$filename != {}} {
	    $current(frame) marker save "\{$filename\}" \
		$marker(format) $marker(system) $marker(sky) \
		$marker(skyformat) $marker(strip) $marker(wcs)
	}
	UnsetWatchCursor
    }
}

proc MarkerDisplay {} {
    global current
    global marker

    if {$current(frame) != {}} {
	SimpleTextDialog markerlist "Region List" 80 20 insert top \
	    [$current(frame) marker list $marker(format) \
		 $marker(system) $marker(sky) \
		 $marker(skyformat) $marker(strip) $marker(wcs)]
    }
}

proc MarkerInfo {} {
    global current
    global marker

    if {$current(frame) != {}} {
	set l [$current(frame) get marker select]
	set i 0
	foreach d $l {
	    incr i
	    if {$i > $marker(maxdialog)} {
		return
	    }
	    MarkerDialog $current(frame) $d
	}
    }
}

proc MarkerDialog {frame id} {

    set type [$frame get marker $id type]

    switch -- $type {
	circle {CircleDialog $frame $id}
	annulus {AnnulusDialog $frame $id}
	panda {PandaDialog $frame $id}
	ellipse {EllipseDialog $frame $id}
	ellipseannulus {EllipseAnnulusDialog $frame $id}
	epanda {EpandaDialog $frame $id}
	box {BoxDialog $frame $id}
	boxannulus {BoxAnnulusDialog $frame $id}
	bpanda {BpandaDialog $frame $id}
	polygon {PolygonDialog $frame $id}
	line {LineDialog $frame $id}
	vector {VectorDialog $frame $id}
	text {TextDialog $frame $id}
	ruler {RulerDialog $frame $id}
	compass {CompassDialog $frame $id}
	projection {ProjectionDialog $frame $id}
	{circle point} -
	{box point} -
	{diamond point} -
	{cross point} -
	{x point} -
	{arrow point} -
	{boxcircle point} {PointDialog $frame $id}
	composite {CompositeDialog $frame $id}
    }
}

proc MarkerCopy {} {
    global current
    global marker

    if {$current(frame) != {}} {
	$current(frame) marker copy
	set marker(copy) $current(frame)
    }
    UpdateEditMenu
}

proc MarkerCut {} {
    global current
    global marker

    if {$current(frame) != {}} {
	$current(frame) marker cut
	set marker(copy) $current(frame)
    }
    UpdateEditMenu
    UpdateGroupDialog
}

proc MarkerUndo {} {
    global current

    if {$current(frame) != {}} {
	$current(frame) marker undo
    }
    UpdateEditMenu
    UpdateGroupDialog
}

proc MarkerPaste {} {
    global current
    global marker

    if {$current(frame) != {}} {
	# same frame?
	if {$current(frame) == $marker(copy)} {
	    $current(frame) marker paste

	    # any special callbacks?
	    foreach id [$current(frame) get marker select] {
		switch -- [$current(frame) get marker $id type] {
		    projection {ProjectionPlotInit $id}
		}
	    }
	} else {
	    if {$marker(paste,system) == {}} {
		if {![MarkerPasteDialog]} {
		    return
		}
	    }

	    global cmd
	    set cmd "[$marker(copy) marker paste $marker(paste,system) $marker(paste,sky)]"
	    $current(frame) marker command ds9 var cmd
	}
    }
	
    UpdateEditMenu
    UpdateGroupDialog
}

proc MarkerPasteDialog {} {
    global mkld
    global current
    global marker
    global menu
    global ds9

    set w ".mkld"

    set marker(paste,system) {}
    set marker(paste,sky) {}

    set mkld(ok) 0
    set mkld(system) WCS
    set mkld(sky) fk5

    DialogCreate $w "Paste Regions" -borderwidth 2
    frame $w.param -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.param $w.buttons  -side top -ipadx 4 -ipady 4 -fill x -expand true

    label $w.param.coordtitle -text "Coordinate System: "
    menubutton $w.param.coordbutton -relief raised \
	-menu $w.param.coordbutton.menu -textvariable mkld(system) \
	-width 8

    menu $w.param.coordbutton.menu -tearoff 0 -selectcolor $menu(selectcolor)
    $w.param.coordbutton.menu add radiobutton -label "WCS" \
	-variable mkld(system) -value "WCS"
    $w.param.coordbutton.menu add cascade -label "Multiple WCS" \
	-menu $w.param.coordbutton.menu.wcs
    $w.param.coordbutton.menu add separator
    $w.param.coordbutton.menu add radiobutton -label "Image" \
	-variable mkld(system) -value "Image"
    $w.param.coordbutton.menu add radiobutton -label "Physical" \
	-variable mkld(system) -value "Physical"
    if {$ds9(amp,det)} {
	$w.param.coordbutton.menu add radiobutton -label "Amplifier" \
	    -variable mkld(system) -value "Amplifier"
	$w.param.coordbutton.menu add radiobutton -label "Detector" \
	    -variable mkld(system) -value "Detector"
    }
    $w.param.coordbutton.menu add separator
    $w.param.coordbutton.menu add radiobutton -label "Equatorial B1950" \
	-variable mkld(sky) -value fk4
    $w.param.coordbutton.menu add radiobutton -label "Equatorial J2000" \
	-variable mkld(sky) -value fk5
    $w.param.coordbutton.menu add radiobutton -label "ICRS" \
	-variable mkld(sky) -value icrs
    $w.param.coordbutton.menu add radiobutton -label "Galactic" \
	-variable mkld(sky) -value galactic
    $w.param.coordbutton.menu add radiobutton -label "Ecliptic" \
	-variable mkld(sky) -value ecliptic

    menu $w.param.coordbutton.menu.wcs -tearoff $menu(tearoff) \
	-selectcolor $menu(selectcolor)
    foreach l {A B C D E F G H I J K L M N O P Q R S T U V W X Y Z} {
	$w.param.coordbutton.menu.wcs add radiobutton -label "WCS $l" \
	    -variable mkld(system) -value "wcs$l"
    }

    grid rowconfigure $w.param 0 -pad 0
    grid $w.param.coordtitle $w.param.coordbutton -padx 4 -sticky w

    button $w.buttons.ok -text "OK" -default active -command {set mkld(ok) 1}
    button $w.buttons.cancel -text "Cancel" -command {set mkld(ok) 0}
    pack $w.buttons.ok -side left -padx 10
    pack $w.buttons.cancel -side right -padx 10

    bind $w <Return> {set mkld(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    DialogWait $w mkld(ok)
    DialogDismiss $w

    if {$mkld(ok)} {
	set marker(paste,system) $mkld(system)
	set marker(paste,sky) $mkld(sky)
    }

    # we want to destroy this window
    destroy $w 

    set r $mkld(ok)
    unset mkld
    return $r
}

proc CompositeCreate {} {
    global current
    global marker

    if {$current(frame) != {}} {
	set cmd "$current(frame) marker create composite"
	append cmd " color = $marker(color)"
	append cmd " width = $marker(width)"
	append cmd " font = \{\"$marker(font) $marker(font,size) $marker(font,style)\"\}"
	append cmd " edit = $marker(edit)"
	append cmd " move = $marker(move)"
	append cmd " rotate = $marker(rotate)"
	append cmd " delete = $marker(delete)"
	append cmd " fixed = $marker(fixed)"
	append cmd " include = $marker(include)"
	append cmd " source = $marker(source)"

	eval $cmd
    }
}

proc CompositeDelete {} {
    global current

    if {$current(frame) != {}} {
	$current(frame) marker composite delete
    }
}

proc UpdateMarkerMenu {} {
    global ds9
    global current
    global marker
    global menu

    if {$current(frame) != {}} {
	set marker(show) [$current(frame) get marker show]
	set marker(preserve) [$current(frame) get marker preserve]

	switch -- $ds9(mode) {
	    pointer {
		if {[$current(frame) get marker select number] == 1} {
		    set marker(color) [$current(frame) get marker color]
		    set marker(edit) [$current(frame) get marker property edit]
		    set marker(move) [$current(frame) get marker property move]
		    set marker(rotate) \
			[$current(frame) get marker property rotate]
		    set marker(delete) \
			[$current(frame) get marker property delete]
		    set marker(fixed) \
			[$current(frame) get marker property fixed]
		    set marker(include) \
			[$current(frame) get marker property include]
		    set marker(source) \
			[$current(frame) get marker property source]

		    set f [$current(frame) get marker font]

		    set marker(font) [lindex $f 0]
		    set marker(font,size) [lindex $f 1]
		    set marker(font,style) [lindex $f 2]
		}
	    }
	}
    }

    UpdateMarkerFormatMenu
}

proc UpdateMarkerFormatMenu {} {
    global marker
    global current
    global ds9
    global menu

    set mm $ds9(mb).region

    switch -- $marker(format) {
	ds9 {
	    UpdateCoordMenu "$current(frame)" $mm.coord
	    AdjustCoord "$current(frame)" marker(system)
	}

	ciao {
	    UpdateCoordMenu "$current(frame)" $mm.coord
	    switch -- $marker(system) {
		detector -
		amplifier -
		physical -
		image {set marker(system) physical}
		wcs -
		default {set marker(system) wcs}
	    }
	    set marker(sky) fk5
	    set marker(skyformat) sexagesimal
	    AdjustCoord "$current(frame)" marker(system)

	    $mm.coord entryconfig "Multiple WCS" -state disabled
	    $mm.coord entryconfig "Image" -state disabled
	    $mm.coord entryconfig "Physical" -state normal
	    if {$ds9(amp,det)} {
		$mm.coord entryconfig "Amplifier" -state disabled
		$mm.coord entryconfig "Detector" -state disabled
	    }

	    $mm.coord entryconfig "Equatorial B1950" -state disabled
	    $mm.coord entryconfig "Equatorial J2000" -state normal
	    $mm.coord entryconfig "ICRS" -state disabled
	    $mm.coord entryconfig "Galactic" -state disabled
	    $mm.coord entryconfig "Ecliptic" -state disabled

	    $mm.coord entryconfig "Degrees" -state disabled
	    $mm.coord entryconfig "Sexagesimal" -state normal
	}

	saotng {
	    UpdateCoordMenu "$current(frame)" $mm.coord
	    switch -- $marker(system) {
		detector -
		amplifier -
		image -
		physical {set marker(system) image}
		wcs -
		default {set marker(system) wcs}
	    }
	    AdjustCoord "$current(frame)" marker(system)

	    $mm.coord entryconfig "Multiple WCS" -state disabled
	    $mm.coord entryconfig "Physical" -state disabled
	    if {$ds9(amp,det)} {
		$mm.coord entryconfig "Amplifier" -state disabled
		$mm.coord entryconfig "Detector" -state disabled
	    }
	}

	saoimage {
	    set marker(system) image

	    SetCoordMenu $mm.coord disabled
	    $mm.coord entryconfig "Image" -state normal
	}

	pros {
	    UpdateCoordMenu "$current(frame)" $mm.coord
	    switch -- $marker(system) {
		detector -
		physical -
		amplifier {set marker(system) physical}
		image {}
		wcs -
		default {set marker(system) wcs}
	    }

	    if {$marker(sky) == "icrs"} {
		set marker(sky) fk5
	    }
	    AdjustCoord "$current(frame)" marker(system)

	    $mm.coord entryconfig "Multiple WCS" -state disabled
	    $mm.coord entryconfig "ICRS" -state disabled
	    if {$ds9(amp,det)} {
		$mm.coord entryconfig "Detector" -state disabled
		$mm.coord entryconfig "Amplifier" -state disabled
	    }
	}
	
	xy {
	    UpdateCoordMenu "$current(frame)" $mm.coord
	    switch -- $marker(system) {
		detector -
		physical -
		amplifier -
		image -
		wcs {}
		default {set marker(system) wcs}
	    }
	    AdjustCoord "$current(frame)" marker(system)

	    $mm.coord entryconfig "Multiple WCS" -state disabled
	}
    }
}

proc UpdateMarkerPrefsFormatMenu {} {
    global prefs
    global marker
    global current
    global ds9

    set mm $ds9(mb).prefs.region
    SetCoordMenu $mm.coord normal

    switch -- $prefs(marker,skyformat) {
	ds9 {}

	ciao {
	    switch -- $prefs(marker,system) {
		detector -
		amplifier -
		physical -
		image {set prefs(marker,system) physical}
		wcs -
		default {set prefs(marker,system) wcs}
	    }
	    set prefs(marker,sky) fk5
	    set prefs(marker,skyformat) sexagesimal

	    $mm.coord entryconfig "Multiple WCS" -state disabled
	    $mm.coord entryconfig "Image" -state disabled
	    $mm.coord entryconfig "Physical" -state normal
	    if {$ds9(amp,det)} {
		$mm.coord entryconfig "Amplifier" -state disabled
		$mm.coord entryconfig "Detector" -state disabled
	    }

	    $mm.coord entryconfig "Equatorial B1950" -state disabled
	    $mm.coord entryconfig "Equatorial J2000" -state normal
	    $mm.coord entryconfig "ICRS" -state disabled
	    $mm.coord entryconfig "Galactic" -state disabled
	    $mm.coord entryconfig "Ecliptic" -state disabled

	    $mm.coord entryconfig "Degrees" -state disabled
	    $mm.coord entryconfig "Sexagesimal" -state normal
	}

	saotng {
	    switch -- $prefs(marker,system) {
		detector -
		amplifier -
		image -
		physical {set prefs(marker,system) image}
		wcs -
		default {set prefs(marker,system) wcs}
	    }

	    $mm.coord entryconfig "Multiple WCS" -state disabled
	    $mm.coord entryconfig "Physical" -state disabled
	    if {$ds9(amp,det)} {
		$mm.coord entryconfig "Amplifier" -state disabled
		$mm.coord entryconfig "Detector" -state disabled
	    }
	}

	saoimage {
	    SetCoordMenu $mm.coord disabled
	    set prefs(marker,system) image
	    $mm.coord entryconfig "Image" -state normal
	}

	pros {
	    switch -- $prefs(marker,system) {
		detector -
		physical -
		amplifier {set prefs(marker,system) physical}
		image {}
		wcs -
		default {set prefs(marker,system) wcs}
	    }
	    if {$prefs(marker,sky) == "icrs"} {
		set prefs(marker,sky) fk5
	    }

	    $mm.coord entryconfig "Multiple WCS" -state disabled
	    $mm.coord entryconfig "Detector" -state disabled
	    if {$ds9(amp,det)} {
		$mm.coord entryconfig "Amplifier" -state disabled
		$mm.coord entryconfig "ICRS" -state disabled
	    }
	}
	
	xy {
	    switch -- $prefs(marker,system) {
		detector -
		physical -
		amplifier -
		image -
		wcs {}
		default {set prefs(marker,system) wcs}
	    }
	    $mm.coord entryconfig "Multiple WCS" -state disabled
	}
    }
}

proc ProcessMarkerCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global current
    global marker

    switch -- [lindex $var $i] {
	movefront {MarkerFront}
	moveback {MarkerBack}
	move {
	    incr i
	    switch -- [lindex $var $i] {
		front {MarkerFront}
		back {MarkerBack}
	    }
	}

	selectall {MarkerSelectAll}
	selectnone {MarkerUnselectAll}
	select {
	    incr i
	    switch -- [lindex $var $i] {
		group {
		    incr i
		    $current(frame) marker "\{[lindex $var $i]\}" select
		}
		all {MarkerSelectAll}
		none {MarkerUnselectAll}
	    }
	}

	deleteall {MarkerDeleteAll}
	delete {
	    incr i
	    switch -- [lindex $var $i] {
		select {MarkerDeleteSelect}
		all {MarkerDeleteAll}
	    }
	}

	format {
	    incr i
	    set marker(format) [lindex $var $i]
	    UpdateMarkerFormatMenu
	}
	coord -
	system {
	    # for backward compatibility
	    incr i
	    switch -- [lindex $var $i] {
		fk4 -
		fk5 -
		icrs -
		galactic -
		ecliptic {
		    incr i
		    set marker(system) wcs
		    set marker(sky) [lindex $var $i]
		}
		
		default {set marker(system) [lindex $var $i]}
	    }
	}
	sky {
	    incr i
	    set marker(sky) [lindex $var $i]
	}
	coordformat -
	skyformat {
	    incr i
	    set marker(skyformat) [lindex $var $i]
	}
	strip {
	    incr i
	    set marker(strip) [FromYesNo [lindex $var $i]]
	}
	wcs {
	    incr i
	    set marker(wcs) [FromYesNo [lindex $var $i]]
	}
	delim {
	    incr i
	    if {[lindex $var $i] != "nl"} {
		set marker(strip) 1
	    } else {
		set marker(strip) 0
	    }
	}
	shape {
	    incr i
	    set marker(shape) [lindex $var $i]
	}
	color {
	    incr i
	    set marker(color) [lindex $var $i] 
	    MarkerColor
	}
	width {
	    incr i
	    set marker(width) [lindex $var $i]
	    MarkerWidth
	}

	tag -
	tags -
	group -
	groups {
	    incr i
	    set tag "\{[lindex $var $i]\}"
	    incr i
	    switch -- [lindex $var $i] {
		color {
		    incr i
		    $current(frame) marker $tag color [lindex $var $i]
		}
		copy {$current(frame) marker $tag copy}
		delete {$current(frame) marker $tag delete}
		cut {$current(frame) marker $tag cut}
		font {
		    incr i
		    $current(frame) marker $tag font "\{[lindex $var $i]\}"
		}
		move {
		    $current(frame) marker $tag move \
			[lindex $var [expr $i+1]] [lindex $var [expr $i+2]]
		    incr i 2
		}
		movefront {$current(frame) marker $tag move front}
		moveback {$current(frame) marker $tag move back}
		property {
		    $current(frame) marker $tag property \
			[lindex $var [expr $i+1]] [lindex $var [expr $i+2]]
		    incr i 2
		}
		select {$current(frame) marker $tag select}
	    }
	}

	copy {MarkerCopy}
	cut {MarkerCut}
	paste {
	    set marker(paste,system) [lindex $var [expr $i+1]] 
	    set marker(paste,sky) [lindex $var [expr $i+2]]
	    MarkerPaste 
	    incr i 2
	}
	undo {MarkerUndo}

	composite {CompositeCreate}
	desolve {CompositeDelete}

	template {
	    incr i
	    set fn [file normalize [lindex $var $i]]
	    if {$fn != {}} {
		LoadTemplateMarker $fn
		FileLastFull templatefbox $fn
	    }
	}
	savetemplate {
	    incr i
	    set fn [file normalize [lindex $var $i]]
	    if {$fn != {}} {
		$current(frame) marker save template "\{$fn\}"
		FileLastFull templatefbox $fn
	    }
	}

	command {
	    incr i
	    $current(frame) marker command $marker(format) \
		"\{[lindex $var $i]\}"
	    UpdateGroupDialog
	}

	save {
	    incr i
	    set fn [file normalize [lindex $var $i]]
	    if {$fn != {}} {
		$current(frame) marker save "\{$fn\}" \
		    $marker(format) $marker(system) $marker(sky) \
		    $marker(skyformat) $marker(strip) $marker(wcs)
		FileLastFull markerfbox $fn
	    }
	}

	file -
	load {
	    incr i
	    set fn [file normalize [lindex $var $i]]
	    if {$fn != {}} {
		MarkerLoad $fn
		FileLastFull markerfbox $fn
	    }
	}
	default {
	    set fn [file normalize [lindex $var $i]]
	    if {$fn != {}} {
		MarkerLoad $fn
		FileLastFull markerfbox $fn
	    }
	}
    }
}
