#  Copyright (C) 1999-2005
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc ProjectionDialog {frame id} {
    global marker

    set w ".marker$id"

    # see if we already have a header window visible
    if [winfo exist $w] {
	raise $w
	return
    }

    # variables
    set marker($frame,$id,tcoord) $marker(dialog,dist,system)
    set marker($frame,$id,tlabel) $marker(dialog,dist,system)
    set marker($frame,$id,tformat) $marker(dialog,dist,format)

    # base
    MarkerBaseLineDialog $frame $id 500 250

    # menus
    MarkerBaseDistMenu $frame $id thick Thick ProjectionThickCB tcoord tformat
    ProjectionMethodMenu $frame $id

    # init
    ProjectionThickCB $frame $id

    # Thick
    label $w.ref.f.title2 -text "Thickness"
    entry $w.ref.f.thick -textvariable marker($frame,$id,thick) -width 13
    label $w.ref.f.tformat -relief groove -width 9 -padx 4 \
	-textvariable marker($frame,$id,tlabel)
    grid $w.ref.f.title2 $w.ref.f.thick x $w.ref.f.tformat -padx 4 -sticky w

    # make sure its opened
    ProjectionPlot $frame $id
}

# actions

proc ProjectionClose {frame id} {
    MarkerBaseLineClose $frame $id
}

proc ProjectionApply {frame id} {
    global marker

    $frame marker $id projection \
	$marker($frame,$id,system) $marker($frame,$id,sky) \
	$marker($frame,$id,x) $marker($frame,$id,y) \
	$marker($frame,$id,x2) $marker($frame,$id,y2) \
    	$marker($frame,$id,thick) $marker($frame,$id,method) \
	$marker($frame,$id,tcoord) $marker($frame,$id,tformat)

    MarkerBaseLineApply $frame $id
}

# callbacks

proc ProjectionCoordCB {frame id} {
    global marker

    MarkerBaseCoordCB $frame $id
    ProjectionEditCB $frame $id

    $frame marker $id projection system $marker($frame,$id,system) \
	$marker($frame,$id,sky)
    ProjectionPlot $frame $id
    ProjectionXAxisTitle $frame $id
}

proc ProjectionEditCB {frame id} {
    global marker

    MarkerBaseLineEditCB $frame $id

    set marker($frame,$id,method) [$frame get marker $id projection method]

    set marker($frame,$id,dist) \
	[$frame get marker $id projection length \
	     $marker($frame,$id,dcoord) $marker($frame,$id,dformat)]

    set marker($frame,$id,thick) \
	[$frame get marker $id projection thick \
	     $marker($frame,$id,tcoord) $marker($frame,$id,tformat)]

    set marker($frame,$id,angle) \
	[$frame get marker $id angle \
		$marker($frame,$id,system) $marker($frame,$id,sky)]
}

proc ProjectionDistCB {frame id} {
    global marker

    MarkerBaseDistCB $frame $id

    set marker($frame,$id,dist) \
	[$frame get marker $id projection length \
	     $marker($frame,$id,dcoord) $marker($frame,$id,dformat)]
}

proc ProjectionThickCB {frame id} {
    global marker
    AdjustDist $frame marker($frame,$id,tcoord)

    set marker($frame,$id,tlabel) $marker($frame,$id,tcoord)
    switch -- $marker($frame,$id,tcoord) {
	image -
	physical -
	amplifier -
	detector {}
	default {
	    if [$frame has wcs $marker($frame,$id,tcoord)] {
		if [$frame has wcs equatorial $marker($frame,$id,tcoord)] {
		    set marker($frame,$id,tlabel) $marker($frame,$id,tformat)
		} else {
		    set name [$frame get wcs name $marker($frame,$id,tcoord)]
		    if {$name != ""} {
			set marker($frame,$id,tlabel) $name
		    }
		}
	    }
	}
    }

    set marker($frame,$id,thick) \
	[$frame get marker $id projection thick \
	     $marker($frame,$id,tcoord) $marker($frame,$id,tformat)]
}

# support

proc ProjectionMethodMenu {frame id} {
    global menu
    global marker

    set mb ".mb$id"

    $mb add cascade -label Method -menu $mb.method
    menu $mb.method -tearoff 0 -selectcolor $menu(selectcolor)
    $mb.method add radiobutton -label "Average" \
	-variable marker($frame,$id,method) -value 1 \
	-command "ProjectionApply $frame $id"
    $mb.method add radiobutton -label "Sum" \
	-variable marker($frame,$id,method) -value 0 \
	-command "ProjectionApply $frame $id"
}

# Plot

proc ProjectionPlotInit {id} {
    global current

    ProjectionPlot $current(frame) $id

    $current(frame) marker $id callback move ProjectionPlot $current(frame)
    $current(frame) marker $id callback edit ProjectionPlot $current(frame)
    $current(frame) marker $id callback update ProjectionPlot $current(frame)
    $current(frame) marker $id callback delete \
	ProjectionPlotDelete $current(frame)
}

# hardcoded into ds9parser.Y
proc ProjectionPlot {frame id} {
    global marker 
    global ap

    set tt proj${id}${frame}
    set xx proj${id}${frame}x
    set yy proj${id}${frame}y
    set xe proj${id}${frame}xe
    set ye proj${id}${frame}ye

    set xc proj${id}${frame}xc
    set yc proj${id}${frame}yc

    global $xx $yy $xe $ye $xc $yc
    if {[APPing $tt]} {
	$frame get marker $id projection $xx $yy $xc $yc 
    } else {
	APDialog $tt Projection Projection "physical coordinates" Counts
	$ap($tt,graph) xaxis configure -command "ProjectionXAxis $frame $id"

	set ap($tt,xdata) $xx
	set ap($tt,ydata) $yy
	set ap($tt,xedata) $xe
	set ap($tt,yedata) $ye

	blt::vector create $xx $yy $xe $ye $xc $yc
	$frame marker $id projection system physical fk5
	$frame get marker $id projection $xx $yy $xc $yc
	
	ProjectionXAxisTitle $frame $id
	APExternal $tt
    }
}

# hardcoded into ds9parser.Y
proc ProjectionPlotDelete {frame id} {
    set xc proj${id}${frame}xc
    set yc proj${id}${frame}yc

    global $xc $yc
    catch {blt::vector destroy $xc $yc}
    # clear any errors
    global errorInfo
    set errorInfo {}

    APDestroy proj${id}${frame}
}

proc ProjectionStats {frame id} {
    global ap

    set yy proj${id}${frame}y
    global $yy

    set min [format "%6.3f" [blt::vector expr min($yy)]]
    set max [format "%6.3f" [blt::vector expr max($yy)]]
    set mean [format "%6.3f" [blt::vector expr mean($yy)]]
    set median [format "%6.3f" [blt::vector expr median($yy)]]
    
    set var [format "%6.3f" [expr [blt::vector expr var($yy)]]]
    set sdev [format "%6.3f" [expr [blt::vector expr sdev($yy)]]]
    
    set title {}
    append title "min $min max $max"
    append title "\nmean $mean median $median"
    append title "\nvar $var sdev $sdev"

    set tt proj${id}${frame}
    $ap($tt,graph) configure -title $title
}

proc ProjectionXAxisTitle {frame id} {
    global ap

    set s [$frame get marker $id projection system]
    set system [lindex $s 0]
    set sky  [lindex $s 1]
    switch -- $system {
	image -
	physical -
	amplifier -
	detector {set title "$system coordinates"}
	default {
	    if {[$frame has wcs equatorial $system]} {
		set title "$sky coordinates"
	    } else {
		set title "[$frame get wcs name $system] coordinates"
	    }
	}
    }
    set tt proj${id}${frame}
    $ap($tt,graph) xaxis configure -title $title
}

proc ProjectionXAxis {frame id w x} {
    set xc proj${id}${frame}xc
    set yc proj${id}${frame}yc

    global $xc $yc
    # sometimes, $x equals $xc length
    if {$x<[$xc length]} {
	set a [format "%6.3f" [expr "$$xc\($x\)"]]
	set b [format "%6.3f" [expr "$$yc\($x\)"]]

	return "$a\n$b"
    } else {
	return {}
    }
}

proc ProjectionDefaultDialog {} {
    global marker
    global ed

    set w ".projection"

    set ed(ok) 0
    set ed(thick) $marker(projection,thick)

    DialogCreate $w "Default Projection" -borderwidth 2
    frame $w.ed  -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.ed $w.buttons -fill x -ipadx 4 -ipady 4

    label $w.ed.title -text "Thickness"
    entry $w.ed.thick -textvariable ed(thick) -width 10
    label $w.ed.unit -text "image" -relief groove -width 9
    
    grid $w.ed.title $w.ed.thick $w.ed.unit -padx 4 -sticky w

    button $w.buttons.ok -text "OK" -default active \
	-command {set ed(ok) 1}
    button $w.buttons.cancel -text "Cancel" -command {set ed(ok) 0}
    pack $w.buttons.ok -side left -padx 10
    pack $w.buttons.cancel -side right -padx 10

    bind $w <Return> {set ed(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    $w.ed.thick select range 0 end
    DialogWait $w ed(ok) $w.ed.thick
    DialogDismiss $w

    if {$ed(ok)} {
	set marker(projection,thick) $ed(thick)
    }

    unset ed
}
