// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#ifndef __frame_h__
#define __frame_h__

#include "framebase.h"
#include "colorscale.h"

// Frame

class Frame : public FrameBase {
 protected:
  FitsImage* fits;           // list of fits images
  int mosaicCount;             // count of fits images
  int sliceCount;             // count of fits images

  int cmapID;                // current colormap id
  float bias;                // current colormap bias
  float contrast;            // current colormap contrast
  int invert;                // current colormap invert

  int colorCount;            // number of dynamic colors
  ColorScale* colorScale;    // current color scale
  unsigned char* colorCells; // current color values
  unsigned short* indexCells; // current color indices

  FrScale frScale;           // scale parameters
  MosaicType mosaicType;
  CoordSystem mosaicSystem;

  Vector WCScdelt;
  double WCSrot;
  Orientation WCSorientation;
  BBox WCSbb;                // WCS bbox in IMAGE coords
  Matrix WCSmatrix;

  Contour* contour;

 private:
  void buildXImage(XImage*, int, int, int, int,
		   double* (FitsImage::*)());

  void psLevel1(PSColorSpace, int, int, float);
  void psLevel2(PSColorSpace, int, int, float);

  void psLevel1Single(PSColorSpace, int, int, float);
  void psLevel1Mosaic(PSColorSpace, int, int, float);
  void psLevel1MosaicFast(PSColorSpace, int, int, float);

  void reset();
  void setCurrentFits(FitsImage* f) {currentFits = f;}
  void setBinCursor(const Vector&);
  void analysisFits();
  void updateBin(const Matrix&);

  BBox imageBBox();
  void align();

  void updateMatrices();
  void updateMagnifierMatrices(Matrix&);
  int doRender() {return currentFits ? 1 : 0;}
  int isIIS();
  void unloadFits();
  int validColorScale() {return colorScale ? 1 : 0;}

 protected:
  void unloadAllFits() {unloadFits();}
  void updateColorCells(unsigned short*, unsigned char*, int);

  virtual void fillXImage(XImage*, int, int, int, int,
			  double* (FitsImage::*)()) =0;
  virtual void fillXImageMosaic(XImage*, int, int, int, int,
				double* (FitsImage::*)()) =0;
  virtual void fillXImageMosaicFast(XImage*, int, int, int, int,
				    double* (FitsImage::*)()) =0;

  virtual void psLevel2Single(PSColorSpace, int, int, float) =0;
  virtual void psLevel2Mosaic(PSColorSpace, int, int, float) =0;
  virtual void psLevel2MosaicFast(PSColorSpace, int, int, float) =0;

 public:
  Frame(Tcl_Interp*, Tk_Canvas, Tk_Item*);
  virtual ~Frame();

  // Fits Commands

  void getFitsSizeCmd();
  void getFitsSizeCmd(CoordSystem, SkyFormat);
  void zoomToFitCmd(double =1);
  void sliceCmd(int);

  // Colormap Commands

  virtual void colormapCmd(int, float, float, int, unsigned short*, 
			   unsigned char*, int) =0;
  virtual void colormapMotionCmd(int, float, float, int, unsigned short*,
				 unsigned char*, int) =0;
  virtual void colormapEndCmd(int, float, float, int, unsigned short*,
			      unsigned char*, int) =0;
  void getColormapCmd();
  void getTypeCmd();

  // Pan Zoom Rotate Orient Commands

  void getWCSZoomCmd(CoordSystem, Precision);
  void wcsZoomCmd(CoordSystem, const Vector&);

  // IIS Commands

  void iisCmd(int, int);
  void iisEraseCmd();
  void iisGetCmd(char*, int, int, int, int);
  void iisSetCmd(const char*, int, int, int, int);
  void iisWCSCmd(const Matrix&, const Vector&, int);

  // Load Commands

  void loadRGBCubeAllocCmd(const char*) {}
  void loadRGBCubeAllocGZCmd(const char*) {}
  void loadRGBCubeChannelCmd(const char*, const char*) {}
  void loadRGBCubeMMapCmd(const char*, LoadMethod) {}
  void loadRGBCubeSMMapCmd(const char*, const char*, LoadMethod) {}
  void loadRGBCubeMMapIncrCmd(const char*, LoadMethod) {}
  void loadRGBCubeShareCmd(ShmType, int, const char*, LoadMethod) {}
  void loadRGBCubeSShareCmd(ShmType, int, int, const char*, LoadMethod) {}
  void loadRGBCubeSocketCmd(int, const char*) {}
  void loadRGBCubeSocketGZCmd(int, const char*) {}
  void loadRGBCubeVarCmd(const char*, const char*, LoadMethod) {}

  void loadRGBCubeSlaveCmd(const char*, FitsFile*) {}

  void loadRGBImageAllocCmd(const char*) {}
  void loadRGBImageAllocGZCmd(const char*) {}
  void loadRGBImageChannelCmd(const char*, const char*) {}
  void loadRGBImageMMapCmd(const char*, LoadMethod) {}
  void loadRGBImageMMapIncrCmd(const char*, LoadMethod) {}
  void loadRGBImageShareCmd(ShmType, int, const char*,LoadMethod) {}
  void loadRGBImageSocketCmd(int, const char*) {}
  void loadRGBImageSocketGZCmd(int, const char*) {}
  void loadRGBImageVarCmd(const char*, const char*, LoadMethod) {}

  void loadArrRGBCubeAllocCmd(const char*) {}
  void loadArrRGBCubeAllocGZCmd(const char*) {}
  void loadArrRGBCubeChannelCmd(const char*, const char*) {}
  void loadArrRGBCubeMMapCmd(const char*) {}
  void loadArrRGBCubeMMapIncrCmd(const char*) {}
  void loadArrRGBCubeShareCmd(ShmType, int, const char*) {}
  void loadArrRGBCubeSocketCmd(int, const char*) {}
  void loadArrRGBCubeSocketGZCmd(int, const char*) {}
  void loadArrRGBCubeVarCmd(const char*, const char*) {}

  // RGB Commands

  void getRGBChannelCmd();
  void setRGBChannelCmd(const char*) {}
  void getRGBViewCmd();
  void setRGBViewCmd(int, int, int) {}
  void getRGBSystemCmd();
  void setRGBSystemCmd(CoordSystem) {}
};

#endif
