// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#ifndef __marker_h__
#define __marker_h__

#include <string.h>

#include <iostream>
#include <sstream>
#include <iomanip>
using namespace std;

#include <tk.h>

#include "vector.h"
#include "callback.h"
#include "list.h"
#include "coord.h"

class FrameBase;
class FitsImage;

class Tag {
  char* tag_;

  Tag* previous_;
  Tag* next_;

 public:
  Tag(const char*);
  Tag(const Tag&);
  Tag& operator=(const Tag&);
  ~Tag();

  const char* tag() {return tag_;}
  void set(const char*);
  Tag* previous() {return previous_;}
  void setPrevious(Tag* t) {previous_ = t;}
  Tag* next() {return next_;}
  void setNext(Tag* t) {next_ = t;}
};

class Marker {
public:
  enum Property {NONE=0, SELECT=1, HIGHLITE=2, EDIT=4, MOVE=8, ROTATE=16, 
		 DELETE=32, FIXED=64, INCLUDE=128, SOURCE=256};
  // Select-- user may select the marker
  // Highlite-- user may highlite the marker
  // Edit-- user may edit the marker
  // Move-- user may move the marker
  // Rotate-- user may rotate the marker
  // Delete-- user may delete the marker
  // Fixed-- marker is fixed in size (not scaled based on zoom)
  // Include-- marker is 'included' or 'excluded' ie '+' or '-'
  // Source-- marker is a 'source' or 'background' marker

  enum RenderMode {SRC,XOR};

protected:
  int id;
  char type[64];
  FrameBase* parent;

  Vector center;                // ref coordinates
  BBox bbox;                    // canvas coordinates
  BBox allBBox;                 // canvas coordinates

  double angle;                 // radians
  Matrix flip_;            // non iis needs a flip in Y

  Vector* handle;               // canvas coordinates 
  int numHandle;

  char* colorName;
  int color;
  int lineWidth;
  unsigned short properties;
  int selected;
  int highlited;

  char* text;
  Tk_Font font;              // Tk font
  char* comment;

  Display* display;
  GC gc;

  int doCB;

  List<Tag> tags;
  List<CallBack> callbacks;

  Marker* previous_;
  Marker* next_;

protected:
  Vector* arrow(const Vector&, const Vector&);
  void renderArrow(Drawable, const Vector&, const Vector&);
  void psArrow(const Vector&, const Vector&);

  void setGC(RenderMode);
  virtual void setLineDash();
  void setLineNoDash();
  virtual void psLineDash();
  void psLineNoDash();

  double calcAngle();
  double calcAngle(double);
  void psColor(int, const char*);
  void calcAllBBox();
  void renderHandles(Drawable);

  void listPre(ostream&, CoordSystem, SkyFrame, FitsImage*, int, int);
  virtual void listPost(ostream&, int, int);

  void listCiaoPre(ostream&);
  void listCiaoPost(ostream&, int, int);

  void listProsPost(ostream&, int);

  void listSAOtngPre(ostream&, int);
  void listSAOtngPost(ostream&, int);

  void listSAOimagePre(ostream&);
  void listSAOimagePost(ostream&, int);

  void listProps(ostream&);
  void listProperties(ostream&, int);

  virtual Matrix fwdRefMatrix();
  virtual Matrix fwdCanvasMatrix();
  virtual Matrix bckRefMatrix();
  virtual Matrix bckCanvasMatrix();

public:
  Marker(const Marker&);
  Marker(FrameBase* p, const Vector& v, double ang,
	 const char* clr, int w, const char* f, 
	 const char* t, unsigned short prop, const char* c, 
	 const List<Tag>& tag, const List<CallBack>& cb);
  virtual ~Marker();

  void renderText(Drawable, const Matrix&, RenderMode);
  virtual void render(Drawable, const Matrix&, RenderMode) =0;
  virtual Marker* dup() =0;
  void newIdentity();

  virtual void updateBBox() =0;
  virtual void updateCoords(const Matrix&);

  virtual void draw(Drawable);
  virtual void drawXOR();
  virtual void drawMagnifier(const Matrix&);

  virtual void moveTo(const Vector& v);

  virtual void moveBegin();
  virtual void move(const Vector& v);
  virtual void moveEnd();

  virtual void editBegin(int);
  virtual void edit(const Vector& v, int h);
  virtual void editEnd();

  virtual void rotateBegin();
  virtual void rotate(const Vector& v, int h);
  virtual void rotateEnd();

  virtual void ps(int);
  virtual int isIn(const Vector& v) {return bbox.isIn(v);}

  void setAngle(double);
  double getAngle() {return angle;}

  int getNumHandle() {return numHandle;}
  Vector getHandle(int);

  virtual int isPolygon() {return 0;}

  virtual void list(ostream&, CoordSystem, SkyFrame, SkyFormat, int, int) =0;
  virtual void listCiao(ostream&, CoordSystem, int, int) {}
  virtual void listSAOtng(ostream&,CoordSystem,SkyFrame,SkyFormat, int) {}
  virtual void listPros(ostream&, CoordSystem, SkyFrame, SkyFormat, int) {}
  virtual void listSAOimage(ostream&, int) {}
  virtual void listXY(ostream&, CoordSystem, SkyFrame, SkyFormat, int);

  const BBox& getBBox() {return bbox;}
  const BBox& getAllBBox() {return allBBox;}
  int isVisible(const BBox&);

  void select();
  void unselect();
  void toggleSelect();
  int isSelected() {return selected;}

  void highlite();
  void unhighlite();
  void toggleHighlite();
  int isHighlited() {return highlited;}

  void setColor(const char*);
  const char* getColorName() {return colorName;}

  void setLineWidth(int);
  int getLineWidth() {return lineWidth;}

  void setText(const char*);
  const char* getText() {return text;}
  void setFont(const char*);
  const char* getFont();

  void addTag(const char*);
  void editTag(const char*, const char*);
  void deleteTags();
  void deleteTag(int);
  void deleteTag(const char*);
  const char* getTag();
  const char* getNextTag();
  const char* getTag(int);
  int hasTag(const char*);

  int onHandle(const Vector& v);

  int getId() {return id;}
  const char* getType() {return type;}
  virtual Vector getCenter() {return center;}
  double mapLenFromRef(double, CoordSystem, SkyFormat);

  int canSelect() {return (properties & SELECT) ? 1:0;}
  int canHighlite() {return (properties & HIGHLITE) ? 1:0;}
  int canEdit() {return (properties & EDIT) ? 1:0;}
  int canMove() {return (properties & MOVE) ? 1:0;}
  int canRotate() {return (properties & ROTATE) ? 1:0;}
  int canDelete() {return (properties & DELETE) ? 1:0;}
  int isFixed() {return (properties & FIXED) ? 1:0;}
  void setProperty(unsigned short, int);
  int getProperty(unsigned short);
  unsigned short getProperty() {return properties;}

  int addCallBack(CallBack::Type, const char*, const char*);
  int deleteCallBack(CallBack::Type, const char*);
  void deleteCallBack(CallBack::Type);
  void doCallBack(CallBack::Type);

  Marker* previous() {return previous_;}
  void setPrevious(Marker* m) {previous_ = m;}
  Marker* next() {return next_;}
  void setNext(Marker* m) {next_ = m;}

  void enableCB() {doCB = 1;}
  void disableCB() {doCB = 0;}
  void deleteCBs();

  // special composite funtionality
  virtual void setComposite(const Matrix&, double);
  void setComposite(unsigned short, const char*, int, int);
};

#endif
