// Copyright (C) 1999-2012
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <iostream>
#include <sstream>
#include <iomanip>
using namespace std;

#include "file.h"
#include "outfile.h"
#include "outchannel.h"
#include "outsocket.h"

// Parser Stuff

#undef yyFlexLexer
#define yyFlexLexer ffFlexLexer
#include <FlexLexer.h>

ffFlexLexer* fflexx = NULL;    // used by frlex
static FitsFile* ff = NULL;    // used by frerror
extern int ffparse(void*);

int fflex()
{
  return (fflexx ? fflexx->yylex() : 0);
}

void fferror(const char* m)
{
  if (ff) {
    ff->error(m);
    const char* cmd = fflexx ? fflexx->YYText() : (const char*)NULL;
    if (cmd && cmd[0] != '\n') {
      ff->error(": ");
      ff->error(cmd);
    }
  }
}

FitsFile::FitsFile()
{
  primary_ = NULL;
  managePrimary_ = 0;

  head_ = NULL;
  manageHead_ = 1;

  data_ = NULL;

  ext_ = 0;
  inherit_ = 0;
  byteswap_ = lsb();
  orgFits_ = 1;
  valid_ = 0;

  pName_ = NULL;
  pExt_ = NULL;
  pIndex_ = -1;

  pFilter_ = NULL;
  pBinX_ = NULL;
  pBinY_ = NULL;
  pBinZ_ = NULL;

  pWidth_ = 0;
  pHeight_ = 0;
  pDepth_ = 1;
  pBitpix_ = 0;
  pSkip_ = 0;
  pArch_ = NATIVE;
  
  pHPXOrder_ =-1;
  pHPXSystem_ =-1;
  pHPXLayout_ =-1;
  pHPXColumn_ =-1;
  pHPXQuad_ =-1;

  coord_ =0;
  xvalid_ =0;
  xmin_ =0;
  xmax_ =0;
  yvalid_ =0;
  ymin_ =0;
  ymax_ =0;
  zvalid_ =0;
  zmin_ =0;
  zmax_ =0;
}

FitsFile::~FitsFile()
{
  if (manageHead_ && head_)
    delete head_;

  if (managePrimary_ && primary_)
    delete primary_;

  if (pName_)
    delete [] pName_;

  if (pExt_)
    delete [] pExt_;

  if (pFilter_)
    delete [] pFilter_;

  if (pBinX_)
    delete [] pBinX_;

  if (pBinY_)
    delete [] pBinY_;

  if (pBinZ_)
    delete [] pBinZ_;
}

void FitsFile::parse(const char* fn)
{
  if (fn) {
    string x(fn);
    istringstream str(x);
    fflexx = new ffFlexLexer(&str);
    ff = this;
    ffparse(this);

    delete fflexx;
    fflexx = NULL;
  }

  if (!pBinX_ && !pBinY_) {
    char *env;
    if ((env = getenv("DS9_BINKEY"))) {
      string x(env);
      istringstream str(x);
      fflexx = new ffFlexLexer(&str);
      ff = this;
      ffparse(this);

      delete fflexx;
      fflexx = NULL;
    }
  }

  if (!pWidth_ && !pHeight_ && !pBitpix_) {
    char *env;
    if ((env = getenv("DS9_ARRAY"))) {
      string x(env);
      istringstream str(x);
      fflexx = new ffFlexLexer(&str);
      ff = this;
      ffparse(this);

      delete fflexx;
      fflexx = NULL;
    }
  }
}

void FitsFile::error(const char* m)
{
  // do nothing for now
  // cerr << m << endl;
}

int FitsFile::findEnd(const char* blk)
{
  for (int j=0; j<FTY_BLOCK; j+=FTY_CARDLEN)
    // only check for 4 chars
    if (!strncmp("END ", blk+j,4))
      return 1;

  return 0;
}

void FitsFile::setpName(const char* n)
{
  if (pName_)
    delete [] pName_;

  pName_ = dupstr(n);
}

void FitsFile::setpExt(const char* n)
{
  if (pExt_)
    delete [] pExt_;

  pExt_ = dupstr(n);
}

void FitsFile::setpBinX(const char* x)
{
  if (pBinX_)
    delete [] pBinX_;

  pBinX_ = dupstr(x);
}

void FitsFile::setpBinY(const char* y)
{
  if (pBinY_)
    delete [] pBinY_;

  pBinY_ = dupstr(y);
}

void FitsFile::setpBinZ(const char* z)
{
  if (pBinZ_)
    delete [] pBinZ_;

  pBinZ_ = dupstr(z);
}

void FitsFile::setpFilter(const char* filt)
{
  if (pFilter_)
    delete [] pFilter_;

  pFilter_ = dupstr(filt);
}

Vector FitsFile::getColMinMax(const char* name)
{
  if (isBinTable()) {
    FitsTableHDU* hdu = (FitsTableHDU*)(head()->hdu());
    FitsColumn* col = hdu->find(name);
    if (col) {
      if (!col->hasMinMax()) {
	double zmin =  DBL_MAX;
	double zmax = -DBL_MIN;
	int rowlen = hdu->width();
	int numrow = hdu->rows();

	char* ptr = (char*)data();
	for (int i=0; i<numrow; i++, ptr+=rowlen) {
	  // for memory models that support internal paging
	  ptr = page(ptr, rowlen);

	  register double z = col->value(ptr);
	  if (z < zmin)
	    zmin = z;
	  if (z > zmax)
	    zmax = z;
	}
	// for memory models that support internal paging
	resetpage();

	col->setMin(zmin);
	col->setMax(zmax);
	return Vector(zmin,zmax);
      }
      else
	return Vector(col->getMin(), col->getMax());
    }
  }
  return Vector();
}

void FitsFile::setColMinMax(const char* name, const Vector& lim)
{
  if (isBinTable()) {
    FitsTableHDU* hdu = (FitsTableHDU*)(head()->hdu());
    FitsColumn* col = hdu->find(name);
    if (col) {
      Vector ll=lim;
      col->setMin(ll[0]);
      col->setMax(ll[1]);
    }
  }
}

Vector FitsFile::getColDim(const char* name)
{
  if (isBinTable()) {
    FitsTableHDU* hdu = (FitsTableHDU*)(head()->hdu());
    FitsColumn* col = hdu->find(name);
    if (col) {
      if (col->hasTLMinTLMax()) {
	Vector lim = col->dimension();
	col->setMin(lim[0]);
	col->setMax(lim[1]);
	return lim;
      }
      else
	return getColMinMax(name);
    }
  }
  else
    return Vector();
}

int FitsFile::validArrayParams()
{
  if (!pWidth_ || !pHeight_ || !pBitpix_)
    return 0;

  // check for valid bitpix
  switch (pBitpix_) {
  case 8:
  case 16:
  case -16:
  case 32:
  case 64:
  case -32:
  case -64:
    break;
  default:
    return 0;
  }

  return 1;
}

void FitsFile::setArrayByteSwap()
{
  switch (pArch_) {
  case NATIVE:
    byteswap_ = 0;
    break;
  case BIGENDIAN:
    byteswap_ = lsb();
    break;
  case LITTLEENDIAN:
    byteswap_ = !lsb();
    break;
  }
}

int FitsFile::find(const char* name, FitsHead* alt)
{
  if (alt)
    return alt->find(name) ? 1 : 0;

  if (head_)
    if (head_->find(name))
      return 1;
    else
      if (primary_ && inherit_)
	if (primary_->find(name))
	  return 1;

  return 0;
}

int FitsFile::getLogical(const char* name, int def, FitsHead* alt)
{
  if (alt)
    return alt->getLogical(name,def);

  if (head_) {
    int r = head_->getLogical(name,def);
    if (r != def)
      return r;
    else
      if (primary_ && inherit_)
	return primary_->getLogical(name,def);
  }

  return def;
}

int FitsFile::getInteger(const char* name, int def, FitsHead* alt)
{
  if (alt)
    return alt->getInteger(name,def);

  if (head_) {
    int r = head_->getInteger(name,def);
    if (r != def)
      return r;
    else
      if (primary_ && inherit_)
	return primary_->getInteger(name,def);
  }

  return def;
}

double FitsFile::getReal(const char* name, double def, FitsHead* alt)
{
  if (alt)
    return alt->getReal(name,def);

  if (head_) {
    double r = head_->getReal(name,def);
    if (r != def)
      return r;
    else
      if (primary_ && inherit_)
	return primary_->getReal(name,def);
  }

  return def;
}

void FitsFile::getComplex(const char* name, double* real, double* img,
			  double rdef, double idef, FitsHead* alt)
{
  if (alt) {
    alt->getComplex(name, real, img, rdef, idef);
    return;
  }

  if (head_) {
    head_->getComplex(name, real, img, rdef, idef);
    if (*real != rdef || *img != idef)
      return;
    else
      if (primary_ && inherit_) {
	primary_->getComplex(name, real, img, rdef, idef);
	return;
      }
  }

  *real = rdef;
  *img = idef;
}

char* FitsFile::getString(const char* name, FitsHead* alt)
{
  if (alt)
    return alt->getString(name);

  if (head_) {
    char* r = head_->getString(name);
    if (r)
      return r;
    else
      if (primary_ && inherit_)
	return primary_->getString(name);
  }

  return NULL;
}

char* FitsFile::getComment(const char* name, FitsHead* alt)
{
  if (alt)
    return alt->getComment(name);

  if (head_) {
    char* r = head_->getComment(name);
    if (r)
      return r;
    else
      if (primary_ && inherit_)
	return primary_->getComment(name);
  }

  return NULL;
}

char* FitsFile::getKeyword(const char* name, FitsHead* alt)
{
  if (alt)
    return alt->getKeyword(name);

  if (head_) {
    char* r = head_->getKeyword(name);
    if (r)
      return r;
    else
      if (primary_ && inherit_)
	return primary_->getKeyword(name);
  }

  return NULL;
}

// save fits image -- as a primary hdu image

int FitsFile::saveFitsImageFile(const char* fn, int compress)
{
  if (!valid_ || !isImage())
    return 0;

  if (!compress) {
    OutFitsFile str(fn);
    if (str.valid())
      return saveFitsImage(str);
  }
  else {
    OutFitsFileGZ str(fn);
    if (str.valid())
      return saveFitsImage(str);
  }
  return 0;
}

int FitsFile::saveFitsImageChannel(Tcl_Interp* interp, const char* ch, 
				   int compress)
{
  if (!valid_ || !isImage())
    return 0;

  OutFitsChannel str(interp, ch);
  if (str.valid())
    return saveFitsImage(str);
  return 0;
}

int FitsFile::saveFitsImageSocket(int s, int compress)
{
  if (!valid_ || !isImage())
    return 0;

  if (!compress) {
    OutFitsSocket str(s);
    if (str.valid())
      return saveFitsImage(str);
  }
  else {
    OutFitsSocketGZ str(s);
    if (str.valid())
      return saveFitsImage(str);
  }
  return 0;
}

int FitsFile::saveFitsImage(OutFitsStream& str)
{
  // write header
  // we have a problem here. the header may be an xtension, so lets force a
  // first line of 'SIMPLE' and then write the rest of the header
  {
    char buf[80];
    memset(buf,' ',80);
    memcpy(buf,"SIMPLE  =                    T / Fits standard",46);
    str.write(buf, 80);

    // we could have another problem with cubes. catch any NAXIS keywords
    char* ptr = head()->cards()+80;
    char* end = head()->cards() + head()->headbytes();
    while (ptr<end) {
      if (!strncmp(ptr,"NAXIS ",6)) {
	memset(buf,' ',80);
	memcpy(buf,"NAXIS   =                    2 / number of data axes",52);
	str.write(buf, 80);
      } 
      else if (!strncmp(ptr,"NAXIS3",6)) {
	memset(buf,' ',80);
	memcpy(buf,"NAXIS3  =                    0 / length of data axis",52);
	str.write(buf, 80);
      } 
      else
	str.write(ptr, 80);

      ptr+=80;
    }
  }

  // write valid data
  // our data may be short (mmap or bad fits), so write valid data
  // then write the pad, becareful with arch, if array

  FitsImageHDU* hdu = (FitsImageHDU*)(head()->hdu());
  size_t imgbytes = hdu->imgbytes();
  size_t allbytes = imgbytes;
  size_t datablocks = (allbytes + (FTY_BLOCK-1))/FTY_BLOCK;
  size_t databytes = datablocks * FTY_BLOCK;
  size_t padbytes = databytes - allbytes;

  if (orgFits_)
    str.write((char*)data(), imgbytes);
  else {
    switch (pArch_) {
    case NATIVE:
      if (!lsb())
	str.write((char*)data(), imgbytes);
      else
	str.writeSwap((char*)data(), imgbytes, head()->bitpix());
      break;
    case BIGENDIAN:
      str.write((char*)data(), imgbytes);
      break;
    case LITTLEENDIAN:
      str.writeSwap((char*)data(), imgbytes, head()->bitpix());
      break;
    }
  }

  // we may need to add a buffer to round out to block size
  int diff = padbytes;
  if (diff>0) {
    char* buf = new char[diff];
    memset(buf,'\0',diff);
    str.write(buf, diff);
    delete [] buf;
  }
  return 1;
}

int FitsFile::saveArrayFile(const char* fn, ArchType endian)
{
  if (!valid_ || !isImage())
    return 0;

  OutFitsFile str(fn);
  if (str.valid())
    return saveArray(str, endian);

  return 0;
}

int FitsFile::saveArrayChannel(Tcl_Interp* interp, const char* ch, 
			       ArchType endian)
{
  if (!valid_ || !isImage())
    return 0;

  OutFitsChannel str(interp, ch);
  if (str.valid())
    return saveArray(str, endian);

  return 0;
}

int FitsFile::saveArraySocket(int s, ArchType endian)
{
  if (!valid_ || !isImage())
    return 0;

  OutFitsSocket str(s);
  if (str.valid())
    return saveArray(str, endian);

  return 0;
}

int FitsFile::saveArray(OutFitsStream& str, ArchType endian)
{
  switch (endian) {
  case NATIVE:
  case BIGENDIAN:
    {
      if (orgFits_)
	str.write((char*)data(), head()->allbytes());
      else {
	switch (pArch_) {
	case NATIVE:
	  if (!lsb())
	    str.write((char*)data(), head()->allbytes());
	  else
	    str.writeSwap((char*)data(), head()->allbytes(), head()->bitpix());
	  break;
	case BIGENDIAN:
	  str.write((char*)data(), head()->allbytes());
	  break;
	case LITTLEENDIAN:
	  str.writeSwap((char*)data(), head()->allbytes(), head()->bitpix());
	  break;
	}
      }
    }
    break;
  case LITTLEENDIAN:
    {
      if (orgFits_)
	str.writeSwap((char*)data(), head()->allbytes(), head()->bitpix());
      else {
	switch (pArch_) {
	case NATIVE:
	  if (!lsb())
	    str.writeSwap((char*)data(), head()->allbytes(), head()->bitpix());
	  else
	    str.write((char*)data(), head()->allbytes());
	  break;
	case BIGENDIAN:
	  str.writeSwap((char*)data(), head()->allbytes(), head()->bitpix());
	  break;
	case LITTLEENDIAN:
	  str.write((char*)data(), head()->allbytes());
	  break;
	}
      }
    }
    break;
  }

  return 1;
}

// save Fits Table -- save priminary hdu, then extension hdu, then table

int FitsFile::saveFitsTableFile(const char* fn, int compress)
{
  if (!valid_ || !isBinTable())
    return 0;

  if (!compress) {
    OutFitsFile str(fn);
    if (str.valid())
      return saveFitsTable(str);
  }
  else {
    OutFitsFileGZ str(fn);
    if (str.valid())
      return saveFitsTable(str);
  }
  return 0;
}

int FitsFile::saveFitsTableChannel(Tcl_Interp* interp, const char* ch, 
				   int compress)
{
  if (!valid_ || !isBinTable())
    return 0;

  OutFitsChannel str(interp, ch);
  if (str.valid())
    return saveFitsTable(str);
  return 0;
}

int FitsFile::saveFitsTableSocket(int s, int compress)
{
  if (!valid_ || !isBinTable())
    return 0;

  if (!compress) {
    OutFitsSocket str(s);
    if (str.valid())
      return saveFitsTable(str);
  }
  else {
    OutFitsSocketGZ str(s);
    if (str.valid())
      return saveFitsTable(str);
  }
  return 0;
}

int FitsFile::saveFitsTable(OutFitsStream& str)
{
  // primary header
  str.write(primary()->cards(), primary()->headbytes());

  // now, ext header
  str.write(head()->cards(), head()->headbytes());

  // write valid data
  // our data may be short (mmap or bad fits), so write valid data
  // then write the pad, becareful with arch, if array

  if (orgFits_)
    str.write((char*)data(), head()->allbytes());
  else {
    switch (pArch_) {
    case NATIVE:
      if (!lsb())
	str.write((char*)data(), head()->allbytes());
      else
	str.writeSwap((char*)data(), head()->allbytes(), head()->bitpix());
      break;
    case BIGENDIAN:
      str.write((char*)data(), head()->allbytes());
      break;
    case LITTLEENDIAN:
      str.writeSwap((char*)data(), head()->allbytes(), head()->bitpix());
      break;
    }
  }

  // we may need to add a buffer to round out to block size
  int diff = head()->padbytes();
  if (diff>0) {
    char* buf = new char[diff];
    memset(buf,'\0',diff);
    str.write(buf, diff);
    delete [] buf;
  }
  return 1;
}


