// Copyright (C) 1999-2012
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "frame3d.h"
#include "fitsimage.h"
#include "ps.h"
#include "NaN.h"
#include "sigbus.h"

#include <pthread.h>

void render3dTimer(void* ptr) {
  int rr = ((Frame3d*)ptr)->updateImage();
  if (rr) {
    Tcl_TimerToken tt = Tcl_CreateTimerHandler(250,render3dTimer,ptr);
    ((Frame3d*)ptr)->setTimer(tt);
  }
  else
    ((Frame3d*)ptr)->setTimer(0);
}

Frame3d::Frame3d(Tcl_Interp* i, Tk_Canvas c, Tk_Item* item) 
  : Frame3dBase(i,c,item)
{
  context = new Context();
  context->parent(this);

  currentContext = context;
  keyContext = context;
  keyContextSet =1;

  cmapID = 1;
  bias = 0.5;
  contrast = 1.0;

  colorCount = 0;
  colorScale = NULL;
  colorCells = NULL;
  indexCells = NULL;

  thread =NULL;
  targ =NULL;
  nrays =0;
  xid =NULL;
  yid =NULL;
  timer =0;

  // scope is alway global
  keyContext->frScale.setClipScope(FrScale::GLOBAL);
}

Frame3d::~Frame3d()
{
  if (context)
    delete context;

  if (colorScale)
    delete colorScale;

  if (colorCells)
    delete [] colorCells;

  if (indexCells)
    delete [] indexCells;

  if (thread)
    delete [] thread;
  if (targ)
    delete [] targ;

  if (xid)
    delete [] xid;
  if (yid)
    delete [] yid;
}

unsigned char* Frame3d::fillImage(int width, int height, Coord::InternalSystem sys)
{
  unsigned char* img;
  switch (sys) {
  case Coord::WIDGET:
    zbuf = &zbufWidget_;
    mkzbuf = &mkzbufWidget_;
    if (syncUpdate)
      img = fillImageJoin(width, height, sys);
    else
      img = fillImageDetach(width, height, sys);
    break;
  case Coord::PANNER:
    zbuf = &zbufPanner_;
    mkzbuf = &mkzbufPanner_;
    img = fillImageJoin(width, height, sys);
    break;
  case Coord::MAGNIFIER:
    zbuf = &zbufMagnifier_;
    mkzbuf = &mkzbufMagnifier_;
    img = fillImageJoin(width, height, sys);
    break;
  case Coord::PS:
    zbuf = &zbufPS_;
    mkzbuf = &mkzbufPS_;
    img = fillImageJoin(width, height, sys);
    break;
  }

  if (!img) {
    internalError("fillImage Error");
    return NULL;
  }

  // no need to carry this around
  switch (sys) {
  case Coord::WIDGET:
  case Coord::PANNER:
    break;
  case Coord::MAGNIFIER:
    if (zbufMagnifier_)
      delete [] zbufMagnifier_;
    zbufMagnifier_ = NULL;
    if (mkzbufMagnifier_)
      delete [] mkzbufMagnifier_;
    mkzbufMagnifier_ = NULL;
    break;
  case Coord::PS:
    if (zbufPS_)
      delete [] zbufPS_;
    zbufPS_ = NULL;
    if (mkzbufPS_)
      delete [] mkzbufPS_;
    mkzbufPS_ = NULL;
    break;
  }

  return img;
}

void* raytrace(void* arg)
{
  t_arg* targ = (t_arg*)arg;
  Frame3dBase::RenderMethod renderMethod = targ->renderMethod;
  int width = targ->width;
  Coord::InternalSystem sys = targ->sys;
  float** zbuf = targ->zbuf;
  unsigned char** mkzbuf = targ->mkzbuf;
  Context* context = targ->context;

  int* xid = targ->xid;
  int* yid = targ->yid;
  int start = targ->start;
  int stop = targ->stop;
  int zstart = targ->zstart;
  int zstop = targ->zstop;

  // this will be incorrect for multiple ext/file cubes
  //  int srcd = context->fits->depth();
  int srcd = context->naxis(2);
  double* mm = context->fits->matrixToData3d(sys).mm();

  FitsImage* ptr = context->fits;

  // if more than 3 axes, walk it forward
  int num = context->calcSlice();
  for (int ii=1; ii<num; ii++)
    if (ptr)
      ptr = ptr->nextSlice();

  // slice jump vector
  FitsImage* sjv[srcd];
  FitsImage* sptr = ptr;
  int ii=0;
  while (sptr) {
    sjv[ii++] = sptr;
    if (sptr)
      sptr = sptr->nextSlice();
  }

  FitsBound* params = 
    context->fits->getDataParams(context->frScale.scanMode());
  int srcw = context->fits->width();

  targ->rays =0;
  for (int ll=start; ll<=stop; ll++, targ->rays++) {
    int& jj = yid[ll];
    int& ii = xid[ll];

    double ii0 = ii*mm[0];
    double ii1 = ii*mm[1];
    double ii2 = ii*mm[2];
    double jj4 = jj*mm[4];
    double jj5 = jj*mm[5];
    double jj6 = jj*mm[6];

    int cnt=0;
    float acc=0;
    float max = -FLT_MAX;

    int kks = zstart;
    int kkt = zstop;

    // good abort point
    if (targ->abort) {
      targ->done =1;
      return 0;
    }

    int inside =0;
    for (int kk=kks; kk<kkt; kk++) {
      double xx = ii0 + jj4 + kk*mm[8]  + mm[12];
      double yy = ii1 + jj5 + kk*mm[9]  + mm[13];
      double zz = ii2 + jj6 + kk*mm[10] + mm[14];

      if (xx>=params->xmin && xx<params->xmax && 
	  yy>=params->ymin && yy<params->ymax &&
	  zz>=params->zmin && zz<params->zmax) {
	float value = sjv[int(zz)]->getValueDouble(long(yy)*srcw+long(xx));
	inside =1;

	// ignore nan
	if (!isnanf(value)) {
	  if (value>max)
	    max = value;
	  cnt++;
	  acc+=value;
	}
      }
      else {
	// early determination
	if (inside)
	  break;
      }
    }

    if (cnt) {
      float* dest = *zbuf + jj*width + ii;
      unsigned char* mkdest = *mkzbuf + jj*width + ii;

      switch (renderMethod) {
      case Frame3dBase::MIP:
	*dest =max;
	break;
      case Frame3dBase::AIP:
	*dest =acc/cnt;
	break;
      }

      *mkdest=1;
    }
  }

  targ->done=1;
  return 0;
}

unsigned char* Frame3d::fillImageJoin(int width, int height, Coord::InternalSystem sys)
{
  if (!*zbuf) {
    *zbuf = new float[width*height];
    if (!(*zbuf)) {
      internalError("fillImageJoin Error");
      return NULL;
    }
    memset(*zbuf, 0, width*height*sizeof(float));

    *mkzbuf = new unsigned char[width*height];
    if (!(*mkzbuf)) {
      internalError("fillImageJoin Error");
      return NULL;
    }
    memset(*mkzbuf, 0, width*height);

    BBox3d bb = imageBounds(width,height,sys);
    int ww = bb.ur[0]-bb.ll[0];
    int hh = bb.ur[1]-bb.ll[1];
    // local var overide
    int nrays = ww*hh;
    float incr = nrays/threads_;
    int* xid = new int[nrays];
    int* yid = new int[nrays];
    int x=bb.ll[0]+.5; // don't know why;
    int y=bb.ll[1]+.5; // don't know why

    // init array
    for (int jj=0; jj<hh; jj++) {
      for (int ii=0; ii<ww; ii++) {
	xid[jj*ww+ii] = ii+x;
	yid[jj*ww+ii] = jj+y;
      }
    }

    // local var overide
    pthread_t thread[threads_];
    t_arg targ[threads_];

    for (int ii=0; ii<threads_; ii++) {
      targ[ii].renderMethod = renderMethod_;
      targ[ii].width = width;
      targ[ii].sys = sys;
      targ[ii].zbuf = zbuf;
      targ[ii].mkzbuf = mkzbuf;
      targ[ii].context = context;

      targ[ii].xid = xid;
      targ[ii].yid = yid;
      targ[ii].start = incr*ii;
      if (ii+1<threads_)
	targ[ii].stop = incr*(ii+1)-1;
      else
	targ[ii].stop = nrays-1;
      targ[ii].zstart = bb.ll[2];
      targ[ii].zstop = bb.ur[2];

      targ[ii].rays =0;
      targ[ii].abort =0;
      targ[ii].done =0;
    }

    for (int ii=0; ii<threads_; ii++) {
      int rr = pthread_create(&thread[ii], NULL, raytrace, &targ[ii]);
      if (rr)
	internalError("Unable to Create Thread");
    }

    // clean up threads
    for (int ii=0; ii<threads_; ii++) {
      int rr = pthread_join(thread[ii], NULL);
      if (rr)
	internalError("Unable to Join Thread");
    }

    if (xid)
      delete [] xid;
    if (yid)
      delete [] yid;
  }

  return fillImageColor(width, height);
}

unsigned char* Frame3d::fillImageDetach(int width, int height,
					Coord::InternalSystem sys)
{
  if (!*zbuf) {
    *zbuf = new float[width*height];
    if (!(*zbuf)) {
      internalError("fillImageDetach Error");
      return NULL;
    }
    memset(*zbuf, 0, width*height*sizeof(float));

    *mkzbuf = new unsigned char[width*height];
    if (!(*mkzbuf)) {
      internalError("fillImageDetach Error");
      return NULL;
    }
    memset(*mkzbuf, 0, width*height);

    BBox3d bb = imageBounds(width,height,sys);
    int ww = bb.ur[0]-bb.ll[0];
    int hh = bb.ur[1]-bb.ll[1];
    nrays = ww*hh;
    float incr = nrays/threads_;
    if (xid)
      delete [] xid;
    xid = new int[nrays];
    if (yid)
      delete [] yid;
    yid = new int[nrays];
    int x=bb.ll[0]+.5; // don't know why
    int y=bb.ll[1]+.5; // don't know why

    // init array
    for (int jj=0; jj<hh; jj++) {
      for (int ii=0; ii<ww; ii++) {
	xid[jj*ww+ii] = ii+x;
	yid[jj*ww+ii] = jj+y;
      }
    }

    // randomize array
    for (int kk=nrays-1; kk>0; kk--) {
      int ll = rand() % (kk+1); // 0 <= ll <= kk
      if (ll!=kk) {
	int tx = xid[kk];
	int ty = yid[kk];
	xid[kk]=xid[ll];
	yid[kk]=yid[ll];
	xid[ll]=tx;
	yid[ll]=ty;
      }
    }

    // init threads
    thread = new pthread_t[threads_];
    targ = new t_arg[threads_];

    for (int ii=0; ii<threads_; ii++) {
      targ[ii].renderMethod = renderMethod_;
      targ[ii].width = width;
      targ[ii].sys = sys;
      targ[ii].zbuf = zbuf;
      targ[ii].mkzbuf = mkzbuf;
      targ[ii].context = context;

      targ[ii].xid = xid;
      targ[ii].yid = yid;
      targ[ii].start = incr*ii;
      if (ii+1<threads_)
	targ[ii].stop = incr*(ii+1)-1;
      else
	targ[ii].stop = nrays-1;

      targ[ii].zstart = bb.ll[2];
      targ[ii].zstop = bb.ur[2];

      targ[ii].rays =0;
      targ[ii].abort =0;
      targ[ii].done =0;
    }

    for (int ii=0; ii<threads_; ii++) {
      int rr = pthread_create(&thread[ii], NULL, raytrace, &targ[ii]);
      if (rr)
	internalError("Unable to Create Thread");
    }

    // start the timer, if needed
    if (!timer)
      render3dTimer(this);
  }

  return fillImageColor(width, height);
}

void Frame3d::cancelImage()
{
  // abort any threads
  if (thread) {
    // set cancel flag
    for (int ii=0; ii<threads_; ii++)
      targ[ii].abort =1;

    // now wait until done
    for (int ii=0; ii<threads_; ii++) {
      int rr = pthread_join(thread[ii], NULL);
      if (rr)
	internalError("Unable to Join Thread");
    }

    delete [] thread;
    thread = NULL;
  }

  if (targ) {
    delete [] targ;
    targ = NULL;
  }

  if (xid)
    delete [] xid;
  xid =NULL;
  if (yid)
    delete [] yid;
  yid =NULL;

  // delete buffers
  if (zbufWidget_)
    delete [] zbufWidget_;
  zbufWidget_ = NULL;
  if (mkzbufWidget_)
    delete [] mkzbufWidget_;
  mkzbufWidget_ = NULL;

  if (zbufPanner_)
    delete [] zbufPanner_;
  zbufPanner_ = NULL;
  if (mkzbufPanner_)
    delete [] mkzbufPanner_;
  mkzbufPanner_ = NULL;

  if (zbufMagnifier_)
    delete [] zbufMagnifier_;
  zbufMagnifier_ = NULL;
  if (mkzbufMagnifier_)
    delete [] mkzbufMagnifier_;
  mkzbufMagnifier_ = NULL;

  if (zbufPS_)
    delete [] zbufPS_;
  zbufPS_ = NULL;
  if (mkzbufPS_)
    delete [] mkzbufPS_;
  mkzbufPS_ = NULL;
}

int Frame3d::updateImage()
{
  // anything running?
  if (!thread)
    return 0;

  // we done yet?
  int sum=0;
  for (int ii=0; ii<threads_; ii++)
    sum += targ[ii].done;

  if (sum == threads_) {
    for (int ii=0; ii<threads_; ii++) {
      int rr = pthread_join(thread[ii], NULL);
      if (rr)
	internalError("Unable to Join Thread");
    }

    nrays =0;
    if (thread)
      delete [] thread;
    thread = NULL;
    if (targ)
      delete [] targ;
    targ = NULL;
    if (xid)
      delete [] xid;
    xid = NULL;
    if (yid)
      delete [] yid;
    yid =NULL;

    // clear progress bar
    Tcl_SetVar2(interp,"threed","status","0",TCL_GLOBAL_ONLY);

    // update image
    updateNow(BASE);
    updateMagnifier();

    return 0;
  }

  //progressbar
  int rays=0;
  for (int ii=0; ii<threads_; ii++)
    rays += targ[ii].rays;

  ostringstream str;
  str << int(float(rays)/nrays*100.) << ends;
  Tcl_SetVar2(interp,"threed","status",str.str().c_str(),
	      TCL_GLOBAL_ONLY);

  // update image
  update(BASE);
  updateMagnifier();

  return 1;
}

unsigned char* Frame3d::fillImageColor(int width, int height)
{
  unsigned char* img = new unsigned char[width*height*3];
  memset(img, 0, width*height*3);

  int length = colorScale->size() - 1;
  const unsigned char* table = colorScale->psColors();

  double ll = keyContext->fits->getLowDouble();
  double hh = keyContext->fits->getHighDouble();
  double diff = hh - ll;

  register unsigned char red = (unsigned char)bgColor->red;
  register unsigned char green = (unsigned char)bgColor->green;
  register unsigned char blue = (unsigned char)bgColor->blue; 

  unsigned char* dest = img;
  float* src = *zbuf;
  unsigned char* mksrc = *mkzbuf;

  for (int jj=0; jj<height; jj++) {
    for (int ii=0; ii<width; ii++, dest+=3, src++, mksrc++) {
      *dest = red;
      *(dest+1) = green;
      *(dest+2) = blue;

      if (*mksrc) {
	float value = *src;

	// will not see nan
	if (value <= ll) {
	  *(dest+2) = table[0];
	  *(dest+1) = table[1];
	  *dest = table[2];
	}
	else if (value >= hh) {
	  *(dest+2) = table[length*3];
	  *(dest+1) = table[length*3+1];
	  *dest = table[length*3+2];
	}
	else {
	  int l = (int)(((value - ll)/diff * length) + .5);
	  *(dest+2) = table[l*3];
	  *(dest+1) = table[l*3+1];
	  *dest = table[l*3+2];
	}
      }
    }
  }

  return img;
}

BBox3d Frame3d::imageBounds(int width, int height, Coord::InternalSystem sys)
{
  Matrix3d mx = keyContext->fits->matrixToData3d(sys).invert() *
    Translate3d(.5,.5,.5);

  FitsBound* params = keyContext->fits->getDataParams(keyContext->frScale.scanMode());
  int& xmin = params->xmin;
  int& xmax = params->xmax;
  int& ymin = params->ymin;
  int& ymax = params->ymax;
  int& zmin = params->zmin;
  int& zmax = params->zmax;

  Vector3d llf = Vector3d(xmin,ymin,zmin) * mx;
  Vector3d lrf = Vector3d(xmax,ymin,zmin) * mx;
  Vector3d urf = Vector3d(xmax,ymax,zmin) * mx;
  Vector3d ulf = Vector3d(xmin,ymax,zmin) * mx;

  Vector3d llb = Vector3d(xmin,ymin,zmax) * mx;
  Vector3d lrb = Vector3d(xmax,ymin,zmax) * mx;
  Vector3d urb = Vector3d(xmax,ymax,zmax) * mx;
  Vector3d ulb = Vector3d(xmin,ymax,zmax) * mx;

  BBox3d bb(llf);
  bb.bound(lrf);
  bb.bound(urf);
  bb.bound(ulf);
  bb.bound(llb);
  bb.bound(lrb);
  bb.bound(urb);
  bb.bound(ulb);
  bb.clip(Vector3d(width,height,zdepth_));
  return bb;
}

void Frame3d::reset()
{
  cmapID = 1;
  bias = 0.5;
  contrast = 1.0;
  keyContext->frScale.resetScanMode();
  keyContext->updateClip();

  Base::reset();
}

void Frame3d::updateColorCells(unsigned short* index, 
			     unsigned char* cells, int cnt)
{
  // the colorbar widget will pass us a pointer to the indexCells

  colorCount = cnt;
  if (indexCells)
    delete [] indexCells;
  indexCells = new unsigned short[cnt];
  if (!indexCells) {
    internalError("Unable to Alloc indexCells");
    return;
  }
  memcpy(indexCells, index, cnt*sizeof(unsigned short));

  // copy the rgb vales to the colorCells array (for postscript printing)

  if (colorCells)
    delete [] colorCells;
  colorCells = new unsigned char[cnt*3];
  if (!colorCells) {
    internalError("Unable to Alloc colorCells");
    return;
  }
  memcpy(colorCells, cells, cnt*3);
}

void Frame3d::pushMatrices()
{
  Base::pushMatrices();

  FitsImage* ptr = keyContext->fits;
  while (ptr) {
    FitsImage* sptr = ptr;
    while (sptr) {
      sptr->updateMatrices(refToWidget3d);
      sptr = sptr->nextSlice();
    }
    ptr = ptr->nextMosaic();
  }
}

void Frame3d::pushPannerMatrices()
{
  Base::pushPannerMatrices();

  FitsImage* ptr = keyContext->fits;
  while (ptr) {
    FitsImage* sptr = ptr;
    while (sptr) {
      sptr->updatePannerMatrices(refToPanner3d);
      sptr = sptr->nextSlice();
    }
    ptr = ptr->nextMosaic();
  }
}

void Frame3d::pushMagnifierMatrices()
{
  Base::pushMagnifierMatrices();

  FitsImage* ptr = keyContext->fits;
  while (ptr) {
    FitsImage* sptr = ptr;
    while (sptr) {
      sptr->updateMagnifierMatrices(refToMagnifier3d);
      sptr = sptr->nextSlice();
    }
    ptr = ptr->nextMosaic();
  }
}

void Frame3d::pushPSMatrices(float scale, int width, int height)
{
  Base::pushPSMatrices(scale, width, height);

  Matrix3d mx = psMatrix(scale, width, height);
  FitsImage* ptr = keyContext->fits;
  while (ptr) {
    FitsImage* sptr = ptr;
    while (sptr) {
      sptr->updatePS(mx);
      sptr = sptr->nextSlice();
    }
    ptr = ptr->nextMosaic();
  }
}

void Frame3d::unloadFits()
{
  if (DebugPerf)
    cerr << "Frame3d::unloadFits" << endl;

  // kill any active threads
  cancelImage();

  keyContext->unload();

  Base::unloadFits();
}

// Commands

void Frame3d::getColorbarCmd()
{
  ostringstream str;
  str << cmapID << ' ' << bias << ' ' << contrast << ' ' << invert << ends;
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}

void Frame3d::getRGBChannelCmd()
{
  Tcl_AppendResult(interp, "red", NULL);
}

void Frame3d::getRGBViewCmd()
{
  Tcl_AppendResult(interp, "1 1 1", NULL);
}

void Frame3d::getRGBSystemCmd()
{
  Tcl_AppendResult(interp, "image", NULL);
}

void Frame3d::getTypeCmd()
{
  Tcl_AppendResult(interp, "3d", NULL);
}
