
// Copyright (c) 1996-2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Malolan Chetlur     
//          Umesh Kumar V. Rajasekaran
//          Narayanan Thondugulam
//	    Magnus Danielson	cfmd@swipnet.se

//---------------------------------------------------------------------------

#include "IIRScram_ConcurrentGenerateForStatement.hh"
#include "IIR_Identifier.hh"
#include "IIR_ConstantDeclaration.hh"
#include "IIR_Label.hh"
#include "IIR_DeclarationList.hh"
#include "IIR_ComponentDeclaration.hh"
#include "IIR_ComponentInstantiationStatement.hh"
#include "IIR_BlockStatement.hh"
#include "IIR_ProcessStatement.hh"
#include "IIR_SignalDeclaration.hh"
#include "IIR_AliasDeclaration.hh"
#include "IIR_ConfigurationSpecification.hh"
#include "IIR_Designator.hh"
#include "IIR_EntityDeclaration.hh"
#include "IIR_ScalarTypeDefinition.hh"
#include "IIR_LibraryDeclaration.hh"
#include "symbol_table.hh"
#include "set.hh"
#include "published_header_file.hh"
#include "published_cc_file.hh"
#include "sstream-wrap.hh"

IIRScram_ConcurrentGenerateForStatement::~IIRScram_ConcurrentGenerateForStatement(){}

void 
IIRScram_ConcurrentGenerateForStatement::_publish_vhdl(ostream &_vhdl_out) {

  _publish_vhdl_stmt_label(_vhdl_out);

  _vhdl_out << " for ";
   get_generate_parameter_specification()->_publish_vhdl(_vhdl_out);
  _vhdl_out << " in ";
  get_generate_parameter_specification()->get_subtype()->_publish_vhdl_range(_vhdl_out);
  _vhdl_out << " generate\n";

  if(block_declarative_part.num_elements() != 0) {
    block_declarative_part._publish_vhdl_decl(_vhdl_out);
  }

  _vhdl_out << "   begin\n";

  concurrent_statement_part._publish_vhdl(_vhdl_out);

  _vhdl_out << "   end generate ";
  if(get_label() != NULL) {
    get_label()->_publish_vhdl(_vhdl_out);
  }
  _vhdl_out << ";\n";

}

void 
IIRScram_ConcurrentGenerateForStatement::_type_check(){
  block_declarative_part._type_check_attribute_specifications( concurrent_statement_part );
  ASSERT( get_generate_parameter_specification() != NULL );  
  get_generate_parameter_specification()->set_subtype( _type_check_iteration_scheme( get_generate_parameter_specification() ) );
}

void 
IIRScram_ConcurrentGenerateForStatement::_type_check_instantiate_statements(){
  // Put our declarations back into scope.
  ASSERT( get_generate_parameter_specification() != NULL );
  _get_symbol_table()->open_scope( this );
  _get_symbol_table()->make_visible( get_generate_parameter_specification() );
  concurrent_statement_part._type_check_instantiate_statements();
  _get_symbol_table()->close_scope( this );
}

// void 
// IIRScram_ConcurrentGenerateForStatement::_reopen_scope( symbol_table *sym_tab ){
//   sym_tab->reopen_scope( this );
// }

void
IIRScram_ConcurrentGenerateForStatement::_publish_cc_generate_elaborate( published_file &_cc_out ){
  IIR_ArchitectureStatement *arch_stmt;
  const string temp = _get_current_architecture_name();
  const string old_current_name = _get_current_publish_name();
  SCRAM_CC_REF( _cc_out, "IIRScram_ConcurrentGenerateForStatement::_publish_cc_generate_elaborate" );
  _set_current_publish_name( "SG" );

  arch_stmt =  concurrent_statement_part.first();

  while (arch_stmt != NULL) {
    // Need to check if generate statements can have other concurrent
    // statements that will need elaboration. If so that check should
    // be added to the following if condition.

    switch (arch_stmt->get_kind()) {
    case IIR_CONCURRENT_GENERATE_FOR_STATEMENT:
    case IIR_BLOCK_STATEMENT:
    case IIR_CONCURRENT_GENERATE_IF_STATEMENT:
      arch_stmt->_publish_cc_elaborate( _cc_out );
      break;

    case IIR_PROCESS_STATEMENT:
    case IIR_COMPONENT_INSTANTIATION_STATEMENT:
      // Prevent the error message from popping up.
      break;
      
    default:
      cerr << "ERROR! IIRScram_ConcurrentGenerateForStatement::"
	   << "_publish_cc_generate_elaborate( _cc_out ): unknown conc_statement "
	   << "type |" << arch_stmt->get_kind_text() << "|" << endl;
      break;
    }
    arch_stmt = concurrent_statement_part.successor(arch_stmt);
  }

  _set_current_architecture_name( temp );
  _set_current_publish_name( old_current_name );
}

void
IIRScram_ConcurrentGenerateForStatement::_publish_cc_init( published_file &_cc_out, IIR *scope ){

  SCRAM_CC_REF( _cc_out, "IIRScram_ConcurrentGenerateForStatement::_publish_cc_init" );

  get_generate_parameter_specification()->_publish_cc_lvalue( _cc_out );
  _cc_out << " = " <<
    get_generate_parameter_specification()->_get_cc_type_name();
  _cc_out << "(ObjectBase::VARIABLE, proc->";
  _publish_cc_scoping_prefix( _cc_out.get_stream(), scope, this );
  _cc_out << "getGenerateConstant(1));\n";
}

void
IIRScram_ConcurrentGenerateForStatement::_publish_cc_universal_generate_range( published_file &_cc_out )
{

  SCRAM_CC_REF( _cc_out, "IIRScram_ConcurrentGenerateForStatement::_publish_cc_universal_generate_range" );

  _cc_out << "(";
  
  if (get_generate_parameter_specification()->get_subtype()->_is_ascending_range()) {
    ((IIR_ScalarTypeDefinition *) get_generate_parameter_specification()->get_subtype())->get_right()->_publish_cc_value( _cc_out );
    _cc_out << " - ";
    ((IIR_ScalarTypeDefinition *) get_generate_parameter_specification()->get_subtype())->get_left()->_publish_cc_value( _cc_out );
    _cc_out << " + 1";
  }
  else {
    ((IIR_ScalarTypeDefinition *) get_generate_parameter_specification()->get_subtype())->get_left()->_publish_cc_value( _cc_out );
    _cc_out << " - ";
    ((IIR_ScalarTypeDefinition *) get_generate_parameter_specification()->get_subtype())->get_right()->_publish_cc_value( _cc_out );
    _cc_out << " + 1";
  }
  
  _cc_out << ")";
}

void
IIRScram_ConcurrentGenerateForStatement::_publish_cc_assign_to_loop_constant( published_file &_cc_out,
									      const char *localIterator){

  SCRAM_CC_REF( _cc_out, "IIRScram_ConcurrentGenerateForStatement::_publish_cc_assign_to_loop_constant" );

  _cc_out << "  assignVariable(";
  get_generate_parameter_specification()->_publish_cc_lvalue( _cc_out );
  _cc_out << ", ";
  get_generate_parameter_specification()->get_subtype()->_publish_cc_kernel_type( _cc_out.get_stream() );
  _cc_out << "(ObjectBase::VARIABLE, " << localIterator
	  << "), nullInfo, nullInfo);\n";
}

IIR_DeclarationList*
IIRScram_ConcurrentGenerateForStatement::_get_declaration_list() {
  return &block_declarative_part;
}

void
IIRScram_ConcurrentGenerateForStatement::_publish_cc_generate_for_loop( published_file &_cc_out )
{
  IIR_ScalarTypeDefinition *range;

  SCRAM_CC_REF( _cc_out, "IIRScram_ConcurrentGenerateForStatement::_publish_cc_generate_for_loop" );

  ASSERT ( get_generate_parameter_specification()->get_subtype()->_is_iir_scalar_type_definition() == TRUE );
  range = (IIR_ScalarTypeDefinition *) get_generate_parameter_specification()->get_subtype();
  
  _cc_out << "  for(i = ";
  range->get_left()->_publish_cc_value( _cc_out );
  _cc_out << "; (i ";

  if (range->_is_ascending_range() == TRUE) {
    _cc_out << " <= ";
  }
  else {
    _cc_out << " >= ";
  }

  range->get_right()->_publish_cc_value( _cc_out );
  _cc_out << "); ";

  if (range->_is_ascending_range() == TRUE) {
    _cc_out << "i++";
  }
  else {
    _cc_out << "i--";
  }

  _cc_out << ")  ";
}

void
IIRScram_ConcurrentGenerateForStatement::_publish_cc( published_file &_cc_out ) {
  ostringstream ent_arch_generate_for;
  ostringstream arch_name;

  IIR *temp                   = _get_current_publish_node();
  const string old_architecture_name = _get_current_architecture_name();
  const string old_current_name      = _get_current_publish_name();
  PublishedUnit saved_publishing_unit = _get_currently_publishing_unit();

  _set_currently_publishing_unit(GENERATE_FOR);
  _set_current_publish_node( this );
  
  ent_arch_generate_for << _get_current_entity_name()
			<< "_" << _get_current_architecture_name();
  _publish_cc_enclosing_stmt_to_architecture_path(ent_arch_generate_for);

  ent_arch_generate_for << *_get_label()->_get_declarator() << "_waits";
  
  published_header_file header_file( _get_work_library()->_get_path_to_directory(), 
				     ent_arch_generate_for.str(),
				     this );

  SCRAM_CC_REF( header_file, "IIRScram_ConcurrentGenerateForStatement::_publish_cc" );

  header_file << "class State;\n\n";

  published_cc_file cc_file( _get_work_library()->_get_path_to_directory(), 
			     ent_arch_generate_for.str(),
			     this );

  SCRAM_CC_REF( _cc_out, "IIRScram_ConcurrentGenerateForStatement::_publish_cc" );
  _set_current_publish_name( "SG" );
  concurrent_statement_part._publish_cc( cc_file );

  _set_current_publish_name( old_current_name );
  _set_current_architecture_name( old_architecture_name );
  
  SCRAM_CC_REF( header_file, "IIRScram_ConcurrentGenerateForStatement::_publish_cc" );
  
  _publish_cc_declarations( header_file );
  
  _set_current_publish_node( temp );

  _set_currently_publishing_unit(saved_publishing_unit);
}

void
IIRScram_ConcurrentGenerateForStatement::_publish_cc_declarations( published_file &_cc_out ) {
  ostringstream ent_arch_generate;

  ent_arch_generate << _get_current_entity_name() << "_"
		    << _get_current_architecture_name();
  _publish_cc_enclosing_stmt_to_architecture_path(ent_arch_generate);
  ent_arch_generate << "_" << *_get_label();

  published_header_file header_file( _get_work_library()->_get_path_to_directory(), 
				     ent_arch_generate.str(),
				     this );
  SCRAM_CC_REF( header_file, "IIRScram_ConcurrentGenerateForStatement::_publish_cc_declarations" );
  
  header_file << "#include\"";
  header_file << _get_current_entity_name() << "_" << _get_current_architecture_name();
  _publish_cc_enclosing_stmt_to_architecture_path( header_file.get_stream() );
  header_file  << ".hh\"\n\n";
  
  block_declarative_part._publish_cc( header_file );
  
  SCRAM_CC_REF( header_file, "IIRScram_ConcurrentGenerateForStatement::_publish_cc_declarations" );

  header_file << "extern " <<
    get_generate_parameter_specification()->_get_cc_type_name();
  header_file << " ";
  get_generate_parameter_specification()->_publish_cc_lvalue( header_file );
  header_file << ";\n";
  
  published_cc_file cc_file( _get_work_library()->_get_path_to_directory(), 
			     ent_arch_generate.str(),
			     this );
  SCRAM_CC_REF( cc_file, "IIRScram_ConcurrentGenerateForStatement::_publish_cc_declarations" );

  cc_file << "#include \"" << ent_arch_generate.str() << ".hh\"\n\n";
  
  block_declarative_part._publish_cc_decl( cc_file );

  // Publish definition for the generate constant guy
  cc_file << get_generate_parameter_specification()->_get_cc_type_name();
  cc_file << " ";
  get_generate_parameter_specification()->_publish_cc_lvalue( cc_file );
  cc_file << "(ObjectBase::VARIABLE);\n";
}

void
IIRScram_ConcurrentGenerateForStatement::_publish_cc_class( published_file &_cc_out ) {
  _publish_cc_class_generate_stmt( _cc_out );
}

void
IIRScram_ConcurrentGenerateForStatement::_publish_cc_headerfile() {
  ostringstream filename;

  _publish_cc_class_name(filename);
  
  published_header_file header_file( _get_work_library()->_get_path_to_directory(), 
				     filename.str(),
				     this );
  SCRAM_CC_REF( header_file, "IIRScram_ConcurrentGenerateForStatement::_publish_cc_headerfile" );

  header_file << "#include \"";
  ASSERT ( _get_enclosing_scope() != NULL );
  _get_enclosing_scope()->_publish_cc_class_name(header_file.get_stream());
  header_file << ".hh\"\n";
  
  header_file << "#include \"_savant_entity_elab.hh\"\n";

  // Include the current declarative region 
  header_file << "#include \"";
  header_file << _get_current_entity_name() << "_"
	      << _get_current_architecture_name();
  _publish_cc_enclosing_stmt_to_architecture_path(header_file.get_stream());
  header_file.get_stream() << "_" << *_get_label() << ".hh\"\n";

  // Request the Hierarchy.hh for our hierarchy stuff
  IIRScram::_publish_cc_include( header_file, "tyvis/Hierarchy.hh" );

  //  _publish_cc_headers( header_file );
  header_file << "\n";
  _publish_cc_class_includes( header_file, &concurrent_statement_part );
  header_file << "\n";
  _publish_cc_class( header_file );
}

void
IIRScram_ConcurrentGenerateForStatement::_publish_cc_elaborate( published_file &_cc_out ){
  PublishedUnit saved_publishing_unit = _get_currently_publishing_unit();

  SCRAM_CC_REF( _cc_out, "IIRScram_ConcurrentGenerateForStatement::_publish_cc_elaborate" );

  _set_currently_publishing_unit(GENERATE_FOR);
  _publish_cc_headerfile();
  _publish_cc_ccfile();
  _publish_cc_generate_elaborate( _cc_out );
  _set_currently_publishing_unit(saved_publishing_unit);
}

void
IIRScram_ConcurrentGenerateForStatement::_publish_cc_ccfile(){
  ostringstream filename;

  _publish_cc_class_name(filename);

  published_cc_file cc_file( _get_work_library()->_get_path_to_directory(), 
			     filename.str(),
			     this );
  SCRAM_CC_REF( cc_file, "IIRScram_ConcurrentGenerateForStatement::_publish_cc_ccfile" );

  cc_file << "#include \"" << filename.str() << ".hh\"\n\n";
  _publish_cc_headerfiles_for_cc( cc_file );

  _publish_cc_static_members( cc_file );
  _publish_cc_constructor( cc_file );
  _publish_cc_destructor( cc_file, &concurrent_statement_part );
  _publish_cc_instantiate( cc_file );
  _publish_cc_createNetInfo( cc_file );
  _publish_cc_connect( cc_file );
  _publish_cc_getGenerateConstant( cc_file );
  _publish_cc_resetGenerateConstant( cc_file );
}

void
IIRScram_ConcurrentGenerateForStatement::_publish_cc_static_members( published_file &_cc_out ){

  SCRAM_CC_REF( _cc_out, "IIRScram_ConcurrentGenerateForStatement::_publish_cc_static_members" );

  _cc_out << "int\n"
	  << _get_cc_elaboration_class_name() 
	  << "::generateConstant = 0;\n\n"

	  << "ArrayInfo::ArrayDirn_t\n"
	  << _get_cc_elaboration_class_name()
	  << "::generateDirection = ";
  if( get_generate_parameter_specification()->_is_ascending_range() ){
    _cc_out << "ArrayInfo::to";
  }
  else {
    _cc_out << "ArrayInfo::downto";
  }
  _cc_out << ";\n\n";
}

void
IIRScram_ConcurrentGenerateForStatement::_publish_cc_getGenerateConstant( published_file &_cc_out )
{

  SCRAM_CC_REF( _cc_out, "IIRScram_ConcurrentGenerateForStatement::_publish_cc_getGenerateConstant" );

  _cc_out << "int\n" 
	  << _get_cc_elaboration_class_name()
	  << "::getGenerateConstant(int flag)  {\n"
	  << "  int returnValue = generateConstant;\n\n"
	  << "  if (flag)  {\n"
	  << "    generateConstant += ((generateDirection == ArrayInfo::to) ? 1 : -1);\n"
	  << "   }\n\n  return returnValue;\n}\n\n";
}
  
void
IIRScram_ConcurrentGenerateForStatement::_publish_cc_resetGenerateConstant( published_file &_cc_out ){

  SCRAM_CC_REF( _cc_out, "IIRScram_ConcurrentGenerateForStatement::_publish_cc_resetGenerateConstant" );

  _cc_out << "void "
	  << _get_cc_elaboration_class_name()
	  << "::resetGenerateConstant()  {\n"
	  << "  generateConstant = ";
  ((IIR_ScalarTypeDefinition *) get_generate_parameter_specification()->get_subtype())->get_left()->_publish_cc_value( _cc_out );
  _cc_out << ";\n}\n\n";
}

void
IIRScram_ConcurrentGenerateForStatement::_publish_cc_headerfiles_for_cc( published_file &_cc_out ){

  SCRAM_CC_REF( _cc_out, 
		"IIRScram_ConcurrentGenerateForStatement::_publish_cc_headerfiles_for_cc" );

  _publish_cc_headerfiles_for_cc_generate_statement( _cc_out );
}

void
IIRScram_ConcurrentGenerateForStatement::_publish_cc_constructor( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_ConcurrentGenerateForStatement::_publish_cc_constructor" );

  _publish_cc_constructor_with_no_arguments( _cc_out );
}

void
IIRScram_ConcurrentGenerateForStatement::_publish_cc_constructor_with_no_arguments( published_file &_cc_out )
{

  SCRAM_CC_REF( _cc_out, "IIRScram_ConcurrentGenerateForStatement::_publish_cc_constructor_with_no_arguments" );
  _cc_out << _get_cc_elaboration_class_name() << "::"
	  << _get_cc_elaboration_class_name() << "(";

  ASSERT ( _get_enclosing_scope() != NULL );
  _get_enclosing_scope()->_publish_cc_class_name( _cc_out.get_stream() );
  _cc_out << "* outerScope)  {\n";
  _cc_out << "  generateConstant = ";
  ((IIR_ScalarTypeDefinition *) get_generate_parameter_specification()->get_subtype())->get_left()->_publish_cc_value( _cc_out );
  _cc_out << ";\n";
  _cc_out << "  enclosingScope = outerScope;\n";
  
  _publish_cc_object_pointers_init( _cc_out );
  _cc_out << "}\n";
}

void
IIRScram_ConcurrentGenerateForStatement::_publish_cc_signal_objects_init( published_file &_cc_out,
									  IIR_Boolean first_decl_flag ){
  IIR_Declaration *decl;
  int first = 0;

  SCRAM_CC_REF( _cc_out, 
		"IIRScram_ConcurrentGenerateForStatement::_publish_cc_signal_objects_init" );

  decl = block_declarative_part.first();

  while (decl != NULL) {
    if (decl->get_kind() == IIR_SIGNAL_DECLARATION) {
      if ( first == 0 && first_decl_flag == TRUE ){
	_cc_out << ":\n";
	first = 1;
      }
      else {
	_cc_out << ",\n";
      }
      
      decl->_publish_cc_elaborate( _cc_out );
      _cc_out << "(ObjectBase::SIGNAL_NETINFO";

      if (((decl->get_subtype()->_is_array_type() == TRUE) ||
	   (decl->get_subtype()->_is_record_type() ==  TRUE)) &&
	  (decl->get_subtype()->_is_access_type() == FALSE)) {
	decl->get_subtype()->_publish_cc_object_type_info( _cc_out );
	_cc_out << ", ";
	decl->get_subtype()->_publish_cc_resolution_function_id( _cc_out );
	
	if (decl->get_subtype()->_is_anonymous() == TRUE) {
	  _cc_out << ", ";
	  decl->get_subtype()->_publish_cc_range( _cc_out );
	}
      }      
      _cc_out <<  ")";
      
      if( decl->_get_implicit_declarations() != NULL
	  && decl->_get_implicit_declarations()->num_elements() != 0 ) {
	IIR_Declaration* imp_decl = decl->_get_implicit_declarations()->get_element();
	while (imp_decl != NULL) {
	  if (imp_decl->get_kind() == IIR_SIGNAL_DECLARATION) {
	    _cc_out << ",\n";
	    imp_decl->_publish_cc_elaborate( _cc_out );
	    _cc_out << "(ObjectBase::SIGNAL_NETINFO";

	    if (((imp_decl->get_subtype()->_is_array_type() == TRUE) ||
		 (imp_decl->get_subtype()->_is_record_type() ==  TRUE)) &&
		(imp_decl->get_subtype()->_is_access_type() ==  FALSE)) {
	      imp_decl->get_subtype()->_publish_cc_object_type_info(_cc_out);
	      _cc_out << ", ";
	      imp_decl->get_subtype()->_publish_cc_resolution_function_id( _cc_out );
	      
	      if (imp_decl->get_subtype()->_is_anonymous() == TRUE) {
		_cc_out << ", ";
		imp_decl->get_subtype()->_publish_cc_range( _cc_out );
	      }
	    }
	    _cc_out << ")";
	  }
	  imp_decl = decl->_get_implicit_declarations()->get_next_element();
	}
      }
    }
    else if (decl->get_kind() == IIR_ALIAS_DECLARATION) {
      if (((IIR_AliasDeclaration *) decl)->get_name()->_is_signal()) {
	if( first == 0 && first_decl_flag == 1 ){
	  _cc_out << ":\n";
	  first = 1;
	}
	else {
	  _cc_out << ",\n";
	}
	((IIR_AliasDeclaration *) decl)->_publish_cc_elaborate_alias_init( _cc_out );
      }
    }
    
    decl = block_declarative_part.successor(decl);
  }
}

void
IIRScram_ConcurrentGenerateForStatement::_publish_cc_object_pointers_init( published_file &_cc_out ){
  const string temp = _get_current_architecture_name();
  const string old_current_name = _get_current_publish_name();

  IIR_ArchitectureStatement *arch_stmt;
  IIR_Boolean found = FALSE;
  IIR* tmpNode;

  SCRAM_CC_REF( _cc_out, "IIRScram_ConcurrentGenerateForStatement::_publish_cc_object_pointers_init" );

  arch_stmt = concurrent_statement_part.first();

  while (arch_stmt != NULL) {
    switch(arch_stmt->get_kind()) {
    case IIR_PROCESS_STATEMENT:
      if (arch_stmt->_get_label() != NULL) {
	_cc_out.get_stream() << *(arch_stmt->_get_label());
      }
      else {
	_cc_out << "ANON_PROCESS" << arch_stmt;
      }
      _cc_out << "_elab_obj = NULL;\n";
      break;

    case IIR_COMPONENT_INSTANTIATION_STATEMENT:
      IIR_ComponentDeclaration *componentName;
      componentName = (IIR_ComponentDeclaration *) arch_stmt->get_instantiated_unit();
      ASSERT(componentName->get_kind() == IIR_COMPONENT_DECLARATION);

      _cc_out.get_stream() << *arch_stmt->_get_label();
      _cc_out << "_elab_obj = new "
	      << _get_cc_elaboration_class_name()
	      << "[";
      _publish_cc_universal_generate_range( _cc_out );
      _cc_out << "](";
      arch_stmt->_get_generic_map_aspect()->_publish_cc_lvalue( _cc_out );
      _cc_out << ");\n";

      _cc_out << "for(int i = 0; (i < ";
      _publish_cc_universal_generate_range( _cc_out );
      _cc_out << "); i++)  {\n";

      tmpNode = _get_current_publish_node();
      _set_current_publish_node( this );     
      arch_stmt->_publish_cc_concurrent_stmt_init( _cc_out, &block_declarative_part );
      _set_current_publish_node( tmpNode );
      _cc_out << "}\n";
      
      break;

    case IIR_BLOCK_STATEMENT:      {
      _set_current_publish_name( "SG" );

      arch_stmt->_get_label()->_publish_cc_elaborate( _cc_out );
      _cc_out << "_elab_obj = new "
	      << arch_stmt->_get_cc_elaboration_class_name() << "[";
      _publish_cc_universal_generate_range( _cc_out );
      _cc_out << "](this";
      
      if ( ((IIR_BlockStatement *) arch_stmt)->generic_map_aspect.num_elements() > 0) {
	_cc_out << ", \n";
	tmpNode = _get_current_publish_node();
	_set_current_publish_node( arch_stmt );
	((IIR_BlockStatement *) arch_stmt)->generic_map_aspect._publish_cc_generic_map_aspect_for_conc_stmts( _cc_out );
	_set_current_publish_node( tmpNode );
      }
      
      _cc_out << ");\n";
      
      _set_current_architecture_name( temp );
      _set_current_publish_name( old_current_name );
      }
      break;
    case IIR_CONCURRENT_GENERATE_FOR_STATEMENT:
    case IIR_CONCURRENT_GENERATE_IF_STATEMENT:{
	_set_current_publish_name( "SG" );
      
      arch_stmt->_get_label()->_publish_cc_elaborate( _cc_out );
      _cc_out << "_elab_obj = new "
	      << arch_stmt->_get_cc_elaboration_class_name() << "[";
      _publish_cc_universal_generate_range( _cc_out );
      _cc_out << "](this);\n";

      _set_current_architecture_name( temp );
      _set_current_publish_name( old_current_name );
      }
    break;

    default:
      cerr << "ERROR! IIRScram_ConcurrentGenerateForStatement::"
	   << "_publish_cc_object_pointers_init( _cc_out ): unknown conc_statement "
	   << "type |" << arch_stmt->get_kind_text() << "|" << endl;
      break;
    }

    found = FALSE;
    arch_stmt = concurrent_statement_part.successor(arch_stmt);
  }
}

void
IIRScram_ConcurrentGenerateForStatement::_publish_cc_instantiate( published_file &_cc_out )
{
  IIR_ArchitectureStatement *arch_stmt;

  SCRAM_CC_REF( _cc_out, "IIRScram_ConcurrentGenerateForStatement::_publish_cc_instantiate" );
  
  _cc_out << "void\n"
	  << _get_cc_elaboration_class_name()
	  << "::instantiate(Hierarchy * hier)  {\n";

  _cc_out << "  int generateLeft = ";
  ((IIR_ScalarTypeDefinition *) get_generate_parameter_specification()->get_subtype())->get_left()->_publish_cc_value( _cc_out );
  _cc_out << ";\n\n";
  _cc_out << "int i;\n";
  
  arch_stmt = concurrent_statement_part.first();
  while (arch_stmt != NULL) {
    if (arch_stmt->get_kind() == IIR_PROCESS_STATEMENT) {
      ostringstream objectname;
      if (arch_stmt->_get_label() != NULL) {
	objectname << *(arch_stmt->_get_label());
      }
      else {
	objectname << "ANON_PROCESS" << arch_stmt;
      }
      _cc_out << objectname.str() << "_elab_obj = new " << objectname.str();
      _cc_out << " *[";
      _publish_cc_universal_generate_range( _cc_out );
      _cc_out << "]";
      _cc_out << ";\n";
      
      _cc_out << "  for(i = 0; (i < ";
      _publish_cc_universal_generate_range( _cc_out );
      _cc_out << "); i++)  {\n";
      _cc_out << "    " << objectname.str() << "_elab_obj[i] = new ";
      _cc_out << objectname.str() << "(this);\n" ;
      _cc_out << "}\n";
      
      _set_number_of_processes( _get_number_of_processes() + 1 );
    }
    else if ((arch_stmt->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT)||
	     (arch_stmt->_is_concurrent_generate_statement() == TRUE) ||
	     (arch_stmt->_is_block_statement() == TRUE)) {
      _publish_cc_generate_for_loop( _cc_out );
      _cc_out << "{\n";
      _publish_cc_assign_to_loop_constant( _cc_out );
      (arch_stmt->_get_label())->_publish_cc_elaborate( _cc_out );
      _cc_out << "_elab_obj[i - generateLeft].instantiate(hier);\n";
      _cc_out << "  }\n";
    }
    else {
      cerr << "ERROR! IIRScram_ConcurrentGenerateForStatement::"
	   << "_publish_cc_instantiate( _cc_out ): unknown conc_statement "
	   << "type |" << arch_stmt->get_kind_text() << "|" << endl;
    }
    arch_stmt = concurrent_statement_part.successor(arch_stmt);
  }
  _cc_out << "createNetInfo();\n";
  _cc_out << "}\n";
}

void
IIRScram_ConcurrentGenerateForStatement::_publish_cc_body_for_component_createNetInfo( published_file &_cc_out,
										       IIR_ArchitectureStatement *arch_stmt){
  ostringstream labelname;
  ostringstream objectname;
  int wanted_instantiation = 1;
  IIR_Boolean found = false;
  IIR_EntityDeclaration* bindingEntity = 0;
  int noofelements = 0;

  SCRAM_CC_REF( _cc_out, 
		"IIRScram_ConcurrentGenerateForStatement::_publish_cc_body_for_component_createNetInfo" );
  
  labelname << *arch_stmt->_get_label() << "_elab_obj[i].";
  _set_current_elab_name( labelname.str() );
  IIR* componentName = arch_stmt->get_instantiated_unit();
      
  if(componentName->get_kind() == IIR_COMPONENT_DECLARATION) {
    IIR_Declaration* decl = block_declarative_part.first();
    while(decl != NULL) {
      if(decl->get_kind() == IIR_CONFIGURATION_SPECIFICATION) {
	IIR_TextLiteral* local_componentName = componentName->_get_declarator();
	IIR_TextLiteral* binding_componentName = ((IIR_ConfigurationSpecification*)decl)->get_component_name()->_get_declarator();
	if(IIR_TextLiteral::_cmp(local_componentName, 
				 binding_componentName) == 0) {
	  IIR_Designator* designator = decl->_get_instantiation_list()->first();
	  
	  while(designator != NULL) {
	    switch(designator->get_kind()) {
	    case IIR_DESIGNATOR_EXPLICIT:
	      wanted_instantiation = IIR_TextLiteral::_cmp(((IIR_Declaration *) designator->_get_name())->get_declarator(), arch_stmt->_get_label()->get_declarator());
	      if(wanted_instantiation == 0) {
		found = TRUE;
		if(((IIR_ConfigurationSpecification*)decl)->port_map_aspect.num_elements() == 0) {
		  _set_current_publish_node( componentName );
		  objectname << "(( ";
		  IIR_LibraryUnit* lib_unit =((IIR_ConfigurationSpecification*)decl)->get_entity_aspect(); 
		  if(lib_unit != NULL) {
		    lib_unit->_publish_cc_binding_name(objectname);
		    bindingEntity = lib_unit->_get_entity();
		  }
		  objectname << "_elab*) " << *arch_stmt->_get_label();
		  objectname << "_elab_obj->boundedEntity)->";
		  _set_current_publish_name( objectname.str() );
		  bindingEntity->port_clause._publish_cc_port_map_aspect( _cc_out );
		}
		else {
		  _set_current_publish_node( componentName ); 
		  objectname << "(( ";
		  IIR_LibraryUnit* lib_unit =((IIR_ConfigurationSpecification*)decl)->get_entity_aspect(); 
		  if(lib_unit != NULL) {
		    lib_unit->_publish_cc_binding_name(objectname);
		    bindingEntity = lib_unit->_get_entity();
		  }
		  objectname << "_elab*) " << *(arch_stmt->_get_label());
		  objectname << "_elab_obj->boundedEntity)->";
		  _set_current_publish_name( objectname.str() );
		  noofelements = ((IIR_ConfigurationSpecification*)decl)->port_map_aspect.num_elements();
		  if(noofelements != 0) {
		    ((IIR_ConfigurationSpecification*)decl)->port_map_aspect._publish_cc_elaborate( _cc_out );
		  }
		  else {
		    IIR_LibraryUnit* binding_unit =((IIR_ConfigurationSpecification*)decl)->get_entity_aspect();
		    if(binding_unit!= NULL) {
		      binding_unit->_get_entity()->port_clause._publish_cc_port_map_aspect( _cc_out );
		    }
		  }
		}
	      }
	      break;
	    case IIR_DESIGNATOR_BY_ALL:
	    case IIR_DESIGNATOR_BY_OTHERS:
	      found = TRUE;
	      if(((IIR_ConfigurationSpecification*)decl)->port_map_aspect.num_elements() == 0) {
		_set_current_publish_node( componentName );
		objectname << "(( ";
		IIR_LibraryUnit* lib_unit =((IIR_ConfigurationSpecification*)decl)->get_entity_aspect(); 
		if(lib_unit != NULL) {
		  lib_unit->_publish_cc_binding_name(objectname);
		  bindingEntity = lib_unit->_get_entity();
		}
		objectname << "_elab*) " << *(arch_stmt->_get_label());
		objectname << "_elab_obj->boundedEntity)->";
		_set_current_publish_name( objectname.str() );
		bindingEntity->port_clause._publish_cc_port_map_aspect( _cc_out );
	      }
	      else {
		_set_current_publish_node( componentName );
		objectname << "(( ";
		IIR_LibraryUnit* lib_unit =((IIR_ConfigurationSpecification*)decl)->get_entity_aspect();
		if(lib_unit != NULL) {
		  lib_unit->_publish_cc_binding_name(objectname);
		  bindingEntity = lib_unit->_get_entity();
		}
		objectname << "_elab*) " << *(arch_stmt->_get_label());
		objectname << "_elab_obj->boundedEntity)->";
		_set_current_publish_name( objectname.str() );
		noofelements = ((IIR_ConfigurationSpecification*)decl)->port_map_aspect.num_elements();
		if(noofelements != 0) {
		  ((IIR_ConfigurationSpecification*)decl)->port_map_aspect._publish_cc_elaborate( _cc_out );
		}
		else {
		  IIR_LibraryUnit* binding_unit =((IIR_ConfigurationSpecification*)decl)->get_entity_aspect();
		  if(binding_unit!= NULL) {
		    binding_unit->_get_entity()->port_clause._publish_cc_port_map_aspect( _cc_out );
		  }
		}
	      }
	      break;

	    default:
	      break;
	    }
	    if(found == TRUE) {
	      designator = NULL;
	    }
	    else {
	      designator =decl->_get_instantiation_list()->successor(designator);
	    }
	  }
	}
      }
      if(found == TRUE) {
	decl = NULL;
      }
      else {
	decl = block_declarative_part.successor(decl);
      }
    }
  }
  
  objectname << *arch_stmt->_get_label() << "_elab_obj";
  _set_current_elab_name( objectname.str() );
  arch_stmt->_publish_createNetInfo( _cc_out );
}

void
IIRScram_ConcurrentGenerateForStatement::_publish_cc_createNetInfo( published_file &_cc_out ){
  IIR_ArchitectureStatement *arch_stmt;
  IIR_Boolean found = false;
  int wanted_instantiation = 1;
  const string tmp  = _get_current_elab_name();
  const string tmp2 = _get_current_publish_name();
  IIR* tmpNode   = _get_current_publish_node();
  PublishedUnit old_publishing_unit = _get_currently_publishing_unit();

  SCRAM_CC_REF( _cc_out, "IIRScram_ConcurrentGenerateForStatement::_publish_cc_createNetInfo" );

  _set_currently_publishing_unit(GENERATE_FOR);
  
  _cc_out << "void\n"
	  << _get_cc_elaboration_class_name()
	  << "::createNetInfo() {\n"
	  << "  int generateLeft = ";
  ((IIR_ScalarTypeDefinition *) get_generate_parameter_specification()->get_subtype())->get_left()->_publish_cc_value( _cc_out );
  _cc_out << ";\n\n";
  _cc_out << "int i;\n";
  
  arch_stmt = concurrent_statement_part.first();
  while (arch_stmt != NULL) {
    ostringstream objectname;
    ostringstream labelname;
    _set_current_publish_name( "" );
    _set_current_publish_node( this );
    
    if (arch_stmt->get_kind() == IIR_PROCESS_STATEMENT) {
      _publish_cc_generate_for_loop( _cc_out );
      _cc_out << "{\n";
      _publish_cc_assign_to_loop_constant( _cc_out );
      
      if( arch_stmt->_get_label() != NULL ){
	objectname << *arch_stmt->_get_label();
      }
      else {
	objectname << "ANON_PROCESS" << arch_stmt;
      }
      objectname << "_elab_obj";
      _set_current_elab_name( objectname.str() );
      arch_stmt->_publish_createNetInfo( _cc_out );
      _cc_out << "}\n";
    }
    else if ((arch_stmt->get_kind() == IIR_BLOCK_STATEMENT) ||
	     (arch_stmt->get_kind() == IIR_CONCURRENT_GENERATE_FOR_STATEMENT)||
	     (arch_stmt->get_kind() == IIR_CONCURRENT_GENERATE_IF_STATEMENT)) {
      // Nothing to be done for these guys
    }
    else if (arch_stmt->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT) {
      _publish_cc_generate_for_loop( _cc_out );
      _cc_out << "{\n";
      _publish_cc_assign_to_loop_constant( _cc_out );
      _publish_cc_body_for_component_createNetInfo( _cc_out, arch_stmt );
      _cc_out << "  }\n";
    }
    else {
      cerr << "ERROR! IIRScram_ConcurrentGenerateForStatement::"
	   << "_publish_cc_createNetInfo( _cc_out ): unknown conc_statement "
	   << "type |" << arch_stmt->get_kind_text() << "| in arch" << endl;
    }
    _set_current_elab_name( "" );
    found = FALSE;
    wanted_instantiation = 1;
    arch_stmt = concurrent_statement_part.successor(arch_stmt);
  }
  _cc_out << "}\n\n";
  _set_current_elab_name( tmp );
  _set_current_publish_name( tmp2 );
  _set_current_publish_node( tmpNode );

  _set_currently_publishing_unit(old_publishing_unit);
}

void
IIRScram_ConcurrentGenerateForStatement::_publish_cc_connect( published_file &_cc_out ){
  PublishedUnit oldUnit = _get_currently_publishing_unit();

  SCRAM_CC_REF( _cc_out, "IIRScram_ConcurrentGenerateForStatement::_publish_cc_connect" );

  _set_currently_publishing_unit(GENERATE_FOR);
  
  _cc_out << "void" << NL()
	  << _get_cc_elaboration_class_name()
	  << OS("::connect( int inputsignals, int outputsignals, ... ){")
	  << "int noOfSignals = inputsignals + outputsignals;"
	  << "va_list ap;" << NL();

  _cc_out << "int generateLeft = ";
  static_cast<IIR_ScalarTypeDefinition *>(get_generate_parameter_specification()->get_subtype())->get_left()->_publish_cc_value( _cc_out );
  _cc_out << ";" << NL();
  
  _cc_out << "va_start(ap, outputsignals);" << NL();
  _cc_out << OS("for(i = 0; (i < noOfSignals); i++) {");
  _cc_out << "addToFanOut( va_arg(ap, VHDLType*) );" << NL();
  _cc_out << CS("}") << NL();
  _cc_out << "va_end(ap);" << NL();
  
  IIR_ArchitectureStatement *arch_stmt;
  arch_stmt = concurrent_statement_part.first();
  while (arch_stmt != NULL) {
    ostringstream objectname;
    if (arch_stmt->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT) {
      _publish_cc_generate_for_loop( _cc_out );
      _cc_out << OS("{");
      _publish_cc_assign_to_loop_constant( _cc_out );
      objectname << *arch_stmt->_get_label() << "_elab_obj[i]";
      _set_current_elab_name( objectname.str() );
      static_cast<IIR_ComponentInstantiationStatement*>(arch_stmt)->_publish_connect( _cc_out );
      _cc_out << CS("}") << NL();
    }
    else if (arch_stmt->_is_concurrent_generate_statement() == TRUE) {
      _publish_cc_generate_for_loop( _cc_out );
      _cc_out << OS("{");
      _publish_cc_assign_to_loop_constant( _cc_out );
      
      arch_stmt->_get_label()->_publish_cc_elaborate( _cc_out );
      if (_get_currently_publishing_unit() == GENERATE_FOR) {
	_cc_out << "_elab_obj[i - generateLeft].connect(0, 0);" << NL();
      }
      else {
	_cc_out << "->connect(0, 0);" << NL();
      }
      _cc_out << CS("}") << NL();
    }
    else if (arch_stmt->_is_block_statement() == TRUE) {
      _publish_cc_generate_for_loop( _cc_out );
      _cc_out << OS("{");
      _publish_cc_assign_to_loop_constant( _cc_out );
      static_cast<IIR_BlockStatement *>(arch_stmt)->_publish_cc_connect_call( _cc_out );
      _cc_out << CS("}") << NL();
    }

    arch_stmt = concurrent_statement_part.successor( arch_stmt );
  }
  _cc_out << CS("}");

  _set_currently_publishing_unit(oldUnit);
}


void
IIRScram_ConcurrentGenerateForStatement::_publish_cc_generate_for_loop_with_zero( published_file &_cc_out,
										  char *variable_name){
  IIR_ScalarTypeDefinition *range;

  SCRAM_CC_REF( _cc_out, "IIRScram_ConcurrentGenerateForStatement::_publish_cc_generate_for_loop_with_zero" );

  ASSERT ( get_generate_parameter_specification()->get_subtype()->_is_iir_scalar_type_definition() == TRUE );
  range = (IIR_ScalarTypeDefinition *) get_generate_parameter_specification()->get_subtype();
  
  _cc_out << "  for(int " << variable_name << " = ";
  if ( range->_is_ascending_range() == TRUE ){
    _cc_out << "0";
  }
  else {
    _cc_out << "(";
    range->get_left()->_publish_cc_value( _cc_out );
    _cc_out << " - ";
    range->get_right()->_publish_cc_value( _cc_out );
    _cc_out << ")";
  }
  _cc_out << "; (" << variable_name << " ";

  if (range->_is_ascending_range() == TRUE) {
    _cc_out << " <= ";
  }
  else {
    _cc_out << " >= ";
  }

  if ( range->_is_ascending_range() == TRUE ){
    _cc_out << "(";
    range->get_right()->_publish_cc_value( _cc_out );
    _cc_out << " - ";
    range->get_left()->_publish_cc_value( _cc_out );
    _cc_out << ")";
  }
  else {
    _cc_out << "0";
  }
  _cc_out << "); ";

  if (range->_is_ascending_range() == TRUE) {
    _cc_out << variable_name << "++";
  }
  else {
    _cc_out << variable_name << "--";
  }

  _cc_out << ")  ";
}

IIR_Label *
IIRScram_ConcurrentGenerateForStatement::_find_instantiate_label( IIR_SimpleName *to_find ){
  return concurrent_statement_part._find_instantiate_label( to_find );
}

visitor_return_type *IIRScram_ConcurrentGenerateForStatement::_accept_visitor(node_visitor *visitor, visitor_argument_type *arg) {
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_ConcurrentGenerateForStatement(this, arg);
};
