/****************************************************************************
**  SCALASCA    http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2013                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2010-2013                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  Copyright (c) 2003-2008                                                **
**  University of Tennessee, Innovative Computing Laboratory               **
**                                                                         **
**  See the file COPYRIGHT in the package base directory for details       **
****************************************************************************/

/**
 * @file  epk_mpiwrap_topo.c
 *
 * @brief C interface wrappers for topologies
 */

#include <stdlib.h>
#include <mpi.h>

#include "elg_error.h"

#include "esd_def.h"
#include "esd_event.h"

#include "epk_defs_mpi.h"
#include "epk_mpicom.h"
#include "epk_mpiwrap.h"

/**
 * @name C wrappers
 * @{
 */

#if defined(HAS_MPI_CART_CREATE)
/**
 * Measurement wrapper for MPI_Cart_create
 * @note Manually adapted wrapper
 * @note C interface
 * @note Introduced with MPI 1.0
 * @ingroup topo
 */
int MPI_Cart_create(MPI_Comm  comm_old,
                    int       ndims,
                    int*      dims,
                    int*      periodv,
                    int       reorder,
                    MPI_Comm* comm_cart)
{
  const int event_gen_active = IS_EVENT_GEN_ON_FOR(TOPO);
  int       return_val;

  if (event_gen_active)
  {
    EVENT_GEN_OFF();
    esd_enter(epk_mpi_regid[EPK__MPI_CART_CREATE]);
  }

  return_val = PMPI_Cart_create(comm_old,
                                ndims,
                                dims,
                                periodv,
                                reorder,
                                comm_cart);

  if (*comm_cart != MPI_COMM_NULL)
  {
    int  cid, my_rank, i;
    int* coordv;

    /* Create new topology object  and set name */
    EPIK_TOPOL * topology = EPIK_Cart_create("MPI Cartesian topology", ndims);

    epk_comm_create(*comm_cart);

    /* get the topid and cid */
    cid = epk_comm_id(*comm_cart);

    /* find the rank of the calling process */
    PMPI_Comm_rank(*comm_cart, &my_rank);

    /* assign the cartesian topology dimension parameters */
    for (i = 0; i < ndims; i++)
    {
      EPIK_Cart_add_dim(topology, dims[i], periodv[i], "");
    }

    /* allocate space for coordv and ucoordv */
    coordv = calloc(topology->num_dims, sizeof(elg_ui4));
    if (coordv == NULL)
    {
      elg_error();
    }

    /* get the coordinates of the calling process in coordv */
    PMPI_Cart_coords(*comm_cart, my_rank, ndims, coordv);

    /* assign the coordinates */
    for (i = 0; i < topology->num_dims; i++)
    {
      topology->coords[i] = (elg_ui4)coordv[i];
    }

    /* create the cartesian topology definition record */
    /* Note: cannot call EPIK_Cart_commit because it does NOT record "cid" */
    topology->topid = esd_def_cart(cid, topology);

    /* create the coordinates definition record */
    /* Could call EPIK_Cart_coords_commit, but does just produce 1 extra
     * useless fun ction call */
    esd_def_coords(topology);
  }

  if (event_gen_active)
  {
    esd_exit(epk_mpi_regid[EPK__MPI_CART_CREATE]);
    EVENT_GEN_ON();
  }

  return return_val;
} /* MPI_Cart_create */

#endif

#if defined(HAS_MPI_CART_SUB)
/**
 * Measurement wrapper for MPI_Cart_sub
 * @note Auto-generated by wrapgen from template: comm_mgnt.w
 * @note C interface
 * @note Introduced with MPI 1.0
 * @ingroup topo
 */
int MPI_Cart_sub(MPI_Comm  comm,
                 int*      remain_dims,
                 MPI_Comm* newcomm)
{
  const int event_gen_active = IS_EVENT_GEN_ON_FOR(TOPO);
  int       return_val;

  if (event_gen_active)
  {
    EVENT_GEN_OFF();
    esd_enter(epk_mpi_regid[EPK__MPI_CART_SUB]);
  }

  return_val = PMPI_Cart_sub(comm, remain_dims, newcomm);
  if (*newcomm != MPI_COMM_NULL)
  {
    epk_comm_create(*newcomm);
  }

  if (event_gen_active)
  {
    esd_exit(epk_mpi_regid[EPK__MPI_CART_SUB]);
    EVENT_GEN_ON();
  }

  return return_val;
}

#endif

#if defined(HAS_MPI_GRAPH_CREATE)
/**
 * Measurement wrapper for MPI_Graph_create
 * @note Auto-generated by wrapgen from template: comm_mgnt.w
 * @note C interface
 * @note Introduced with MPI 1.0
 * @ingroup topo
 */
int MPI_Graph_create(MPI_Comm  comm_old,
                     int       nnodes,
                     int*      index,
                     int*      edges,
                     int       reorder,
                     MPI_Comm* newcomm)
{
  const int event_gen_active = IS_EVENT_GEN_ON_FOR(TOPO);
  int       return_val;

  if (event_gen_active)
  {
    EVENT_GEN_OFF();
    esd_enter(epk_mpi_regid[EPK__MPI_GRAPH_CREATE]);
  }

  return_val = PMPI_Graph_create(comm_old,
                                 nnodes,
                                 index,
                                 edges,
                                 reorder,
                                 newcomm);
  if (*newcomm != MPI_COMM_NULL)
  {
    epk_comm_create(*newcomm);
  }

  if (event_gen_active)
  {
    esd_exit(epk_mpi_regid[EPK__MPI_GRAPH_CREATE]);
    EVENT_GEN_ON();
  }

  return return_val;
}

#endif

#if defined(HAS_MPI_DIST_GRAPH_CREATE) && !defined(NO_MPI_TOPO)
/**
 * Measurement wrapper for MPI_Dist_graph_create
 * @note Auto-generated by wrapgen from template: comm_mgnt.w
 * @note C interface
 * @note Introduced with MPI 2.2
 * @ingroup topo
 */
int MPI_Dist_graph_create(MPI_Comm  comm_old,
                          int       n,
                          int       sources[],
                          int       degrees[],
                          int       destinations[],
                          int       weights[],
                          MPI_Info  info,
                          int       reorder,
                          MPI_Comm* newcomm)
{
  const int event_gen_active = IS_EVENT_GEN_ON_FOR(TOPO);
  int       return_val;

  if (event_gen_active)
  {
    EVENT_GEN_OFF();
    esd_enter(epk_mpi_regid[EPK__MPI_DIST_GRAPH_CREATE]);
  }

  return_val = PMPI_Dist_graph_create(comm_old,
                                      n,
                                      sources,
                                      degrees,
                                      destinations,
                                      weights,
                                      info,
                                      reorder,
                                      newcomm);
  if (*newcomm != MPI_COMM_NULL)
  {
    epk_comm_create(*newcomm);
  }

  if (event_gen_active)
  {
    esd_exit(epk_mpi_regid[EPK__MPI_DIST_GRAPH_CREATE]);
    EVENT_GEN_ON();
  }

  return return_val;
}

#endif

#if defined(HAS_MPI_DIST_GRAPH_CREATE_ADJACENT) && !defined(NO_MPI_TOPO)
/**
 * Measurement wrapper for MPI_Dist_graph_create_adjacent
 * @note Auto-generated by wrapgen from template: comm_mgnt.w
 * @note C interface
 * @note Introduced with MPI 2.2
 * @ingroup topo
 */
int MPI_Dist_graph_create_adjacent(MPI_Comm  comm_old,
                                   int       indegree,
                                   int       sources[],
                                   int       sourceweights[],
                                   int       outdegree,
                                   int       destinations[],
                                   int       destweights[],
                                   MPI_Info  info,
                                   int       reorder,
                                   MPI_Comm* newcomm)
{
  const int event_gen_active = IS_EVENT_GEN_ON_FOR(TOPO);
  int       return_val;

  if (event_gen_active)
  {
    EVENT_GEN_OFF();
    esd_enter(epk_mpi_regid[EPK__MPI_DIST_GRAPH_CREATE_ADJACENT]);
  }

  return_val = PMPI_Dist_graph_create_adjacent(comm_old,
                                               indegree,
                                               sources,
                                               sourceweights,
                                               outdegree,
                                               destinations,
                                               destweights,
                                               info,
                                               reorder,
                                               newcomm);
  if (*newcomm != MPI_COMM_NULL)
  {
    epk_comm_create(*newcomm);
  }

  if (event_gen_active)
  {
    esd_exit(epk_mpi_regid[EPK__MPI_DIST_GRAPH_CREATE_ADJACENT]);
    EVENT_GEN_ON();
  }

  return return_val;
}

#endif

#if defined(HAS_MPI_CART_COORDS) && !defined(NO_MPI_EXTRA) \
  && !defined(NO_MPI_TOPO)
/**
 * Measurement wrapper for MPI_Cart_coords
 * @note Auto-generated by wrapgen from template: std.w
 * @note C interface
 * @note Introduced with MPI 1.0
 * @ingroup topo
 */
int MPI_Cart_coords(MPI_Comm comm,
                    int      rank,
                    int      maxdims,
                    int*     coords)
{
  int return_val;

  if (IS_EVENT_GEN_ON_FOR(TOPO))
  {
    EVENT_GEN_OFF();
    esd_enter(epk_mpi_regid[EPK__MPI_CART_COORDS]);

    return_val = PMPI_Cart_coords(comm, rank, maxdims, coords);

    esd_exit(epk_mpi_regid[EPK__MPI_CART_COORDS]);
    EVENT_GEN_ON();
  }
  else
  {
    return_val = PMPI_Cart_coords(comm, rank, maxdims, coords);
  }

  return return_val;
}

#endif
#if defined(HAS_MPI_CART_GET) && !defined(NO_MPI_EXTRA) \
  && !defined(NO_MPI_TOPO)
/**
 * Measurement wrapper for MPI_Cart_get
 * @note Auto-generated by wrapgen from template: std.w
 * @note C interface
 * @note Introduced with MPI 1.0
 * @ingroup topo
 */
int MPI_Cart_get(MPI_Comm comm,
                 int      maxdims,
                 int*     dims,
                 int*     periods,
                 int*     coords)
{
  int return_val;

  if (IS_EVENT_GEN_ON_FOR(TOPO))
  {
    EVENT_GEN_OFF();
    esd_enter(epk_mpi_regid[EPK__MPI_CART_GET]);

    return_val = PMPI_Cart_get(comm, maxdims, dims, periods, coords);

    esd_exit(epk_mpi_regid[EPK__MPI_CART_GET]);
    EVENT_GEN_ON();
  }
  else
  {
    return_val = PMPI_Cart_get(comm, maxdims, dims, periods, coords);
  }

  return return_val;
}

#endif
#if defined(HAS_MPI_CART_MAP) && !defined(NO_MPI_EXTRA) \
  && !defined(NO_MPI_TOPO)
/**
 * Measurement wrapper for MPI_Cart_map
 * @note Auto-generated by wrapgen from template: std.w
 * @note C interface
 * @note Introduced with MPI 1.0
 * @ingroup topo
 */
int MPI_Cart_map(MPI_Comm comm,
                 int      ndims,
                 int*     dims,
                 int*     periods,
                 int*     newrank)
{
  int return_val;

  if (IS_EVENT_GEN_ON_FOR(TOPO))
  {
    EVENT_GEN_OFF();
    esd_enter(epk_mpi_regid[EPK__MPI_CART_MAP]);

    return_val = PMPI_Cart_map(comm, ndims, dims, periods, newrank);

    esd_exit(epk_mpi_regid[EPK__MPI_CART_MAP]);
    EVENT_GEN_ON();
  }
  else
  {
    return_val = PMPI_Cart_map(comm, ndims, dims, periods, newrank);
  }

  return return_val;
}

#endif
#if defined(HAS_MPI_CART_RANK) && !defined(NO_MPI_EXTRA) \
  && !defined(NO_MPI_TOPO)
/**
 * Measurement wrapper for MPI_Cart_rank
 * @note Auto-generated by wrapgen from template: std.w
 * @note C interface
 * @note Introduced with MPI 1.0
 * @ingroup topo
 */
int MPI_Cart_rank(MPI_Comm comm,
                  int*     coords,
                  int*     rank)
{
  int return_val;

  if (IS_EVENT_GEN_ON_FOR(TOPO))
  {
    EVENT_GEN_OFF();
    esd_enter(epk_mpi_regid[EPK__MPI_CART_RANK]);

    return_val = PMPI_Cart_rank(comm, coords, rank);

    esd_exit(epk_mpi_regid[EPK__MPI_CART_RANK]);
    EVENT_GEN_ON();
  }
  else
  {
    return_val = PMPI_Cart_rank(comm, coords, rank);
  }

  return return_val;
}

#endif
#if defined(HAS_MPI_CART_SHIFT) && !defined(NO_MPI_EXTRA) \
  && !defined(NO_MPI_TOPO)
/**
 * Measurement wrapper for MPI_Cart_shift
 * @note Auto-generated by wrapgen from template: std.w
 * @note C interface
 * @note Introduced with MPI 1.0
 * @ingroup topo
 */
int MPI_Cart_shift(MPI_Comm comm,
                   int      direction,
                   int      disp,
                   int*     rank_source,
                   int*     rank_dest)
{
  int return_val;

  if (IS_EVENT_GEN_ON_FOR(TOPO))
  {
    EVENT_GEN_OFF();
    esd_enter(epk_mpi_regid[EPK__MPI_CART_SHIFT]);

    return_val = PMPI_Cart_shift(comm,
                                 direction,
                                 disp,
                                 rank_source,
                                 rank_dest);

    esd_exit(epk_mpi_regid[EPK__MPI_CART_SHIFT]);
    EVENT_GEN_ON();
  }
  else
  {
    return_val = PMPI_Cart_shift(comm,
                                 direction,
                                 disp,
                                 rank_source,
                                 rank_dest);
  }

  return return_val;
}

#endif
#if defined(HAS_MPI_CARTDIM_GET) && !defined(NO_MPI_EXTRA) \
  && !defined(NO_MPI_TOPO)
/**
 * Measurement wrapper for MPI_Cartdim_get
 * @note Auto-generated by wrapgen from template: std.w
 * @note C interface
 * @note Introduced with MPI 1.0
 * @ingroup topo
 */
int MPI_Cartdim_get(MPI_Comm comm,
                    int*     ndims)
{
  int return_val;

  if (IS_EVENT_GEN_ON_FOR(TOPO))
  {
    EVENT_GEN_OFF();
    esd_enter(epk_mpi_regid[EPK__MPI_CARTDIM_GET]);

    return_val = PMPI_Cartdim_get(comm, ndims);

    esd_exit(epk_mpi_regid[EPK__MPI_CARTDIM_GET]);
    EVENT_GEN_ON();
  }
  else
  {
    return_val = PMPI_Cartdim_get(comm, ndims);
  }

  return return_val;
}

#endif
#if defined(HAS_MPI_DIST_GRAPH_NEIGHBORS) && !defined(NO_MPI_EXTRA) \
  && !defined(NO_MPI_TOPO)
/**
 * Measurement wrapper for MPI_Dist_graph_neighbors
 * @note Auto-generated by wrapgen from template: std.w
 * @note C interface
 * @note Introduced with MPI 2.2
 * @ingroup topo
 */
int MPI_Dist_graph_neighbors(MPI_Comm comm,
                             int      maxindegree,
                             int      sources[],
                             int      sourceweights[],
                             int      maxoutdegree,
                             int      destinations[],
                             int      destweights[])
{
  int return_val;

  if (IS_EVENT_GEN_ON_FOR(TOPO))
  {
    EVENT_GEN_OFF();
    esd_enter(epk_mpi_regid[EPK__MPI_DIST_GRAPH_NEIGHBORS]);

    return_val = PMPI_Dist_graph_neighbors(comm,
                                           maxindegree,
                                           sources,
                                           sourceweights,
                                           maxoutdegree,
                                           destinations,
                                           destweights);

    esd_exit(epk_mpi_regid[EPK__MPI_DIST_GRAPH_NEIGHBORS]);
    EVENT_GEN_ON();
  }
  else
  {
    return_val = PMPI_Dist_graph_neighbors(comm,
                                           maxindegree,
                                           sources,
                                           sourceweights,
                                           maxoutdegree,
                                           destinations,
                                           destweights);
  }

  return return_val;
}

#endif
#if defined(HAS_MPI_DIST_GRAPH_NEIGHBORS_COUNT) \
  && !defined(NO_MPI_EXTRA) && !defined(NO_MPI_TOPO)
/**
 * Measurement wrapper for MPI_Dist_graph_neighbors_count
 * @note Auto-generated by wrapgen from template: std.w
 * @note C interface
 * @note Introduced with MPI 2.2
 * @ingroup topo
 */
int MPI_Dist_graph_neighbors_count(MPI_Comm comm,
                                   int*     indegree,
                                   int*     outdegree,
                                   int*     weighted)
{
  int return_val;

  if (IS_EVENT_GEN_ON_FOR(TOPO))
  {
    EVENT_GEN_OFF();
    esd_enter(epk_mpi_regid[EPK__MPI_DIST_GRAPH_NEIGHBORS_COUNT]);

    return_val = PMPI_Dist_graph_neighbors_count(comm,
                                                 indegree,
                                                 outdegree,
                                                 weighted);

    esd_exit(epk_mpi_regid[EPK__MPI_DIST_GRAPH_NEIGHBORS_COUNT]);
    EVENT_GEN_ON();
  }
  else
  {
    return_val = PMPI_Dist_graph_neighbors_count(comm,
                                                 indegree,
                                                 outdegree,
                                                 weighted);
  }

  return return_val;
}

#endif
#if defined(HAS_MPI_GRAPH_GET) && !defined(NO_MPI_EXTRA) \
  && !defined(NO_MPI_TOPO)
/**
 * Measurement wrapper for MPI_Graph_get
 * @note Auto-generated by wrapgen from template: std.w
 * @note C interface
 * @note Introduced with MPI 1.0
 * @ingroup topo
 */
int MPI_Graph_get(MPI_Comm comm,
                  int      maxindex,
                  int      maxedges,
                  int*     index,
                  int*     edges)
{
  int return_val;

  if (IS_EVENT_GEN_ON_FOR(TOPO))
  {
    EVENT_GEN_OFF();
    esd_enter(epk_mpi_regid[EPK__MPI_GRAPH_GET]);

    return_val = PMPI_Graph_get(comm, maxindex, maxedges, index, edges);

    esd_exit(epk_mpi_regid[EPK__MPI_GRAPH_GET]);
    EVENT_GEN_ON();
  }
  else
  {
    return_val = PMPI_Graph_get(comm, maxindex, maxedges, index, edges);
  }

  return return_val;
}

#endif
#if defined(HAS_MPI_GRAPH_MAP) && !defined(NO_MPI_EXTRA) \
  && !defined(NO_MPI_TOPO)
/**
 * Measurement wrapper for MPI_Graph_map
 * @note Auto-generated by wrapgen from template: std.w
 * @note C interface
 * @note Introduced with MPI 1.0
 * @ingroup topo
 */
int MPI_Graph_map(MPI_Comm comm,
                  int      nnodes,
                  int*     index,
                  int*     edges,
                  int*     newrank)
{
  int return_val;

  if (IS_EVENT_GEN_ON_FOR(TOPO))
  {
    EVENT_GEN_OFF();
    esd_enter(epk_mpi_regid[EPK__MPI_GRAPH_MAP]);

    return_val = PMPI_Graph_map(comm, nnodes, index, edges, newrank);

    esd_exit(epk_mpi_regid[EPK__MPI_GRAPH_MAP]);
    EVENT_GEN_ON();
  }
  else
  {
    return_val = PMPI_Graph_map(comm, nnodes, index, edges, newrank);
  }

  return return_val;
}

#endif
#if defined(HAS_MPI_GRAPH_NEIGHBORS) && !defined(NO_MPI_EXTRA) \
  && !defined(NO_MPI_TOPO)
/**
 * Measurement wrapper for MPI_Graph_neighbors
 * @note Auto-generated by wrapgen from template: std.w
 * @note C interface
 * @note Introduced with MPI 1.0
 * @ingroup topo
 */
int MPI_Graph_neighbors(MPI_Comm comm,
                        int      rank,
                        int      maxneighbors,
                        int*     neighbors)
{
  int return_val;

  if (IS_EVENT_GEN_ON_FOR(TOPO))
  {
    EVENT_GEN_OFF();
    esd_enter(epk_mpi_regid[EPK__MPI_GRAPH_NEIGHBORS]);

    return_val = PMPI_Graph_neighbors(comm, rank, maxneighbors, neighbors);

    esd_exit(epk_mpi_regid[EPK__MPI_GRAPH_NEIGHBORS]);
    EVENT_GEN_ON();
  }
  else
  {
    return_val = PMPI_Graph_neighbors(comm, rank, maxneighbors, neighbors);
  }

  return return_val;
}

#endif
#if defined(HAS_MPI_GRAPH_NEIGHBORS_COUNT) && !defined(NO_MPI_EXTRA) \
  && !defined(NO_MPI_TOPO)
/**
 * Measurement wrapper for MPI_Graph_neighbors_count
 * @note Auto-generated by wrapgen from template: std.w
 * @note C interface
 * @note Introduced with MPI 1.0
 * @ingroup topo
 */
int MPI_Graph_neighbors_count(MPI_Comm comm,
                              int      rank,
                              int*     nneighbors)
{
  int return_val;

  if (IS_EVENT_GEN_ON_FOR(TOPO))
  {
    EVENT_GEN_OFF();
    esd_enter(epk_mpi_regid[EPK__MPI_GRAPH_NEIGHBORS_COUNT]);

    return_val = PMPI_Graph_neighbors_count(comm, rank, nneighbors);

    esd_exit(epk_mpi_regid[EPK__MPI_GRAPH_NEIGHBORS_COUNT]);
    EVENT_GEN_ON();
  }
  else
  {
    return_val = PMPI_Graph_neighbors_count(comm, rank, nneighbors);
  }

  return return_val;
}

#endif
#if defined(HAS_MPI_GRAPHDIMS_GET) && !defined(NO_MPI_EXTRA) \
  && !defined(NO_MPI_TOPO)
/**
 * Measurement wrapper for MPI_Graphdims_get
 * @note Auto-generated by wrapgen from template: std.w
 * @note C interface
 * @note Introduced with MPI 1.0
 * @ingroup topo
 */
int MPI_Graphdims_get(MPI_Comm comm,
                      int*     nnodes,
                      int*     nedges)
{
  int return_val;

  if (IS_EVENT_GEN_ON_FOR(TOPO))
  {
    EVENT_GEN_OFF();
    esd_enter(epk_mpi_regid[EPK__MPI_GRAPHDIMS_GET]);

    return_val = PMPI_Graphdims_get(comm, nnodes, nedges);

    esd_exit(epk_mpi_regid[EPK__MPI_GRAPHDIMS_GET]);
    EVENT_GEN_ON();
  }
  else
  {
    return_val = PMPI_Graphdims_get(comm, nnodes, nedges);
  }

  return return_val;
}

#endif
#if defined(HAS_MPI_TOPO_TEST) && !defined(NO_MPI_EXTRA) \
  && !defined(NO_MPI_TYPE)
/**
 * Measurement wrapper for MPI_Topo_test
 * @note Auto-generated by wrapgen from template: std.w
 * @note C interface
 * @note Introduced with MPI 1.0
 * @ingroup topo
 */
int MPI_Topo_test(MPI_Comm comm,
                  int*     status)
{
  int return_val;

  if (IS_EVENT_GEN_ON_FOR(TOPO))
  {
    EVENT_GEN_OFF();
    esd_enter(epk_mpi_regid[EPK__MPI_TOPO_TEST]);

    return_val = PMPI_Topo_test(comm, status);

    esd_exit(epk_mpi_regid[EPK__MPI_TOPO_TEST]);
    EVENT_GEN_ON();
  }
  else
  {
    return_val = PMPI_Topo_test(comm, status);
  }

  return return_val;
}

#endif
#if defined(HAS_MPI_DIMS_CREATE) && !defined(NO_MPI_EXTRA) \
  && !defined(NO_MPI_TOPO)
/**
 * Measurement wrapper for MPI_Dims_create
 * @note Auto-generated by wrapgen from template: std.w
 * @note C interface
 * @note Introduced with MPI 1.0
 * @ingroup topo
 */
int MPI_Dims_create(int  nnodes,
                    int  ndims,
                    int* dims)
{
  int return_val;

  if (IS_EVENT_GEN_ON_FOR(TOPO))
  {
    EVENT_GEN_OFF();
    esd_enter(epk_mpi_regid[EPK__MPI_DIMS_CREATE]);

    return_val = PMPI_Dims_create(nnodes, ndims, dims);

    esd_exit(epk_mpi_regid[EPK__MPI_DIMS_CREATE]);
    EVENT_GEN_ON();
  }
  else
  {
    return_val = PMPI_Dims_create(nnodes, ndims, dims);
  }

  return return_val;
}

#endif
/**
 * @}
 */
