/*
 * scamper_do_tbit.c
 *
 * $Id: scamper_do_tbit.c,v 1.22 2010/05/13 03:31:41 mjl Exp $
 *
 * Copyright (C) 2009-2010 Ben Stasiewicz
 * Copyright (C) 2010 University of Waikato
 *
 * This file implements algorithms described in the tbit-1.0 source code,
 * as well as the papers:
 *
 *  "On Inferring TCP Behaviour"
 *      by Jitendra Padhye and Sally Floyd
 *  "Measuring the Evolution of Transport Protocols in the Internet" by
 *      by Alberto Medina, Mark Allman, and Sally Floyd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef lint
static const char rcsid[] =
  "$Id: scamper_do_tbit.c,v 1.22 2010/05/13 03:31:41 mjl Exp $";
#endif

#include <sys/types.h>

#if defined(_MSC_VER)
typedef unsigned __int8 uint8_t;
typedef unsigned __int16 uint16_t;
typedef unsigned __int32 uint32_t;
#define __func__ __FUNCTION__
#endif

#ifndef _WIN32
#include <sys/socket.h>
#include <sys/time.h>
#endif

#if defined(__linux__)
#define __FAVOR_BSD
#endif

#ifndef _WIN32
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/ip6.h>
#include <netinet/ip_icmp.h>
#include <netinet/icmp6.h>
#include <netinet/tcp.h>
#include <unistd.h>
#endif

#if defined(__APPLE__)
#include <stdint.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

#ifdef _WIN32
#include <winsock2.h>
#include <ws2tcpip.h>
#endif

#include <assert.h>

#if defined(DMALLOC)
#include <dmalloc.h>
#endif

#include "scamper.h"
#include "scamper_addr.h"
#include "scamper_list.h"
#include "scamper_task.h"
#include "scamper_fds.h"
#include "scamper_dl.h"
#include "scamper_dlhdr.h"
#include "scamper_firewall.h"
#include "scamper_rtsock.h"
#include "scamper_if.h"
#include "scamper_probe.h"
#include "scamper_getsrc.h"
#include "scamper_tcp4.h"
#include "scamper_tcp6.h"
#include "scamper_queue.h"
#include "scamper_file.h"
#include "scamper_options.h"
#include "scamper_debug.h"
#include "utils.h"
#include "mjl_list.h"
#include "scamper_tbit.h"
#include "scamper_do_tbit.h"

/* Default test parameters */
#define TBIT_DEFAULT_RETX         3
#define TBIT_DEFAULT_TIMEOUT      10000

typedef struct tbit_options
{
  uint8_t   app;
  uint8_t   type;
  uint8_t   syn_retx;
  uint8_t   dat_retx;
  uint8_t   ptb_retx;
  uint16_t  mss;
  uint16_t  mtu;
  uint16_t  sport;
  uint16_t  dport;
  char     *url;
} tbit_options_t;

typedef struct tbit_segment
{
  struct timeval  tx;
  uint32_t        seq;
  uint8_t        *data;
  uint16_t        len;
} tbit_segment_t;

typedef struct tbit_frag
{
  uint16_t         off;
  uint8_t         *data;
  uint16_t         datalen;
} tbit_frag_t;

typedef struct tbit_frags
{
  struct timeval   tv;
  uint32_t         id;
  tbit_frag_t    **frags;
  int              fragc;
  uint8_t          gotlast;
} tbit_frags_t;

typedef struct tbit_probe
{
  uint8_t type;
  int     wait;
  union
  {
    struct tp_tcp
    {
      uint8_t  flags;
      uint32_t seq;
      uint32_t ack;
      uint16_t len;
    } tcp;
  } un;
} tbit_probe_t;

typedef struct tbit_state
{
  scamper_fd_t               *rt;
  scamper_fd_t               *dl;
  scamper_dlhdr_t            *dlhdr;
  scamper_firewall_entry_t   *fw;
  uint8_t                     mode;
  uint8_t                     flags;
  uint8_t                     attempt;
  struct timeval              timeout;
  uint16_t                    ipid;

  slist_t                    *tx;
  slist_t                    *segments;
  uint32_t                    seq;
  uint32_t                    expected_seq;

  tbit_frags_t              **frags;
  int                         fragc;

  uint8_t                    *ptb_data;
  uint16_t                    ptb_datalen;
  uint16_t                    ptb_count;
} tbit_state_t;

/* The callback functions registered with the tbit task */
static scamper_task_funcs_t tbit_funcs;

/* Source port to use in our probes */
static uint16_t default_sport;

/* Address cache used to avoid reallocating the same address multiple times */
extern scamper_addrcache_t *addrcache;

#define TBIT_STATE_FLAG_FIN_SEEN      0x01
#define TBIT_STATE_FLAG_FIN_ACKED     0x02
#define TBIT_STATE_FLAG_SEEN_DATA     0x04
#define TBIT_STATE_FLAG_SEEN_220      0x08
#define TBIT_STATE_FLAG_NODF          0x10
#define TBIT_STATE_FLAG_RST_SEEN      0x20

/* Options that tbit supports */
#define SCAMPER_TBIT_OPT_DPORT       1
#define SCAMPER_TBIT_OPT_MSS         2
#define SCAMPER_TBIT_OPT_MTU         3
#define SCAMPER_TBIT_OPT_APP         4
#define SCAMPER_TBIT_OPT_SPORT       5
#define SCAMPER_TBIT_OPT_TYPE        6
#define SCAMPER_TBIT_OPT_URL         7
#define SCAMPER_TBIT_OPT_USERID      8

static const scamper_option_in_t opts[] = {
  {'d', NULL, SCAMPER_TBIT_OPT_DPORT,    SCAMPER_OPTION_TYPE_NUM},
  {'m', NULL, SCAMPER_TBIT_OPT_MSS,      SCAMPER_OPTION_TYPE_NUM},
  {'M', NULL, SCAMPER_TBIT_OPT_MTU,      SCAMPER_OPTION_TYPE_NUM},
  {'p', NULL, SCAMPER_TBIT_OPT_APP,      SCAMPER_OPTION_TYPE_STR},
  {'s', NULL, SCAMPER_TBIT_OPT_SPORT,    SCAMPER_OPTION_TYPE_NUM},
  {'t', NULL, SCAMPER_TBIT_OPT_TYPE,     SCAMPER_OPTION_TYPE_STR},
  {'u', NULL, SCAMPER_TBIT_OPT_URL,      SCAMPER_OPTION_TYPE_STR},
  {'U', NULL, SCAMPER_TBIT_OPT_USERID,   SCAMPER_OPTION_TYPE_NUM},
};
static const int opts_cnt = SCAMPER_OPTION_COUNT(opts);

static const uint8_t MODE_RTSOCK    = 1; /* waiting for rtsock */
static const uint8_t MODE_DLHDR     = 2; /* waiting for dlhdr to use */
static const uint8_t MODE_DONE      = 3; /* test finished */
static const uint8_t MODE_SYN       = 4; /* waiting for syn/ack */
static const uint8_t MODE_FIN       = 5; /* send fin, wait for ack */
static const uint8_t MODE_DATA      = 6; /* connection established */
static const uint8_t MODE_PMTUD     = 7; /* sending PTBs */

#define TBIT_PROBE_TYPE_TCP 1
#define TBIT_PROBE_TYPE_PTB 2

/* Note : URL is only valid for HTTP tests. */
const char *scamper_do_tbit_usage(void)
{
  return "tbit [-t type] [-p app] [-d dport] [-s sport] [-m mss] [-M mtu]\n"
         "     [-O option] [-u url]";
}

static void tbit_queue(scamper_task_t *task)
{
  tbit_state_t *state = task->state;

  if(slist_count(state->tx) > 0)
    scamper_queue_probe(task->queue);
  else if(state->mode == MODE_DONE)
    scamper_queue_done(task->queue, 0);
  else
    scamper_queue_wait_tv(task->queue, &state->timeout);

  return;
}

/*
 * tbit_result:
 *
 * record the result, and then begin to gracefully end the connection.
 */
static void tbit_result(scamper_task_t *task, uint8_t result)
{
  scamper_tbit_t *tbit = task->data;
  tbit_state_t *state = task->state;
  char buf[16], addr[64];
  char *res;
  int d = 0;

  switch(result)
    {
    case SCAMPER_TBIT_RESULT_NONE: res = "none"; d = 1; break;
    case SCAMPER_TBIT_RESULT_TCP_NOCONN: res = "tcp-noconn"; d = 1; break;
    case SCAMPER_TBIT_RESULT_TCP_NOCONN_RST: res = "tcp-noconn-rst";d=1;break;
    case SCAMPER_TBIT_RESULT_TCP_RST: res = "tcp-rst"; d = 1; break;
    case SCAMPER_TBIT_RESULT_TCP_ERROR: res = "tcp-error"; d = 1; break;
    case SCAMPER_TBIT_RESULT_ERROR: res = "sys-error"; d = 1; break;
    case SCAMPER_TBIT_RESULT_ABORTED: res = "aborted"; d = 1; break;
    case SCAMPER_TBIT_RESULT_PMTUD_NOACK: res = "pmtud-noack"; d = 1; break;
    case SCAMPER_TBIT_RESULT_PMTUD_NODATA: res = "pmtud-nodata"; break;
    case SCAMPER_TBIT_RESULT_PMTUD_TOOSMALL: res = "pmtud-toosmall"; break;
    case SCAMPER_TBIT_RESULT_PMTUD_NODF: res = "pmtud-nodf"; break;
    case SCAMPER_TBIT_RESULT_PMTUD_FAIL: res = "pmtud-fail"; d = 1; break;
    case SCAMPER_TBIT_RESULT_PMTUD_SUCCESS: res = "pmtud-success"; break;
    case SCAMPER_TBIT_RESULT_PMTUD_CLEARDF: res = "pmtud-cleardf"; break;
    default: snprintf(buf, sizeof(buf), "%d", result); res = buf; break;
    }

  if(tbit->result == SCAMPER_TBIT_RESULT_NONE)
    {
      tbit->result = result;
      scamper_addr_tostr(tbit->dst, addr, sizeof(addr));
      scamper_debug(__func__, "%s %s", addr, res);
    }

  if(d == 0)
    {
      state->mode = MODE_FIN;
      scamper_queue_probe(task->queue);
    }
  else
    {
      state->mode = MODE_DONE;
      scamper_queue_done(task->queue, 0);
    }

  return;
}

static void tbit_classify(scamper_task_t *task)
{
  scamper_tbit_t *tbit = task->data;
  tbit_state_t *state = task->state;
  scamper_tbit_pmtud_t *pmtud = tbit->data;
  int ipv6 = 0;

  if(tbit->result != SCAMPER_TBIT_RESULT_NONE)
    {
      if(state->flags & TBIT_STATE_FLAG_RST_SEEN)
	{
	  state->mode = MODE_DONE;
	  scamper_queue_done(task->queue, 0);
	}
      return;
    }

  if(tbit->dst->type == SCAMPER_ADDR_TYPE_IPV6)
    ipv6 = 1;

  /*
   * if we receive a reset, then the measurement is finished.
   * if we have sent two PTBs, we class it as a failure.
   */
  if(state->flags & TBIT_STATE_FLAG_RST_SEEN)
    {
      if(state->ptb_count >= 2)
	tbit_result(task, SCAMPER_TBIT_RESULT_PMTUD_FAIL);
      else
	tbit_result(task, SCAMPER_TBIT_RESULT_TCP_RST);
    }
  /* if we haven't seen any data */
  else if((state->flags & TBIT_STATE_FLAG_SEEN_DATA) == 0)
    tbit_result(task, SCAMPER_TBIT_RESULT_PMTUD_NODATA);
  /* if we were trying to solicit a fragmentation header but couldn't */
  else if(state->ptb_count == 0 && ipv6 && pmtud->mtu < 1280)
    tbit_result(task, SCAMPER_TBIT_RESULT_NONE);
  /* if we sent any PTBs but didn't see an appropriate reduction, fail */
  else if(state->ptb_count > 0)
    tbit_result(task, SCAMPER_TBIT_RESULT_PMTUD_FAIL);
  /* if the IP DF was not set on sufficiently large packets */
  else if(ipv6 == 0 && state->flags & TBIT_STATE_FLAG_NODF)
    tbit_result(task, SCAMPER_TBIT_RESULT_PMTUD_NODF);
  /* otherwise, we just didn't see a sufficiently large packet */
  else 
    tbit_result(task, SCAMPER_TBIT_RESULT_PMTUD_TOOSMALL);

  return;
}

static void tbit_handleerror(scamper_task_t *task, int error)
{
  scamper_tbit_t *tbit = task->data;
  tbit_state_t *state = task->state;
  tbit->result = SCAMPER_TBIT_RESULT_ERROR;
  if(state != NULL) state->mode = MODE_DONE;
  scamper_queue_done(task->queue, 0);
  return;
}

static void tbit_frags_free(tbit_frags_t *frags)
{
  int i;

  if(frags == NULL)
    return;

  if(frags->frags != NULL)
    {
      for(i=0; i<frags->fragc; i++)
	{
	  free(frags->frags[i]->data);
	  free(frags->frags[i]);
	}
      free(frags->frags);
    }
  free(frags);
  return;
}

static int tbit_frags_cmp(const void *va, const void *vb)
{
  const tbit_frags_t *a = *((const tbit_frags_t **)va);
  const tbit_frags_t *b = *((const tbit_frags_t **)vb);
  if(a->id < b->id) return -1;
  if(a->id > b->id) return  1;
  return 0;
}

static int tbit_frag_cmp(const void *va, const void *vb)
{
  const tbit_frag_t *a = *((const tbit_frag_t **)va);
  const tbit_frag_t *b = *((const tbit_frag_t **)vb);
  if(a->off < b->off) return -1;
  if(a->off > b->off) return  1;
  return 0;
}

static int tbit_reassemble(scamper_task_t *task, scamper_dl_rec_t **out,
			   scamper_dl_rec_t *dl)
{
  scamper_tbit_t *tbit = task->data;
  tbit_state_t *state = task->state;
  scamper_tbit_pmtud_t *pmtud;
  scamper_dl_rec_t *newp = NULL;
  tbit_frags_t fmfs, *frags;
  tbit_frag_t fmf, *frag, *next;
  uint8_t *data = NULL;
  size_t off;
  uint16_t mtu;
  int i, rc, pos, ipv4 = 0, ipv6 = 0;

  /* empty fragment? */
  if(dl->dl_ip_datalen == 0 || dl->dl_ip_proto != IPPROTO_TCP)
    {
      scamper_debug(__func__, "ignoring fragment %d %d",
		    dl->dl_ip_datalen, dl->dl_ip_proto);
      return 0;
    }

  if(dl->dl_af == AF_INET)
    ipv4 = 1;
  else
    ipv6 = 1;

  /* if we are doing path mtu discovery, the fragment might not be accepted */
  if(tbit->type == SCAMPER_TBIT_TYPE_PMTUD)
    {
      pmtud = tbit->data;
      mtu = pmtud->mtu;
      
      if(ipv4 || pmtud->mtu > 1280)
	mtu = pmtud->mtu;
      else
	mtu = 1280;

      /*
       * if the packet is larger than the psuedo mtu, we can't reassemble
       * it since in theory we didn't receive it.
       * if the fragment offset is zero, pass it back to trigger a PTB.
       */
      if((ipv6 || SCAMPER_DL_IS_IP_DF(dl)) && dl->dl_net_rawlen > mtu)
	{
	  if(dl->dl_ip_off == 0)
	    *out = dl;
	  return 0;
	}
    }

  /* see if we have other fragments for this packet. if not, create new rec */
  if(ipv4)
    fmfs.id = dl->dl_ip_id;
  else
    fmfs.id = dl->dl_ip6_id;
  pos = array_findpos((void **)state->frags,state->fragc,&fmfs,tbit_frags_cmp);
  if(pos >= 0)
    {
      frags = state->frags[pos];
    }
  else
    {
      if((frags = malloc_zero(sizeof(tbit_frags_t))) == NULL)
	{
	  printerror(errno, strerror, __func__, "could not malloc frags");
	  goto err;
	}
      frags->id = fmfs.id;
      rc=array_insert((void ***)&state->frags,&state->fragc,frags,
		      tbit_frags_cmp);
      if(rc != 0)
	{
	  printerror(errno, strerror, __func__, "could not insert frags");
	  goto err;
	}
      pos = array_findpos((void **)state->frags, state->fragc, frags,
			  tbit_frags_cmp);
      assert(pos != -1);
    }

  /* add the fragment to the collection */
  fmf.off = dl->dl_ip_off;
  frag = array_find((void **)frags->frags,frags->fragc,&fmf,tbit_frag_cmp);
  if(frag == NULL)
    {
      if((frag = malloc_zero(sizeof(tbit_frags_t))) == NULL)
	{
	  printerror(errno, strerror, __func__, "could not malloc frag");
	  goto err;
	}
      frag->off = fmf.off;

      if((frag->data = memdup(dl->dl_ip_data, dl->dl_ip_datalen)) == NULL)
	{
	  printerror(errno, strerror, __func__, "could not dup %d",
		     dl->dl_ip_datalen);
	  goto err;
	}
      frag->datalen = dl->dl_ip_datalen;

      if(array_insert((void ***)&frags->frags, &frags->fragc, frag,
		      tbit_frag_cmp) != 0)
	{
	  printerror(errno, strerror, __func__, "could not add frag");
	  goto err;
	}

      if(SCAMPER_DL_IS_IP_MF(dl) == 0)
	frags->gotlast = 1;
    }

  /* can't reassemble a packet without the last fragment */
  if(frags->gotlast == 0 || frags->fragc < 2)
    {
      return 0;
    }

  for(i=0; i<frags->fragc-1; i++)
    {
      frag = frags->frags[i];
      next = frags->frags[i+1];

      if(frag->off + frag->datalen != next->off)
	{
	  return 0;
	}
    }

  frag = frags->frags[frags->fragc-1];
  if((data = malloc(frag->off + frag->datalen)) == NULL)
    {
      printerror(errno, strerror, __func__, "could not malloc data");
      goto err;
    }
  for(i=0, off=0; i<frags->fragc; i++)
    {
      frag = frags->frags[i];
      memcpy(data+off, frag->data, frag->datalen);
      off += frag->datalen;
    }
  array_remove((void **)state->frags, &state->fragc, pos);
  tbit_frags_free(frags);

  if((newp = malloc_zero(sizeof(scamper_dl_rec_t))) == NULL)
    {
      printerror(errno, strerror, __func__, "could not malloc newp");
      goto err;
    }

  timeval_cpy(&newp->dl_tv, &dl->dl_tv);
  newp->dl_type       = SCAMPER_DL_TYPE_RAW;
  newp->dl_net_type   = SCAMPER_DL_REC_NET_TYPE_IP;
  newp->dl_ifindex    = dl->dl_ifindex;
  newp->dl_af         = dl->dl_af;
  newp->dl_ip_hl      = dl->dl_ip_hl;
  newp->dl_ip_proto   = dl->dl_ip_proto;
  newp->dl_ip_size    = dl->dl_ip_hl + off;
  newp->dl_ip_id      = dl->dl_ip_id;
  newp->dl_ip6_id     = dl->dl_ip6_id;
  newp->dl_ip_tos     = dl->dl_ip_tos;
  newp->dl_ip_ttl     = dl->dl_ip_ttl;
  newp->dl_ip_src     = dl->dl_ip_src;
  newp->dl_ip_dst     = dl->dl_ip_dst;
  newp->dl_ip_flow    = dl->dl_ip_flow;
  newp->dl_ip_data    = data;
  newp->dl_ip_datalen = off;
  newp->dl_ip_flags   = SCAMPER_DL_IP_FLAG_REASS;

  if(sizeof(struct tcphdr) > newp->dl_ip_datalen)
    {
      free(newp);
      free(data);
      return 0;
    }

  newp->dl_tcp_sport   = bytes_ntohs(data+0);
  newp->dl_tcp_dport   = bytes_ntohs(data+2);
  newp->dl_tcp_seq     = bytes_ntohl(data+4);
  newp->dl_tcp_ack     = bytes_ntohl(data+8);
  newp->dl_tcp_hl      = (data[12] >> 4) * 4;
  newp->dl_tcp_flags   = data[13];
  newp->dl_tcp_win     = bytes_ntohs(data+14);
  newp->dl_tcp_datalen = newp->dl_ip_datalen - newp->dl_tcp_hl;
  if(newp->dl_tcp_datalen > 0)
    newp->dl_tcp_data  = data + newp->dl_tcp_hl;

  *out = newp;
  return 0;

 err:
  if(newp != NULL) free(newp);
  if(data != NULL) free(data);
  return -1;
}

static void tp_free(tbit_probe_t *tp)
{
  if(tp == NULL)
    return;
  free(tp);
  return;
}

static tbit_probe_t *tp_alloc(tbit_state_t *state, uint8_t type)
{
  tbit_probe_t *tp;

  if((tp = malloc_zero(sizeof(tbit_probe_t))) == NULL)
    {
      printerror(errno, strerror, __func__, "could not malloc tp");
      return NULL;
    }
  if(slist_tail_push(state->tx, tp) == NULL)
    {
      printerror(errno, strerror, __func__, "could not queue tp");
      free(tp);
      return NULL;
    }

  tp->type = type;
  return tp;
}

static tbit_probe_t *tp_tcp(tbit_state_t *state, uint32_t seq, uint32_t ack,
			    uint16_t len)
{
  tbit_probe_t *tp;

  if((tp = tp_alloc(state, TBIT_PROBE_TYPE_TCP)) == NULL)
    return NULL;

  tp->un.tcp.flags = TH_ACK;
  tp->un.tcp.seq   = seq;
  tp->un.tcp.ack   = ack;
  tp->un.tcp.len   = len;

  return tp;
}

static tbit_probe_t *tp_ptb(tbit_state_t *state)
{
  return tp_alloc(state, TBIT_PROBE_TYPE_PTB);
}

static void tbit_segment_free(tbit_segment_t *seg)
{
  if(seg == NULL)
    return;
  if(seg->data != NULL)
    free(seg->data);
  free(seg);
  return;
}

static int tbit_segment(tbit_state_t *state, const uint8_t *data, uint16_t len)
{
  tbit_segment_t *seg = NULL;
  slist_node_t *node;
  uint32_t seq;

  if((node = slist_head_node(state->segments)) == NULL)
    {
      seq = state->seq;
    }
  else
    {
      while(slist_node_next(node) != NULL)
	node = slist_node_next(node);
      seg = slist_node_item(node);
      seq = seg->seq + seg->len;
    }
  seg = NULL;

  if((seg = malloc_zero(sizeof(tbit_segment_t))) == NULL)
    {
      printerror(errno, strerror, __func__, "could not malloc seg");
      goto err;
    }
  if((seg->data = memdup(data, len)) == NULL)
    {
      printerror(errno, strerror, __func__, "could not malloc seg->data");
      goto err;
    }
  seg->len = len;
  if(slist_tail_push(state->segments, seg) == NULL)
    {
      printerror(errno, strerror, __func__, "could not add seg");
      goto err;
    }

  return 0;

 err:
  tbit_segment_free(seg);
  return -1;
}

static int tbit_app_http_rx(scamper_task_t *task, uint8_t *data, uint16_t dlen)
{
  static const char *http_ua =
    "Mozilla/5.0 (X11; U; FreeBSD i386; en-US; rv:1.9.1.2) "
    "Gecko/20090806 Firefox/3.5.2";
  scamper_tbit_t *tbit = task->data;
  scamper_tbit_app_http_t *http = tbit->app_data;
  tbit_state_t *state = task->state;
  char buf[512];
  size_t off;

  if(state->mode == MODE_SYN)
    {
      off = 0;
      string_concat(buf, sizeof(buf), &off, "GET %s HTTP/1.0\r\n", http->file);
      if(http->host != NULL)
	string_concat(buf, sizeof(buf), &off, "Host: %s\r\n", http->host);
      string_concat(buf, sizeof(buf), &off,
		    "Connection: Keep-Alive\r\n"
		    "Accept: */*\r\n"
		    "User-Agent: %s\r\n\r\n", http_ua);

      if(tbit_segment(state, (const uint8_t *)buf, off) != 0)
	return -1;
      return off;
    }

  return 0;
}

static int tbit_app_smtp_rx(scamper_task_t *task, uint8_t *data, uint16_t dlen)
{
  scamper_tbit_t *tbit = task->data;
  tbit_state_t *state = task->state;
  char buf[8192], hostname[256], *data_cpy = NULL;
  size_t off = 0, tmp, namelen;
  int i;

  if(state->mode != MODE_DATA || state->flags & TBIT_STATE_FLAG_SEEN_220)
    return 0;

  /* From the 220, can we determine which MTA we are dealing with? */
  if(dlen >= 3 && strncmp((char*)data, "220", 3) == 0)
    {
      if((data_cpy = malloc(dlen + 1)) == NULL)
	goto quit;
      memcpy(data_cpy, data, dlen);
      data_cpy[dlen] = '\0';
      state->flags |= TBIT_STATE_FLAG_SEEN_220;

      if(strcasestr(data_cpy, "sendmail") != NULL)
	{
	  string_concat(buf, sizeof(buf), &off, "HELP\r\nHELP EHLO\r\n");
	}
      else if(strcasestr(data_cpy, "postfix") != NULL)
	{
	  /* send multiple EHLOs */
	  if(gethostname(hostname, sizeof(hostname)) != 0 ||
	     (namelen = strlen(hostname)) == 0)
	    goto quit;
	  for(i=0; i<20; i++)
	    string_concat(buf, sizeof(buf), &off, "EHLO %s\r\n", hostname);
	}
      else if(strcasestr(data_cpy, "exim") != NULL)
	{
	  /* Send one EHLO with a really long domain name */
	  if(gethostname(hostname, sizeof(hostname)) != 0 ||
	     (namelen = strlen(hostname)) == 0)
	    goto quit;
	  if(tbit->server_mss > 1280)
	    tmp = tbit->server_mss;
	  else
	    tmp = 1280;
	  string_concat(buf, sizeof(buf), &off, "EHLO ");
	  while(off + namelen + namelen < 1280)
	    string_concat(buf, sizeof(buf), &off, "%s.", hostname);
	  string_concat(buf, sizeof(buf), &off, "%s\r\n", hostname);
	}
      free(data_cpy); data_cpy = NULL;
    }

  /*
   * Send a quit if the response wasn't a 220 from either Sendmail,
   * Postfix or Exim
   */
 quit:
  if(off == 0)
    string_concat(buf, sizeof(buf), &off, "QUIT\r\n");

  /* Create the TCP segment */
  if(tbit_segment(state, (uint8_t *)buf, off) != 0)
    return -1;

  return off;
}

static int tbit_app_dns_rx(scamper_task_t *task, uint8_t *data, uint16_t dlen)
{
  /* recursive DNS request for the TXT record on tbit.staz.net.nz */
  static const uint8_t dns_request[] = {
    0x00, 0x22, 0x05, 0x4a,
    0x01, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x04, 0x74,
    0x62, 0x69, 0x74, 0x04,
    0x73, 0x74, 0x61, 0x7a,
    0x03, 0x6e, 0x65, 0x74,
    0x02, 0x6e, 0x7a, 0x00,
    0x00, 0x10, 0x00, 0x01,
  };
  tbit_state_t *state = task->state;

  if(state->mode == MODE_SYN)
    {
      if(tbit_segment(state, dns_request, sizeof(dns_request)) != 0)
	return -1;
      return sizeof(dns_request);
    }

  return 0;
}

static int tbit_app_ftp_rx(scamper_task_t *task, uint8_t *data, uint16_t dlen)
{
  return 0;
}

static int tbit_app_rx(scamper_task_t *task, uint8_t *data, uint16_t len)
{
  static int (* const func[])(scamper_task_t *, uint8_t *, uint16_t) = {
    NULL,
    tbit_app_http_rx,
    tbit_app_smtp_rx,
    tbit_app_dns_rx,
    tbit_app_ftp_rx,
  };
  scamper_tbit_t *tbit = task->data;

  assert(tbit->app_proto != 0);
  assert(tbit->app_proto <= 4);

  return func[tbit->app_proto](task, data, len);
}

/*
 * dl_syn:
 *
 * handles the response to a SYN - It should be a SYN/ACK.
 */
static void dl_syn(scamper_task_t *task, scamper_dl_rec_t *dl)
{
  scamper_tbit_t *tbit = task->data;
  tbit_state_t *state = task->state;
  tbit_probe_t *tp = NULL;
  int rc;

  /*
   * make sure it has the SYN/ACK flags set and acknowledges the expected
   * sequence number.
   */
  if((dl->dl_tcp_flags & (TH_SYN|TH_ACK)) != (TH_SYN|TH_ACK) ||
     dl->dl_tcp_ack != state->seq + 1)
    {
      if(dl->dl_tcp_flags & TH_RST)
	tbit_result(task, SCAMPER_TBIT_RESULT_TCP_NOCONN_RST);
      else
	tbit_result(task, SCAMPER_TBIT_RESULT_TCP_NOCONN);
      return;
    }

  tbit->server_mss = dl->dl_tcp_mss;

  /* increment our sequence number, and remember the seq we expect from them */
  state->seq++;
  state->expected_seq = dl->dl_tcp_seq + 1;

  /* send a syn-ack, figure out if we have data to send */
  if((rc = tbit_app_rx(task, NULL, 0)) < 0 ||
     tp_tcp(state, state->seq, state->expected_seq, 0) == NULL)
    {
      tbit_handleerror(task, errno);
      return;
    }

  /* send our request if there is one */
  if(rc > 0)
    {
      if((tp = tp_tcp(state, state->seq, state->expected_seq, rc)) == NULL)
	{
	  tbit_handleerror(task, errno);
	  return;
	}
      tp->wait = TBIT_DEFAULT_TIMEOUT;
      state->attempt = 0;
    }

  state->mode = MODE_DATA;
  tbit_queue(task);
  return;
}

static void timeout_rt(scamper_task_t *task)
{
  tbit_result(task, SCAMPER_TBIT_RESULT_ERROR);
  return;
}

static void timeout_dlhdr(scamper_task_t *task)
{
  tbit_result(task, SCAMPER_TBIT_RESULT_ERROR);
  return;
}

static void timeout_syn(scamper_task_t *task)
{
  scamper_tbit_t *tbit = task->data;
  tbit_state_t *state = task->state;
  if(state->attempt >= tbit->syn_retx)
    tbit_result(task, SCAMPER_TBIT_RESULT_TCP_NOCONN);
  return;
}

static void dl_fin(scamper_task_t *task, scamper_dl_rec_t *dl)
{
  scamper_tbit_t *tbit = task->data;
  tbit_state_t *state = task->state;
  tbit_probe_t *tp;

  if((dl->dl_tcp_flags & TH_ACK) == 0)
    return;

  timeval_add_s(&state->timeout, &dl->dl_tv, 70);

  /* see if the remote host has acked our fin */
  if(dl->dl_tcp_ack == state->seq + 1 &&
     (state->flags & TBIT_STATE_FLAG_FIN_ACKED) == 0)
    {
      state->flags |= TBIT_STATE_FLAG_FIN_ACKED;
      state->seq++;
    }

  /* check if the remote host has received our ack to their fin */
  if(dl->dl_tcp_seq == state->expected_seq + 1)
    {
      if((state->flags & TBIT_STATE_FLAG_FIN_SEEN) != 0 &&
	 (state->flags & TBIT_STATE_FLAG_FIN_ACKED) != 0)
	{
	  if(tbit->result == SCAMPER_TBIT_RESULT_NONE)
	    tbit_classify(task);

	  state->mode = MODE_DONE;
	  tbit_queue(task);
	}
      return;
    }

  if(dl->dl_tcp_seq != state->expected_seq)
    {
      if(tp_tcp(state, state->seq, state->expected_seq, 0) != NULL)
	tbit_queue(task);
      return;
    }

  /* if there is nothing to ack, then don't generate an ack */
  if(dl->dl_tcp_datalen == 0 && (dl->dl_tcp_flags & TH_FIN) == 0)
    return;

  state->expected_seq += dl->dl_tcp_datalen;

  if((tp = tp_tcp(state, state->seq, state->expected_seq, 0)) == NULL)
    {
      tbit_handleerror(task, errno);
      return;
    }

  if((dl->dl_tcp_flags & TH_FIN) != 0)
    {
      state->flags |= TBIT_STATE_FLAG_FIN_SEEN;
      tp->un.tcp.ack++;
    }

  if((state->flags & TBIT_STATE_FLAG_FIN_ACKED) == 0)
    tp->un.tcp.flags |= TH_FIN;

  if((state->flags & TBIT_STATE_FLAG_FIN_SEEN) != 0 &&
     (state->flags & TBIT_STATE_FLAG_FIN_ACKED) != 0)
    {
      if(tbit->result == SCAMPER_TBIT_RESULT_NONE)
	tbit_classify(task);
      state->mode = MODE_DONE;
    }

  tbit_queue(task);
  return;
}

static void timeout_fin(scamper_task_t *task)
{
  scamper_tbit_t *tbit = task->data;
  tbit_state_t *state = task->state;

  if(tbit->result == SCAMPER_TBIT_RESULT_NONE)
    tbit_classify(task);
  state->mode = MODE_DONE;
  tbit_queue(task);
  return;
}

/*
 * dl_data_pmtud
 *
 * this function is tasked with reading data from the end host until
 * it sends a packet that we can send a PTB for.  when that happens,
 * we send the PTB and then go into a mode where this function will never
 * be called again.
 */
static int dl_data_pmtud(scamper_task_t *task, scamper_dl_rec_t *dl)
{
  scamper_tbit_t *tbit = task->data;
  tbit_state_t *state = task->state;
  scamper_tbit_pmtud_t *pmtud = tbit->data;
  tbit_probe_t *tp;
  int rc, ipv4 = 0, ipv6 = 0, skip = 0, frag = 0;

  /* if it is out of sequence, then send an ack for what we want */
  if(dl->dl_tcp_seq != state->expected_seq)
    {
      if(tp_tcp(state, state->seq, state->expected_seq, 0) == NULL)
	goto err;
      return 0;
    }

  if(dl->dl_af == AF_INET)
    ipv4 = 1;
  else
    ipv6 = 1;

  if(SCAMPER_DL_IS_IP_FRAG(dl) != 0)
    frag = 1;

  /*
   * skip over tcp packets without data, packets that were allowed to be
   * reassembled, data packets without the DF bit set, data packets below
   * the MTU, and ipv6 packets with the fragmentation header set
   */
  if(dl->dl_tcp_datalen == 0 || SCAMPER_DL_IS_IP_REASS(dl) != 0)
    skip = 1;
  else if(ipv4 && SCAMPER_DL_IS_IP_DF(dl) == 0)
    skip = 1;
  else if((ipv4 || pmtud->mtu >= 1280) && dl->dl_net_rawlen <= pmtud->mtu)
    skip = 1;
  else if(ipv6 && pmtud->mtu < 1280 && frag != 0 && dl->dl_net_rawlen <= 1280)
    skip = 1;

  if(dl->dl_tcp_datalen > 0)
    state->flags |= TBIT_STATE_FLAG_SEEN_DATA;

  if(skip)
    {
      if(dl->dl_tcp_datalen > 0)
	{
	  state->expected_seq += dl->dl_tcp_datalen;

	  if(ipv4 && SCAMPER_DL_IS_IP_DF(dl) == 0 &&
	     dl->dl_net_rawlen > pmtud->mtu && SCAMPER_DL_IS_IP_REASS(dl) == 0)
	    state->flags |= TBIT_STATE_FLAG_NODF;
	}

      if(dl->dl_tcp_datalen > 0 || (dl->dl_tcp_flags & TH_FIN) != 0)
	{
	  if((rc = tbit_app_rx(task, dl->dl_tcp_data, dl->dl_tcp_datalen)) < 0)
	    goto err;
	  if((tp = tp_tcp(state, state->seq, state->expected_seq, rc)) == NULL)
	    goto err;
	  if(rc > 0)
            {
	      tp->wait = TBIT_DEFAULT_TIMEOUT;
	      state->attempt = 0;
            }

	  if((dl->dl_tcp_flags & TH_FIN) != 0)
	    {
	      tp->un.tcp.flags |= TH_FIN;
	      tp->un.tcp.ack++;
	      state->mode = MODE_FIN;
	      state->flags |= TBIT_STATE_FLAG_FIN_SEEN;
	    }
	}

      return 0;
    }

  state->mode = MODE_PMTUD;

  if(dl->dl_af == AF_INET)
    state->ptb_datalen = dl->dl_ip_hl + 8;
  else if(dl->dl_net_rawlen >= 1280-40-8)
    state->ptb_datalen = 1280 - 40 - 8;
  else
    state->ptb_datalen = dl->dl_net_rawlen;

  if((state->ptb_data = memdup(dl->dl_net_raw,state->ptb_datalen)) == NULL)
    {
      printerror(errno, strerror, __func__, "could not dup quote");
      goto err;
    }

  if(tp_ptb(state) == NULL)
    goto err;
  state->attempt = 0;

  return 0;

 err:
  tbit_handleerror(task, errno);
  return -1;
}

static void timeout_data_pmtud(scamper_task_t *task)
{
  tbit_result(task, SCAMPER_TBIT_RESULT_PMTUD_NODATA);
  return;
}

static void dl_data(scamper_task_t *task, scamper_dl_rec_t *dl)
{
  tbit_state_t *state = task->state;
  tbit_segment_t *seg;
  uint32_t ab;

  /* ensure the acknowledgement field is set */
  if((dl->dl_tcp_flags & TH_ACK) == 0)
    return;

  timeval_add_s(&state->timeout, &dl->dl_tv, 70);

  /* remove segment data from the send queue */
  if(dl->dl_tcp_ack > state->seq)
    {
      if((seg = slist_head_get(state->segments)) == NULL)
	return;

      ab = dl->dl_tcp_ack - state->seq;
      if(ab >= seg->len)
	{
	  ab = seg->len;
	  slist_head_pop(state->segments);
	  tbit_segment_free(seg);
	}
      else
	{
	  memmove(seg->data, seg->data+ab, seg->len-ab);
	}
      state->seq += ab;
    }

  dl_data_pmtud(task, dl);
  tbit_queue(task);
  return;
}

static void timeout_data(scamper_task_t *task)
{
  scamper_tbit_t *tbit = task->data;
  tbit_state_t *state = task->state;
  tbit_segment_t *seg;
  tbit_probe_t *tp;

  if((seg = slist_head_get(state->segments)) != NULL)
    {
      if(state->attempt >= tbit->dat_retx)
	{
	  timeout_data_pmtud(task);
	  return;
	}
      if((tp=tp_tcp(state, state->seq, state->expected_seq, seg->len)) == NULL)
	goto err;
      tp->wait = TBIT_DEFAULT_TIMEOUT;
      tbit_queue(task);
    }
  else
    {
      tbit_classify(task);
    }

  return;

 err:
  tbit_handleerror(task, errno);
  return;
}

/*
 * timeout_pmtud
 *
 * did not observe remote TCP changing behaviour.
 */
static void timeout_pmtud(scamper_task_t *task)
{
  tbit_classify(task);
  return;
}

/*
 * Checks the response to a PTB message.
 */
static void dl_pmtud(scamper_task_t *task, scamper_dl_rec_t *dl)
{
  scamper_tbit_t *tbit = task->data;
  tbit_state_t *state = task->state;
  scamper_tbit_pmtud_t *pmtud = tbit->data;
  tbit_probe_t *tp;
  uint16_t mtu = pmtud->mtu;
  int ipv4 = 0;
  int success = 0;

  if((dl->dl_tcp_flags & TH_ACK) == 0)
    return;

  /* no packet size restriction for fragmentation header technique */
  if(tbit->dst->type == SCAMPER_ADDR_TYPE_IPV4)
    ipv4 = 1;
  else if(pmtud->mtu < 1280)
    mtu = 0;

  /* if an out of sequence packet is received, ack it */
  if(dl->dl_tcp_seq != state->expected_seq)
    {
      if(dl->dl_net_rawlen > mtu)
	return;
      if(tp_tcp(state, state->seq, state->expected_seq, 0) == NULL)
	goto err;
      tbit_queue(task);
      return;
    }

  if(mtu != 0)
    {
      if(dl->dl_net_rawlen <= mtu || (ipv4 && SCAMPER_DL_IS_IP_DF(dl) == 0))
	success = 1;
    }
  else
    {
      if(SCAMPER_DL_IS_IP_FRAG(dl) || SCAMPER_DL_IS_IP_REASS(dl))
	success = 1;
    }

  if(success)
    {
      state->expected_seq += dl->dl_tcp_datalen;

      if(ipv4 && SCAMPER_DL_IS_IP_DF(dl) == 0)
	tbit_result(task, SCAMPER_TBIT_RESULT_PMTUD_CLEARDF);
      else
	tbit_result(task, SCAMPER_TBIT_RESULT_PMTUD_SUCCESS);

      if((tp = tp_tcp(state, state->seq, state->expected_seq, 0)) == NULL)
	{
	  tbit_handleerror(task, errno);
	  return;
	}
      if((dl->dl_tcp_flags & TH_FIN) != 0)
	{
	  tp->un.tcp.flags |= TH_FIN;
	  tp->un.tcp.ack++;
	  state->flags |= TBIT_STATE_FLAG_FIN_SEEN;
	}

      tbit_queue(task);
      return;
    }

  if(pmtud->ptb_retx != 0 && state->attempt >= pmtud->ptb_retx)
    {
      tbit_result(task, SCAMPER_TBIT_RESULT_PMTUD_FAIL);
      return;
    }

  /* send another PTB */
  if(tp_ptb(state) == NULL)
    goto err;

  tbit_queue(task);
  return;

 err:
  tbit_handleerror(task, errno);
  return;
}

/*
 * do_tbit_handle_dl
 *
 * for each packet received, check that the addresses and ports make sense,
 * and that the packet is not a reset.
 */
static void do_tbit_handle_dl(scamper_task_t *task, scamper_dl_rec_t *dl)
{
  static void (* const func[])(scamper_task_t *, scamper_dl_rec_t *) =
    {
      NULL,
      NULL,          /* MODE_RTSOCK */
      NULL,          /* MODE_DLHDR */
      NULL,          /* MODE_DONE */
      dl_syn,        /* MODE_SYN */
      dl_fin,        /* MODE_FIN */
      dl_data,       /* MODE_DATA */
      dl_pmtud,      /* MODE_PMTUD */
    };

  scamper_tbit_t *tbit = task->data;
  tbit_state_t *state = task->state;
  scamper_tbit_pkt_t *pkt = NULL;
  scamper_dl_rec_t *newp = NULL;

  if(dl->dl_ip_off != 0 || SCAMPER_DL_IS_IP_MF(dl))
    {
      scamper_dl_rec_frag_print(dl);

      pkt = scamper_tbit_pkt_alloc(SCAMPER_TBIT_PKT_DIR_RX, dl->dl_net_raw,
				   dl->dl_net_rawlen, &dl->dl_tv);
      if(pkt == NULL || scamper_tbit_record_pkt(tbit, pkt) != 0)
	goto err;

      if(tbit_reassemble(task, &newp, dl) != 0)
	goto err;
      if(newp == NULL)
	return;
      dl = newp;
    }

  /* Unless it is an inbound TCP packet for the flow, ignore it */
  if(SCAMPER_DL_IS_TCP(dl) == 0 ||
     dl->dl_tcp_sport != tbit->dport || dl->dl_tcp_dport != tbit->sport ||
     scamper_addr_raw_cmp(tbit->dst, dl->dl_ip_src) != 0 ||
     scamper_addr_raw_cmp(tbit->src, dl->dl_ip_dst) != 0)
    {
      goto done;
    }

  if(newp == NULL)
    {
      scamper_dl_rec_tcp_print(dl);

      /* Record the packet */
      pkt = scamper_tbit_pkt_alloc(SCAMPER_TBIT_PKT_DIR_RX, dl->dl_net_raw,
				   dl->dl_net_rawlen, &dl->dl_tv);
      if(pkt == NULL || scamper_tbit_record_pkt(tbit, pkt) != 0)
	{
	  if(pkt != NULL) scamper_tbit_pkt_free(pkt);
	  tbit_handleerror(task, errno);
	  return;
	}
    }

  if(func[state->mode] != NULL)
    {
      /* If a reset packet is received, abandon the measurement */
      if((dl->dl_tcp_flags & TH_RST) != 0 && state->mode != MODE_SYN)
	{
	  state->flags |= TBIT_STATE_FLAG_RST_SEEN;
	  tbit_classify(task);
	  return;
	}

      func[state->mode](task, dl);
    }

 done:
  if(newp != NULL && newp != dl)
    {
      if(newp->dl_ip_data != NULL)
	free(newp->dl_ip_data);
      free(newp);
    }
  return;

 err:
  tbit_handleerror(task, errno);
  return;
}

static void do_tbit_handle_timeout(scamper_task_t *task)
{
  /* Array of timeout functions */
  static void (* const func[])(scamper_task_t *) =
    {
      NULL,
      timeout_rt,         /* MODE_RTSOCK */
      timeout_dlhdr,      /* MODE_DLHDR */
      NULL,               /* MODE_DONE */
      timeout_syn,        /* MODE_SYN */
      timeout_fin,        /* MODE_FIN */
      timeout_data,       /* MODE_DATA */
      timeout_pmtud,      /* MODE_PMTUD */
    };
  tbit_state_t *state = task->state;

  /* Call the appropriate timeout function */
  if(func[state->mode] != NULL)
    func[state->mode](task);

  return;
}

static void tbit_handle_dlhdr(scamper_task_t *task, scamper_dlhdr_t *dlhdr)
{
  tbit_state_t *state = task->state;

  if(dlhdr == NULL)
    {
      scamper_queue_done(task->queue, 0);
      return;
    }

  state->dlhdr = dlhdr;
  state->mode  = MODE_SYN;
  scamper_queue_probe(task->queue);
  return;
}

static int tbit_handle_rt(scamper_task_t *task, scamper_rt_rec_t *rt)
{
  scamper_tbit_t *tbit = task->data;
  tbit_state_t *state = task->state;
  scamper_addr_t *dst = tbit->dst;
  uint16_t mtu;

  /*
   * scamper needs the datalink to transmit packets; try and get a
   * datalink on the ifindex specified.
   */
  if((state->dl = scamper_fd_dl(rt->ifindex)) == NULL)
    {
      scamper_debug(__func__, "could not get dl for %d", rt->ifindex);
      return -1;
    }

  /* Calculate the MSS to advertise */
  if(tbit->client_mss == 0)
    {
      if(scamper_if_getmtu(rt->ifindex, &mtu) != 0)
        {
	  scamper_debug(__func__, "could not get the interface mtu");
	  return -1;
        }

      if(tbit->dst->type == SCAMPER_ADDR_TYPE_IPV4)
	tbit->client_mss = mtu - 40;
      else if(tbit->dst->type == SCAMPER_ADDR_TYPE_IPV6)
	tbit->client_mss = mtu - 60;

      scamper_debug(__func__, "using client mss = %hu", tbit->client_mss);
    }

  /*
   * determine the underlying framing to use with each probe packet that will
   * be sent on the datalink.
   */
  state->mode = MODE_DLHDR;
  if(scamper_dlhdr_get(task, state->dl, dst, rt->gw, tbit_handle_dlhdr) != 0)
    {
      return -1;
    }

  return 0;
}

static void do_tbit_handle_rt(scamper_task_t *task, scamper_rt_rec_t *rt)
{
  tbit_state_t *state = task->state;

  if(state->mode != MODE_RTSOCK)
    return;

  assert(state->rt != NULL);
  scamper_fd_free(state->rt);
  state->rt = NULL;

  /* if there was a problem getting the ifindex, handle that */
  if(rt->error != 0 || rt->ifindex < 0)
    {
      printerror(errno, strerror, __func__, "could not get ifindex");
      goto err;
    }

  if(tbit_handle_rt(task, rt) != 0)
    goto err;

  if(state->mode != MODE_SYN)
    scamper_queue_wait(task->queue, 1000);

  return;

err:
  tbit_handleerror(task, errno);
  return;
}

static void do_tbit_write(scamper_file_t *sf, scamper_task_t *task)
{
  scamper_file_write_tbit(sf, (scamper_tbit_t *)task->data);
  return;
}

static void tbit_state_free(tbit_state_t *state)
{
  tbit_segment_t *seg;
  tbit_probe_t *tp;
  int i;

  if(state == NULL)
    return;

  if(state->fw != NULL)
    scamper_firewall_entry_free(state->fw);

  if(state->rt != NULL)
    scamper_fd_free(state->rt);

  if(state->dl != NULL)
    scamper_fd_free(state->dl);

  if(state->dlhdr != NULL)
    scamper_dlhdr_free(state->dlhdr);

  if(state->ptb_data != NULL)
    free(state->ptb_data);

  if(state->segments != NULL)
    {
      while((seg = slist_head_pop(state->segments)) != NULL)
	tbit_segment_free(seg);
      slist_free(state->segments);
    }

  if(state->tx != NULL)
    {
      while((tp = slist_head_pop(state->tx)) != NULL)
	tp_free(tp);
      slist_free(state->tx);
    }

  if(state->frags != NULL)
    {
      for(i=0; i<state->fragc; i++)
	tbit_frags_free(state->frags[i]);
      free(state->frags);
    }

  free(state);
  return;
}

static int tbit_state_alloc(scamper_task_t *task)
{
  scamper_tbit_t *tbit = task->data;
  tbit_state_t *state;
  uint16_t seq;

  if((state = malloc_zero(sizeof(tbit_state_t))) == NULL)
    {
      printerror(errno, strerror, __func__, "could not malloc state");
      goto err;
    }
  task->state = state;

  if((state->segments = slist_alloc()) == NULL)
    {
      printerror(errno, strerror, __func__, "could not create segments list");
      goto err;
    }
  if((state->tx = slist_alloc()) == NULL)
    {
      printerror(errno, strerror, __func__, "could not create tx list");
      goto err;
    }

  /*
   * generate a random 16 bit sequence number so we don't have to deal
   * with sequence number wrapping for now.
   */
  if(random_u16(&seq) != 0)
    {
      printerror(errno, strerror, __func__, "could not get random isn");
      goto err;
    }
  state->seq = seq;

  if(tbit->dst->type == SCAMPER_ADDR_TYPE_IPV4)
    random_u16(&state->ipid);

  if((state->rt = scamper_fd_rtsock()) == NULL)
    {
      printerror(errno, strerror, __func__, "could not get rtsock");
      goto err;
    }

  state->mode = MODE_RTSOCK;
  return 0;

err:
  return -1;
}

static void do_tbit_free(scamper_task_t *task)
{
  scamper_tbit_t *tbit = task->data;
  tbit_state_t *state = task->state;
  if(state != NULL) tbit_state_free(state);
  if(tbit != NULL) scamper_tbit_free(tbit);
  return;
}

static int tbit_tx_tcp(scamper_task_t *task, scamper_probe_t *pr,
		       tbit_probe_t *tp)
{
  scamper_tbit_t *tbit  = task->data;
  tbit_state_t   *state = task->state;
  tbit_segment_t *seg;

  pr->pr_ip_proto  = IPPROTO_TCP;
  pr->pr_tcp_sport = tbit->sport;
  pr->pr_tcp_dport = tbit->dport;
  pr->pr_tcp_win   = tbit->client_mss * 5;
  pr->pr_tcp_seq   = state->seq;

  if(state->mode == MODE_SYN)
    {
      pr->pr_tcp_flags = TH_SYN;
      pr->pr_tcp_mss   = tbit->client_mss;
      state->attempt++;
      return 1;
    }

  if(tp != NULL)
    {
      if(tp->un.tcp.len > 0)
	{
	  if((seg = slist_head_get(state->segments)) == NULL)
	    return 0;

	  pr->pr_data = seg->data;
	  pr->pr_len  = tp->un.tcp.len;
	  state->attempt++;
	}
      pr->pr_tcp_seq = tp->un.tcp.seq;
      pr->pr_tcp_ack = tp->un.tcp.ack;
      pr->pr_tcp_flags = tp->un.tcp.flags;
    }
  else
    {
      pr->pr_tcp_ack = state->expected_seq;
      pr->pr_tcp_flags = TH_ACK;
      if(state->mode == MODE_FIN)
	pr->pr_tcp_flags |= TH_FIN;
    }

  return 1;
}

static int tbit_tx_ptb(scamper_task_t *task, scamper_probe_t *pr,
		       tbit_probe_t *tp)
{
  scamper_tbit_t       *tbit  = task->data;
  tbit_state_t         *state = task->state;
  scamper_tbit_pmtud_t *pmtud = tbit->data;

  if(tbit->dst->type == SCAMPER_ADDR_TYPE_IPV4)
    {
      pr->pr_ip_proto  = IPPROTO_ICMP;
      pr->pr_icmp_type = ICMP_UNREACH;
      pr->pr_icmp_code = ICMP_UNREACH_NEEDFRAG;
    }
  else if(tbit->dst->type == SCAMPER_ADDR_TYPE_IPV6)
    {
      pr->pr_ip_proto  = IPPROTO_ICMPV6;
      pr->pr_icmp_type = ICMP6_PACKET_TOO_BIG;
    }

  pr->pr_icmp_mtu  = pmtud->mtu;
  pr->pr_data      = state->ptb_data;
  pr->pr_len       = state->ptb_datalen;
  state->attempt++;
  state->ptb_count++;

  return 1;
}

static void do_tbit_probe(scamper_task_t *task)
{
  scamper_firewall_rule_t sfw;
  scamper_tbit_t     *tbit = task->data;
  tbit_state_t       *state = task->state;
  scamper_tbit_pkt_t *pkt;
  scamper_probe_t     probe;
  tbit_probe_t       *tp;
  int                 wait, rc;

  if(state == NULL)
    {
      /* Fill in the test start time */
      gettimeofday_wrap(&tbit->start);

      /* Find the source IP address to use in probes */
      if((tbit->src = scamper_getsrc(tbit->dst, 0)) == NULL)
	goto err;

      /* Allocate space to store task state */
      if(tbit_state_alloc(task) != 0)
	goto err;

      state = task->state;
    }

  if(state->mode == MODE_RTSOCK)
    {
      if(scamper_rtsock_getroute(state->rt, tbit->dst) != 0)
	goto err;

      if(scamper_queue_isdone(task->queue))
	return;

      if(state->mode != MODE_SYN)
        {
	  scamper_queue_wait(task->queue, 1000);
	  return;
        }
    }

  if(state->mode == MODE_SYN && state->attempt == 0)
    {
      sfw.type = SCAMPER_FIREWALL_RULE_TYPE_5TUPLE;
      sfw.sfw_5tuple_proto = IPPROTO_TCP;
      sfw.sfw_5tuple_src   = tbit->dst;
      sfw.sfw_5tuple_dst   = tbit->src;
      sfw.sfw_5tuple_sport = tbit->dport;
      sfw.sfw_5tuple_dport = tbit->sport;

      if((state->fw = scamper_firewall_entry_get(&sfw)) == NULL)
	{
	  scamper_debug(__func__, "could not get firewall entry");
	  goto err;
	}
    }

  memset(&probe, 0, sizeof(probe));

  /* Common to all probes */
  probe.pr_dl     = scamper_fd_write_state(state->dl);
  probe.pr_dl_buf = state->dlhdr->buf;
  probe.pr_dl_len = state->dlhdr->len;
  probe.pr_ip_src = tbit->src;
  probe.pr_ip_dst = tbit->dst;
  probe.pr_ip_ttl = 255;

  if(tbit->dst->type == SCAMPER_ADDR_TYPE_IPV4)
    probe.pr_ip_id = state->ipid++;

  for(;;)
    {
      if((tp = slist_head_pop(state->tx)) != NULL)
	{
	  if(tp->type == TBIT_PROBE_TYPE_TCP)
	    rc = tbit_tx_tcp(task, &probe, tp);
	  else if(tp->type == TBIT_PROBE_TYPE_PTB)
	    rc = tbit_tx_ptb(task, &probe, tp);
	  else
	    rc = 0;
	  wait = tp->wait;
	  tp_free(tp);
	}
      else
	{
	  rc = tbit_tx_tcp(task, &probe, NULL);
	  wait = TBIT_DEFAULT_TIMEOUT;
	}

      if(rc == 0)
	{
	  tbit_queue(task);
	  return;
	}
      break;
    }

  /* Send the probe */
  if(scamper_probe(&probe) != 0)
    {
      errno = probe.pr_errno;
      printerror(errno, strerror, __func__, "could not send probe");
      goto err;
    }

  if((pkt = scamper_tbit_pkt_alloc(SCAMPER_TBIT_PKT_DIR_TX, probe.pr_tx_raw,
				   probe.pr_tx_rawlen, &probe.pr_tx))==NULL ||
     scamper_tbit_record_pkt(tbit, pkt) != 0)
    {
      printerror(errno, strerror, __func__, "could not record packet");
      goto err;
    }

  if(wait > 0)
    timeval_add_ms(&state->timeout, &probe.pr_tx, wait);

  tbit_queue(task);
  return;

err:
  tbit_handleerror(task, errno);
  return;
}

static int tbit_arg_param_validate(int optid, char *param, long *out)
{
  long tmp;

  switch(optid)
    {
    case SCAMPER_TBIT_OPT_TYPE:
      if(strcasecmp(param, "pmtud") == 0)
	tmp = SCAMPER_TBIT_TYPE_PMTUD;
      else
	goto err;
      break;

    case SCAMPER_TBIT_OPT_APP:
      if(strcasecmp(param, "smtp") == 0)
	tmp = SCAMPER_TBIT_APP_SMTP;
      else if(strcasecmp(param, "http") == 0)
	tmp = SCAMPER_TBIT_APP_HTTP;
      else if(strcasecmp(param, "dns") == 0)
	tmp = SCAMPER_TBIT_APP_DNS;
      else if(strcasecmp(param, "ftp") == 0)
	tmp = SCAMPER_TBIT_APP_FTP;
      else
	goto err;
      break;

    case SCAMPER_TBIT_OPT_SPORT:
    case SCAMPER_TBIT_OPT_DPORT:
    case SCAMPER_TBIT_OPT_MSS:
    case SCAMPER_TBIT_OPT_MTU:
      if(string_tolong(param, &tmp) != 0 || tmp < 0 || tmp > 65535)
	goto err;
      break;

    case SCAMPER_TBIT_OPT_USERID:
      if(string_tolong(param, &tmp) != 0 || tmp < 0)
	goto err;
      break;

    case SCAMPER_TBIT_OPT_URL:
      break;

    default:
      return -1;
    }

  /* valid parameter */
  if(out != NULL)
    *out = tmp;
  return 0;

 err:
  return -1;
}

int scamper_do_tbit_arg_validate(int argc, char *argv[], int *stop)
{
  return scamper_options_validate(opts, opts_cnt, argc, argv, stop,
				  tbit_arg_param_validate);
}

static int tbit_app_smtp(scamper_tbit_t *tbit, tbit_options_t *o)
{
  if(tbit->dport == 0)
    tbit->dport = 25;
  return 0;
}

static int tbit_app_dns(scamper_tbit_t *tbit, tbit_options_t *o)
{
  if(tbit->dport == 0)
    tbit->dport = 53;
  return 0;
}

static int tbit_app_ftp(scamper_tbit_t *tbit, tbit_options_t *o)
{
  if(tbit->dport == 0)
    tbit->dport = 21;
  return 0;
}

static int tbit_app_http(scamper_tbit_t *tbit, tbit_options_t *o)
{
  char *host;
  char *file;
  char *ptr;

  if(tbit->dport == 0)
    tbit->dport = 80;

  if(o->url == NULL)
    {
      host = NULL; file = "/";
      goto done;
    }

  if(strncasecmp(o->url, "http://", 7) != 0)
    return -1;

  /* extract the domain */
  host = ptr = o->url+7;
  while(*ptr != '\0')
    {
      if(*ptr == '/') break;
      if(isalnum(*ptr) == 0 && *ptr != '-' && *ptr != '.') return -1;
      ptr++;
    }
  if(ptr == host)
    return -1;

  if(*ptr == '\0')
    {
      file = "/";
    }
  else
    {
      memmove(host-1, host, ptr-host);
      host--;
      *(ptr-1) = '\0';
      file = ptr;
    }

 done:
  if((tbit->app_data = scamper_tbit_app_http_alloc(host, file)) == NULL)
    return -1;
  return 0;
}

static int tbit_alloc_pmtud(scamper_tbit_t *tbit, tbit_options_t *o)
{
  scamper_tbit_pmtud_t *pmtud;

  if((pmtud = scamper_tbit_pmtud_alloc()) == NULL)
    return -1;
  tbit->data = pmtud;

  pmtud->ptb_retx = 4;

  if(o->mtu == 0)
    {
      if(tbit->dst->type == SCAMPER_ADDR_TYPE_IPV4)
	pmtud->mtu = 576;
      else
	pmtud->mtu = 1280;
    }
  else
    {
      pmtud->mtu = o->mtu;
    }

  return 0;
}

/*
 * scamper_do_tbit_alloc
 *
 * Given a string representing a tbit task, parse the parameters and assemble
 * a tbit. Return the tbit structure so that it is all ready to go.
 */
void *scamper_do_tbit_alloc(char *str)
{
  scamper_option_out_t *opts_out = NULL, *opt;
  scamper_tbit_t *tbit = NULL;
  tbit_options_t o;
  uint8_t type = SCAMPER_TBIT_TYPE_PMTUD;
  uint32_t userid = 0;
  char *addr;
  long tmp;
  int rc;

  memset(&o, 0, sizeof(o));

  /* Parse the options */
  if(scamper_options_parse(str, opts, opts_cnt, &opts_out, &addr) != 0)
    {
      scamper_debug(__func__, "could not parse options");
      goto err;
    }

  /* If there is no IP address after the options string, then stop now */
  if(addr == NULL)
    {
      scamper_debug(__func__, "no address parameter");
      goto err;
    }

  /* Parse the options, do preliminary sanity checks */
  for(opt = opts_out; opt != NULL; opt = opt->next)
    {
      if(opt->type != SCAMPER_OPTION_TYPE_NULL &&
	 tbit_arg_param_validate(opt->id, opt->str, &tmp) != 0)
	{
	  scamper_debug(__func__, "validation of optid %d failed", opt->id);
	  goto err;
	}

      switch(opt->id)
        {
	case SCAMPER_TBIT_OPT_TYPE:
	  type = (uint8_t)tmp;
	  break;

	case SCAMPER_TBIT_OPT_APP:
	  o.app = (uint8_t)tmp;
	  break;

	case SCAMPER_TBIT_OPT_DPORT:
	  o.dport = (uint16_t)tmp;
	  break;
	
	case SCAMPER_TBIT_OPT_SPORT:
	  o.sport = (uint16_t)tmp;
	  break;

	case SCAMPER_TBIT_OPT_MSS:
	  o.mss = (uint16_t)tmp;
	  break;

	case SCAMPER_TBIT_OPT_MTU:
	  o.mtu = (uint16_t)tmp;
	  break;

	case SCAMPER_TBIT_OPT_URL:
	  o.url = opt->str;
	  break;

	case SCAMPER_TBIT_OPT_USERID:
	  userid = (uint32_t)tmp;
	  break;
        }
    }
  scamper_options_free(opts_out); opts_out = NULL;

  if((tbit = scamper_tbit_alloc()) == NULL)
    {
      printerror(errno, strerror, __func__, "could not alloc tbit");
      goto err;
    }
  if((tbit->dst = scamper_addrcache_resolve(addrcache,AF_UNSPEC,addr)) == NULL)
    {
      printerror(errno, strerror, __func__, "could not resolve %s", addr);
      goto err;
    }
  tbit->type       = type;
  tbit->userid     = userid;
  tbit->client_mss = o.mss;
  tbit->dport      = o.dport;
  tbit->sport      = (o.sport != 0)    ? o.sport    : default_sport;
  tbit->syn_retx   = (o.syn_retx != 0) ? o.syn_retx : TBIT_DEFAULT_RETX;
  tbit->dat_retx   = (o.dat_retx != 0) ? o.dat_retx : TBIT_DEFAULT_RETX;

  if(o.app == 0) o.app = SCAMPER_TBIT_APP_HTTP;
  tbit->app_proto = o.app;
  if(o.app == SCAMPER_TBIT_APP_HTTP)
    rc = tbit_app_http(tbit, &o);
  else if(o.app == SCAMPER_TBIT_APP_SMTP)
    rc = tbit_app_smtp(tbit, &o);
  else if(o.app == SCAMPER_TBIT_APP_DNS)
    rc = tbit_app_dns(tbit, &o);
  else if(o.app == SCAMPER_TBIT_APP_FTP)
    rc = tbit_app_ftp(tbit, &o);
  else
    goto err;
  if(rc != 0)
    goto err;

  if(type == SCAMPER_TBIT_TYPE_PMTUD)
    rc = tbit_alloc_pmtud(tbit, &o);
  else
    goto err;
  if(rc != 0)
    goto err;

  return tbit;

err:
  if(tbit != NULL) scamper_tbit_free(tbit);
  if(opts_out != NULL) scamper_options_free(opts_out);
  return NULL;
}

int scamper_do_tbit_dstaddr(void *data, void *param,
			    int (*foreach)(scamper_addr_t *, void *))
{
  scamper_tbit_t *tbit = (scamper_tbit_t *)data;
  return foreach(tbit->dst, param);
}

void scamper_do_tbit_free(void *data)
{
  scamper_tbit_t *tbit = (scamper_tbit_t *)data;
  scamper_tbit_free(tbit);
  return;
}

scamper_task_t *scamper_do_tbit_alloctask(void *data, scamper_list_t *list,
					  scamper_cycle_t *cycle)
{
  scamper_tbit_t *tbit = (scamper_tbit_t *)data;

  /* associate the list and cycle with the tbit */
  tbit->list  = scamper_list_use(list);
  tbit->cycle = scamper_cycle_use(cycle);

  /* allocate a task structure and store the tbit with it */
  return scamper_task_alloc(data, &tbit_funcs);
}

void scamper_do_tbit_cleanup(void)
{
  return;
}

int scamper_do_tbit_init(void)
{
#ifndef _WIN32
  pid_t pid = getpid();
#else
  DWORD pid = GetCurrentProcessId();
#endif
  default_sport = (pid & 0x7fff) + 0x8000;

  tbit_funcs.probe          = do_tbit_probe;
  tbit_funcs.handle_icmp    = NULL;
  tbit_funcs.handle_dl      = do_tbit_handle_dl;
  tbit_funcs.handle_timeout = do_tbit_handle_timeout;
  tbit_funcs.write          = do_tbit_write;
  tbit_funcs.task_free      = do_tbit_free;
  tbit_funcs.task_addrs     = scamper_do_tbit_dstaddr;
  tbit_funcs.handle_rt      = do_tbit_handle_rt;

  return 0;
}
