/*
 * Schism Tracker - a cross-platform Impulse Tracker clone
 * copyright (c) 2003-2005 Storlek <storlek@rigelseven.com>
 * copyright (c) 2005-2008 Mrs. Brisby <mrs.brisby@nimh.org>
 * copyright (c) 2009 Storlek & Mrs. Brisby
 * copyright (c) 2010 Storlek
 * URL: http://schismtracker.org/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#define NEED_BYTESWAP
#include "headers.h"
#include "slurp.h"
#include "fmt.h"
#include "log.h"

#include "sndfile.h"

/* --------------------------------------------------------------------------------------------------------- */

/* None of the sfx files on Modland are of the 31-instrument type that xmp recognizes.
However, there are a number of 31-instrument files with a different tag, under "SoundFX 2". */

static struct sfxfmt {
        size_t tagpos;
        const char tag[4];
        int nsmp;
        int dunno;
        const char *id;
} sfxfmts[] = {
        {124, "SO31", 31, 4, "SoundFX 2"},
        {124, "SONG", 31, 0, "SoundFX 2 (?)"},
        { 60, "SONG", 15, 0, "SoundFX"},
        {  0, ""    ,  0, 0, NULL},
};


int fmt_sfx_read_info(dmoz_file_t *file, const uint8_t *data, size_t length)
{
        int n;
        for (n = 0; sfxfmts[n].nsmp; n++) {
                if (length >= sfxfmts[n].tagpos + 4
                    && memcmp(data + sfxfmts[n].tagpos, sfxfmts[n].tag, 4) == 0) {
                        file->description = sfxfmts[n].id;
                        /*file->extension = str_dup("sfx");*/
                        file->title = strdup(""); // whatever
                        file->type = TYPE_MODULE_MOD;
                        return 1;
                }
        }
        return 0;
}

/* --------------------------------------------------------------------------------------------------------- */

/* Loader taken mostly from XMP.

Why did I write a loader for such an obscure format? That is, besides the fact that neither Modplug nor
Mikmod support SFX (and for good reason; it's a particularly dumb format) */

int fmt_sfx_load_song(CSoundFile *song, slurp_t *fp, unsigned int lflags)
{
        uint8_t tag[4];
        int n, nord, npat, pat, chan, restart, nsmp = 0;
        uint32_t smpsize[31];
        uint16_t tmp;
        MODCOMMAND *note;
        SONGSAMPLE *sample;
        unsigned int effwarn = 0;
        struct sfxfmt *fmt = sfxfmts;

        do {
                slurp_seek(fp, fmt->tagpos, SEEK_SET);
                slurp_read(fp, tag, 4);
                if (memcmp(tag, fmt->tag, 4) == 0) {
                        nsmp = fmt->nsmp;
                        break;
                }
                fmt++;
        } while (fmt->nsmp);

        if (!nsmp)
                return LOAD_UNSUPPORTED;


        slurp_rewind(fp);
        slurp_read(fp, smpsize, 4 * nsmp);
        slurp_seek(fp, 4, SEEK_CUR); /* the tag again */
        slurp_read(fp, &tmp, 2);
        if (!tmp)
                return LOAD_UNSUPPORTED; // erf
        tmp = 14565 * 122 / bswapBE16(tmp);
        song->m_nDefaultTempo = CLAMP(tmp, 31, 255);

        slurp_seek(fp, 14, SEEK_CUR); /* unknown bytes (reserved?) - see below */

        if (lflags & LOAD_NOSAMPLES) {
                slurp_seek(fp, 30 * nsmp, SEEK_CUR);
        } else {
                for (n = 0, sample = song->Samples + 1; n < nsmp; n++, sample++) {
                        slurp_read(fp, sample->name, 22);
                        sample->name[22] = 0;
                        slurp_read(fp, &tmp, 2); /* seems to be half the sample size, minus two bytes? */
                        tmp = bswapBE16(tmp);
                        sample->nLength = bswapBE32(smpsize[n]);

                        song->Samples[n].nC5Speed = MOD_FINETUNE(slurp_getc(fp)); // ?
                        sample->nVolume = slurp_getc(fp);
                        if (sample->nVolume > 64)
                                sample->nVolume = 64;
                        sample->nVolume *= 4; //mphack
                        sample->nGlobalVol = 64;
                        slurp_read(fp, &tmp, 2);
                        sample->nLoopStart = bswapBE16(tmp);
                        slurp_read(fp, &tmp, 2);
                        tmp = bswapBE16(tmp) * 2; /* loop length */
                        if (tmp > 2) {
                                sample->nLoopEnd = sample->nLoopStart + tmp;
                                sample->uFlags |= CHN_LOOP;
                        } else {
                                sample->nLoopStart = sample->nLoopEnd = 0;
                        }
                }
        }

        /* pattern/order stuff */
        nord = slurp_getc(fp);
        nord = MIN(nord, 127);
        restart = slurp_getc(fp);
        slurp_read(fp, song->Orderlist, nord);
        slurp_seek(fp, 128 - nord, SEEK_CUR);
        npat = 0;
        for (n = 0; n < nord; n++) {
                if (song->Orderlist[n] > npat)
                        npat = song->Orderlist[n];
        }
        npat++;

        /* Not sure what this is about, but skipping a few bytes here seems to make SO31's load right.
        (they all seem to have zero here) */
        slurp_seek(fp, fmt->dunno, SEEK_CUR);

        if (lflags & LOAD_NOPATTERNS) {
                slurp_seek(fp, npat * 1024, SEEK_CUR);
        } else {
                for (pat = 0; pat < npat; pat++) {
                        note = song->Patterns[pat] = csf_allocate_pattern(64, 64);
                        song->PatternSize[pat] = song->PatternAllocSize[pat] = 64;
                        for (n = 0; n < 64; n++, note += 60) {
                                for (chan = 0; chan < 4; chan++, note++) {
                                        uint8_t p[4];
                                        slurp_read(fp, p, 4);
                                        mod_import_note(p, note);
                                        /* Some sfx files have FF FE 00 00 note events (period 4094, sample
                                        240). Judging by where these are placed in the patterns, it appears
                                        that they're supposed to be note-cut or something.
                                        (See SoundFX/- unknown/1st intro.sfx on modland, for example) */
                                        if (p[0] == 255 && p[1] == 254) {
                                                note->note = NOTE_CUT;
                                                note->instr = 0;
                                        }
                                        switch (note->command) {
                                        case 0:
                                                break;
                                        case 1: /* arpeggio */
                                                note->command = CMD_ARPEGGIO;
                                                break;
                                        case 2: /* pitch bend */
                                                if (note->param >> 4) {
                                                        note->command = CMD_PORTAMENTODOWN;
                                                        note->param >>= 4;
                                                } else if (note->param & 0xf) {
                                                        note->command = CMD_PORTAMENTOUP;
                                                        note->param &= 0xf;
                                                } else {
                                                        note->command = 0;
                                                }
                                                break;
                                        case 5: /* volume up */
                                                note->command = CMD_VOLUMESLIDE;
                                                note->param = (note->param & 0xf) << 4;
                                                break;
                                        case 6: /* set volume */
                                                if (note->param > 64)
                                                        note->param = 64;
                                                note->volcmd = VOLCMD_VOLUME;
                                                note->vol = 64 - note->param;
                                                note->command = 0;
                                                note->param = 0;
                                                break;
                                        case 3: /* LED on (wtf!) */
                                        case 4: /* LED off (ditto) */
                                        case 7: /* set step up */
                                        case 8: /* set step down */
                                        default:
                                                effwarn |= (1 << note->command);
                                                note->command = CMD_UNIMPLEMENTED;
                                                break;
                                        }
                                }
                        }
                }
                for (n = 0; n < 16; n++) {
                        if (effwarn & (1 << n))
                                log_appendf(4, " Warning: Unimplemented effect %Xxx", n);
                }

                if (restart < npat)
                        csf_insert_restart_pos(song, restart);
        }

        /* sample data */
        if (!(lflags & LOAD_NOSAMPLES)) {
                for (n = 0, sample = song->Samples + 1; n < fmt->nsmp; n++, sample++) {
                        int8_t *ptr;

                        if (sample->nLength <= 2)
                                continue;
                        ptr = csf_allocate_sample(sample->nLength);
                        slurp_read(fp, ptr, sample->nLength);
                        sample->pSample = ptr;
                }
        }

        /* more header info */
        song->m_dwSongFlags = SONG_ITOLDEFFECTS | SONG_COMPATGXX;
        for (n = 0; n < 4; n++)
                song->Channels[n].nPan = PROTRACKER_PANNING(n); /* ??? */
        for (; n < MAX_CHANNELS; n++)
                song->Channels[n].dwFlags = CHN_MUTE;

        strcpy(song->tracker_id, fmt->id);
        song->m_nStereoSeparation = 64;

//      if (ferror(fp)) {
//              return LOAD_FILE_ERROR;
//      }

        /* done! */
        return LOAD_SUCCESS;
}


/*
most of modland's sfx files have all zeroes for those 14 "unknown" bytes, with the following exceptions:

64 00 00 00 00 00 00 00 00 00 00 00 00 00  d............. - unknown/antitrax.sfx
74 63 68 33 00 00 00 00 00 00 00 00 00 00  tch3.......... - unknown/axel f.sfx
61 6c 6b 00 00 00 00 00 00 00 00 00 00 00  alk........... Andreas Hommel/cyberblast-intro.sfx
21 00 00 00 00 00 00 00 00 00 00 00 00 00  !............. - unknown/dugger.sfx
00 00 00 00 00 0d 00 00 00 00 00 00 00 00  .............. Jean Baudlot/future wars - time travellers - dugger (title).sfx
00 00 00 00 00 00 00 00 0d 00 00 00 00 00  .............. Jean Baudlot/future wars - time travellers - escalator.sfx
6d 65 31 34 00 00 00 00 00 00 00 00 00 00  me14.......... - unknown/melodious.sfx
0d 0d 0d 53 46 58 56 31 2e 38 00 00 00 00  ...SFXV1.8.... AM-FM/sunday morning.sfx
61 6c 6b 00 00 00 00 00 00 00 00 00 00 00  alk........... - unknown/sunday morning.sfx
6f 67 00 00 00 00 00 00 00 00 00 00 00 00  og............ Philip Jespersen/supaplex.sfx
6e 74 20 73 6f 6e 67 00 00 00 00 00 00 00  nt song....... - unknown/sweety.sfx
61 6c 6b 00 00 00 00 00 00 00 00 00 00 00  alk........... - unknown/thrust.sfx
*/

