#
# SchoolTool - common information systems platform for school administration
# Copyright (c) 2005 Shuttleworth Foundation
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
"""
Student Intervention browser views.
"""
import csv
from StringIO import StringIO
import tempfile
import zipfile

from zope.app.form.browser.editview import EditView
from zope.browserpage.viewpagetemplatefile import ViewPageTemplateFile
from zope.catalog.interfaces import ICatalog
from zope.component import getUtility, getMultiAdapter, queryUtility
from zope.intid.interfaces import IIntIds
from zope.publisher.browser import BrowserView
from zope.security.checker import canWrite
from zope.security.proxy import removeSecurityProxy
from zope.traversing.api import getParents
from zope.traversing.browser.absoluteurl import absoluteURL

from zc.table.column import GetterColumn

from schooltool.app.interfaces import ISchoolToolApplication
from schooltool.app.browser import app
from schooltool.app.membership import URIGroup, URIMembership
from schooltool.intervention import InterventionGettext as _
from schooltool.contact.interfaces import IContact
from schooltool.course.interfaces import ISection, IInstructor
from schooltool.intervention import intervention, sendmail, catalog
from schooltool.intervention import interfaces
from schooltool.person.interfaces import IPerson
from schooltool.relationship import getRelatedObjects
from schooltool.schoolyear.interfaces import ISchoolYear
from schooltool.schoolyear.interfaces import ISchoolYearContainer
from schooltool.securitypolicy.crowds import AdministrationCrowd
from schooltool.skin.skin import NavigationViewlet
from schooltool.table.catalog import IndexedTableFormatter, FilterWidget
from schooltool.table.catalog import IndexedGetterColumn
from schooltool.table.catalog import IndexedLocaleAwareGetterColumn
from schooltool.table.table import SchoolToolTableFormatter
from schooltool.table.interfaces import ITableFormatter
from schooltool.term.interfaces import ITerm, IDateManager


class InterventionNavigationViewlet(NavigationViewlet):
    """View class for rendering the Intervention tab."""

    def canRender(self):
        person = IPerson(self.request.principal, None)
        return person is not None

    def url(self):
        person = IPerson(self.request.principal)
        return absoluteURL(person, self.request) + '/intervention_tab'


class StudentTableFormatter(SchoolToolTableFormatter):
    """View class for rendering the student table."""

    def columns(self):
        first_name = GetterColumn(
            name='first_name',
            title=_(u'First Name'),
            getter=lambda i, f: i.first_name,
            cell_formatter=self.cell_formatter,
            subsort=True)
        last_name = GetterColumn(
            name='last_name',
            title=_(u'Last Name'),
            getter=lambda i, f: i.last_name,
            cell_formatter=self.cell_formatter,
            subsort=True)
        return [first_name, last_name]

    def items(self):
        if AdministrationCrowd(None).contains(self.request.principal):
            persons = ISchoolToolApplication(None)['persons']
            return list(persons.values())

        students = set()
        person = IPerson(self.request.principal, None)
        for section in IInstructor(person).sections():
            for student in section.members:
                students.add(student)
        for student in person.advisees:
            students.add(student)
        return list(students)

    def sortOn(self):
        return (("last_name", False),)

    def cell_formatter(self, value, item, formatter):
        app = ISchoolToolApplication(None)
        container = interfaces.IInterventionSchoolYear(app)
        url = absoluteURL(container, formatter.request)
        if item.username not in container:
            # need to auto-vivyfy because we give the user a link to it
            getMultiAdapter((item, ISchoolYear(container)),
                interfaces.IInterventionStudent)
        return '<a href="%s/%s">%s</a>' % (url, item.username, value)


class StudentFilterWidget(FilterWidget):
    """View class for rendering the student filter widget."""

    template = ViewPageTemplateFile('templates/student_filter.pt')
    parameters = ['SEARCH_STUDENT_NAME', 'WITH_GOALS_ONLY']

    def filter(self, items):
        username = IPerson(self.request.principal).username

        if 'CLEAR_STUDENT_SEARCH' in self.request:
            for parameter in self.parameters:
                self.request.form[parameter] = ''
            return items

        if 'SEARCH_STUDENT_NAME' in self.request:
            searchstr = self.request['SEARCH_STUDENT_NAME'].lower()
            new_items = []
            for item in items:
                targetstr = '%s %s' % (item.first_name.lower(),
                    item.last_name.lower())
                if searchstr in targetstr:
                    new_items.append(item)
                else:
                    targetstr = item.username.lower()
                    if searchstr in targetstr:
                        new_items.append(item)
            items = new_items

        if self.request.get('WITH_GOALS_ONLY'):
            app = ISchoolToolApplication(None)
            year = interfaces.IInterventionSchoolYear(app)
            new_items = []
            for item in items:
                username = item.username
                if username in year and len(year[username]['goals']):
                    new_items.append(item)
            items = new_items

        return items


class InterventionTableFormatter(IndexedTableFormatter):
    """View class for rendering the intervention table."""

    def columns(self):
        first_name = IndexedLocaleAwareGetterColumn(
            index='first_name',
            name='first_name',
            title=_(u'First Name'),
            getter=lambda i, f: catalog.getStudentFirstName(i),
            subsort=True)
        last_name = IndexedLocaleAwareGetterColumn(
            index='last_name',
            name='last_name',
            title=_(u'Last Name'),
            getter=lambda i, f: catalog.getStudentLastName(i),
            subsort=True)
        inter_type = GetterColumn(
            name='intervention_type',
            title='',
            getter=self.intervention_type_getter)
        inter = GetterColumn(
            name='intervention',
            title=_(u"Intervention"),
            cell_formatter=self.intervention_cell_formatter,
            getter=interfaces.IInterventionGetter)
        created = IndexedGetterColumn(
            index='created',
            name='created',
            cell_formatter=self.created_cell_formatter,
            title=_(u'Created'),
            getter=lambda i, f: i.created,
            subsort=True)
        return [first_name, last_name, inter_type, inter, created]

    def sortOn(self):
        return (("created", True),)

    def intervention_type_getter(self, item, formatter):
        if interfaces.IInterventionType(item) == 'message':
            return _('Msg:')
        else:
            return _('Goal:')

    def intervention_cell_formatter(self, value, item, formatter):
        url = absoluteURL(IPerson(item), formatter.request)
        year = interfaces.IInterventionSchoolYear(item)
        url += '/schoolyears/%s/%ss/%s' % (year.__name__,
                                           interfaces.IInterventionType(item),
                                           item.__name__)
        return '<a href="%s">%s</a>' % (url, value)

    def created_cell_formatter(self, value, item, formatter):
        return '<span>%s</span>' % value.strftime('%x')


class InterventionFilterWidget(FilterWidget):
    """View class for rendering the intervention filter widget."""

    template = ViewPageTemplateFile('templates/filter.pt')
    parameters = ['SEARCH_NAME', 'GOALS_ONLY']

    def filter(self, items):
        catalog = ICatalog(self.context)
        user_contact = IContact(IPerson(self.request.principal))
        contact_intid = getUtility(IIntIds).getId(user_contact)
        pr_idx = catalog['persons_responsible']
        items = [item for item in items
                 if contact_intid in pr_idx.documents_to_values[item['id']]]

        if 'CLEAR_SEARCH' in self.request:
            for parameter in self.parameters:
                self.request.form[parameter] = ''
            return items

        if 'SEARCH_NAME' in self.request:
            fn_idx = catalog['first_name']
            ln_idx = catalog['last_name']
            searchstr = self.request['SEARCH_NAME'].lower()
            new_items = []
            for item in items:
                targetstr = '%s %s' % (
                    fn_idx.documents_to_values[item['id']].lower(),
                    ln_idx.documents_to_values[item['id']].lower())
                if searchstr in targetstr:
                    new_items.append(item)
            items = new_items

        if self.request.get('GOALS_ONLY'):
            type_idx = catalog['intervention_type']
            items = [item for item in items 
                     if type_idx.documents_to_values[item['id']] == 'goal']

        return items


class InterventionStartupView(BrowserView):
    """Dashboard view containing the user's intreventions."""

    title = _("Intervention Dashboard")
    student_title = _("Search Students")
    inbox_title = _("Inbox")

    def __call__(self):
        if queryUtility(IDateManager).current_term is None:
            template = ViewPageTemplateFile('templates/no_current_term.pt')
            return template(self)
        else:
            return super(InterventionStartupView, self).__call__()

    def createInterventionTableFormatter(self, **kwargs):
        container = ISchoolToolApplication(None)['schooltool.interventions']
        formatter = getMultiAdapter((container, self.request),
                                    ITableFormatter)
        formatter.setUp(**kwargs)
        return formatter

    def setUpTables(self):
        self.available_table = self.createInterventionTableFormatter(
            prefix="intervention", batch_size=40)

    def update(self):
        self.setUpTables()


class InterventionStudentsView(BrowserView):
    """View to search for the user's student intreventions."""

    title = _("Search Student Interventions")
    student_title = _("Students")

    def __call__(self):
        if queryUtility(IDateManager).current_term is None:
            template = ViewPageTemplateFile('templates/no_current_term.pt')
            return template(self)
        else:
            return super(InterventionStudentsView, self).__call__()

    def inboxURL(self):
        return absoluteURL(self.context, self.request)

    def createStudentTableFormatter(self, **kwargs):
        app = ISchoolToolApplication(None)
        container = interfaces.IInterventionSchoolYear(app)
        formatter = getMultiAdapter((container, self.request),
                                    ITableFormatter)
        formatter.setUp(**kwargs)
        return formatter

    def setUpTables(self):
        self.student_table = self.createStudentTableFormatter(
            prefix="student", batch_size=40)

    def update(self):
        self.setUpTables()


class InterventionStudentRedirectView(BrowserView):
    """View for redirecting to the no current term view."""

    def __call__(self):
        template = ViewPageTemplateFile('templates/no_current_term.pt')
        return template(self)


class InterventionStudentView(object):
    """View for managing a student's intervention."""

    @property
    def currentSchoolYear(self):
        return ISchoolYear(self.context.__parent__).title

    @property
    def isCurrentSchoolYear(self):
        app = ISchoolToolApplication(None)
        interventionSchoolYear = interfaces.IInterventionSchoolYear(app)
        return interventionSchoolYear == self.context.__parent__

    @property
    def schoolyears(self):
        results = []
        student = IPerson(self.context)
        app = ISchoolToolApplication(None)
        for year in ISchoolYearContainer(app).values():
            interventionSchoolYear = interfaces.IInterventionSchoolYear(year)
            interventionStudent = getMultiAdapter((student, year),
                interfaces.IInterventionStudent)

            result = {
                'url': absoluteURL(interventionStudent, self.request),
                'title': year.title,
                }
            results.append(result)
        return results

    def learnerOf(self):
        """Get the sections the student is a member of."""
        results = []

        for item in IPerson(self.context).groups:
            if ISection.providedBy(item):
                results.append(item)
            else:
                group_sections = getRelatedObjects(item, URIGroup,
                                                   rel_type=URIMembership)
                for section in group_sections:
                    results.append(section)
        results.sort(key=lambda s: s.__name__)
        return results

    def update(self):
        if 'schoolyear' in self.request:
            title = self.request['schoolyear']
            if title != self.currentSchoolYear:
                for year in self.schoolyears:
                    if year['title'] == title:
                        self.request.response.redirect(year['url'])

        interventionURL = absoluteURL(self.context, self.request)
        student = IPerson(self.context)

        self.studentURL = absoluteURL(student, self.request)
        self.studentName = '%s %s' % (IContact(student).first_name,
                                      IContact(student).last_name)

        self.messages = []
        for k, v in sorted(self.context['messages'].items()):
            if v.status_change:
                continue
            name = ', '.join(intervention.contactsName(v.sender))
            added = v.created
            message = {
                'creation_date': added,
                'url': absoluteURL(v, self.request),
                'text': _('Message from ${sender} sent ${date}',
                          mapping={'sender': name,
                                   'date': added.strftime('%x')})
                }
            self.messages.append(message)
        self.messages.sort(key=lambda m: m['creation_date'], reverse=True)
        self.newMessageURL = interventionURL + '/messages/+/addMessage.html'
        self.allMessagesURL = interventionURL + '/messages/allMessages.html'

        self.goals = []
        for k, v in sorted(self.context['goals'].items()):
            added = v.created.strftime('%x')
            due = v.timeline.strftime('%x')
            if v.goal_met:
                status = _('CLOSED')
            else:
                status = _('OPEN')
            goal = {
                'url': absoluteURL(v, self.request),
                'text': _('Goal ${goal_name} - Added ${date_added} - '
                          'Due ${date_due} - ${status}',
                          mapping={'goal_name': k,
                                   'date_added': added,
                                   'date_due': due,
                                   'status': status})
                }
            self.goals.append(goal)
        self.newGoalURL = interventionURL + '/goals/+/addGoal.html'
        self.allGoalsURL = interventionURL + '/goals/allGoals.html'

        studentId = IPerson(self.context).username
        currentYear = ISchoolYear(self.context.__parent__)
        terms = {}
        for section in self.learnerOf():
            if not canWrite(section, 'title'):
                continue
            if ISchoolYear(ITerm(section)) != currentYear:
                continue
            sheets = terms.setdefault(ITerm(section), [])
            activities = interfaces.IIndependentActivities(section, None)
            if activities is None:
                continue
            for worksheet in activities.values():
                if not worksheet.deployed:
                    continue
                course_title = ', '.join([course.title for course in
                                            section.courses])
                title = _('${worksheet_title} for ${course_title} - '
                          '${section_title}',
                          mapping={'worksheet_title': worksheet.title,
                                   'course_title': course_title,
                                   'section_title': section.title})
                worksheet_url = absoluteURL(worksheet, self.request)
                sheet = {
                    'url': '%s/gradebook/%s/view.html' % (worksheet_url, 
                        studentId),
                    'text': title
                    }
                sheets.append(sheet)
        self.terms = [{'title': term.title, 'sheets': terms[term]}
                      for term in sorted(terms.keys())]

        self.statusMessages = []
        for k, v in sorted(self.context['messages'].items()):
            if not v.status_change:
                continue
            name = ', '.join(intervention.contactsName(v.sender))
            added = v.created.strftime('%x')
            message = {
                'url': absoluteURL(v, self.request),
                'text': _('Change of Status Message from ${sender} '
                          'sent ${date}',
                          mapping={'sender': name, 'date': added})
                }
            self.statusMessages.append(message)
        self.newStatusMessageURL = interventionURL + '/messages/+/addStatusMessage.html'
        self.allStatusMessagesURL = interventionURL + '/messages/allStatusMessages.html'


class SectionInterventionsView(object):
    """Tabbed section view showing links for adding or editing intervention data."""

    def __init__(self, context, request):
        self.context = context
        self.request = request
        self.section = '%s - %s' % (', '.join([course.title for course in
                                               self.context.courses]),
                                    self.context.title)
        self.heading = _("${section}'s Student Interventions",
                         mapping={'section': self.section})
        self.schoolyear = interfaces.IInterventionSchoolYear(
            ISchoolYear(self.context))
        activities = interfaces.IIndependentActivities(self.context, None)
        if activities is None:
            self.worksheets = []
        else:
            activities = removeSecurityProxy(activities)
            self.worksheets = [worksheet for worksheet in activities.values()
                               if worksheet.deployed]
        app = ISchoolToolApplication(None)
        self.activeSchoolYear = interfaces.IInterventionSchoolYear(app)

    def messageLinks(self, student):
        newMessageURL = '%s/messages/%s/+/addMessage.html' % (
            absoluteURL(self.context, self.request), student.__name__)
        allMessagesURL = '%s/messages/allMessages.html' % (
            absoluteURL(student, self.request))
        n_messages = len([m for m in student['messages'].values()
                          if not m.status_change])
        result = {
            'title': _('View Messages (${message_count})',
                       mapping={'message_count': n_messages}),
            'url': allMessagesURL,
            }
        results = [result]
        if self.schoolyear == self.activeSchoolYear:
            result = {
                'title': _('Write New'),
                'url': newMessageURL,
                }
            results.append(result)
        return (results)

    def goalLinks(self, student):
        student_url = absoluteURL(student, self.request)
        allGoalsURL = '%s/goals/allGoals.html' % student_url
        n_goals = len(student['goals'])
        return ({'title': _('View Goals (${goal_count})',
                            mapping={'goal_count': n_goals}),
                 'url': allGoalsURL},
                )

    def reportSheetLinks(self, student):
        studentId = student.__name__
        nexturl = absoluteURL(self.context, self.request) + \
            '/@@interventions.html'
        results = []
        for worksheet in self.worksheets:
            worksheet_url = absoluteURL(worksheet, self.request)
            result = {
                'title': worksheet.title,
                'url': '%s/gradebook/%s?nexturl=%s' % (worksheet_url, 
                    studentId, nexturl)
                }
            results.append(result)
        return results

    @property
    def tabs(self):
        """Each tab is a tuple of tab_name, (tab_title, tab_link_factory)"""
        results = [
            ('messages', (_('Messages'), self.messageLinks)),
            ('goals', (_('Goals'), self.goalLinks)),
            ]
        if len(self.worksheets):
            result = ('sheets', (_('Report Sheets'), self.reportSheetLinks))
            results.append(result)
        return results
        

    def getActiveTabName(self):
        default_tab = 'messages'
        if len(self.worksheets):
            default_tab = 'sheets'
        return self.request.get('TAB', default_tab)

    def iterTabs(self):
        url = absoluteURL(self.context, self.request) + '/interventions.html'
        active_tab = self.getActiveTabName()
        for name, tab in self.tabs:
            title, link_factory = tab
            yield {
                'active': name == active_tab,
                'url': '%s?TAB=%s' % (url, name),
                'title': title
                }

    def listStudents(self):
        tab = dict(self.tabs).get(self.getActiveTabName())
        if tab is None:
            link_factory = lambda student: ()
        else:
            tab_title, link_factory = tab

        rows = []
        for student in self.context.members:
            interventionStudent = getMultiAdapter((student,
                ISchoolYear(self.context)), interfaces.IInterventionStudent)
            row = {
                'name': intervention.contactName(IContact(student)),
                'sortName': student.last_name + student.first_name,
                'url': absoluteURL(interventionStudent, self.request),
                'links': link_factory(interventionStudent),
                }
            rows.append(row)

        return sorted(rows, key=lambda row: row['sortName'])


class InterventionMessageAddView(app.BaseAddView):
    """View for adding a message."""

    def __init__(self, context, request):
        super(InterventionMessageAddView, self).__init__(context, request)
        self.label = _("PLEASE REMEMBER: This information is part of the student's record.")

    def create(self, *args, **kw):
        sender = IContact(self.request.principal._person)
        return self._factory(sender, *args)

    def nextURL(self):
        """If section is in the context heirachcy, return to the section
           interventions view.  Otherwise return to the student's intervention
           center."""
        for parent in getParents(self.context):
            if ISection.providedBy(parent):
                return absoluteURL(parent, self.request) + '/interventions.html'
        interventionStudent = self.context.context.__parent__
        return absoluteURL(interventionStudent, self.request)


class InterventionStatusMessageAddView(app.BaseAddView):
    """View for adding a message."""

    def create(self, *args, **kw):
        sender = IContact(self.request.principal._person)
        kw['status_change'] = True
        return self._factory(sender, *args, **kw)

    def nextURL(self):
        """If section is in the context heirachcy, return to the section
           interventions view.  Otherwise return to the student's intervention
           center."""
        for parent in getParents(self.context):
            if ISection.providedBy(parent):
                return absoluteURL(parent, self.request) + '/interventions.html'
        interventionStudent = self.context.context.__parent__
        return absoluteURL(interventionStudent, self.request)


class InterventionMessageView(object):
    """View for viewing a message."""

    def __init__(self, context, request):
        self.context = context
        self.request = request
        sender = ', '.join(intervention.contactsName(context.sender))
        added = context.created.strftime('%x')
        mapping = {'sender': sender, 'date': added}
        if context.status_change:
            self.heading = _('Change of Status Message from: ${sender} '
                             'sent ${date}', mapping=mapping)
        else:
            self.heading = _('Message from: ${sender} sent ${date}',
                             mapping=mapping)
        self.recipients = ', '.join(intervention.contactsName(
            context.recipients))


class InterventionMessagesView(object):
    """View for viewing all messages."""

    def __init__(self, context, request):
        self.context = context
        self.request = request
        self.messages = []
        for k, v in sorted(self.context.items()):
            if self.skipMessage(v):
                continue
            name = ', '.join(intervention.contactsName(v.sender))
            added = v.created
            heading = _('Message sent by ${sender} on ${date}',
                        mapping={'sender': name,
                                 'date': added.strftime('%x')})
            recipients = ', '.join(intervention.contactsName(v.recipients))
            message = {
                'creation_date': added,
                'heading': heading,
                'recipients': recipients,
                'body': v.body,
            }
            self.messages.append(message)
        self.messages.sort(key=lambda m: m['creation_date'], reverse=True)

    def skipMessage(self, message):
        return message.status_change

    @property
    def heading(self):
        name = intervention.contactName(IContact(IPerson(self.context)))
        return _('Messages regarding ${student}', mapping={'student': name})


class InterventionStatusMessagesView(InterventionMessagesView):
    """View for viewing all change of status messages."""

    def skipMessage(self, message):
        return not message.status_change

    @property
    def heading(self):
        name = intervention.contactName(IContact(IPerson(self.context)))
        return _('Change of Status Messages regarding ${student}',
                 mapping={'student': name})


class InterventionGoalAddView(app.BaseAddView):
    """View for adding a goal."""

    def create(self, *args, **kw):
        creator = IContact(self.request.principal._person)
        kw['creator'] = creator
        return self._factory(*args, **kw)

    def nextURL(self):
        interventionStudent = self.context.context.__parent__
        return absoluteURL(interventionStudent, self.request)


class InterventionGoalEditView(EditView):
    """View for editing a goal"""

    def update(self):
        if 'CANCEL' in self.request:
            self.request.response.redirect(self.nextURL())
        else:
            status = EditView.update(self)
            if 'UPDATE_SUBMIT' in self.request and not self.errors:
                self.request.response.redirect(self.nextURL())
            return status

    def nextURL(self):
        return absoluteURL(self.context, self.request)


class InterventionGoalView(object):
    """View for viewing a goal."""

    def __init__(self, context, request):
        self.context = context
        self.request = request
        self.timeline = context.timeline.strftime('%x')
        self.notified = ''
        if context.notified:
            self.notified = _('notification sent')
        self.persons_responsible = ', '.join(intervention.contactsName(
            context.persons_responsible))
        self.goal_met = _('No')
        if context.goal_met:
            self.goal_met = _('Yes')

    @property
    def heading(self):
        student = removeSecurityProxy(IPerson(self.context))
        name = intervention.contactName(IContact(student))
        added = self.context.created.strftime('%x')
        return _('Goal ${goal_name} for ${student} added ${date}',
                 mapping={'goal_name': self.context.__name__,
                          'student': name, 'date': added})


class InterventionGoalsView(object):
    """View for viewing all goals."""

    def __init__(self, context, request):
        self.context = context
        self.request = request
        self.goals = []
        for k, v in sorted(self.context.items()):
            added = v.created.strftime('%x')
            heading = _('Goal ${goal_name} added ${date}',
                        mapping={'goal_name': k, 'date': added})
            notified = ''
            if v.notified:
                notified = _('notification sent')
            persons_responsible = intervention.contactsName(
                v.persons_responsible)
            goal_met = _('No')
            if v.goal_met:
                goal_met = _('Yes')
            goal = {
                'heading': heading,
                'presenting_concerns': v.presenting_concerns,
                'goal': v.goal,
                'strengths': v.strengths,
                'indicators': v.indicators,
                'intervention': v.intervention,
                'timeline': v.timeline.strftime('%x'),
                'notified': notified,
                'persons_responsible': persons_responsible,
                'goal_met': goal_met,
                'follow_up_notes': v.follow_up_notes,
            }
            self.goals.append(goal)

    @property
    def heading(self):
        return _('Goals for: ${first_name} ${last_name}',
            mapping={'first_name': IContact(IPerson(self.context)).first_name,
                     'last_name': IContact(IPerson(self.context)).last_name})


class NotifyGoalsView(object):
    """View for notifying persons responsible for goals that
       have come due"""

    def __init__(self, context, request):
        self.context = context
        self.request = request
        goalsNotified = sendmail.sendInterventionGoalNotifyEmails()
        self.goals = []
        for notified in goalsNotified:
            name = intervention.contactName(IContact(IPerson(notified)))
            goal = {
                'text': _('${student} goal ${goal_name}',
                          mapping={'student': name,
                                   'goal_name': notified.__name__}),
                'url': absoluteURL(notified, request)
                }
            self.goals.append(goal)


class MessagesCSVView(BrowserView):

    def __call__(self):
        csvfile = StringIO()
        writer = csv.writer(csvfile, quoting=csv.QUOTE_ALL)
        row = ['year', 'student', 'date', 'sender', 'recipients', 'body']
        writer.writerow(row)
        root = interfaces.IInterventionRoot(self.context)
        for year in root.values():
            for student in year.values():
                for message in student['messages'].values():
                    created = message.created.strftime('%Y-%m-%d')
                    sender =  [IPerson(c).username for c in message.sender]
                    recipients = ', '.join(intervention.contactsEmail(
                        message.recipients))
                    body = message.body.replace('\n', '\\n')
                    body = body.replace('\r', '\\r')
                    row = [year.__name__, student.__name__, created,
                           ', '.join(sender), recipients, body]
                    row = [item.encode('utf-8') for item in row]
                    writer.writerow(row)
        return csvfile.getvalue().decode('utf-8')


class GoalsCSVView(BrowserView):

    def __call__(self):
        csvfile = StringIO()
        writer = csv.writer(csvfile, quoting=csv.QUOTE_ALL)
        row = ['year', 'student', 'created', 'creator', 'presenting_concerns',
               'goal', 'strengths', 'indicators', 'intervention', 'timeline',
               'persons_responsible', 'goal_met', 'follow_up_notes', 'notified']
        writer.writerow(row)
        root = interfaces.IInterventionRoot(self.context)
        for year in root.values():
            for student in year.values():
                for goal in student['goals'].values():
                    created = goal.created.strftime('%Y-%m-%d')
                    creator =  [IPerson(c).username for c in goal.creator]
                    timeline = goal.timeline.strftime('%Y-%m-%d')
                    goal_met = unicode(goal.goal_met)
                    notified = unicode(goal.notified)
                    persons_responsible = ', '.join(
                        intervention.contactsEmail(
                        goal.persons_responsible))
                    row = [year.__name__, student.__name__,
                           created, ', '.join(creator),
                           goal.presenting_concerns, goal.goal,
                           goal.strengths, goal.indicators, goal.intervention,
                           timeline, persons_responsible, goal_met,
                           goal.follow_up_notes, notified]
                    clean_row = []
                    for item in row:
                        if item is None:
                            item = u''
                        item = item.replace('\n', '\\n')
                        item = item.replace('\r', '\\r')
                        clean_row.append(item.encode('utf-8'))
                    writer.writerow(clean_row)
        return csvfile.getvalue().decode('utf-8')


class MessagesGoalsCSVView(BrowserView):

    def __call__(self):
        messages = MessagesCSVView(self.context, self.request)()
        goals = GoalsCSVView(self.context, self.request)()

        zfile = StringIO()
        archive = zipfile.ZipFile(zfile, "w")
        messages_file = tempfile.mktemp('.csv')
        open(messages_file, 'w').write(messages.encode('utf-8'))
        archive.write(messages_file, 'messages.csv', zipfile.ZIP_DEFLATED)
        goals_file = tempfile.mktemp('.csv')
        open(goals_file, 'w').write(goals.encode('utf-8'))
        archive.write(goals_file, 'goals.csv', zipfile.ZIP_DEFLATED)
        archive.close()

        data = zfile.getvalue()
        response = self.request.response
        response.setHeader('Content-Type', 'application/zip')
        response.setHeader('Content-Length', len(data))
        return data

