# coding=UTF8
#
# SchoolTool - common information systems platform for school administration
# Copyright (c) 2009 Shuttleworth Foundation
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
"""
Unit tests for schooltool.intervention.generations.evolve4
"""

import unittest
import doctest

from zope.app.testing import setup
from zope.component.hooks import setSite
from zope.interface import implements
from zope.site.folder import Folder

from schooltool.app.interfaces import ISchoolToolApplication
from schooltool.generations.tests import ContextStub
from schooltool.intervention.generations.evolve4 import evolve
from schooltool.intervention.interfaces import IInterventionGoal


class AppStub(Folder):
    implements(ISchoolToolApplication)


class GoalStub(object):
    implements(IInterventionGoal)
    _persons_responsible = []
    __dict__ = {'persons_responsible': ['jdoe']}
    at_one_time_responsible = []


def doctest_evolve4():
    """Evolution to generation 4.

    First, we'll set up the app object:

        >>> context = ContextStub()
        >>> context.root_folder['app'] = app = AppStub()
        >>> app[u'schooltool.interventions'] = {}
        >>> manager = setup.createSiteManager(app)

    We'll set up our test with data that will be effected by running the
    evolve script:

        >>> root = app[u'schooltool.interventions']
        >>> year = root['2009'] = {}
        >>> student = year['jdoe'] = {}

        >>> goals = student['goals'] = {}
        >>> goal1 = goals['1'] = GoalStub()

    Finally, we'll run the evolve script, testing the effected values before and
    after:

        >>> goal1.__dict__['persons_responsible']
        ['jdoe']
        >>> goal1._persons_responsible
        []
        >>> goal1.at_one_time_responsible
        []

        >>> evolve(context)

        >>> 'persons_responsible' in goal1.__dict__
        False
        >>> goal1._persons_responsible
        ['jdoe']
        >>> goal1.at_one_time_responsible
        ['jdoe']

    What if the intervention container doesn't exist yet in the
    application:

        >>> context = ContextStub()
        >>> context.root_folder['app'] = app = AppStub()
        >>> manager = setup.createSiteManager(app)
        >>> evolve(context)

    """


def setUp(test):
    setup.placefulSetUp()
    setSite()

def tearDown(test):
    setup.placefulTearDown()
    setSite()


def test_suite():
    return unittest.TestSuite([
        doctest.DocTestSuite(setUp=setUp, tearDown=tearDown,
                             optionflags=doctest.ELLIPSIS
                             | doctest.NORMALIZE_WHITESPACE
                             | doctest.REPORT_NDIFF
                             | doctest.REPORT_ONLY_FIRST_FAILURE),
        ])

if __name__ == '__main__':
    unittest.main(defaultTest='test_suite')

