
#ifndef __SCHRO_FRAME_H__
#define __SCHRO_FRAME_H__

#include <schroedinger/schro-stdint.h>
#include <schroedinger/schroparams.h>

SCHRO_BEGIN_DECLS

typedef struct _SchroFrame SchroFrame;
typedef struct _SchroFrameComponent SchroFrameComponent;

typedef void (*SchroFrameFreeFunc)(SchroFrame *frame, void *priv);

/* bit pattern:
 *  0x100 - 0: normal, 1: indirect (packed)
 *  0x001 - horizontal chroma subsampling: 0: 1, 1: 2
 *  0x002 - vertical chroma subsampling: 0: 1, 1: 2
 *  0x00c - depth: 0: u8, 1: s16, 2: s32
 *  */
typedef enum _SchroFrameFormat {
  SCHRO_FRAME_FORMAT_U8_444 = 0x00,
  SCHRO_FRAME_FORMAT_U8_422 = 0x01,
  SCHRO_FRAME_FORMAT_U8_420 = 0x03,

  SCHRO_FRAME_FORMAT_S16_444 = 0x04,
  SCHRO_FRAME_FORMAT_S16_422 = 0x05,
  SCHRO_FRAME_FORMAT_S16_420 = 0x07,

  SCHRO_FRAME_FORMAT_S32_444 = 0x08,
  SCHRO_FRAME_FORMAT_S32_422 = 0x09,
  SCHRO_FRAME_FORMAT_S32_420 = 0x0b,

  /* indirectly supported */
  SCHRO_FRAME_FORMAT_YUYV = 0x100, /* YUYV order */
  SCHRO_FRAME_FORMAT_UYVY = 0x101, /* UYVY order */
  SCHRO_FRAME_FORMAT_AYUV = 0x102,
  SCHRO_FRAME_FORMAT_ARGB = 0x103
} SchroFrameFormat;

#define SCHRO_FRAME_FORMAT_DEPTH(format) ((format) & 0xc)
#define SCHRO_FRAME_FORMAT_DEPTH_U8 0x00
#define SCHRO_FRAME_FORMAT_DEPTH_S16 0x04
#define SCHRO_FRAME_FORMAT_DEPTH_S32 0x08

#define SCHRO_FRAME_FORMAT_H_SHIFT(format) ((format) & 0x1)
#define SCHRO_FRAME_FORMAT_V_SHIFT(format) (((format)>>1) & 0x1)

#define SCHRO_FRAME_IS_PACKED(format) (((format)>>8) & 0x1)

struct _SchroFrameComponent {
  void *data;
  int stride;
  int width;
  int height;
  int length;
  int h_shift;
  int v_shift;
};

struct _SchroFrame {
  int refcount;
  SchroFrameFreeFunc free;
  void *regions[3];
  void *priv;

  SchroFrameFormat format;
  int width;
  int height;

  SchroFrameComponent components[3];

  uint32_t frame_number;
};


SchroFrame * schro_frame_new (void);
SchroFrame * schro_frame_new_and_alloc (SchroFrameFormat format, int width,
    int height);
SchroFrame * schro_frame_new_from_data_I420 (void *data, int width, int height);
SchroFrame * schro_frame_new_from_data_YUY2 (void *data, int width, int height);
SchroFrame * schro_frame_new_from_data_AYUV (void *data, int width, int height);
void schro_frame_set_free_callback (SchroFrame *frame,
    SchroFrameFreeFunc free_func, void *priv);
void schro_frame_unref (SchroFrame *frame);
SchroFrame *schro_frame_ref (SchroFrame *frame);
void schro_frame_convert (SchroFrame *dest, SchroFrame *src);
void schro_frame_add (SchroFrame *dest, SchroFrame *src);
void schro_frame_subtract (SchroFrame *dest, SchroFrame *src);
void schro_frame_shift_left (SchroFrame *frame, int shift);
void schro_frame_shift_right (SchroFrame *frame, int shift);
void schro_frame_edge_extend (SchroFrame *frame, int width, int height);
void schro_frame_zero_extend (SchroFrame *frame, int width, int height);

void schro_frame_iwt_transform (SchroFrame *frame, SchroParams *params,
    int16_t *tmp);
void schro_frame_inverse_iwt_transform (SchroFrame *frame, SchroParams *params,
    int16_t *tmp);

void schro_frame_downsample (SchroFrame *dest, SchroFrame *src, int shift);
void schro_frame_upsample_horiz (SchroFrame *dest, SchroFrame *src);
void schro_frame_upsample_vert (SchroFrame *dest, SchroFrame *src);
int schro_frame_calculate_average_luma (SchroFrame *frame);

SchroFrame * schro_frame_convert_to_444 (SchroFrame *frame);


SCHRO_END_DECLS

#endif

