/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_layer_dialog.h"
#include "sg.h"
#include "sg_dialogs.h"

#define PTS_MM 2.8350000
#define PTS_CM 28.3500000
#define PTS_IN 72.0000000
#define PTS_PT 1.0000000
#define PTS_PIX (SG_DPI/PTS_IN)

static GtkWidget *left_entry, *top_entry;
static GtkWidget *width_entry, *height_entry;
static GtkWidget *transparent_item; 
static GtkWidget *bg_button;
static GtkWidget *units_combo;
static GtkWidget *ratio_item;
static gint pts_width, pts_height;
static gint pts_left, pts_top;

/* static GtkWidget *legends_item; */ 
static GtkWidget *color_combo; 
static GtkPlot *plot;
static SGplot *the_plot;
static SGlayer *the_layer;

static gchar *units[] = {"PS points",
                         "mm",
                         "cm",
                         "inches",
                         "pixels",
                         "% of page size",
                         NULL};

void
sg_layer_dialog_update_plot(GtkWidget *widget)
{
  GtkWidget *child;
  SGplot *parent;
  GtkColorCombo *combo;
  GdkColor color;
  gchar *color_name;
  gdouble left, top, width, height;
  gint units;

  width = atof(gtk_entry_get_text(GTK_ENTRY(width_entry)));
  height = atof(gtk_entry_get_text(GTK_ENTRY(height_entry)));
  left = atof(gtk_entry_get_text(GTK_ENTRY(left_entry)));
  top = atof(gtk_entry_get_text(GTK_ENTRY(top_entry)));

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(units_combo)->list)->selection->data;
  units = gtk_list_child_position(GTK_LIST(GTK_COMBO(units_combo)->list), child);

  switch(units){
   case SG_UNIT_MM:
     pts_width = width * PTS_MM;
     pts_height = height * PTS_MM;
     pts_left = left * PTS_MM;
     pts_top = top * PTS_MM;
     break;
   case SG_UNIT_CM:
     pts_width = width * PTS_CM;
     pts_height = height * PTS_CM;
     pts_left = left * PTS_CM;
     pts_top = top * PTS_CM;
     break;
   case SG_UNIT_IN:
     pts_width = width * PTS_IN;
     pts_height = height * PTS_IN;
     pts_left = left * PTS_IN;
     pts_top = top * PTS_IN;
     break;
   case SG_UNIT_PIXEL:
     pts_width = width * PTS_PIX;
     pts_height = height * PTS_PIX;
     pts_left = left * PTS_PIX;
     pts_top = top * PTS_PIX;
     break;
   case SG_UNIT_PT:
     pts_width = width * PTS_PT;
     pts_height = height * PTS_PT;
     pts_left = left * PTS_PT;
     pts_top = top * PTS_PT;
     break;
   default:
     pts_width = width * the_plot->page_width;
     pts_height = height * the_plot->page_height;
     pts_left = left * the_plot->page_width;
     pts_top = top * the_plot->page_height;
     break;
  }

  gtk_plot_move_resize(plot,
                       pts_left / (gdouble) the_plot->page_width,
                       pts_top / (gdouble) the_plot->page_height,
                       pts_width / (gdouble) the_plot->page_width,
                       pts_height / (gdouble) the_plot->page_height);
                       

  combo = GTK_COLOR_COMBO(color_combo);
  color_name = gtk_color_combo_get_color_at(combo, combo->row, combo->column);
  gdk_color_parse(color_name, &color);
  gdk_color_alloc(color_combo->style->colormap, &color);
  plot->background = color;

  gtk_plot_set_transparent(GTK_PLOT(plot), 
            gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(transparent_item)));

/*
  plot->show_legends = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(legends_item));
*/

  parent = (SGplot *)the_layer->parent;
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(parent->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(parent->real_canvas));
}

static void
select_image(GtkWidget *widget, gpointer data)
{
  GdkPixmap *pixmap;

  pixmap = sg_image_import();

  if(pixmap) 
     gtk_plot_set_background_pixmap(GTK_PLOT(the_layer->real_plot), pixmap);
}

static void
new_units(GtkWidget *widget, gpointer data)
{
  GtkWidget *child;
  gchar label[100];
  gint new_units;
  gdouble width, height;
  gdouble left, top;

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(units_combo)->list)->selection->data;
  new_units = gtk_list_child_position(GTK_LIST(GTK_COMBO(units_combo)->list), child);

  switch(new_units){
   case SG_UNIT_MM:
       width = (gdouble)pts_width / (gdouble)PTS_MM;
       height = (gdouble)pts_height / (gdouble)PTS_MM;
       left = (gdouble)pts_left / (gdouble)PTS_MM;
       top = (gdouble)pts_top / (gdouble)PTS_MM;
       break;
   case SG_UNIT_CM:
       width = (gdouble)pts_width / (gdouble)PTS_CM;
       height = (gdouble)pts_height / (gdouble)PTS_CM;
       left = (gdouble)pts_left / (gdouble)PTS_CM;
       top = (gdouble)pts_top / (gdouble)PTS_CM;
       break;
   case SG_UNIT_IN:
       width = (gdouble)pts_width / (gdouble)PTS_IN;
       height = (gdouble)pts_height / (gdouble)PTS_IN;
       left = (gdouble)pts_left / (gdouble)PTS_IN;
       top = (gdouble)pts_top / (gdouble)PTS_IN;
       break;
   case SG_UNIT_PIXEL:
       width = (gdouble)pts_width / (gdouble)PTS_PIX;
       height = (gdouble)pts_height / (gdouble)PTS_PIX;
       left = (gdouble)pts_left / (gdouble)PTS_PIX;
       top = (gdouble)pts_top / (gdouble)PTS_PIX;
       break;
   case SG_UNIT_PT:
       width = (gdouble)pts_width / (gdouble)PTS_PT;
       height = (gdouble)pts_height / (gdouble)PTS_PT;
       left = (gdouble)pts_left / (gdouble)PTS_PT;
       top = (gdouble)pts_top / (gdouble)PTS_PT;
       break;
   default:
       width = (gdouble)pts_width / (gdouble)the_plot->page_width;
       height = (gdouble)pts_height / (gdouble)the_plot->page_height;
       left = (gdouble)pts_left / (gdouble)the_plot->page_width;
       top = (gdouble)pts_top / (gdouble)the_plot->page_height;
       break;
  }

  sprintf(label, "%.*f", 3, width);
  gtk_entry_set_text(GTK_ENTRY(width_entry), label);
  sprintf(label, "%.*f", 3, height);
  gtk_entry_set_text(GTK_ENTRY(height_entry), label);
  sprintf(label, "%.*f", 3, left);
  gtk_entry_set_text(GTK_ENTRY(left_entry), label);
  sprintf(label, "%.*f", 3, top);
  gtk_entry_set_text(GTK_ENTRY(top_entry), label);
}

static void
preserve_ratio(GtkWidget *widget, gpointer data)
{
  GtkWidget *child;
  gint new_width, new_height;
  gdouble width, height;
  gdouble ratio;
  gchar label[100];
  gint units;

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(units_combo)->list)->selection->data;
  units = gtk_list_child_position(GTK_LIST(GTK_COMBO(units_combo)->list), child);


  width = atof(gtk_entry_get_text(GTK_ENTRY(width_entry)));
  height = atof(gtk_entry_get_text(GTK_ENTRY(height_entry)));
 
  if(!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(ratio_item))) return;

  switch(units){
   case SG_UNIT_MM:
     new_width = width * PTS_MM;
     new_height = height * PTS_MM;
     break;
   case SG_UNIT_CM:
     new_width = width * PTS_CM;
     new_height = height * PTS_CM;
     break;
   case SG_UNIT_IN:
     new_width = width * PTS_IN;
     new_height = height * PTS_IN;
     break;
   case SG_UNIT_PIXEL:
     new_width = width * PTS_PIX;
     new_height = height * PTS_PIX;
     break;
   case SG_UNIT_PT:
     new_width = width * PTS_PT;
     new_height = height * PTS_PT;
     break;
   default:
     new_width = width * the_plot->page_width;
     new_height = height * the_plot->page_height;
     break;
  }

  if(widget == width_entry){
      ratio = (gdouble)new_width / (gdouble)pts_width;
      height *= ratio;
      pts_height *= ratio;
  }else{
      ratio = (gdouble)new_height / (gdouble)pts_height;
      width *= ratio;
      pts_width *= ratio;
  }
  
  sprintf(label, "%.*f", 3, width);
  gtk_entry_set_text(GTK_ENTRY(width_entry), label);
  sprintf(label, "%.*f", 3, height);
  gtk_entry_set_text(GTK_ENTRY(height_entry), label);
}

static void
init_dialog()
{
  gchar label[100];

  sg_combo_set_items(GTK_COMBO(units_combo), units);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(units_combo)->list), the_plot->page_units);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(units_combo)->entry), FALSE);

  pts_width = plot->width * the_plot->page_width;
  pts_height = plot->height * the_plot->page_height;
  pts_left = plot->x * the_plot->page_width;
  pts_top = plot->y * the_plot->page_height;
  new_units(NULL, NULL);

  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(transparent_item),
                              gtk_plot_is_transparent(GTK_PLOT(plot))); 

/*
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(legends_item),
                              plot->show_legends); 
*/

  sg_color_combo_init(GTK_COLOR_COMBO(color_combo), GTK_PLOT(plot)->background);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ratio_item), FALSE);
}

GtkWidget *
sg_layer_dialog_new (SGlayer *layer)
{
  GtkWidget *frame;
  GtkWidget *main_box, *main_table;
  GtkWidget *table;
  GtkWidget *box;
  GtkWidget *label;
  GtkRequisition req;

  the_layer = layer;
  plot = GTK_PLOT(layer->real_plot);
  the_plot = SG_PLOT(layer->parent);

  /* Create widgets */
  main_box = gtk_hbox_new (FALSE,5);
  gtk_container_set_border_width(GTK_CONTAINER(main_box), 5);

  main_table = gtk_table_new(2,2,FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(main_table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(main_table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(main_table), 5);
  gtk_box_pack_start (GTK_BOX (main_box), main_table, FALSE, FALSE, 0);
/*----------------------------------------------*/
  frame = gtk_frame_new("Position");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach_defaults(GTK_TABLE(main_table), frame, 0, 1, 0, 1);

  table=gtk_table_new(5, 3, FALSE);  
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  box = gtk_hbox_new(TRUE, 5);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            box, 0, 4, 0, 1);
  label = gtk_label_new("Units ");
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);
  units_combo = gtk_combo_new();
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(units_combo)->entry), FALSE);
  gtk_box_pack_end(GTK_BOX(box), units_combo, FALSE, FALSE, 0);

  gtk_table_attach(GTK_TABLE(table), gtk_hseparator_new(), 0, 4, 1, 2, 
                   GTK_FILL, GTK_SHRINK, 5, 5); 

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new("left offset "), 0, 1, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new("top offset "), 0, 1, 3, 4);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new("width "), 2, 3, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new("height "), 2, 3, 3, 4);
  left_entry = gtk_entry_new();
  top_entry = gtk_entry_new();
  width_entry = gtk_entry_new();
  height_entry = gtk_entry_new();

  gtk_widget_size_request(left_entry, &req);
  req.width /= 2;
  gtk_widget_set_usize(left_entry, req.width, req.height);
  gtk_widget_set_usize(top_entry, req.width, req.height);
  gtk_widget_set_usize(width_entry, req.width, req.height);
  gtk_widget_set_usize(height_entry, req.width, req.height);

  gtk_table_attach_defaults(GTK_TABLE(table), left_entry, 1, 2, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), top_entry, 1, 2, 3, 4);
  gtk_table_attach_defaults(GTK_TABLE(table), width_entry, 3, 4, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), height_entry, 3, 4, 3, 4);

  gtk_table_attach_defaults(GTK_TABLE(table), 
            ratio_item = gtk_check_item_new_with_label("preserve aspect ratio"),
            0, 4, 4, 5);
  
 /*--------------------------------------*/
  frame = gtk_frame_new("Properties");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach_defaults(GTK_TABLE(main_table), frame, 1, 2, 0, 1);

  table=gtk_table_new(3, 3, FALSE);  
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  transparent_item = gtk_check_item_new_with_label("transparent");
  gtk_table_attach_defaults(GTK_TABLE(table), transparent_item, 0, 2, 0, 1);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new("Background"), 0, 1, 1, 2);
  color_combo = gtk_color_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table), color_combo, 1, 2, 1, 2);

/*
  legends_item = gtk_check_item_new_with_label("show legends");
  gtk_table_attach_defaults(GTK_TABLE(table), legends_item, 0, 2, 2, 3);
*/
  bg_button = gtk_button_new_with_label("Browse...");

  gtk_table_attach_defaults(GTK_TABLE(table), gtk_label_new("Use image"), 0, 1, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), bg_button, 1, 2, 2, 3);
/*--------------------------------------*/
  /* connect signals */
  init_dialog();

  sg_entry_set_numeric(GTK_ENTRY(left_entry), 3);
  sg_entry_set_numeric(GTK_ENTRY(top_entry), 3);
  sg_entry_set_numeric(GTK_ENTRY(width_entry), 3);
  sg_entry_set_numeric(GTK_ENTRY(height_entry), 3);

  gtk_signal_connect (GTK_OBJECT (bg_button),
                      "clicked",
                      GTK_SIGNAL_FUNC (select_image),
                      NULL);

  gtk_signal_connect (GTK_OBJECT (GTK_COMBO(units_combo)->entry),
                      "changed",
                      GTK_SIGNAL_FUNC (new_units),
                      NULL);

  gtk_signal_connect (GTK_OBJECT (width_entry),
                      "activate",
                      GTK_SIGNAL_FUNC (preserve_ratio),
                      NULL);

  gtk_signal_connect (GTK_OBJECT (height_entry),
                      "activate",
                      GTK_SIGNAL_FUNC (preserve_ratio),
                      NULL);

  return main_box;
}
