/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_page_dialog.h"
#include "sg.h"

#define PTS_MM 2.8350000
#define PTS_CM 28.3500000
#define PTS_IN 72.0000000
#define PTS_PT 1.0000000
#define PTS_PIX (SG_DPI/PTS_IN)

static GtkWidget *size_combo; 
static GtkWidget *units_combo; 
static GtkWidget *width_entry; 
static GtkWidget *height_entry; 
static GtkWidget *orient_combo; 
static GtkWidget *color_combo; 
static GtkWidget *grid_item; 
static GtkWidget *ratio_item; 
static GtkWidget *antialias_item; 
static SGplot *plot;
static gint pts_width, pts_height;
static gint the_units;

static gchar *units[] = {"PS points",
                         "mm",
                         "cm",
                         "inches",
                         "pixels",
                         NULL};

static gchar *orientations[] = {"Portrait",
                               "Landscape",
                               NULL};

static gchar *page_sizes[] = {"Letter",
                              "Legal",
                              "A4",
                              "Executive",
                              "Custom",
                              NULL};


void
sg_page_dialog_update_plot(GtkWidget *widget)
{
  GtkWidget *child;
  GtkPlotCanvas *canvas;
  GtkColorCombo *combo;
  GdkColor color;
  gchar *color_name;
  gint page_size, units;
  gint orientation;
  gboolean do_art = FALSE;
  gboolean resize = FALSE;

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(size_combo)->list)->selection->data;
  page_size = gtk_list_child_position(GTK_LIST(GTK_COMBO(size_combo)->list),
                                      child);

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(units_combo)->list)->selection->data;
  plot->page_units = units = gtk_list_child_position(GTK_LIST(GTK_COMBO(units_combo)->list), child);

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(orient_combo)->list)->selection->data;
  orientation = gtk_list_child_position(GTK_LIST(GTK_COMBO(orient_combo)->list),
                                        child);

  combo = GTK_COLOR_COMBO(color_combo);
  color_name = gtk_color_combo_get_color_at(combo, combo->row, combo->column);
  gdk_color_parse(color_name, &color);
  gdk_color_alloc(color_combo->style->colormap, &color);

  canvas = GTK_PLOT_CANVAS(plot->real_canvas);

  canvas->background = color;

  canvas->show_grid = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(grid_item));
  do_art = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(antialias_item));
/*
  resize =  = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(ratio_item));
*/

  sg_plot_antialias(plot, do_art);

  if(page_size == GTK_PLOT_CUSTOM){
    gdouble width, height;
    width = atof(gtk_entry_get_text(GTK_ENTRY(width_entry))); 
    height = atof(gtk_entry_get_text(GTK_ENTRY(height_entry))); 
    switch(units){
     case SG_UNIT_MM:
       pts_width = width * PTS_MM; 
       pts_height = height * PTS_MM; 
       break;
     case SG_UNIT_CM:
       pts_width = width * PTS_CM; 
       pts_height = height * PTS_CM; 
       break;
     case SG_UNIT_IN:
       pts_width = width * PTS_IN; 
       pts_height = height * PTS_IN; 
       break;
     case SG_UNIT_PIXEL:
       pts_width = width * PTS_PIX;
       pts_height = height * PTS_PIX;
       break;
     case SG_UNIT_PT:
     default:
       pts_width = width * PTS_PT; 
       pts_height = height * PTS_PT; 
       break;
    }
  }

  sg_plot_set_size(plot, page_size, pts_width, pts_height, orientation);
}

static void
new_units(GtkWidget *widget, gpointer data)
{
  GtkWidget *child;
  gchar label[100];
  gint new_units;
  gdouble width, height;

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(units_combo)->list)->selection->data;
  new_units = gtk_list_child_position(GTK_LIST(GTK_COMBO(units_combo)->list), child);

  switch(new_units){
   case SG_UNIT_MM:
       width = (gdouble)pts_width / (gdouble)PTS_MM; 
       height = (gdouble)pts_height / (gdouble)PTS_MM; 
       break;
   case SG_UNIT_CM:
       width = (gdouble)pts_width / (gdouble)PTS_CM; 
       height = (gdouble)pts_height / (gdouble)PTS_CM; 
       break;
   case SG_UNIT_IN:
       width = (gdouble)pts_width / (gdouble)PTS_IN; 
       height = (gdouble)pts_height / (gdouble)PTS_IN; 
       break;
   case SG_UNIT_PIXEL:
       width = (gdouble)pts_width / (gdouble)PTS_PIX;
       height = (gdouble)pts_height / (gdouble)PTS_PIX;
       break;
   case SG_UNIT_PT:
   default:
       width = (gdouble)pts_width / (gdouble)PTS_PT; 
       height = (gdouble)pts_height / (gdouble)PTS_PT; 
       break;
  }

  sprintf(label, "%.*f", 3, width);
  gtk_entry_set_text(GTK_ENTRY(width_entry), label);
  sprintf(label, "%.*f", 3, height);
  gtk_entry_set_text(GTK_ENTRY(height_entry), label);
}


static void
new_size(GtkCombo *combo, gpointer data)
{
  GtkWidget *child;
  gint new_size;

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(size_combo)->list)->selection->data;
  new_size = gtk_list_child_position(GTK_LIST(GTK_COMBO(size_combo)->list), child);

  gtk_widget_set_sensitive(width_entry, FALSE);
  gtk_widget_set_sensitive(height_entry, FALSE);
  switch(new_size){
   case GTK_PLOT_LETTER:
      pts_width = GTK_PLOT_LETTER_W;
      pts_height = GTK_PLOT_LETTER_H;
      break;
   case GTK_PLOT_LEGAL:
      pts_width = GTK_PLOT_LEGAL_W;
      pts_height = GTK_PLOT_LEGAL_H;
      break;
   case GTK_PLOT_A4:
      pts_width = GTK_PLOT_A4_W;
      pts_height = GTK_PLOT_A4_H;
      break;
   case GTK_PLOT_EXECUTIVE:
      pts_width = GTK_PLOT_EXECUTIVE_W;
      pts_height = GTK_PLOT_EXECUTIVE_H;
      break;
   case GTK_PLOT_CUSTOM:
      gtk_widget_set_sensitive(width_entry, TRUE);
      gtk_widget_set_sensitive(height_entry, TRUE);
      break;
   }

   new_units(NULL, NULL);

}

static void
init_dialog()
{

  pts_width = GTK_PLOT_CANVAS(plot->real_canvas)->pixmap_width;
  pts_height = GTK_PLOT_CANVAS(plot->real_canvas)->pixmap_height;

  sg_combo_set_items(GTK_COMBO(units_combo), units);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(units_combo)->list), plot->page_units);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(units_combo)->entry), FALSE);

  sg_combo_set_items(GTK_COMBO(orient_combo), orientations);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(orient_combo)->list), plot->orientation);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(orient_combo)->entry), FALSE);

  sg_combo_set_items(GTK_COMBO(size_combo), page_sizes);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(size_combo)->list), plot->page_size);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(size_combo)->entry), FALSE);

  sg_color_combo_init(GTK_COLOR_COMBO(color_combo), GTK_PLOT_CANVAS(plot->real_canvas)->background);

  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(grid_item), 
                              GTK_PLOT_CANVAS(plot->real_canvas)->show_grid);
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(antialias_item), plot->antialias);
}

GtkWidget *
sg_page_dialog_new (SGplot *the_plot)
{
  GtkWidget *frame;
  GtkWidget *main_box;
  GtkWidget *table;
  GtkWidget *label;
  GtkRequisition req;

  plot = the_plot;

  /* Create widgets */
  main_box = gtk_hbox_new (FALSE,5);
  gtk_container_set_border_width(GTK_CONTAINER(main_box), 10);

  /* Poperties */

  frame = gtk_frame_new("Page properties");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (main_box), frame, FALSE, FALSE, 0);

  table=gtk_table_new(9, 2, FALSE);  
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  label = gtk_label_new("Page size ");
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
  label = gtk_label_new("Width ");
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);
  label = gtk_label_new("Height ");
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);
  label = gtk_label_new("Units ");
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 3, 4);
  label = gtk_label_new("Orientation ");
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 4, 5);
  label = gtk_label_new("Background ");
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 5, 6);

  size_combo = gtk_combo_new();
  units_combo = gtk_combo_new();
  orient_combo = gtk_combo_new();
  width_entry = gtk_entry_new();
  height_entry = gtk_entry_new();
  color_combo = gtk_color_combo_new();
/*
  ratio_item = gtk_check_item_new_with_label("Preserve aspect ratio");
*/
  grid_item = gtk_check_item_new_with_label("Show guiding grid");
  antialias_item = gtk_check_item_new_with_label("Antialiased");

  gtk_widget_size_request(width_entry, &req);
/*
  req.width = (gdouble)req.width * .75;
*/

  gtk_widget_set_usize(width_entry, req.width, req.height);
  gtk_widget_set_usize(height_entry, req.width, req.height);
  gtk_widget_set_usize(size_combo, req.width, req.height);
  gtk_widget_set_usize(units_combo, req.width, req.height);
  gtk_widget_set_usize(orient_combo, req.width, req.height);

  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(size_combo)->entry), FALSE);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(units_combo)->entry), FALSE);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(orient_combo)->entry), FALSE);

  gtk_table_attach_defaults(GTK_TABLE(table), size_combo, 1, 2, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(table), width_entry, 1, 2, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), height_entry, 1, 2, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), units_combo, 1, 2, 3, 4);
  gtk_table_attach_defaults(GTK_TABLE(table), orient_combo, 1, 2, 4, 5);
  gtk_table_attach_defaults(GTK_TABLE(table), color_combo, 1, 2, 5, 6);
/*
  gtk_table_attach_defaults(GTK_TABLE(table), ratio_item, 0, 2, 6, 7);
*/
  gtk_table_attach_defaults(GTK_TABLE(table), grid_item, 0, 2, 6, 7);
  gtk_table_attach_defaults(GTK_TABLE(table), antialias_item, 0, 2, 7, 8);

  /* connect signals */
  gtk_signal_connect (GTK_OBJECT (GTK_COMBO(size_combo)->entry), 
                      "changed",
                      GTK_SIGNAL_FUNC (new_size),
                      NULL);

  gtk_signal_connect (GTK_OBJECT (GTK_COMBO(units_combo)->entry), 
                      "changed",
                      GTK_SIGNAL_FUNC (new_units),
                      NULL);

  init_dialog();

  sg_entry_set_numeric(GTK_ENTRY(width_entry), 3);
  sg_entry_set_numeric(GTK_ENTRY(height_entry), 3);
/*
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ratio_item), TRUE);
*/

  return main_box;
}
