//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [pos_car,vel_car,jacob]=CL_co_sph2car(pos_sph,vel_sph)
// Spherical coordinates to cartesian coordinates
//
// Calling Sequence
// [pos_car [,vel_car,jacob]] = CL_co_sph2car(pos_sph [,vel_sph])
//
// Description
// <itemizedlist><listitem>
// This function converts cartesian coordinates into spherical coordinates.  
// <para> The input arguments can consist in position only, or position and velocity, in which case
// the derivatives of the spherical coordinates with respect to time are computed as well. </para> 
// <para/><inlinemediaobject><imageobject><imagedata fileref="spherical_coord.gif"/></imageobject></inlinemediaobject>
// </listitem>
// <listitem>
// Notes: 
// <para> - The transformation jacobian is computed if the corresponding output argument exists. </para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Parameters
// pos_sph : [lon;lat;r] Position vector in spherical coordinates [m,rad] (3xN)
// vel_sph : (optional) [d(lon)/dt;d(lat)/dt;d(r)/dt] Derivatives / time of spherical coordinates [rad/s,m/s] (3xN)
// pos_car: [X;Y;Z] Position vector in cartesian coordinates [m] (3xN)
// vel_car: (optional) [Vx,Vy,Vz] Velocity vector in cartesian coordinates [m/s] (3xN)
// jacob:  (optional) Transformation jacobian (6x6xN)
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// 1 Mecanique Spatiale, Cnes - Cepadues Editions, Tome I, section 3.2.3 (Les reperes de l'espace et du temps, Relations entre les coordonnees)
// 2 CNES - MSLIB FORTRAN 90, Volume T (mt_geoc_car)
//
// See also
// CL_co_car2sph
// CL_co_car2ell
// CL_co_ell2car
//
// Examples
// // Example 1
// pos_sph = [0,0,%CL_eqRad;%pi/2,%pi/2,%CL_eqRad]';
// pos_car = CL_co_sph2car(pos_sph);
//
// // Example 2 :
// pos_sph = [0.090715;5.362077;6377951.7];
// vel_sph = [1.8332e-5;2.2060e-4;-181.488];
// [pos_car,vel_car,jacob1] = CL_co_sph2car(pos_sph,vel_sph);

// Declarations:


// Code:

[lhs,rhs] = argn();

select lhs
  case 1
    compute_velocity = %f
    compute_jacob = %f
  case 2
    compute_velocity = %t
    compute_jacob = %f
    if ~exists('vel_sph','local') then CL__error('give velocity as input argument'); end
  case 3
    compute_velocity = %t;
    compute_jacob = %t;
    if ~exists('vel_sph','local') then CL__error('give velocity as input argument'); end
  else
    CL__error('check number of output arguments')
end

//get latitude, longitude and distance
lat = pos_sph(2,:);
lon = pos_sph(1,:);
d = pos_sph(3,:);

rcos1 = cos(lat)
rsin1 = sin(lat)
rcos2 = cos(lon)
rsin2 = sin(lon)

rc1c2 = rcos1.*rcos2
rs1s2 = rsin1.*rsin2
rs1c2 = rsin1.*rcos2
rc1s2 = rcos1.*rsin2

//transform into cartesian coordinates
x = d.*rc1c2;
y = d.*rc1s2;
z = d.*rsin1;
pos_car = [x;y;z];

if compute_velocity
  lat_v = vel_sph(2,:)
  lon_v = vel_sph(1,:)
  d_v = vel_sph(3,:)

  vx = d_v.*rc1c2 - z.*rcos2.*lat_v - y.*lon_v
  vy = d_v.*rc1s2 - z.*rsin2.*lat_v + x.*lon_v
  vz = d_v.*rsin1 + d.*rcos1.*lat_v
  vel_car = [vx;vy;vz]
end

if compute_jacob
  jacob = zeros(6,6,size(pos_sph,2))
  //x partial derivative
  jacob(1,2,:) = -d.*rs1c2
  jacob(1,1,:) = -y
  jacob(1,3,:) = rc1c2
  //y partial derivative
  jacob(2,2,:) = -d.*rs1s2
  jacob(2,1,:) = x
  jacob(2,3,:) = rc1s2
  //z partial derivative
  jacob(3,2,:) = d.*rcos1
  jacob(3,3,:) = rsin1
  //vx partial derivative
  jacob(4,2,:) = -vz.*rcos2 + z.*rsin2.*lon_v
  jacob(4,1,:) = -vy
  jacob(4,3,:) = -rs1c2.*lat_v - rc1s2.*lon_v
  jacob(4,5,:) = -d.*rs1c2
  jacob(4,4,:) = -y
  jacob(4,6,:) = rc1c2
  //vy partial derivative
  jacob(5,2,:) = -vz.*rsin2 - z.*rcos2.*lon_v
  jacob(5,1,:) = vx
  jacob(5,3,:) = rc1c2.*lon_v - rs1s2.*lat_v
  jacob(5,5,:) = -d.*rs1s2
  jacob(5,4,:) = x
  jacob(5,6,:) = rc1s2
  //vz partial derivative
  jacob(6,2,:) = d_v.*rcos1 - z.*lat_v
  jacob(6,3,:) = rcos1.*lat_v
  jacob(6,5,:) = d.*rcos1
  jacob(6,6,:) = rsin1

  if size(pos_sph,2)==1 then jacob=jacob(:,:,1); end
end

endfunction
