//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [semi_ang_ecl] = CL_gm_betaEclipse(sma,betaa,er)
// Sun eclipse angle as function of beta for circular orbits.
//
// Calling Sequence
// [semi_ang_ecl] = CL_gm_betaEclipse(sma,betaa [,er])
//
// Description
// <itemizedlist><listitem>
// This function computes the half-angle that characterizes the portion of the orbit 
// path where the Sun is eclipsed by the planet as seen from the satellite. 
// <para> The eclipse semi-angle (<emphasis role="bold">semi_ang_ecl</emphasis>) is computed as a function of semi major axis (<emphasis role="bold">sma</emphasis>) and beta angle (<emphasis role="bold">betaa</emphasis>: signed angle between the orbit plane and the Sun direction). </para>
// <para>If the Sun is never eclipsed, the semi-angle is set to 0.</para>
// <inlinemediaobject><imageobject><imagedata fileref="betaEclipse.gif"/></imageobject></inlinemediaobject>
// </listitem>
// <listitem>
// Notes: 
// <para> - A spherical planet and a circular orbit are assumed. </para>
// <para> - See also CL_gm_eclipseCir for more detailed results on the beginning/end of eclipse. </para>  
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Parameters
// sma: Semi major axis [m] (1xN)
// betaa: Beta angle [rad] (1xN)
// er: (optional) Planet radius (default is %CL_eqRad) [m]
// semi_ang_ecl: in-orbit eclipse semi-angle [rad] (1xN)
//
// See also
// CL_gm_eclipseCir
// CL_gm_raan2beta
// CL_mod_moonSunG50
//
// Authors
// CNES - DCT/SB
//
// Examples
// // Example 1
// betaa=CL_deg2rad([10;20]);
// sma=[%CL_eqRad+350000;%CL_eqRad+400000];
// CL_gm_betaEclipse(sma,betaa)
//
// // Example 2
// cjd = 20050;
// [r_sun,rs] = CL_mod_moonSunG50(cjd,'s');
// r_sph = CL_co_car2sph(r_sun);
// alpha_sun = r_sph(1);
// delta_sun = r_sph(2);
// raan = %pi/4;
// inc = CL_deg2rad(98.7);
// betaa = CL_gm_raan2beta(inc,raan,alpha_sun,delta_sun)
//
// sma = 6978.e3;
// semi_ang_ecl = CL_gm_betaEclipse(sma,betaa)
// //mean motion J2
// xnmoy = CL_op_paramsJ2('mm',sma,0,inc);
// tot_duration_minutes = 2*semi_ang_ecl/xnmoy/60.0

// Declarations:
global %CL_eqRad;

// Code:

if ~exists('er','local') then er=%CL_eqRad; end

semi_ang_ecl=zeros(betaa);

//Indices sans exception numerique
ind = find(~(abs(betaa)==(%pi/2.))&(sqrt(1-(er./sma).^2)./cos(betaa)<1));

//Demi amplitude de l'eclipse
semi_ang_ecl(ind)=acos(sqrt(1-(er./sma(ind)).^2)./cos(betaa(ind)));

//Limite de validite de la formule
ind=find(abs(betaa)>asin(er./sma));
semi_ang_ecl(ind)=0;


endfunction
