//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [psoE] = CL_kp_M2Ecir(ex,ey,psoM)
// Mean arg. of latitude to eccentric arg. of latitude (Kepler's equation)
//
// Calling Sequence
// psoE = CL_kp_M2Ecir(ex,ey,psoM)
//
// Description
// <itemizedlist><listitem>
// This function solves the 'generalized' Kepler's equation for elliptical orbits. 
// <para> It computes the eccentric argument of latitude <emphasis role="bold">psoE</emphasis> from the mean 
// argument of latitude <emphasis role="bold">psoM</emphasis> and the eccentricity vector components 
// <emphasis role="bold">ex</emphasis> and <emphasis role="bold">ey</emphasis>. </para>
// <para><inlinemediaobject><imageobject><imagedata fileref="kep_gen.gif"/></imageobject></inlinemediaobject> </para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Parameters
// ex: x component of eccentricity vector (PxN or Px1)
// ey: y component of eccentricity vector (PxN or Px1)
// psoM: Mean argument of latitude (=w+M) [rad] (PxN or Px1)
// psoE: Eccentric argument of latitude (=w+E) [rad] (PxN)
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// 1 CNES - MSLIB FORTRAN 90, Volume V (mv_kepler_gene)
//
// See also
// CL_kp_M2E
//
// Examples
// pom = 2*%pi/3;
// ecc = 0.001;
// ex = ecc * cos(pom);
// ey = ecc * sin(pom);
// M = (0:4)*%pi/4;
// E = CL_kp_M2Ecir(ex,ey,pom+M) - pom; 
//
// // Now using CL_kp_M2E :
// E = CL_kp_M2E(ecc,M) 

// Declarations:
global %CL_epsilon;

// Code:

Nex = size(ex,2);
Ney = size(ey,2);
NpsoM = size(psoM,2);
N = max(Nex,Ney,NpsoM);

if ~( (Nex==N|Nex==1) & (Ney==N|Ney==1) & (NpsoM==N|NpsoM==1))
   CL__error('Wrong size of input arguments'); 
end

// adjust the sizes
if (Nex == 1); ex = ex * ones(1,N); end
if (Ney == 1); ey = ey * ones(1,N); end
if (NpsoM == 1); psoM = psoM * ones(1,N); end 


// test on eccentricity
ecc2 = (ex.^2) + (ey.^2); 
ecc = sqrt(ecc2);        

if (find(ecc >= 1-%CL_epsilon.parab) <> [])
   CL__error("Not elliptical orbit"); 
end


// initialisation de encke
// =======================
cos_psoM = cos(psoM);
sin_psoM = sin(psoM);

x = (ex .* cos_psoM) + (ey .* sin_psoM);
y = (ex .* sin_psoM) - (ey .* cos_psoM);

// initially:  
// r = (e2-x)/y;  
// az = e * sign(y) / sqrt(1+r^2)
// f0 = psoM + az - (az.^4 .*x .* (1-ecc2)) / (6 .* y); 
// => improved to avoid division by 0

az = 1 ./ sqrt(1 + ecc2 - 2 * x); 
psoE = psoM + az.*y - (az.^4 .*x .* (1-ecc2)) .* (y.^3) / 6; 

// -------------------------------------------
// resolution iterative par methode de newton
// f(x) = 0
// xi+1 = xi - (f(xi) / f'(xi))
// -------------------------------------------

nb_max_iter = 20;          // max number of iterations
errmax = 100 * %eps;       // accuracy 

err1 = ones(psoE) * %inf;  // initial value for accuracy variables
err2 = ones(psoE) * %inf; 

K = 1:length(psoE);        // indices to be dealt with
iter = 0;                  // iter = current number of iterations

while (K ~= [] & iter <= nb_max_iter) 

  psoE1 = psoE; // previous value

  psoE(K) = psoE(K) - (psoE(K) - ex(K).*sin(psoE(K)) + ey(K).*cos(psoE(K)) - psoM(K)) ./ ...
            (1 - ex(K).*cos(psoE(K)) - ey(K).*sin(psoE(K)));

  err1(K) = abs( psoE1(K)-psoE(K) ) ./ max(abs(psoE(K)),1);
  err2(K) = abs( psoM(K) - psoE1(K) + ex(K).*sin(psoE1(K)) - ey(K).*cos(psoE1(K)) ) ./ ...
            max(abs(psoM(K)),1);

  K = find( err1 > errmax | err2 > errmax )
  iter = iter + 1; 

end

if (K ~= []) 
   CL__error('Maximum number of iterations reached'); 
end


endfunction

