//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [d_ecc_dt,d_pom_dt] = CL_op_frozenOrbitDer(sma,ecc,inc,pom,er,mu,zonals)
// Eccentricity and perigee drift over time for an orbit close to a frozen orbit
//
// Calling Sequence
// [d_ecc_dt,d_pom_dt] = CL_op_frozenOrbitDer(sma,ecc,inc,pom,[er,[mu,[zonals]]])
//
// Description
// <itemizedlist><listitem>
// This function computes eccentricity and periapsis argument derivatives with respect to time. These derivatives
// correspond to a frozen orbit in presence of perturbations coming from terms J2 to J3 of gravitational potential.
// Eccentricity and periapsis argument (that are close to the values for a frozen orbit) must be given as input.
// <para>You can optionally give the zonal coefficients J1/J2/J3 to be used  : default is %CL_j1jn(1:3).</para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Parameters
// sma: semi major axis [m] (1x1 : same semi major axis for all) or (1xM)
// ecc: eccentricity (1x1 : same eccentricity for all) or (1xM)
// inc: inclination [rad] (1x1 : same inclination for all) or (1xM)
// pom: argument of perigee (1x1 : same argument of perigee for all) or (1xM)
// er: (optional) equatorial radius [m] (default is %CL_eqRad)
// mu: (optional) geocentric gravitational constant [m^3/s^2] (default value is %CL_mu)
// zonals: (optional) vector of zonals coefficients J1 to Jn (troncated to J3) to be used (default is %CL_j1jn(1:3)) (1 x N)
// d_ecc_dt: (optional depending on model) eccentricity derivative with respect to time [s^-1] (1xM)
// d_pom_dt: (optional depending on model) periapsis argument derivative with respect to time [rad/s] (1xM)
//
// Authors
// CNES - DCT/SB
//
// Examples
// sma = [%CL_eqRad+350.e3 %CL_eqRad+700.e3];
// inc = CL_deg2rad([51.6 91.6]);
// [ecc,pom] = CL_op_frozenOrbit(sma,inc);
// ecc = 0.999*ecc;
// pom = 0.999*pom;
// [d_ecc_dt,d_pom_dt] = CL_op_frozenOrbitDer(sma,ecc,inc,pom)


	function [d_ecc_dt,d_pom_dt] = CL_op_perGelD(a,ecc,xi,pom,er,mu,j2,j3)

	//calcul des derivees de l'eccentricite et de l'argument du perigee au sens des termes J2 et J3 du potentiel terrestre


	// Declarations:


	// Code:

	sinxi = sin(xi)
	cosxi = cos(xi)
	cospom = cos(pom)
	sinpom = sin(pom)

	xn = CL_kp_params('mm',a,mu) //mean motion

	//calculation of eccentricity derivative
	exp1 = 1.5*xn .* (er./a).^3 .* j3 ./ (1-ecc.^2).^2
	exp2 = sinxi .* cospom .* (1.25.*(sinxi).^2 + 1)
	d_ecc_dt = exp1.*exp2

	//calculation of periapsis argument derivative
	exp11 = -0.75*xn .* (er./a).^2 .* j2 ./ (1-ecc.^2).^2 .* (1-5*cosxi.^2)
	exp22 = -1.5*xn .* (er./a).^3 .* j3 ./ (ecc.*(1-ecc.^2).^3) .* sinpom./sinxi
	exp3 = (1.25*sinxi.^2 - 1) .* sinxi.^2
	exp4 = ecc.^2 .* (1-8.75*sinxi.^2 .* cosxi.^2)
	d_pom_dt = exp11 + exp22.*(exp3+exp4)

	endfunction


// Declarations:
global %CL_eqRad;
global %CL_mu;
global %CL_j1jn;


// Code:

Na = size(sma,2); //number of semi major axis given
Ninc = size(inc,2); //number of inclinations given
Necc = size(ecc,2); //number of inclinations given
Npom = size(pom,2); //number of inclinations given

N = max(Na,Ninc,Necc,Npom); //max of sizes

coherence = (Na==N|Na==1) & (Ninc==N|Ninc==1) & (Necc==N|Necc==1) & (Npom==N|Npom==1); //check coherence for vectorization
if ~coherence then CL__error('bad size of input arguments'); end

if N~=1
  if Na==1 then sma=sma*ones(1,N); end
  if Ninc==1 then inc=inc*ones(1,N); end
  if Necc==1 then ecc=ecc*ones(1,N); end
  if Npom==1 then pom=pom*ones(1,N); end
end

if ~exists('er','local') then er=%CL_eqRad; end
if ~exists('mu','local') then mu=%CL_mu; end
if ~exists('zonals','local') then zonals=%CL_j1jn(1:3); end

Nzonals = length(zonals);
if(Nzonals <3)
  CL__error('zonals must be a vector of size 3 (or more)');
end

[d_ecc_dt,d_pom_dt] = CL_op_perGelD(sma,ecc,inc,pom,er,mu,zonals(2),zonals(3))


endfunction
