//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [delta_v,dv1,dv2,anv1,anv2] = CL_man_hohmannG(ai,ei,af,ef,posman1,rotation,mu)
// Delta V for a generalized Hohmann transfer
//
// Calling Sequence
// [delta_v,dv1,dv2,anv1,anv2] = CL_man_hohmannG(ai,ei,af,ef [,posman1,rotation,mu])
//
// Description
// <itemizedlist><listitem>
// This function computes the maneuvers of a generalized Hohmann transfer.
// The 2 velocity increments transform an initial orbit defined by semi-major axis and eccentricity into a final orbit also defined by semi-major axis and eccentricity. 
// <para>The first velocity increment can be at periapsis or apoapsis of the initial orbit, depending on the value of <emphasis role="bold">posman1</emphasis>. </para>
// <para>The second velocity increment is 180 degrees apart.</para>
// <para><emphasis role="bold">delta_v</emphasis> is the sum of the norms of the velocity increments required 
// (|<emphasis role="bold">dv1</emphasis>| + |<emphasis role="bold">dv2</emphasis>|).</para>
// <para>Velocity increments are expressed in spherical coordinates in the QSW frame: 
// [lambda; phi; dv], where lambda is the in-plane angle (+%pi: towards planet and +%pi/2: 
// ~along velocity), phi is the out-of-plane angle, positive towards the angular momentum vector 
// (the angular momentum vector is perpendicular to the orbit plane and oriented according to 
// right hand rule), dv is the norm of the velocity increment. </para>
// <para><emphasis role="bold">anv1</emphasis> is the true anomaly at the location of  the first velocity increment, and <emphasis role="bold">anv2</emphasis> is the anomaly at the second location (and relative to the intermediate orbit).</para>
// <para><emphasis role="bold">posman1</emphasis> is used to define the location of the first maneuver (0->periapsis, 1->apoapsis)</para>
// <para><emphasis role="bold">rotation</emphasis> defines whether the final orbit should have its perigee rotated 180deg relative to the initial orbit (0->no rotation, 1->180 degree rotation)</para>
// <para/><inlinemediaobject><imageobject><imagedata fileref="hohmanng.gif"/></imageobject></inlinemediaobject></listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Parameters
// ai : Semi-major axis of initial orbit [m] (1xN)
// ei: Eccentricity of initial orbit (1xN)
// af : Semi-major axis of final orbit [m] (1xN)
// ef: Eccentricity of final orbit (1xN)
// posman1: (optional) Flag indicating the location of the first maneuver (0->periapsis, 1->apoapsis; default is 0) (1xN)
// rotation: (optional) Flag indicating whether the final orbit should have its perigee rotated 180deg relative to the initial orbit (0->no rotation, 1->rotation ; default is 0) (1xN)
// mu : (optionnal) Gravitational constant. [m^3/s^2] (default value is %CL_mu)
// delta_v : Total delta-v required = |dv1| + |dv2|. [m/s] (1xN)
// dv1 : First velocity increment in spherical coordinates in the QSW frame [lambda;phi;|dv|] [rad,rad,m/s] (3xN)
// dv2 : Second velocity increment in spherical coordinates in the QSW frame [lambda;phi;|dv|] [rad,rad,m/s] (3xN)
// anv1: True anomaly at the location of the first velocity increment (in the initial orbit). [rad] (1xN)
// anv2: True anomaly at the location of the second velocity increment (in the intermediate orbit). [rad] (1xN)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_man_biElliptic
// CL_man_hohmann
// CL_man_sma
//
// Examples
// // Maneuver at apogee, no rotation
// ai = 7200.e3;
// af = 7000.e3;
// ei = 0.1;
// ef = 0.1;
// [delta_v,dv1,dv2,anv1,anv2] = CL_man_hohmannG(ai,ei,af,ef,1,0)
// // Check results :
// kep = [ai ; ei ; %pi/2 ; 0 ; 0 ; anv1];
// kep1 = CL_man_applyDv(kep,dv1);
// kep1(6) = anv2;
// kep2 = CL_man_applyDv(kep1,dv2)
//
// // Maneuver at apogee, with rotation
// ai = 7200.e3;
// af = 7000.e3;
// ei = 0.1;
// ef = 0.1;
// [delta_v,dv1,dv2,anv1,anv2] = CL_man_hohmannG(ai,ei,af,ef,1,1)
// // Check results :
// kep = [ai ; ei ; %pi/2 ; 0 ; 0 ; anv1];
// kep1 = CL_man_applyDv(kep,dv1);
// kep1(6) = anv2;
// kep2 = CL_man_applyDv(kep1,dv2)


// Declarations:
global %CL_mu;

// Code:

if ~exists('posman1','local') then posman1=0; end
if ~exists('rotation','local') then rotation=0; end
if ~exists('mu','local') then mu=%CL_mu; end

Nai = size(ai,2);
Nei = size(ei,2);
Naf = size(af,2);
Nef = size(ef,2);
Nposman1 = size(posman1,2);
Nrotation = size(rotation,2);
N = max(Nai,Naf,Nei,Nef,Nposman1,Nrotation);
coherence = (Nai==N | Nai==1) & (Nei==N | Nei==1) &..
            (Naf==N | Naf==1) & (Nef==N | Nef==1) &..
            (Nposman1==N | Nposman1==1) & (Nrotation==N | Nrotation==1);
if ~coherence then CL__error('bad input argument size'); end
if N~=1
  if Nai==1 then ai=ai*ones(1,N); end
  if Naf==1 then af=af*ones(1,N); end
  if Nei==1 then ei=ei*ones(1,N); end
  if Nef==1 then ef=ef*ones(1,N); end
  if Nposman1==1 then posman1=posman1*ones(1,N); end
  if Nrotation==1 then rotation=rotation*ones(1,N); end
end

posman2 = zeros(posman1);
ii = find(posman1 == 0 & rotation ==0);
posman2(ii) = ones(posman2(ii));
ii = find(posman1 == 1 & rotation ==1);
posman2(ii) = ones(posman2(ii));


rman1 = zeros(1,N);
rman2 = zeros(1,N);

//calculation of radius at maneuver point
//first maneuver
i1_11 = find(ei~=0 & posman1~=0);  //maneuver at periapsis and not circular orbit
i1_12 = find(ei~=0 & posman1==0);  //maneuver at apoapsis and not circular orbit
i1_2 = find(ei==0); //circular orbit
rman1(i1_11) = ai(i1_11).*(1-ei(i1_11));
rman1(i1_12) = ai(i1_12).*(1+ei(i1_12));
rman1(i1_2) = ai(i1_2);
//second maneuver
i2_11 = find(ef~=0 & posman2~=0);  //maneuver at periapsis and not circular orbit
i2_12 = find(ef~=0 & posman2==0);  //maneuver at apoapsis and not circular orbit
i2_2 = find(ef==0); //circular orbit
rman2(i2_11) = af(i2_11).*(1-ef(i2_11));
rman2(i2_12) = af(i2_12).*(1+ef(i2_12));
rman2(i2_2) = af(i2_2);

atr = (rman1+rman2)*0.5;

[dv1,anv1] = CL__man_raps(rman1,ai,atr,mu);
[dv2,anv2] = CL__man_raps(rman2,atr,af,mu);

delta_v = dv1(3,:) + dv2(3,:);

endfunction

