//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [v1,v2] = CL_man_lambert(pos1,pos2,delta_t,direction,mu)
// Lambert's problem
//
// Calling Sequence
// [v1,v2] = CL_man_lambert(pos1,pos2,delta_t,direction [,mu])
//
// Description
// <itemizedlist><listitem>
// The function computes the velocity vectors <emphasis role="bold">v1</emphasis> and <emphasis role="bold">v2</emphasis>, given the position vectors <emphasis role="bold">r1</emphasis> and <emphasis role="bold">r2</emphasis>, the time of flight <emphasis role="bold">delta_t</emphasis>, and the direction of motion <emphasis role="bold">direction</emphasis>.
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Parameters
// pos1 : Initial position vector [m] (3xN)
// pos2 : Final position vector [m] (3xN)
// delta_t : Time of flight from r1 to r2 [s] (1xN)
// direction : 'pro' if the transfer orbit is prograde, 'retro' if the transfer orbit is retrograde (1xN)
// mu : (optional) Gravitational constant (default is %CL_mu) [m^3/s^2]
// v1 : Initial velocity vector [m/s] (3xN)
// v2 : Final velocity vector [m/s] (3xN)
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// 1 Orbital Mechanics for Engineering Students, H D Curtis, Section 5.3 and Appendix D.11 (algorithm 5.2)
// 2 Modern astrodynamics Fundamentals and perturbation methods, V Bond and M Allman, Chapter 6.2
//
// See also
// CL_stumpS
// CL_stumpC
//
// Examples
// r1 = [ 5000.e3 10000.e3 2100.e3]';
// r2 = [-14600.e3 2500.e3 7000.e3]';
// dt = 3600; //one hour
// direction = 'pro';
// [v1, v2]=CL_man_lambert([r1 r1],[r2 r2],[dt dt],[direction direction], %CL_mu);

// Declarations:
global %CL_mu;

// Code:

if ~exists('mu','local') then mu=%CL_mu; end

N = size(pos1,2);

//magnitudes of pos1 and pos2
r1 = CL_colNorm(pos1);
r2 = CL_colNorm(pos2);

c12 = CL_cross(pos1,pos2);
theta = acos(CL_dot(pos1,pos2)./r1./r2);

//Determine whether the orbit is prograde or retrograde:
ii = find( (direction=='pro')&(c12(3,:)<=0) )
jj = find( (direction=='retro')&(c12(3,:)>=0) )
kk = find( (direction~='pro')&(direction~='retro') )
theta(ii) = 2*%pi - theta(ii);
theta(jj) = 2*%pi - theta(jj);
direction(kk) = 'pro' //prograde trajectory assumed

//Equation 5.35:
A = sin(theta).*sqrt(r1.*r2./(1 - cos(theta)));

//Determine approximately where F(z,t) changes sign, and
//use that value of z as the starting value for Equation 5.45:
z = -100.*ones(1,N);
aux = real(FF(z,delta_t,r1,r2,A,mu))
tt = find(aux<0)
while tt~=[]
  z(tt) = z(tt) + 0.1;
  aux = real(FF(z,delta_t,r1,r2,A,mu))
  tt = find(aux<0)
end

//Set an error tolerance and a limit on the number of iterations:
tol = 1.e-8;
nmax = 5000;

//Iterate on Equation 5.45 until z is determined to within
//the error tolerance:
ratio = ones(1,N);
n = zeros(1,N);
pp = find( (abs(ratio)>tol) & (n<=nmax) )
while pp~=[]
  n(pp) = n(pp) + 1;
  ratio(pp) = FF(z(pp),delta_t(pp),r1(pp),r2(pp),A(pp),mu)./dFdz(z(pp),A(pp),r1(pp),r2(pp));
  z(pp) = z(pp) - ratio(pp);
  pp = find( (abs(ratio)>tol) & (n<=nmax) )
end

//Report if the maximum number of iterations is ecceeded:
if find(n>=nmax)~=[] then CL__error('Number of iterations ecceeds nmax'); end

//Equation 5.46a:
f = ( 1 - yy(z,r1,r2,A)./r1 ).*.ones(3,1);

//Equation 5.46b:
g = ( A.*sqrt(yy(z,r1,r2,A)./mu) ).*.ones(3,1);

//Equation 5.46d:
gCL_dot = ( 1 - yy(z,r1,r2,A)./r2 ).*.ones(3,1);

//Equation 5.28:
v1 = 1 ./g.*(pos2 - f.*pos1);

//Equation 5.29:
v2 = 1 ./g.*(gCL_dot.*pos2 - pos1);

endfunction

//auxiliar functions
function dum = yy(z,r1,r2,A)

// Declarations:


// Code:

  dum = r1 + r2 + A .* (z.*SS(z)-1) ./ sqrt(CC(z));
endfunction

function dum = FF(z,t,r1,r2,A,mu)

// Declarations:


// Code:

  dum = ( yy(z,r1,r2,A)./CC(z) ).^1.5 .* SS(z) + A.*sqrt(yy(z,r1,r2,A)) - sqrt(mu).*t;
endfunction

function dum = dFdz(z,A,r1,r2)

// Declarations:


// Code:

  ii = find(z==0)
  jj = find(z~=0)
  dum(ii) = sqrt(2) ./ 40 .* yy(0,r1(ii),r2(ii),A(ii))^1.5 + A(ii) ./ 8 .* ...
           ( sqrt(yy(0,r1(ii),r2(ii),A(ii))) + A(ii) .* sqrt(1/2 ./ yy(0,r1(ii),r2(ii),A(ii))) );
  dum(jj) = (yy(z(jj),r1(jj),r2(jj),A(jj))./CC(z(jj))).^1.5 .* ...
            ( 1/2 ./ z(jj).*(CC(z(jj)) - 3.*SS(z(jj))./ 2 ./CC(z(jj)))+ 3.*SS(z(jj)).^2 ./ 4 ./ CC(z(jj))) ...
            + A(jj)./ 8 .* ...
            (3.*SS(z(jj))./CC(z(jj)).*sqrt(yy(z(jj),r1(jj),r2(jj),A(jj))) + A(jj).*sqrt(CC(z(jj))./yy(z(jj),r1(jj),r2(jj),A(jj))));

endfunction

function dum = CC(z)

// Declarations:


// Code:

  dum=CL_stumpC(z);
endfunction

function dum = SS(z)

// Declarations:


// Code:

  dum=CL_stumpS(z);
endfunction
