//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [vmin, vmax, nb1, nb2] = CL_graduate(xmi, xma, ns1, ns2)
// Simple graduation calculation
//
// Calling Sequence
// [vmin, vmax, nb1, nb2] = CL_graduate(xmi, xma, ns1, ns2)
//
// Description
// <itemizedlist>
// <listitem>
// The function computes uniformly distributed graduation values. 
// <para>The steps are computed such that the corresponding values (vmin+k*(vmax-vmin)/nb1, k=0:nb1) are of the form: k*10^n, with k in [1,2,3,5,6]. The substeps are then chosen accordingly. For instance, if k=1, possible values for nb2 are 1, 2, 5, 10, and so on... </para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Parameters
// xmi: User data minimum value.
// xma: User data maximum value.
// ns1: (optional) Approximate number of steps wanted. Default is 7. 
// ns2: (optional) Approximate number of sub-steps wanted. Default is 5. 
// vmin: Interval minimum value (vmin <= xmi).
// vmax: Interval maximum value (vmax >= xma).
// nb1: Number of steps.
// nb2: Number of sub-steps.
//
// Authors
// CNES - DCT/SB (AL)
//
// See also
// CL_autoLevels
//
// Examples
// [vmin, vmax, nb1, nb2] = CL_graduate(0, 6)
// [vmin, vmax, nb1, nb2] = CL_graduate(1.14, 2.27, 5, 5)

// Declarations:


// Code:

if ~exists('ns1', 'local'); ns1=7; end
if ~exists('ns2', 'local'); ns2=5; end

n1 = max(ns1-2, 1);
n2 = max(ns1+2, 1);

xmin = min(xmi, xma);
xmax = max(xmi, xma);

ampl = xmax - xmin;

if (ampl <= 0)
  xmin = xmin - 0.1*xmin;
  xmax = xmax + 0.1*xmax;
  ampl = xmax - xmin;
  if (ampl <= 0) // if xmax == xmin
    xmin = -1;
    xmax = 1;
    ampl = xmax - xmin;
  end
end

pas = ampl ./ (n1:n2);  
fact = (10.^floor(log10(pas)));
pas = pas ./ fact;

valref = [1,2,3,5,6,10]; // step = valref(k) * 10^n. 

A = abs( [ pas-valref(1)*ones(pas);
           pas-valref(2)*ones(pas);
           pas-valref(3)*ones(pas);
           pas-valref(4)*ones(pas);
           pas-valref(5)*ones(pas);
           pas-valref(6)*ones(pas);
 ]);

[delta, I] = min(A);
pas = valref(I(1)) * fact(I(2));

vmin = floor(xmin/pas) * pas;
vmax = ceil(xmax/pas) * pas;
nb1 = round((vmax-vmin)/pas);
i = I(1);

// substeps
Asub = [ 1,2,5,10;  // values for steps multiple of 1
         1,2,4,10;  // values for steps multiple of 2
         1,3,6,12;  // values for steps multiple of 3
         1,1,5,10;  // values for steps multiple of 5
         1,3,6,12;  // values for steps multiple of 6
         1,2,5,10 ] // values for steps multiple of 10

[v, k] = min(abs(Asub(i,:) - ns2));
nb2 = Asub(i,k);

endfunction
