//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function M = CL_rot_angles2matrix(axes,angles)
// Rotation angles to transformation matrix
//
// Calling Sequence
// M = CL_rot_angles2matrix(axes,angles)
//
// Description
// <itemizedlist><listitem>
// Computes the transformation matrix <emphasis role="bold">M</emphasis> that results 
// from the combination of 1 to 3 elementary rotations, each elementary rotation 
// being described by an axis number (1=x-axis, 2=y-axis, 3=z-axis) and an angle. 
// </listitem>
// <listitem>
// <emphasis role="bold">Important notes:</emphasis> 
// <para> M is the <emphasis role="bold">frame transformation matrix</emphasis> and its 
// definition is consistent with conventions on reference frame transformations.  </para>
// <para> - Let u be the coordinates of some vector in some reference frame. The coordinates of
// the rotated vector are given by: </para>
// <para> R(u) = M' * u (M transposed) </para>
// <para> - Conversely, let F1 be some reference frame, F2 be the reference frame obtained 
// after applying the successive rotations to F1, u1 the coordinates relative to F1 
// of some vector, then the coordinates relative to F2 of the same vector are 
// given by:  </para>
// <para> u2 = M * u1 </para>
// </listitem>
// </itemizedlist>
//
// Parameters
// axes : Axis numbers: 1=x-axis, 2=y-axis or 3=z-axis (1xP or Px1 with P=1,2 or 3)
// angles : Rotation angles around respective axes [rad] (PxN) 
// M : Transformation matrix (3x3xN).
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_rot_quat2matrix
// CL_rot_matrix2quat
//
// Examples
// // 30deg rotation around X: 
// M = CL_rot_angles2matrix([1],[%pi/3])
//
// // 30deg rotation around X followed by 90deg around Y: 
// M = CL_rot_angles2matrix([1,2],[%pi/3;%pi/2])
//
// // Meaning of M:   
// M = CL_rot_angles2matrix(3, 10*%pi/180)
// u = [1;0;0]  // coordinates of vector u in initial frame
// v = M*u      // coordinates of vector u in final frame
// urot = M'*u  // image of rotation of u in initial frame
//

// Declarations:


// Code:

// validity checking
Nrot = length(axes);
N = size(angles,2); 

// same number of axes as angles
if (size(angles,1) <> Nrot | Nrot < 1 | Nrot > 3)
  CL__error("Invalid size for axes or angles"); 
end

// same number of axes as angles
I = find(axes <> 1 & axes <> 2 & axes <> 3); // check integer number as well
if (~isempty(I))
  CL__error("Invalid axis number");
end

select Nrot

  case 1
    select axes(1)
    // attention: la fonction Sscilab 'hypermat' accepte les elements colonne
    //           par colonne, M11;M21;M31;M12;M22...
    case 1
      M = hypermat([3 3 N] , [ones(angles); zeros(angles); zeros(angles); zeros(angles); ...
          cos(angles); -sin(angles); zeros(angles); sin(angles); cos(angles)]);
    case 2
      M = hypermat([3 3 N] , [cos(angles); zeros(angles); sin(angles); zeros(angles); ...
          ones(angles); zeros(angles); -sin(angles); zeros(angles); cos(angles)]);
    else // 3
      M = hypermat([3 3 N] , [cos(angles); -sin(angles); zeros(angles); sin(angles); ...
          cos(angles); zeros(angles); zeros(angles); zeros(angles); ones(angles)]);
    end

  case 2
    i = axes(1);
    j = axes(2);
    M = CL_rot_angles2matrix(j,angles(2,:))*CL_rot_angles2matrix(i,angles(1,:));

  else // 3
    i = axes(1);
    j = axes(2);
    k = axes(3);
    M = CL_rot_angles2matrix(k,angles(3,:))*CL_rot_angles2matrix(j,angles(2,:)) ...
        * CL_rot_angles2matrix(i,angles(1,:));

end


if (N == 1) 
   M = M(:,:,1); 
end

endfunction
