//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [mean_cir_t2, osc_cir_t2] = CL_ex_eckHech(t1, mean_cir_t1, t2,er,mu,zonals )
// Eckstein Hechler orbit propagation analytical model
//
// Calling Sequence
// [mean_cir_t2,osc_cir_t2] = CL_ex_eckHech(t1,mean_cir_t1,t2 [,er,mu,zonals])
//
// Description
// <itemizedlist><listitem>
// The function computes the mean orbital elements <emphasis role="bold">mean_cir_t2</emphasis> and the osculating orbital elements <emphasis role="bold">osc_cir_t2</emphasis>
// at time <emphasis role="bold">t2</emphasis> given the mean orbital elements <emphasis role="bold">mean_cir_t1</emphasis> at time <emphasis role="bold">t1</emphasis>.
// <para>Zonals coefficients up to J6 are taken into account. </para>
// </listitem>
// <listitem>
// <para> The function works in the following cases: </para>
// <para> - 1 initial time and 1 set of initial orbital elements, and N final times. </para>
// <para> - N (or 1) initial times and N sets of initial orbital elements, and N (or 1) final times. </para>
// </listitem>
// <listitem> The orbital elements are the following:  
// <para><inlinemediaobject><imageobject><imagedata fileref="cir_par.gif"/></imageobject></inlinemediaobject></para>
// </listitem>
// <listitem>Warnings :
// <para> - This function does not work for inclinations close to the critical inclinations (63.43494882 deg and 116.5650512 deg)</para>
// <para> - This function nominally works for eccentricities smaller than 5.e-3</para>
// <para> - This function works but with a lesser accuracy for eccentricities between 5.e-3 and 0.1</para>
// <para> - This function does not work for eccentricities greater than 0.1</para></listitem>
// </itemizedlist>
//
// Parameters
// t1: Initial time [days] (1xN or 1x1)
// mean_cir_t1: Circular adapted mean orbital elements at time t1 [sma;ex;ey;inc;raan;pom+anm] (6xN or 6x1). 
// t2: Final time [days] (1xN or 1x1). 
// er: (optional) Equatorial radius [m] (default is %CL_eqRad)
// mu: (optional) Gravitational constant [m^3/s^2] (default value is %CL_mu)
// zonals: (optional) Vector of zonal coefficients J1 to Jn (troncated to J6) to be used (default is %CL_j1jn(1:6)) (Nz x 1)
// mean_cir_t2: Circular adapted mean orbital elements at t2 [sma;ex;ey;inc;raan;pom+anm] (6xN)
// osc_cir_t2: Circular adapted osculating orbital elements at t2 [sma;ex;ey;inc;raan;pom+anm] (6xN)
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// 1 CNES - MSLIB FORTRAN 90, Volume E (me_eck_hech)
//
// See also
// CL_ex_lyddane
// CL_ex_meanEckHech
//
// Examples
// // Example 1 : 1 orbit propagated to several times:
// t1 = 12584
// mean_cir_t1 = [42166712;2.9e-3;1.2e-3;CL_deg2rad([97.2;125;5])]
// t2 = 12587:0.2:12588;
// [mean_cir_t2,osc_cir_t2] = CL_ex_eckHech(t1,mean_cir_t1,t2)
//
// // Example 2 : several orbits propagated to one final time:
// t1 = 12584
// mean_cir_t1 = [42166712;2.9e-3;1.2e-3;CL_deg2rad([97.2;125;5])]
// mean_cir_t1 = [mean_cir_t1,mean_cir_t1*0.95]
// t2 = 12587;
// [mean_cir_t2,osc_cir_t2] = CL_ex_eckHech(t1,mean_cir_t1,t2)


//************************************************************************
//
// But:  Modele analytique d'extrapolation d'orbite de ECKSTEIN-HECHLER.
// ===
//
// Note d'utilisation:  L'excentricite  doit etre superieure >= 0 et < 0.1.
// ==================   Le domaine d'utilisation du modele est .. [ 0E+0 , 5E-3 [.
//                      Le domaine de precision degradee est .... [ 5E-3 , 1E-1 [.
//                      Le domaine d'erreur est ................. [ 1E-1 , +Oo  [.
//                      L'inclinaison  doit etre > 0. et < pi et non proches des inclinaisons critiques
//                      ( pm_i_critique_non_retro, pm_i_critique_retro ) a pm_eps_i_critique pres.
//
// Historique:
// ==========
//   + Version 2.0 (SP 341 ed01 rev00): creation a partir de la routine MEEHJ6 de la MSLIB f77
//   + Version 3.1 (DE globale 439 ed01 rev00) : ajout des champs %biblio et %message pour le code retour
//   + Version 4.1 (DE globale 482 ed01 rev00): Corrections qualite
//   + Version 4.1 (DE globale 482 ed01 rev00): Corrections qualite
//   + Version 6.0 (DE globale 618 ed01 rev00): ajout de tests sur mu et les coefficients CN0
//************************************************************************

//initialisations

// Declarations:
if(~exists('%CL_criticalInc1')) then global %CL_criticalInc1; end;
if(~exists('%CL_criticalInc2')) then global %CL_criticalInc2; end;
if(~exists('%CL_eqRad')) then global %CL_eqRad; end;
if(~exists('%CL_j1jn')) then global %CL_j1jn; end;
if(~exists('%CL_mu')) then global %CL_mu; end;

// Code:

lhs = argn(1);

compute_osc_cir_t2 = %f;
if (lhs == 2); compute_osc_cir_t2 = %t; end

if (lhs > 2) 
  CL__error('too many outputs');
end

if ~exists('mu','local') then mu=%CL_mu; end
if ~exists('er','local') then er=%CL_eqRad; end
if ~exists('zonals','local')
   zonals = %CL_j1jn(1:6); 
else
   zonals_tmp = zonals(:); // column vector 
   zonals = zeros(6,1); 
   nz = min(6, length(zonals_tmp)); 
   zonals(1:nz) = zonals_tmp(1:nz);
end

// initialisation de la valeur epsilon * 100
eps100 = 100 * %eps
eccel = 0.05

// controle des donnees
// demi grand axe
if or( mean_cir_t1(1,:) <= 0 )
  CL__error("Error : semi major axis < or = 0");
end
// eccentricite
ecc = sqrt(mean_cir_t1(2,:).^2+(mean_cir_t1(3,:).^2));
if or( ecc > 0.1 )
  CL__error("Error : eccentricity higher than 0.1");
end
//if or( ecc > 5.e-3)
//  warning("Warning : lesser precision for eccentricities between 5e-3 and 0.1");
//end

// inclinaison
if or( mean_cir_t1(4,:) < 0 )
  CL__error("negative inclination");
end
if or( mean_cir_t1(4,:) > %pi )
  CL__error("inclination > pi");
end

// inclinaison proche de l'inclinaison critique.
eps_i_critique = 4e-5; // obtenu experimentalement : au dela ca plante
if (abs(mean_cir_t1(4,:) - %CL_criticalInc1) < eps_i_critique | ...
     abs(mean_cir_t1(4,:) - %CL_criticalInc2) < eps_i_critique) then
   CL__error("inclination close (less than "+string(eps_i_critique)+" rad) to the critical value");
end

// Empty final times
if (t2 == []) 
  mean_cir_t2 = [];
  osc_cir_t2 = [];
  return; // *** EXIT ***
end


cn0 = -zonals;

// parametres d'orbite adaptes circulaires
// recuperation valeurs parametres meanens en entree
// ................................................
a0   = mean_cir_t1(1,:);
i0   = mean_cir_t1(4,:);
om0  = mean_cir_t1(5,:);
ex0  = mean_cir_t1(2,:);                         // e cos(w)
ey0  = mean_cir_t1(3,:);                         // e sin(w)
l0   = mean_cir_t1(6,:);                         // w + M

// calculs preliminaires
// .....................
// calcul de: 
// gi = cn0(i) * ( %CL_eqRad / a0 )**i
// bi = sin(i0)^i

q  = er ./ a0;

c  = cos(i0);
b1 = sin(i0);
b2 = b1 .^2;
b4 = b1 .^4; 
b6 = b1 .^6; 

g2 = cn0(2) * q^2;
g3 = cn0(3) * q^3;
g4 = cn0(4) * q^4;
g5 = cn0(5) * q^5;
g6 = cn0(6) * q^6;

xnot  = sqrt(mu./(a0.^3)) .* (t2 - t1) * 86400.;  

// demi grand axe
am = a0;

// vecteur excentricite
delta_pom  = -0.75 .* g2 .* ( 4 - 5 * b2 );
delta_pomp = 7.5 * g4 .* ( 1 - (31/8) * b2 + (49/16) * b4 ) - ...
             13.125 * g6 .* (1 - 8 * b2 + (129/8) * b4 - (297/32) * b6);
x          = ( delta_pom + delta_pomp ) .* xnot; 
cx         = cos(x); 
sx         = sin(x); 
q          = (3/32) ./ delta_pom;
eps1       = q .* g4 .* b2 .* ( 30 - 35 * b2 ) - 175 * q .* g6 .* b2 .* ...
             ( 1 - 3 * b2 + (33/16) .* b4 );
q          = (3/8) * b1 ./ delta_pom;
eps2       = q .* g3 .* ( 4 - 5 * b2 ) - q .* g5 .* ( 10 - 35 * b2 + 26.25 * b4 );
exm        = ex0 .* cx - ( 1 - eps1 ) .* ey0 .* sx + eps2 .* sx; 
eym        = ( 1 + eps1 ) .* ex0 .* sx + ( ey0 - eps2 ) .* cx + eps2; 

// inclinaison
im = i0;

// ascension droite du noeud
q   = 1.5 * g2 - 2.25 * (g2.^2) .* ( 2.5 - (19/6) * b2 ) + ...
      (15/16) * g4 .* ( 7 * b2 - 4 ) + ...
      (105/32) * g6 .* ( 2 - 9 * b2 + (33/4) * b4 );
omm = om0 + q .* c .* xnot; 

// argument de latitude
delta_l = 1 - 1.5 * g2 .* ( 3 - 4 * b2 );
q   = delta_l + 2.25 * (g2.^2) .* ( 9 - (263/12) * b2 + (341/24) * b4 ) + ...
      (15/16) * g4 .* ( 8 - 31 * b2 + 24.5 * b4 ) + ...
      (105/32) * g6 .* ( -(10/3) + 25 * b2 - 48.75 * b4 + 27.5 .* b6 );
lm  = l0 + q .* xnot; 


// termes periodiques
// ..................

// calcul des cli = cos(i*lm)  sli = sin(i*lm)

cl = cos([lm; 2*lm; 3*lm; 4*lm; 5*lm; 6*lm]); 
sl = sin([lm; 2*lm; 3*lm; 4*lm; 5*lm]); 


// autres quantites
qq = -1.5 * g2 ./ delta_l;
qh =  (3/8) * ( eym - eps2 ) ./ delta_pom; 
ql =  (3/8) * exm ./ b1 ./ delta_pom; 

// sortie des elements meanens a la date t2
// ....................
mean_cir_t2 = [ am .* ones(t2); exm; eym; im .* ones(t2); omm; lm];


// MODELE POUR LES OSCULATEURS
// ===========================
if compute_osc_cir_t2

  // termes periodiques sur a
  // ............
  // contribution de j2
  f       = ( 2 - 3.5 * b2 ) .* exm .* cl(1,:) + ( 2 - 2.5 * b2 ) .* eym .* sl(1,:) + ...
            b2 .* cl(2,:) + 3.5 * b2 .* ( exm .* cl(3,:) + eym .* sl(3,:) );
  delta_a = qq .* f;

  // contribution j2 carre
  q       = 0.75 * (g2.^2) .* b2;
  f       = 7 * ( 2 - 3 * b2 ) .* cl(2,:) + b2 .* cl(4,:);
  delta_a = delta_a + q .* f;

  // contribution de j3
  q       = -0.75 * g3 .* b1;
  f       = ( 4 - 5 * b2 ) .* sl(1,:) + (5/3) * b2 .* sl(3,:);
  delta_a = delta_a + q .* f;

  // contribution de j4
  q       = 0.25 * g4 .* b2;
  f       = ( 15 - 17.5 * b2 ) .* cl(2,:) + 4.375 * b2 .* cl(4,:);
  delta_a = delta_a + q .* f;

  // contribution de j5
  q       = 3.75 * g5 .* b1;
  f       = ( 2.625 * b4 - 3.5 * b2 + 1 ) .* sl(1,:) + ...
            (7/6) * b2 .* ( 1 - 1.125 * b2 ) .* sl(3,:) + (21/ 80) * b4 .* sl(5,:);
  delta_a = delta_a + q .* f;

  // contribution de j6
  q       = (105/16) * g6 .* b2;
  f       = ( 3 * b2 - 1 - (33/16) * b4 ) .* cl(2,:) + ...
            0.75 * ( 1.1 * b4 - b2 ) .* cl(4,:) - (11/80) * b4 .* cl(6,:);
  delta_a = delta_a + q .* f;

  // termes periodiques sur ex
  // .............
  // contribution de j2
  f        = ( 1 - 1.25 * b2 ) .* cl(1,:) + 0.5 * ( 3 - 5 * b2 ) .* exm .* cl(2,:) + ...
             ( 2 - 1.5 * b2 ) .* eym .* sl(2,:) + (7/12) * b2 .* cl(3,:) + ...
             (17/8) * b2 .* ( exm .* cl(4,:) + eym .* sl(4,:) );
  delta_ex = qq .* f;

  // termes periodiques sur ey
  // .............
  // contribution j2
  f        = ( 1 - 1.75 * b2 ) .* sl(1,:) + ( 1 - 3 * b2 ) .* exm .* sl(2,:) + ...
             ( 2 * b2 - 1.5 ) .* eym .* cl(2,:) + (7/12) * b2 .* sl(3,:) + ...
             (17/ 8) * b2 .* ( exm .* sl(4,:) - eym .* cl(4,:) );
  delta_ey = qq .* f;

  // termes periodiques sur om
  // .............
  // contribution de j2
  q         = -qq .* c;
  f         = 3.5 * exm .* sl(1,:) - 2.5 * eym .* cl(1,:) - 0.5 * sl(2,:) + ...
              (7/6) * ( eym .* cl(3,:) - exm .* sl(3,:) );
  delta_gom = q .* f;

  // contribution de j3
  f         = g3 .* c .* ( 4 - 15 * b2 );
  delta_gom = delta_gom + ql .* f;

  // contribution de j5
  f         = 2.5 * g5 .* c .* ( 4 - 42 * b2 + 52.5 * b4 );
  delta_gom = delta_gom - ql .* f;

  // termes periodiques sur xi
  // .............
  // contribution de j2
  q       = 0.5 * qq .* b1 .* c;
  f       = eym .* sl(1,:) - exm .* cl(1,:) + cl(2,:) + ..
            (7/3) * ( exm .* cl(3,:) + eym .* sl(3,:) );
  delta_i = q .* f;

  // contribution de j3
  f       = g3 .* c .* ( 4 - 5 * b2 );
  delta_i = delta_i - qh .* f;

  // contribution de j5
  f       = 2.5 * g5 .* c .* ( 4 - 14 * b2 + 10.5 * b4 );
  delta_i = delta_i + qh .* f;

  // termes periodiques sur xl
  // .............
  // contribution de j2
  f        = ( 7 - (77/8) * b2 ) .* exm .* sl(1,:) + ...
             ( (55/8) * b2 - 7.5 ) .* eym .* cl(1,:) + ...
             ( 1.25 * b2 - 0.5 ) .* sl(2,:) + ...
             ( (77/24) * b2 - (7/6) ) .* ( exm .* sl(3,:) - eym .* cl(3,:) );
  ddelta_l = qq .* f;

  // contribution de j3
  f        = g3 .* ( 53 * b2 - 4 - 57.5 * b4 );
  ddelta_l = ddelta_l + ql .* f;

  // contribution de j5
  f        = 2.5 * g5 .* ( 4 - 96 * b2 + 269.5 * b4 - 183.75 * b6 );
  ddelta_l = ddelta_l + ql .* f;

  // parametres osculateurs
  // ...........
  ex = exm + delta_ex;
  ey = eym + delta_ey;
  om = omm + delta_gom;
  xi = im + delta_i;
  xl = lm + ddelta_l;

  // sortie optionnelle des parametres osculateurs a la date t2
  // .............................
  osc_cir_t2 = [ am .* ( 1  + delta_a ); ex; ey; xi; om; xl];
  osc_cir_t2(5:6,:) = CL_rMod(osc_cir_t2(5:6,:), mean_cir_t2(5:6,:)-%pi, mean_cir_t2(5:6,:)+%pi); 
  
end


endfunction
