//  Copyright (c) CNES  2011
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [jacob] = CL__fr_jacobian(mat, omega)
// Jacobian of a frame (position/velocity) transformation
//
// Calling Sequence
// jacob = CL__fr_jacobian(mat, omega)
//
// Description
// <itemizedlist><listitem>
// <p>Computes the jacobian of the transformation (pos1,vel1) to (pos2,vel2): </p> 
// <p>pos2 = mat * pos1 </p> 
// <p>vel2 = mat * (vel1 - omega ^ pos1) </p>
// <p>jacob = d(pv2)/d(pv1), more precisely jacob(i,j,:) = d(pv2(i,:)/d(pv1(j,:))</p>
// <p></p></listitem>
// </itemizedlist>
//
// Parameters
// mat: Transformation matrix from frame1 to frame2 (pos2 = mat * pos1) (3x3xN)
// omega: Angular velocity vector of frame2 wrt frame1 with coordinates in frame1 (3xN)
// jacob: Jacobian of the transformation: (pos1,vel1) to (pos2,vel2) (6x6xN)
//
// Authors
// CNES - DCT/SB
//

// Declarations:

// Code:

// W: 3x3xn matrix such that: W * X = omega ^ X 
// pos2 = mat * pos1 
// vel2 = mat * (vel1 - W * pos1) 
// => jacob = [  mat,     0; 
//              -mat*W,  mat  ] 

n = size(omega,2); 

W = zeros(3,3,n);
W(1,2,1:n) = -omega(3,:);
W(1,3,1:n) =  omega(2,:);
W(2,1,1:n) =  omega(3,:);
W(2,3,1:n) = -omega(1,:);
W(3,1,1:n) = -omega(2,:);
W(3,2,1:n) =  omega(1,:);

jacob = zeros(6,6,n);
jacob(1:3,1:3,1:n) = mat;
jacob(4:6,4:6,1:n) = mat;
jacob(4:6,1:3,1:n) = -mat*W;

endfunction
