//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [pos,vel,jacob] = CL__oe_kep2carHyperb(kep,mu)


// Declarations:

// Code:
[lhs,rhs]=argn();

// Check number of input arguments. ALL obligatory (including mu)
if (rhs <> 2)
  CL__error("Invalid number of input arguments"); 
end

select lhs
  case 3
    compute_jacob=%t
  case 2
    compute_jacob=%f
  case 1
    compute_jacob=%f
  else
    CL__error('Invalid number of output arguments')
end

N = size(kep,2)

a    = kep(1,:)
ecc  = kep(2,:)
ai   = kep(3,:)
apom = kep(4,:)
agom = kep(5,:)
am   = kep(6,:)

ah = CL_kp_M2E(ecc,am)// calcul anomalie eccentrique H par equation de kepler   e * sinh(H) - H = M

// 1- calcul vecteur position r = a (ld0*p + mu0*q)
//        et vecteur vitesse rv = sqrt(mu/a) (nu0*p + ki0*q)

// calcul de variables intermediaires
cai = cos(ai)       //
sai = sin(ai)       //
cgo = cos(agom)     //
sgo = sin(agom)     //    cosinus et sinus des angles i,gomega,pomega
cpo = cos(apom)     //
spo = sin(apom)     //
cgocpo = cgo .* cpo
sgocpo = sgo .* cpo
sgospo = sgo .* spo
cgospo = cgo .* spo

chah = cosh(ah) //cosinus et
shah = sinh(ah) //sinus hyperboliques de H

v1 = sqrt( (ecc.^2)-1 ) //calcul des variables sqrt(e**2-1) et
v2 = (ecc.*chah) - 1    //e*cosh(H)-1

p = zeros(3,N)
p(1,:) = cgocpo - (cai .* sgospo)   //
p(2,:) = sgocpo + (cai .* cgospo)   //
p(3,:) = sai .* spo                 // calcul des composantes des vecteurs p et q en fonction de
q = zeros(3,N)
q(1,:) = - cgospo - (cai .* sgocpo) // angles i,pomega,gomega
q(2,:) = - sgospo + (cai .* cgocpo) //
q(3,:) = sai .* cpo                 //

rld0 = ecc - chah   // calcul de ld0 =e-coshH et de
rmu0 = shah .* v1    // mu0 = sinhH * sqrt(e**2-1)

sqmusa = sqrt(mu./a)      // calcul de sqmusa = sqrt(mu/a),
rnu0 = -shah ./ v2        //  de nu0 = -sinhH/(e*coshH-1)
rki0 =  (chah .* v1)./v2  // et de ki0 = coshH*sqrt(e**2-1) / (e*coshH-1)

pos = a.*.ones(3,1) .* ((rld0.*.ones(3,1) .* p) + (rmu0.*.ones(3,1) .* q))     // calcul des composantes des vecteurs position
vel = sqmusa.*.ones(3,1) .* ((rnu0.*.ones(3,1) .* p) + (rki0.*.ones(3,1) .* q))//et vitesse

// 2- calcul du jacobien de la transformation
if compute_jacob

   // calcul de variables intermediaires
   xdpai = zeros(3,N)
   xdpai(1,:) = sai  .* sgospo
   xdpai(2,:) = -sai .* cgospo
   xdpai(3,:) = cai  .* spo       // derivees des composantes de p et q par rapport a i
   xdqai = zeros(3,N)
   xdqai(1,:) = sai  .* sgocpo
   xdqai(2,:) = -sai .* cgocpo
   xdqai(3,:) = cai  .* cpo
   xdpgo = zeros(3,N)
   xdpgo(1,:) = - sgocpo - (cai .* cgospo)
   xdpgo(2,:) = cgocpo - (cai .* sgospo)
   xdpgo(3,:) = 0            // derivees des composantes de p et q par rapport a gomega
   xdqgo = zeros(3,N)
   xdqgo(1,:) = sgospo - (cai .* cgocpo)
   xdqgo(2,:) = - cgospo - (cai .* sgocpo)
   xdqgo(3,:) = 0

   xdldex =  ones(1,N)
   xdmuex = shah .* (ecc./v1)    // calcul des derivees de ld0 et mu0 par rapport a e et H
   xdldah = -shah
   xdmuah = chah .* v1

   // calcul derivees de H par rapport a M et e
   xdaham = 1.0 ./ v2          // dH/dam = 1 / (ecc*coshH-1) et
   xdahex = - shah .* xdaham          // dH/dex =-sinH / (-1+ecc*coshH)

   // calcul des derivees de nu0 et ki0 par rapport a e et H
   // avec la variable : xdaham = 1. / v2
   xdnuex = chah .* shah .* xdaham .* xdaham                  // calcul de: xdnuex = (chah*shah) / (v2**2)
   xdkiex = xdaham .* chah .* ((ecc./v1) - (chah.*v1.*xdaham))  // calcul de: xdkiex=((ecc*chah)/(v1*v2)) - (((chah**2)*v1)/(v2**2))
   xdnuah = xdaham .* (-chah + (ecc.*shah.*shah.*xdaham))      // calcul de: xdnuah = (-chah/v2) + ((ecc*(shah**2)) / (v2**2))
   // calcul de: xdkiah = v1 * ((shah/v2) - ((ecc*chah*shah) / (v2**2)))
   xdkiah = v1 .* xdaham .* shah .* (1 - (ecc.*chah.*xdaham))

   // calcul des derivees partielles des vecteurs position et
   // vitesse par rapport a a,e,i,pomega,gomega,m
   jacob = zeros(6,6,N);
   //        ---par rapport a a
   jacob(1:3,1,1:N) = (rld0.*.ones(3,1) .* p) + (rmu0.*.ones(3,1) .* q)
   jacob(4:6,1,1:N) = (-0.5.*sqmusa./a).*.ones(3,1) .*((rnu0.*.ones(3,1) .* p) + (rki0.*.ones(3,1) .* q))

   //        ---par rapport a e ( = derivee/e + derivee/H * derivee H/e)
   jacob(1:3,2,1:N) = a.*.ones(3,1) .* (((xdldex.*.ones(3,1) .*p) + (xdmuex.*.ones(3,1) .*q)) + ...
                    ((xdldah.*.ones(3,1) .*p) + (xdmuah.*.ones(3,1) .*q)) .* (xdahex.*.ones(3,1)))
   jacob(4:6,2,1:N) = sqmusa.*.ones(3,1) .* (((xdnuex.*.ones(3,1) .*p) + (xdkiex.*.ones(3,1) .*q)) + ...
                    ((xdnuah.*.ones(3,1) .*p) + (xdkiah.*.ones(3,1) .*q)) .* (xdahex.*.ones(3,1)))

   //        ---par rapport a i
   jacob(1:3,3,1:N) = a.*.ones(3,1) .* ((rld0.*.ones(3,1) .* xdpai) + (rmu0.*.ones(3,1) .* xdqai))
   jacob(4:6,3,1:N) = sqmusa.*.ones(3,1) .* ((rnu0.*.ones(3,1) .* xdpai) + (rki0.*.ones(3,1) .* xdqai))

   //        ---par rapport a pomega (q = derivee de p /pomega)
   jacob(1:3,4,1:N) = a.*.ones(3,1) .* ((rld0.*.ones(3,1) .* q) - (rmu0.*.ones(3,1) .* p))
   jacob(4:6,4,1:N) = sqmusa.*.ones(3,1) .* ((rnu0.*.ones(3,1) .* q) - (rki0.*.ones(3,1) .* p))

   //        ---par rapport a gomega
   jacob(1:3,5,1:N) = a.*.ones(3,1) .* ((rld0.*.ones(3,1) .* xdpgo) + (rmu0.*.ones(3,1) .* xdqgo))
   jacob(4:6,5,1:N) = sqmusa.*.ones(3,1) .* ((rnu0.*.ones(3,1) .* xdpgo) + (rki0.*.ones(3,1) .* xdqgo))

   //        ---par rapport a M ( = derivee/H * derivee de H/M)
   jacob(1:3,6,1:N) = a.*.ones(3,1) .* ((xdldah.*.ones(3,1) .* p) + (xdmuah.*.ones(3,1) .* q)) .* (xdaham.*.ones(3,1))
   jacob(4:6,6,1:N) = sqmusa.*.ones(3,1) .* ((xdnuah.*.ones(3,1) .* p) + (xdkiah.*.ones(3,1) .* q)) .* (xdaham.*.ones(3,1))

end

endfunction
