//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [cir,jacob] = CL_oe_kep2cir(kep)
// Keplerian to circular adapted orbital elements
//
// Calling Sequence
// [cir,jacob] = CL_oe_kep2cir(kep)
//
// Description
// <itemizedlist><listitem>
// <p>Converts "classical" Keplerian orbital elements to orbital elements adapted to near-circular orbits.</p>
// <p>The transformation jacobian is optionally computed.</p>
// <p>See <link linkend="Orbital Elements">Orbital elements</link> for more details.</p>
// </listitem>
// </itemizedlist>
//
// Parameters
// kep: Keplerian orbital elements [sma;ecc;inc;pom;raan;anm] [m,rad] (6xN)
// cir: Orbital elements adapted to near-circular orbits [sma;ex;ey;inc;raan;pom+anm] [m,rad] (6xN)
// jacob: (optional) Transformation jacobian (See <link linkend="Orbital Elements">Orbital elements</link> for more details) (6x6xN)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_oe_cir2kep
// CL_oe_kep2car
//
// Examples
// // Example 1
// kep = [42166.e3; 1e-4; 1; %pi/4; 2; %pi/4];
// [cir,jacob1] = CL_oe_kep2cir(kep);
// [kep2,jacob2] = CL_oe_cir2kep(cir); // kep2: same as kep
// jacob2 * jacob1 // => identity


// Declarations:
global %CL__PRIV; 
if (~exists("%CL_epsilon")); %CL_epsilon = %CL__PRIV.epsilon; end

// Code:
// Handle [] cases
if (kep == [])
  cir = [];
  jacob = [];
  return;
end

// check args
if (find(kep(1,:) <= 0 | kep(2,:) < 0 | kep(2,:) > 1-%CL_epsilon.parab) <> [])
  CL__error("Invalid orbital elements"); 
end

[lhs,rhs] = argn();
if (lhs >= 2)
  compute_jacob = %t;
else
  compute_jacob = %f;
end

cir = zeros(kep);

cir(1,:) = kep(1,:);                      // sma = sma
cir(2,:) = kep(2,:) .* cos(kep(4,:));     // ex = e*cos(w)
cir(3,:) = kep(2,:) .* sin(kep(4,:));     // ex = e*sin(w)
cir(4,:) = kep(3,:);                      // inc = inc
cir(5,:) = kep(5,:);                      // raan = raan
cir(6,:) = kep(4,:) + kep(6,:);           // pso = w+M


// Jacobian computation
if (compute_jacob)
  // jacob = d(cir)/d(kep)
  // jacob(i,j) = d(cir_i)/d(kep_j)
  //
  // da/da = 1
  //
  // d(ex)/d(e) = cos(w)
  // d(ey)/d(e) = sin(w)
  // d(ex)/d(w) = -ey
  // d(ey)/d(w) = ex
  //
  // d(pso)/d(w) = 1
  // d(pso)/d(M) = 1


  N = size(kep,2);
  jacob = hypermat([6,6,N]);

  jacob(1,1,1:N) = 1; // da
  
  jacob(2,2,1:N) = cos(kep(4,:)); // dex/de
  jacob(2,4,1:N) = - cir(3,:); // dex/w
  
  jacob(3,2,1:N) = sin(kep(4,:)); // dey/de
  jacob(3,4,1:N) = cir(2,:); // dey/dw
  
  jacob(4,3,1:N) = 1 ; // di/di
  
  jacob(5,5,1:N) = 1; // dgom/dgom
  
  jacob(6,4,1:N) = 1; // dpso/dw
  jacob(6,6,1:N) = 1; // dpso/dM
  
  // conversion to matrix type
  if (N == 1); jacob = jacob(:,:,1); end
end

endfunction
