//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [y,ydot] = CL_integrate(x,ydotn, yini,meth,n)
// Evaluation of integral
//
// Calling Sequence
// [y,ydot] = CL_integrate(x,ydotn [,yini, meth, n])
//
// Description
// <itemizedlist><listitem>
// <p>Evaluates the 1st or 2nd integral of a function defined by a data series.</p>
// <p></p>
// <p>The nth integral is evaluated using derivatives (<b>ydotn</b>) and initial conditions (<b>yini</b>). </p>
// <p>- For the fist integral (<b>n</b>=1), ydotn is the first derivative of y at abscissae x, and yini gives the 
// value of y at first abscissa. The function returns the values of y at abscissae x. </p>
// <p>- For the 2nd integral (<b>n</b>=2), ydotn is the second derivative of y at abscissae x, and yini gives the 
// value of y and ydot at first abscissa. The function returns the values of y and its first derivative ydot 
// at abscissae x. </p>
// <p></p>
// <p>There are three methods to evaluate the integrals:</p>
// <p>- meth = "step_l" (l as lower bound): The result is the integral of a constant function by intervals. 
// The constant value between x(k) and x(k+1) is ydotn(k).</p>
// <inlinemediaobject><imageobject><imagedata fileref="integrate_step_l.gif"/></imageobject></inlinemediaobject>
// <p>- meth = "step_m" (m as middle): The result is the integral of a constant function by intervals. 
// The constant value between x(k) and x(k+1) is (ydotn(k)+ydotn(k+1)/2. </p>
// <inlinemediaobject><imageobject><imagedata fileref="integrate_step_m.gif"/></imageobject></inlinemediaobject>
// <p>- meth = "lin" (linear): The result is the integral of a linear function by intervals (polynomial (P) 
// of degree 1). P(xk)=ydotn(k) and P(xk+1)=ydotn(k+1). </p>
// <inlinemediaobject><imageobject><imagedata fileref="integrate_lin.gif"/></imageobject></inlinemediaobject>
// <p></p></listitem>
// </itemizedlist>
//
// Parameters
// x: Abscissae (in strictly increasing order) (1xN or PxN)
// ydotn: Values of nth derivative at abscissae x (1xN or PxN)
// yini: (optional) Initial conditions for y (y0 or [y0, ydot0]). Default is 0. (1xn or Pxn)
// meth: (string, optional) Integration method. Available methods are "step_l", "step_m" and "lin". Default is "lin". (1x1)
// n: (integer, optional) Integration order (= number of successive integrations): 1 or 2. Default is 1. (1x1)
// y: Integrated values (after n successive integrations) at abscissae x (PxN)
// ydot: First derivatives of y at abscissae x ([] if n == 1) (PxM)
//
// Authors
// CNES - DCT/SB
//
// Examples
// // Example 1: Simple integration (linear method)
// x = linspace(0,2*%pi,100);
// ydot = -sin(2*x);
// yini = 1/2;
// y = CL_integrate(x, ydot, yini);
// yref = cos(2*x)/2; 
// plot2d(x, y-yref)
//
// // Example 2: Double integration (step method)
// x = linspace(0,2*%pi,100);
// ydot2 = -sin(2*x);
// yini = [0 , 1/2];
// [y, ydot] = CL_integrate(x, ydot2, yini, meth="step_m", n=2);
// ydotref = cos(2*x)/2; 
// yref = sin(2*x)/4; 
// plot2d(x, y-yref)
// plot2d(x, ydot-ydotref)




// Integration method (example for order 2): 
// we have x_0, x_1, ... x_N : abscissae
// and:    D_1, D_2, D_N : constant values on each interval
// (<=> second derivative = constant)
//
// we compute : I(k-1,k) = integral(k-1, k), k = 1..N
// I(k-1,k) = D_k * (x_k - x_k-1) = ydot_k - ydot_k-1, k=1..N
// => ydot_1 = ydot_0 + I(0,1)
// => ydot_2 = ydot_1 + I(1,2) = ydot_0 + I(0,1) + I(1,2) 
// => ... 
//
// Second integral: 
// I2(k-1,k) = 1/2 * D_k * (x_k - x_k-1)^2 + 
//              ydot_k-1 * (x_k - x_k-1) 
//           = y_k - y_k-1, k=1..N
// => y_k = ... , k=1..N, knowing y_0


// Single integration of constant values on intervals
// x = abscissae (PxN)
// D = constant values of first derivative on each interval. (Px(N-1))
//            D1       D2       D3      DN-1
//        |--------|--------|--------|--------|
//       x1       x2       x3       x4        xN
// yini = Initial value of y at first abscissa (Px1)
function y = evalInt1(x,D,yini)
  dx = x(:,2:$) - x(:,1:$-1);  // dim N-1
  dy = D .* dx;                // dim N-1
  y = cumsum([yini, dy],"c");  // dim N
endfunction
  
// Double integration of constant values on intervals
// x = abscissae (PxN)
// D = constant values of second derivative on each interval (Px(N-1))
// yini = Initial value of y at first abscissa (Px1)
// ydotini = Initial value of first derivative of y at first abscissa (Px1)
function [y,ydot] = evalInt2(x,D,yini,ydotini)
  dx = x(:,2:$) - x(:,1:$-1);                 // dim N-1
  dydot = D .* dx;                            // dim N-1
  ydot = cumsum([ydotini, dydot],"c");        // dim N
  dy = 0.5 * D .* dx.^2 + ydot(:,1:$-1) .* dx;  // dim N-1
  y = cumsum([yini, dy],"c");                 // dim N
endfunction
  
// Triple integration of constant values on intervals
// x = abscissae (PxN)
// D = constant values of 3rd derivative on each interval (Px(N-1))
// yini = Initial value of y at first abscissa (Px1)
// ydotini = Initial value of first derivative of y at first abscissa (Px1)
// ydot2ini = Initial value of second derivative of y at first abscissa (Px1)
function [y,ydot,ydot2] = evalInt3(x,D,yini,ydotini,ydot2ini)
  dx = x(:,2:$) - x(:,1:$-1);                       // dim N-1
  dydot2 = D .* dx;                                 // dim N-1
  ydot2 = cumsum([ydot2ini, dydot2],"c");           // dim N
  dydot = 0.5 * D .* dx.^2 + ydot2(:,1:$-1) .* dx;    // dim N-1
  ydot = cumsum([ydotini, dydot],"c");              // dim N
  dy = (1/6) * D .* dx.^3 + 0.5 * ydot2(:,1:$-1) .* dx.^2 + ydot(:,1:$-1) .* dx;  // dim N-1 
  y = cumsum([yini, dy],"c");                       // dim N
endfunction
  
  
  
// ---------------------------
// MAIN 
// ---------------------------

if (~exists("n", "local")); n = 1; end
if (n <> 1 & n <> 2); CL__error("Wrong value for n"); end
if (~exists("yini", "local")); yini = zeros(1,n); end
if (~exists("meth", "local")); meth = "lin"; end
 

P = max(size(x,1), size(ydotn,1), size(yini,1));
 
// Check and resize arguments if necessary
[x, ydotn, N] = CL__checkInputs(x,[1,P], ydotn,[1,P]);
if (size(x,1) == 1); x = ones(P,1) * x; end
if (size(ydotn,1) == 1); ydotn = ones(P,1) * ydotn; end 
if (size(yini,1) == 1); yini = ones(P,1) * yini; end 

// Check for other possible errors
if (size(yini,1) <> P | size(yini,2) <> n); CL__error("Wrong size for yini"); end
if (typeof(meth) <> "string"); CL__error("Wrong type for meth. String expected"); end;
if (size(meth,"*") <> 1); CL__error("Wrong type for meth. String expected"); end;

// Check that abscissae are in strictly increasing order 
if (find(x(:,2:$) - x(:,1:$-1) <= 0))
  error("Abscissae not in strictly increasing order");
end

ydot = [];

// Special cases (N=0 or N=1)
if (N == 0)
  y = [];
  ydot = [];
  return;  // <-- RETURN !
end

if (N == 1)
  y = yini(:,1);
  if (n==2)
    ydot = yini(:,2);
  end
  return;  // <-- RETURN !
end


if (meth == "step_m" | meth == "step_l")
  if (meth == "step_l")
    // The derivative is constant in each interval.
    // The constant is the beginning values.
    D = ydotn(:,1:$-1);
  elseif (meth == "step_m")
    // The derivative is constant in each interval.
    // The constant is the mean of beginning/end values.
    D = (ydotn(:,2:$) + ydotn(:,1:$-1))/2;
  end
    
  if (n == 1)
    y = evalInt1(x, D, yini);
  elseif (n == 2)
    [y, ydot] = evalInt2(x, D, yini(:,1), yini(:,2));
  end
    
elseif (meth == "lin")
  
  // The integrated function is linear (a*x+b) in each interval.
  // And we integrate n+1 times.
  D = (ydotn(:,2:$) - ydotn(:,1:$-1)) ./ (x(:,2:$) - x(:,1:$-1));
    
  if (n == 1)
    y = evalInt2(x, D, yini(:,1), ydotn(:,1));
  elseif (n == 2)
    [y, ydot] = evalInt3(x, D, yini(:,1), yini(:,2), ydotn(:,1));
  end
  
else
  CL__error("Unkown method of integration");
end


endfunction
