//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.
 
function [psoE] = CL_kp_M2Ecir(ex,ey,psoM)
// Mean arg. of latitude to eccentric arg. of latitude (Kepler's equation)
//
// Calling Sequence
// psoE = CL_kp_M2Ecir(ex,ey,psoM)
//
// Description
// <itemizedlist><listitem>
// <p>Solves the 'generalized' Kepler's equation for elliptical orbits. </p>
// <p>It computes the eccentric argument of latitude <b>psoE</b> from the mean 
// argument of latitude <b>psoM</b> and the eccentricity vector components 
// <b>ex</b> and <b>ey</b>. </p>
// <p><inlinemediaobject><imageobject><imagedata fileref="kep_gen.gif"/></imageobject></inlinemediaobject> </p>
// </listitem>
// </itemizedlist>
//
// Parameters
// ex: x component of eccentricity vector (PxN or Px1)
// ey: y component of eccentricity vector (PxN or Px1)
// psoM: Mean argument of latitude (=w+M) [rad] (PxN or Px1)
// psoE: Eccentric argument of latitude (=w+E) [rad] (PxN)
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// 1) CNES - MSLIB FORTRAN 90, Volume V (mv_kepler_gene)
//
// See also
// CL_kp_M2E
//
// Examples
// pom = 2*%pi/3;
// ecc = 0.001;
// ex = ecc * cos(pom);
// ey = ecc * sin(pom);
// M = (0:4)*%pi/4;
// E = CL_kp_M2Ecir(ex,ey,pom+M) - pom; 
//
// // Now using CL_kp_M2E :
// E = CL_kp_M2E(ecc,M) 

// Declarations:
global %CL__PRIV; 
if (~exists("%CL_epsilon")); %CL_epsilon = %CL__PRIV.epsilon; end

// Code:
epsilon = %CL_epsilon.parab

// check / resize
nr = size(ex,1); 
[ex,ey,psoM] = CL__checkInputs(ex,nr,ey,nr,psoM,nr); 

// check eccentricity
ecc2 = (ex.^2) + (ey.^2); 

if (find(ecc2 >= (1-epsilon)^2) <> [])
   CL__error("Invalid eccentricity vector"); 
end



// Encke initialization
// --------------------
x = (ex .* cos(psoM)) + (ey .* sin(psoM));
y = (ex .* sin(psoM)) - (ey .* cos(psoM));

// initially:  
// r = (ecc2-x)/y;  
// az = ecc * sign(y) / sqrt(1+r^2)
// f0 = psoM + az - (az.^4 .*x .* (1-ecc2)) / (6 .* y); 
// => improved to avoid division by 0

az = 1 ./ sqrt(1 + ecc2 - 2 * x); 
psoE = psoM + az.*y - (az.^4 .*x .* (1-ecc2)) .* (y.^3) / 6; 

// -------------------------------------------
// iterative methode (Newton)
// f(x) = 0
// xi+1 = xi - (f(xi) / f'(xi))
// -------------------------------------------

nb_max_iter = 20;          // max number of iterations
errmax = 100 * %eps;       // accuracy 

err1 = ones(psoE) * %inf;  // initial value for accuracy variables
err2 = ones(psoE) * %inf; 

K = 1:length(psoE);        // indices to be dealt with
iter = 0;                  // iter = current number of iterations

while (K <> [] & iter <= nb_max_iter) 

  psoE1 = psoE; // previous value

  psoE(K) = psoE(K) - (psoE(K) - ex(K).*sin(psoE(K)) + ey(K).*cos(psoE(K)) - psoM(K)) ./ ...
            (1 - ex(K).*cos(psoE(K)) - ey(K).*sin(psoE(K)));

  err1(K) = abs( psoE1(K)-psoE(K) ) ./ max(abs(psoE(K)),1);
  err2(K) = abs( psoM(K) - psoE1(K) + ex(K).*sin(psoE1(K)) - ey(K).*cos(psoE1(K)) ) ./ ...
            max(abs(psoM(K)),1);

  K = find( err1 > errmax | err2 > errmax )
  iter = iter + 1; 

end

if (K <> []) 
   CL__error('Maximum number of iterations reached'); 
end


endfunction

