//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [vel1,vel2] = CL_man_lambert(pos1,pos2,delta_t,direction,mu)
// Lambert's problem
//
// Calling Sequence
// [vel1,vel2] = CL_man_lambert(pos1,pos2,delta_t [,direction] [,mu])
//
// Description
// <itemizedlist><listitem>
// <p>Computes the velocity vectors <b>vel1</b> and <b>vel2</b>, 
// given the position vectors <b>pos1</b> and <b>pos2</b>, 
// the time of flight <b>delta_t</b>, and the direction of motion <b>direction</b>.</p>
// </listitem>
// </itemizedlist>
//
// Parameters
// pos1 : Initial position vector [m] (3xN)
// pos2 : Final position vector [m] (3xN)
// delta_t : Time of flight from r1 to r2 [s] (1xN or 1x1)
// direction : (string, optional) 'pro' if the transfer orbit is prograde, 'retro' if the transfer orbit is retrograde (default is 'pro')
// mu : (optional) Gravitational constant (default is %CL_mu) [m^3/s^2]
// vel1 : Initial velocity vector [m/s] (3xN)
// vel2 : Final velocity vector [m/s] (3xN)
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// 1) Orbital Mechanics for Engineering Students, H D Curtis, Section 5.3 and Appendix D.11 (algorithm 5.2)
// 2) Modern astrodynamics Fundamentals and perturbation methods, V Bond and M Allman, Chapter 6.2
//
// See also
// CL_stumpS
// CL_stumpC
//
// Examples
// dt = 1000; // seconds
// kep1 = [7000.e3; 0.1; 1; 0; 0; 0]; 
// kep2 = CL_ex_kepler(0, kep1, dt/86400); 
// [pos1, vel1] = CL_oe_kep2car(kep1); 
// [pos2, vel2] = CL_oe_kep2car(kep2);
// [vel1b, vel2b] = CL_man_lambert(pos1, pos2, dt); 
// CL_norm(vel1-vel1b) + CL_norm(vel2-vel2b)


// Declarations:
global %CL__PRIV; 
if (~exists("%CL_mu")); %CL_mu = %CL__PRIV.mu; end

// Code:
// Auxiliary functions : 

  function dum = yy(z,r1,r2,A)
    dum = real(r1 + r2 + A .* (z.*CL_stumpS(z)-1) ./ sqrt(CL_stumpC(z)));
  endfunction

  function dum = FF(z,t,r1,r2,A,mu)
    dum = real(( yy(z,r1,r2,A)./CL_stumpC(z) ).^1.5 .* CL_stumpS(z) + A.*sqrt(yy(z,r1,r2,A)) - sqrt(mu).*t);
  endfunction
  
  function dum = dFdz(z,A,r1,r2)
    ii = find(z==0)
    jj = find(z~=0)
    dum(ii) = real( sqrt(2) ./ 40 .* yy(0,r1(ii),r2(ii),A(ii))^1.5 + A(ii) ./ 8 .* ...
           ( sqrt(yy(0,r1(ii),r2(ii),A(ii))) + A(ii) .* sqrt(1/2 ./ yy(0,r1(ii),r2(ii),A(ii))) ) );
    dum(jj) = real( (yy(z(jj),r1(jj),r2(jj),A(jj))./CL_stumpC(z(jj))).^1.5 .* ...
            ( 1/2 ./ z(jj).*(CL_stumpC(z(jj)) - 3.*CL_stumpS(z(jj))./ 2 ./CL_stumpC(z(jj)))+ 3.*CL_stumpS(z(jj)).^2 ./ 4 ./ CL_stumpC(z(jj))) ...
            + A(jj)./ 8 .* ...
            (3.*CL_stumpS(z(jj))./CL_stumpC(z(jj)).*sqrt(yy(z(jj),r1(jj),r2(jj),A(jj))) + A(jj).*sqrt(CL_stumpC(z(jj))./yy(z(jj),r1(jj),r2(jj),A(jj)))) );
  endfunction


if ~exists('direction','local') then direction='pro'; end;
if ~exists('mu','local') then mu=%CL_mu; end;


N = size(pos1,2);

// Consistency of inputs :
Npos2 = size(pos2,2);
Ndelta_t = size(delta_t,2);
if( Ndelta_t == 1) then delta_t = delta_t * ones(1,N); end;
if( Npos2 ~= N) then CL__error('pos1 and pos2 must be of same size'); end;
if( Ndelta_t ~= N) then CL__error('delta_t must be of size 1 or the same size as pos1'); end;

// Magnitudes of pos1 and pos2
r1 = CL_norm(pos1);
r2 = CL_norm(pos2);

c12 = CL_cross(pos1,pos2);
cos_theta = CL_dot(pos1,pos2) ./ (r1 .* r2); 
sin_theta = CL_norm(c12) ./ (r1 .* r2); 

if (direction == 'pro')
  I = find(c12(3,:) < 0);
  sin_theta(I) = -sin_theta(I);
elseif (direction == 'retro')
  I = find(c12(3,:) > 0);
  sin_theta(I) = -sin_theta(I);
else
  CL__error("Invalid direction: ''pro'' or ''retro'' expected");
end

// If pos1 and pos2 are (nearly) aligned : %nan
eps_theta = 1.e-12
I = find(abs(cos_theta) >= 1-eps_theta); 
cos_theta(I) = %nan; 

// Equation 5.35:
A = sin_theta .* sqrt(r1.*r2 ./ (1 - cos_theta));

// Modification of initial algorithm : 
// Before : systematic search of the initial guess such that FF(z) ~= 0 
// with a step of 0.1 starting from -100
// Now : Dichotomy to find the initial guess 

// Dichotomy to find the initial guess
// The solution to FF(z,delta_t,r1,r2,A,mu) = 0 is looked for in [zmin, zmax] 
z_min = -100 ;
z_max = (2*%pi)^2 - 3e-6 ; // such that CL_stumpC(z) > %eps
a = z_min * ones(1,N); // lower bound of dichotomy
b = z_max * ones(1,N); // upper bound of dichotomy
fa = FF(a,delta_t,r1,r2,A,mu); 
fb = FF(b,delta_t,r1,r2,A,mu); 

// 10 iterations => solution at the end of the dichotomy has an 
// accuracy of (z_max-zmin)/2^10 ~= 0.1
for i = 1:10 
  c = (a + b)/2; 
  fc = FF(c,delta_t,r1,r2,A,mu); 
  K = find (fc .* fa >= 0); 
  a(K) = c(K); fa(K) = fc(K);  
  K = find (fc .* fb >= 0); 
  b(K) = c(K); fb(K) = fc(K);
end

// Newton resolution : Iterate on Equation 5.45 until z is determined to within
// the error tolerance (tol)
// NB: The algorithm is assumed to converge given the accuracy of the initial guess (~0.1) 
tol = 1.e-8;  // error tolerance
nmax = 100;   // limit on the number of iterations
z = (a+b)/2;  // initial guess
ratio = ones(1,N);
n = 0;
I = 1 : N;
while ( I ~= [] & n<=nmax)
  n = n + 1;
  ratio(I) = FF(z(I),delta_t(I),r1(I),r2(I),A(I),mu)./dFdz(z(I),A(I),r1(I),r2(I));
  z(I) = z(I) - ratio(I);
  I = find( (abs(ratio)>tol) );
end

// Error if the maximum number of iterations has been exceeded:
if (I ~= []) then CL__error('Maximum number of iterations exceeded'); end

// Equation 5.46a:
f = ( 1 - yy(z,r1,r2,A)./r1 ).*.ones(3,1);

// Equation 5.46b:
g = ( A.*sqrt(yy(z,r1,r2,A)./mu) ).*.ones(3,1);

// Equation 5.46d:
gdot = ( 1 - yy(z,r1,r2,A)./r2 ).*.ones(3,1);

// Equation 5.28:
vel1 = 1 ./g.*(pos2 - f.*pos1);

// Equation 5.29:
vel2 = 1 ./g.*(gdot.*pos2 - pos1);

endfunction



