/*
 * Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 * Copyright (C) 2007 - INRIA - Vincent Couvert
 * Copyright (C) 2007 - INRIA - Marouane BEN JELLOUL
 * 
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at    
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 *
 */

package org.scilab.modules.gui.textbox;

import java.awt.Color;
import java.awt.Font;

import org.scilab.modules.gui.bridge.ScilabBridge;
import org.scilab.modules.gui.events.callback.CallBack;
import org.scilab.modules.gui.text.ScilabText;
import org.scilab.modules.gui.utils.Position;
import org.scilab.modules.gui.utils.Size;

/**
 * Class for Scilab EditBoxs in GUIs
 * @author Vincent COUVERT
 * @author Marouane BEN JELLOUL
 */
public class ScilabTextBox extends ScilabText implements TextBox {
	
	private SimpleTextBox component;
	
	/**
	 * Constructor
	 */
	protected ScilabTextBox() {
		component = ScilabBridge.createTextBox();
	}

	/**
	 * Creates a Scilab TextBox object
	 * @return the created TextBox
	 */
	public static TextBox createTextBox() {
		return new ScilabTextBox();
	}
	
	/**
	 * Gets this Bridge component object
	 * @return this Bridge component object
	 */
	public SimpleTextBox getAsSimpleTextBox() {
		return component;
	}
	
	/**
	 * Sets the text of a TextBox
	 * @param newText the text to set to the TextBox
	 */
	public void setText(String newText) {
		ScilabBridge.setText(this, newText);
	}
	
	/**
	 * Gets the text of a TextBox
	 * @return text of the TextBox
	 */
	public String getText() {
		return ScilabBridge.getText(this);
	}
	
	/**
	 * Draws a TextBox
	 */
	public void draw() {
		ScilabBridge.draw(this);
	}
	
	/**
	 * Sets the dimensions (width and height) of a swing Scilab element
	 * @param newSize the dimensions to set to the element
	 * @see org.scilab.modules.gui.uielement.UIElement#setDims(org.scilab.modules.gui.utils.Size)
	 */
	public void setDims(Size newSize) {
		ScilabBridge.setDims(this, newSize); 
	}
	
	/**
	 * Gets the dimensions (width and height) of a swing Scilab element
	 * @return the dimensions (width and height) of the element
	 * @see org.scilab.modules.gui.uielement.UIElement#getDims(org.scilab.modules.gui.utils.Size)
	 */
	public Size getDims() {
		return ScilabBridge.getDims(this); 
	}
	
	/**
	 * Gets the position (X-coordinate and Y-coordinate) of a Scilab element
	 * @return the position of the element
	 * @see org.scilab.modules.gui.uielement.UIElement#getPosition()
	 */
	public Position getPosition() {
		return ScilabBridge.getPosition(this); 
	}

	/**
	 * Sets the position (X-coordinate and Y-coordinate) of a Scilab element
	 * @param newPosition the position we want to set to the element
	 * @see org.scilab.modules.gui.uielement.UIElement#setPosition(org.scilab.modules.gui.utils.Position)
	 */
	public void setPosition(Position newPosition) {
		ScilabBridge.setPosition(this, newPosition);
	}

	/**
	 * Gets the visibility status of an element
	 * @return the visibility status of the element (true if the element is visible, false if not)
	 */
	public boolean isVisible() {
		return ScilabBridge.isVisible(this); 
	}
	
	/**
	 * Sets the visibility status of a element
	 * @param newVisibleState the visibility status we want to set for the element
	 * 			(true if the element is visible, false if not)
	 */
	public void setVisible(boolean newVisibleState) {
		ScilabBridge.setVisible(this, newVisibleState);
	}

	/**
	 * Set the Background color of the textbox
	 * @param color the Color
	 */
	public void setBackground(Color color) {
		ScilabBridge.setBackground(this, color);
	}

	/**
	 * Get the Background color of the textbox
	 * @return the Color
	 */
	public Color getBackground() {
		return ScilabBridge.getBackground(this);
	}
	
	/**
	 * Set the Foreground color of the textbox
	 * @param color the Color
	 */
	public void setForeground(Color color) {
		ScilabBridge.setForeground(this, color);
	}

	/**
	 * Get the Foreground color of the textbox
	 * @return the Color
	 */
	public Color getForeground() {
		return ScilabBridge.getForeground(this);
	}

	/**
	 * Set the font of the textbox.
	 * @param font the font
	 */
	public void setFont(Font font) {
		ScilabBridge.setFont(this, font);
	}
	
	/**
	 * Get the font of the textbox.
	 * @return the font
	 */
	public Font getFont() {
		return ScilabBridge.getFont(this);
	}

	/**
	 * Set the status of the TextBox
	 * @param status true if the TextBox is enabled
	 * @see org.scilab.modules.gui.widget.Widget#setEnabled(boolean)
	 */
	public void setEnabled(boolean status) {
		ScilabBridge.setEnabled(this, status);
	}

	/**
	 * Gets the enable status of a TextBox
	 * @return the enable status of the TextBox (true if the TextBox is enabled, false if not)
	 */
	public boolean isEnabled() {
		return ScilabBridge.isEnabled(this);
	}

	/**
	 * Add a callback to the TextBox
	 * @param callback the callback to set.
	 */
	public void setCallback(CallBack callback) {
		ScilabBridge.setCallback(this, callback);
	}

	/**
	 * Set the horizontal alignment for the TextBox text
	 * @param alignment the value for the alignment (See ScilabAlignment.java)
	 */
	public void setHorizontalAlignment(String alignment) {
		ScilabBridge.setHorizontalAlignment(this, alignment);
	}

	/**
	 * Set the vertical alignment for the TextBox text
	 * @param alignment the value for the alignment (See ScilabAlignment.java)
	 */
	public void setVerticalAlignment(String alignment) {
		ScilabBridge.setVerticalAlignment(this, alignment);
	}
	
	/**
	 * Set the Relief of the TextBox
	 * @param reliefType the type of the relief to set (See ScilabRelief.java)
	 */
	public void setRelief(String reliefType) {
		ScilabBridge.setRelief(this, reliefType);
	}

	/**
	 * Delete the TextBox
	 */
	public void destroy() {
		ScilabBridge.destroy(this);
	}

	/**
	 * Give the focus to the ContextMenu
	 */
	public void requestFocus() {
		ScilabBridge.requestFocus(this);
	}

}
