/*
 * Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 * Copyright (C) 2007 - INRIA - Sylvestre LEDRU
 * Copyright (C) 2009 - DIGITEO - Allan CORNET
 * 
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at    
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 *
 */
/*--------------------------------------------------------------------------*/
#include <string.h>
#include "javasci_SciAbstractArray.h"
#include "javasci_globals.h"
#include "call_scilab.h"
#include "api_scilab.h"
#include "freeArrayOfString.h"
#ifdef _MSC_VER
#include "strdup_windows.h"
#endif
/*--------------------------------------------------------------------------*/
static int JNI_getMatrixOfInteger(JNIEnv *env, jobject obj_this, jclass class_this, int Rows, int Cols, char *name);
static int JNI_getMatrixOfDouble(JNIEnv *env, jobject obj_this, jclass class_this, char *name);
static int JNI_getMatrixOfComplex(JNIEnv *env, jobject obj_this, jclass class_this, char *name);
static int JNI_getMatrixOfString(JNIEnv *env, jobject obj_this, jclass class_this, int Rows, int Cols, char *name);
static int JNI_getMatrixOfBoolean(JNIEnv *env, jobject obj_this, jclass class_this, int Rows, int Cols, char *name);
/*--------------------------------------------------------------------------*/
static int JNI_setMatrixOfInteger(JNIEnv *env, jobject obj_this, jclass class_this, int Rows, int Cols, char *name);
static int JNI_setMatrixOfDouble(JNIEnv *env, jobject obj_this, jclass class_this, int Rows, int Cols, char *name);
static int JNI_setMatrixOfComplex(JNIEnv *env, jobject obj_this, jclass class_this, int Rows, int Cols, char *name);
static int JNI_setMatrixOfString(JNIEnv *env, jobject obj_this, jclass class_this, int Rows, int Cols, char *name);
static int JNI_setMatrixOfBoolean(JNIEnv *env, jobject obj_this, jclass class_this, int Rows, int Cols, char *name);
/*--------------------------------------------------------------------------*/
static char* detectSignatureTypeFromObjectName(JNIEnv *env, jobject obj_this);
/*--------------------------------------------------------------------------*/
/*
* Class:     javasci_SciAbstractArray
* Method:    Initialize
* Signature: ()V
*/
JNIEXPORT void JNICALL Java_javasci_SciAbstractArray_Initialize (JNIEnv *env, jclass cl)
{
	if ( GetInterfState() == 0) { EnableInterf(); Initialize();} 
}
/*--------------------------------------------------------------------------*/
/*
* Class:     javasci_SciAbstractArray
* Method:    getNumberOfRowsFromScilab
* Signature: (Ljava/lang/String;)I
*/
JNIEXPORT jint JNICALL Java_javasci_SciAbstractArray_getNumberOfRowsFromScilab(JNIEnv *env, jobject obj_this, jstring name)
{
	SciErr sciErr;
	const char *cname = NULL;
	jint row = -1;
	int dimension[2];

	cname = (*env)->GetStringUTFChars(env, name, NULL);

	sciErr = getNamedVarDimension(pvApiCtx, (char*)cname, &dimension[0], &dimension[1]);
	if(sciErr.iErr)
	{
		// bug 6388
		// fprintf(stderr,"%s", getErrorMessage(sciErr));
		// fprintf(stderr,"Error in Java_javasci_SciAbstractArray_Get (1).\n");
		(*env)->ReleaseStringUTFChars(env, name , cname);
		
		return -1;
	}

	row = (jint)dimension[0];

	(*env)->ReleaseStringUTFChars(env, name , cname);

	return row;
}
/*--------------------------------------------------------------------------*/
/*
* Class:     javasci_SciAbstractArray
* Method:    getColFromScilab
* Signature: (Ljava/lang/String;)I
*/
JNIEXPORT jint JNICALL Java_javasci_SciAbstractArray_getNumberOfColsFromScilab(JNIEnv *env, jobject obj_this, jstring name)
{
	SciErr sciErr;
	const char *cname = NULL;
	jint col = -1;
	int dimension[2];

	cname = (*env)->GetStringUTFChars(env, name, NULL);

	sciErr = getNamedVarDimension(pvApiCtx, (char*)cname, &dimension[0], &dimension[1]);
	if(sciErr.iErr)
	{
		// bug 6388
		// fprintf(stderr,"%s", getErrorMessage(sciErr));
		// fprintf(stderr,"Error in Java_javasci_SciAbstractArray_Get (1).\n");
		(*env)->ReleaseStringUTFChars(env, name , cname);
		return -1;
	}

	col = (jint)dimension[1];

	(*env)->ReleaseStringUTFChars(env, name , cname);

	return col;
}
/*--------------------------------------------------------------------------*/
/*
* Class:     javasci_SciAbstractArray
* Method:    Job
* Signature: (Ljava/lang/String;)Z
*/
JNIEXPORT jboolean JNICALL Java_javasci_SciAbstractArray_Job(JNIEnv *env, jobject obj_this, jstring job)
{
	const char *cjob = NULL;
	jboolean bOK = 0;

	/* get the field value */
	cjob = (*env)->GetStringUTFChars(env, job, NULL);

	if (strlen(cjob) >= MAX_STR)
	{
		fprintf(stderr, "Error in Java_javasci_SciAsbtractArray_Job routine (line too long).\n");
		bOK = (jboolean)JNI_FALSE;
	}
	else
	{
		if (SendScilabJob((char*)cjob))
		{
			fprintf(stderr, "Error in Java_javasci_SciAsbtractArray_Job routine.\n");
			bOK = JNI_FALSE;
		}
		else bOK = JNI_TRUE;
	}

	(*env)->ReleaseStringUTFChars(env, job , cjob);

	return bOK;
}
/*--------------------------------------------------------------------------*/
/*
* Class:     javasci_SciAbstractArray
* Method:    Get
* Signature: ()V
*/
JNIEXPORT void JNICALL Java_javasci_SciAbstractArray_Get(JNIEnv *env, jobject obj_this)
{
	SciErr sciErr;
	char *signatureType = detectSignatureTypeFromObjectName(env, obj_this);

	jclass class_Mine = (*env)->GetObjectClass(env, obj_this);
	jfieldID id_name =  (*env)->GetFieldID(env, class_Mine, "name","Ljava/lang/String;");
	jfieldID id_m = (*env)->GetFieldID(env, class_Mine, "m", "I");
	jfieldID id_n = (*env)->GetFieldID(env, class_Mine, "n", "I");

	jstring jname = (jstring) (*env)->GetObjectField(env, obj_this, id_name);
	jint jm = (*env)->GetIntField(env, obj_this, id_m);
	jint jn = (*env)->GetIntField(env, obj_this, id_n);
	const char *cname = (*env)->GetStringUTFChars(env, jname, NULL);

	int dimension[2];

	sciErr = getNamedVarDimension(pvApiCtx, (char*)cname, &dimension[0], &dimension[1]);
	if(sciErr.iErr)
	{
		fprintf(stderr,"%s", getErrorMessage(sciErr));
		(*env)->ReleaseStringUTFChars(env, jname , cname);
		fprintf(stderr,"Error in Java_javasci_SciAbstractArray_Get (1).\n");
		return;
	}

	if (dimension[0] != jm)
	{
		fprintf(stderr,"Error in Java_javasci_SciAbstractArray_Get (2).\n");
		(*env)->ReleaseStringUTFChars(env, jname , cname);
		return;
	}

	if (dimension[1] != jn)
	{
		fprintf(stderr,"Error in Error in Java_javasci_SciAbstractArray_Get (3).\n");
		(*env)->ReleaseStringUTFChars(env, jname , cname);
		return;
	}
	if (strcmp(signatureType,"I") == 0)
	{
		JNI_getMatrixOfInteger(env, obj_this, class_Mine, dimension[0], dimension[1], (char*)cname);
	}
	else if (strcmp(signatureType,"D") == 0)
	{
		JNI_getMatrixOfDouble(env, obj_this, class_Mine, (char*)cname);
	}
	else if (strcmp(signatureType,"Z") == 0)
	{
		JNI_getMatrixOfBoolean(env, obj_this, class_Mine, (int)jm, (int)jn, (char*)cname);
	}
	else if (strcmp(signatureType,"CO") == 0)
	{
		JNI_getMatrixOfComplex(env, obj_this, class_Mine, (char*)cname);
	}
	else if (strcmp(signatureType,"Ljava/lang/String;") == 0)
	{
		JNI_getMatrixOfString(env, obj_this, class_Mine, (int)jm, (int)jn, (char*)cname);
	}
	else
	{
		fprintf(stderr,"Error in Java_javasci_SciAbstractArray_Get (invalid type).\n");
	}
	(*env)->ReleaseStringUTFChars(env, jname , cname);
}
/*--------------------------------------------------------------------------*/
/**
* Detect which Scilab object we are dealing with
* might need some cleaning
* @return Scilab Type of the object we are dealing with
*/
static char* detectSignatureTypeFromObjectName(JNIEnv *env, jobject obj_this)
{
	if((*env)->IsInstanceOf(env, obj_this,  (*env)->FindClass(env,"javasci/SciDoubleArray")) == JNI_TRUE)
	{
		return "D";
	}

	if((*env)->IsInstanceOf(env, obj_this,  (*env)->FindClass(env,"javasci/SciBooleanArray")) == JNI_TRUE)
	{
		return "Z";
	}

	if((*env)->IsInstanceOf(env, obj_this,  (*env)->FindClass(env,"javasci/SciComplexArray")) == JNI_TRUE)
	{
		return "CO";
	}

	if((*env)->IsInstanceOf(env, obj_this,  (*env)->FindClass(env,"javasci/SciIntegerArray")) == JNI_TRUE)
	{
		return "I";
	}

	if((*env)->IsInstanceOf(env, obj_this,  (*env)->FindClass(env,"javasci/SciStringArray")) == JNI_TRUE)
	{
		return "Ljava/lang/String;";
	}

	return "";
}
/*--------------------------------------------------------------------------*/
/*
* Class:     javasci_SciAbstractArray
* Method:    Send
* Signature: ()V
*/
JNIEXPORT void JNICALL Java_javasci_SciAbstractArray_Send(JNIEnv *env, jobject obj_this)
{
	char *signatureType = detectSignatureTypeFromObjectName(env, obj_this);

	jclass class_Mine = (*env)->GetObjectClass(env, obj_this);
	jfieldID id_name =  (*env)->GetFieldID(env, class_Mine, "name","Ljava/lang/String;");
	jfieldID id_m = (*env)->GetFieldID(env, class_Mine, "m", "I");
	jfieldID id_n = (*env)->GetFieldID(env, class_Mine, "n", "I");

	jstring jname = (jstring) (*env)->GetObjectField(env, obj_this, id_name);
	jint jm = (*env)->GetIntField(env, obj_this, id_m);
	jint jn = (*env)->GetIntField(env, obj_this, id_n);
	const char *cname = (*env)->GetStringUTFChars(env, jname, NULL);

	if (strcmp(signatureType,"I") == 0)
	{
		JNI_setMatrixOfInteger(env, obj_this, class_Mine, (int)jm, (int)jn, (char*)cname);
	}
	else if (strcmp(signatureType,"D") == 0)
	{
		JNI_setMatrixOfDouble(env, obj_this, class_Mine, (int)jm, (int)jn, (char*)cname);
	}
	else if (strcmp(signatureType,"Z") == 0)
	{
		JNI_setMatrixOfBoolean(env, obj_this, class_Mine, (int)jm, (int)jn, (char*)cname);
	}
	else if (strcmp(signatureType,"CO") == 0)
	{
		JNI_setMatrixOfComplex(env, obj_this, class_Mine, (int)jm, (int)jn, (char*)cname);
	}
	else if (strcmp(signatureType,"Ljava/lang/String;") == 0)
	{
		JNI_setMatrixOfString(env, obj_this, class_Mine, (int)jm, (int)jn, (char*)cname);
	}
	else
	{
		fprintf(stderr,"Error in Java_javasci_SciAbstractArray_Set (invalid type).\n");
	}
	(*env)->ReleaseStringUTFChars(env, jname , cname);
}
/*--------------------------------------------------------------------------*/
static int JNI_getMatrixOfInteger(JNIEnv *env, jobject obj_this, jclass class_this, int Rows, int Cols, char *name)
{
	SciErr sciErr;
	int i = 0;
	int cm = 0, cn = 0;
	jfieldID id_x = (*env)->GetFieldID(env, class_this, "x", "[I");
	jintArray jx = (*env)->GetObjectField(env, obj_this, id_x);
	jint *cx = (*env)->GetIntArrayElements(env, jx, NULL);
	int *icx = (int*)MALLOC(sizeof(int) * (Rows * Cols));

	if (icx == NULL)
	{
		fprintf(stderr,"Error in JNI_getMatrixOfInteger (1).\n");
		(*env)->ReleaseIntArrayElements(env, jx, cx, 0);
		return 1;
	}

	for (i = 0; i < (Rows * Cols); i++) icx[i] = (int)cx[i];

	sciErr = readNamedMatrixOfInteger32(pvApiCtx, name, &cm, &cn, icx);
	if(sciErr.iErr)
	{
		fprintf(stderr,"%s", getErrorMessage(sciErr));
		FREE(icx); icx = NULL;
		fprintf(stderr,"Error in JNI_getMatrixOfInteger (2).\n");
		(*env)->ReleaseIntArrayElements(env, jx, cx, 0);
		return 1;
	}

	FREE(icx); icx = NULL;

	(*env)->ReleaseIntArrayElements(env, jx, cx, 0);
	return 0;
}
/*--------------------------------------------------------------------------*/
static int JNI_getMatrixOfDouble(JNIEnv *env, jobject obj_this, jclass class_this, char *name)
{
	SciErr sciErr;
	int cm = 0, cn = 0;
	jfieldID id_x = (*env)->GetFieldID(env, class_this, "x", "[D");
	jdoubleArray jx = (*env)->GetObjectField(env, obj_this, id_x);
	double *cx = (*env)->GetDoubleArrayElements(env, jx, NULL);

	sciErr = readNamedMatrixOfDouble(pvApiCtx, name, &cm, &cn, cx);
	if(sciErr.iErr)
	{
		fprintf(stderr,"%s", getErrorMessage(sciErr));
		fprintf(stderr,"Error in JNI_getMatrixOfDouble.\n");
		(*env)->ReleaseDoubleArrayElements(env, jx, cx, 0);
		return 1;
	}

	(*env)->ReleaseDoubleArrayElements(env, jx, cx, 0);
	return 0;
}
/*--------------------------------------------------------------------------*/
static int JNI_getMatrixOfComplex(JNIEnv *env, jobject obj_this, jclass class_this, char *name)
{
	SciErr sciErr;
	int cm = 0, cn = 0;
	jfieldID id_x = (*env)->GetFieldID(env, class_this, "x", "[D");
	jfieldID id_y = (*env)->GetFieldID(env, class_this, "y", "[D");

	/* get the field value */
	jdoubleArray jx = (*env)->GetObjectField(env, obj_this, id_x);
	jdoubleArray jy = (*env)->GetObjectField(env, obj_this, id_y);

	double *cx = (*env)->GetDoubleArrayElements(env, jx, NULL);
	double *cy = (*env)->GetDoubleArrayElements(env, jy, NULL);

	sciErr = readNamedComplexMatrixOfDouble(pvApiCtx, name, &cm, &cn, cx, cy);
	if(sciErr.iErr)
	{
		fprintf(stderr,"%s", getErrorMessage(sciErr));
		fprintf(stderr,"Error in JNI_getMatrixOfComplex.\n");
		(*env)->ReleaseDoubleArrayElements(env, jx, cx, 0);
		(*env)->ReleaseDoubleArrayElements(env, jy ,cy, 0);
		return 1;
	}

	(*env)->ReleaseDoubleArrayElements(env, jx, cx, 0);
	(*env)->ReleaseDoubleArrayElements(env, jy, cy, 0);
	return 0;
}
/*--------------------------------------------------------------------------*/
static int JNI_getMatrixOfString(JNIEnv *env, jobject obj_this, jclass class_this, int Rows, int Cols, char *name)
{
	SciErr sciErr;
	jfieldID id_x = (*env)->GetFieldID(env, class_this, "x", "[Ljava/lang/String;");
	jobjectArray jx = (*env)->GetObjectField(env, obj_this, id_x);

	int *pLength = NULL;
	char **pStrings = NULL;
	int cm = 0, cn = 0;
	int i = 0;

	pLength = (int*)MALLOC(sizeof(int)*(Rows * Cols));
	if (pLength == NULL)
	{
		fprintf(stderr,"Error in JNI_getMatrixOfString (1).\n");
		return 1;
	}

	sciErr = readNamedMatrixOfString(pvApiCtx, name, &cm, &cn, pLength, pStrings);
	if(sciErr.iErr)
	{
		fprintf(stderr,"%s", getErrorMessage(sciErr));
		fprintf(stderr,"Error in JNI_getMatrixOfString (2).\n");
		return 1;
	}

	pStrings = (char**)MALLOC(sizeof(char*)*(cm * cn));
	if (pStrings == NULL)
	{
		fprintf(stderr,"Error in JNI_getMatrixOfString (3).\n");
		return 1;
	}

	for (i = 0; i < cm * cn; i++)
	{
		pStrings[i] = (char*)MALLOC(sizeof(char)* (pLength[i] + 1));
		if (pStrings[i] == NULL)
		{
			fprintf(stderr,"Error in JNI_getMatrixOfString (4).\n");
			freeArrayOfString(pStrings, i);
			FREE(pLength);
			return 1;
		}
	}

	sciErr = readNamedMatrixOfString(pvApiCtx, name, &cm, &cn, pLength, pStrings);
	if(sciErr.iErr)
	{
		fprintf(stderr,"%s", getErrorMessage(sciErr));
		if (pLength) {FREE(pLength); pLength = NULL;}
		fprintf(stderr,"Error in JNI_getMatrixOfString (5).\n");
		return 1;
	}

	if (pLength) {FREE(pLength); pLength = NULL;}

	for (i = 0; i < cm * cn; i++)
	{
		jstring jelement = (*env)->NewStringUTF(env, pStrings[i]);
		(*env)->SetObjectArrayElement(env, jx,  i, jelement);
	}

	freeArrayOfString(pStrings, cm * cn);

	return 0;
}
/*--------------------------------------------------------------------------*/
static int JNI_getMatrixOfBoolean(JNIEnv *env, jobject obj_this, jclass class_this, int Rows, int Cols, char *name)
{
	SciErr sciErr;
	jfieldID id_x = (*env)->GetFieldID(env, class_this, "x", "[Z");
	jbooleanArray jx = (*env)->GetObjectField(env, obj_this, id_x);
	jboolean *cx = (*env)->GetBooleanArrayElements(env, jx, NULL); 
	int i = 0;
	int cm = 0, cn = 0;
	int *CX = (int*) MALLOC(sizeof(int)*(Rows * Cols));

	if (CX == NULL)
	{
		fprintf(stderr,"Error in JNI_getMatrixOfBoolean (1).\n");
		(*env)->ReleaseBooleanArrayElements(env, jx, cx, 0);
		return 1;
	}

	sciErr = readNamedMatrixOfBoolean(pvApiCtx, name, &cm, &cn, CX);
	if(sciErr.iErr)
	{
		fprintf(stderr,"%s", getErrorMessage(sciErr));
		fprintf(stderr,"Error in JNI_getMatrixOfBoolean (2).\n");
		(*env)->ReleaseBooleanArrayElements(env, jx, cx, 0);
		return 1;
	}

	for(i = 0; i < cm*cn; i++)
	{
		cx[i] = (jboolean)CX[i];
	}

	FREE(CX); CX = NULL;

	(*env)->ReleaseBooleanArrayElements(env, jx, cx, 0);
	return 0;
}
/*--------------------------------------------------------------------------*/
static int JNI_setMatrixOfInteger(JNIEnv *env, jobject obj_this, jclass class_this, int Rows, int Cols, char *name)
{
	SciErr sciErr;
	jfieldID id_x = (*env)->GetFieldID(env, class_this, "x", "[I");
	jintArray jx = (*env)->GetObjectField(env, obj_this, id_x);
	jint *cx = (*env)->GetIntArrayElements(env, jx, NULL);
	int *icx = NULL;
	int i = 0;

	if ( (Rows * Cols) == 0 ) icx = (int*)MALLOC(sizeof(int) * (1));
	else icx = (int*)MALLOC(sizeof(int) * (Rows * Cols));

	if (icx == NULL)
	{
		fprintf(stderr,"Error in JNI_setMatrixOfInteger (1).\n");
		(*env)->ReleaseIntArrayElements(env, jx ,cx, 0);
		return 1;
	}

	for (i = 0; i < (Rows * Cols); i++) icx[i] = (int)cx[i];

	sciErr = createNamedMatrixOfInteger32(pvApiCtx, name, Rows, Cols, icx);
	if(sciErr.iErr)
	{
		fprintf(stderr,"%s", getErrorMessage(sciErr));
		FREE(icx); icx = NULL;
		fprintf(stderr,"Error in JNI_setMatrixOfInteger (2).\n");
		(*env)->ReleaseIntArrayElements(env, jx ,cx, 0);
		return 1; 
	}

	FREE(icx); icx = NULL;

	(*env)->ReleaseIntArrayElements(env, jx, cx, 0);
	return 0;
}
/*--------------------------------------------------------------------------*/
static int JNI_setMatrixOfDouble(JNIEnv *env, jobject obj_this, jclass class_this, int Rows, int Cols, char *name)
{
	SciErr sciErr;
	jfieldID id_x = (*env)->GetFieldID(env, class_this, "x", "[D");
	jdoubleArray jx = (*env)->GetObjectField(env, obj_this, id_x);
	double *cx = (*env)->GetDoubleArrayElements(env, jx, NULL);

	sciErr = createNamedMatrixOfDouble(pvApiCtx, name, Rows, Cols, cx);
	if(sciErr.iErr)
	{
		fprintf(stderr,"%s", getErrorMessage(sciErr));
		fprintf(stderr,"Error in JNI_setMatrixOfDouble.\n");
		(*env)->ReleaseDoubleArrayElements(env, jx ,cx, 0);
		return 1; 
	}

	(*env)->ReleaseDoubleArrayElements(env, jx, cx, 0);
	return 0;
}
/*--------------------------------------------------------------------------*/
static int JNI_setMatrixOfComplex(JNIEnv *env, jobject obj_this, jclass class_this, int Rows, int Cols, char *name)
{
	SciErr sciErr;
	jfieldID id_x = (*env)->GetFieldID(env, class_this, "x", "[D");
	jfieldID id_y = (*env)->GetFieldID(env, class_this, "y", "[D");

	jdoubleArray jx = (*env)->GetObjectField(env, obj_this, id_x);
	jdoubleArray jy = (*env)->GetObjectField(env, obj_this, id_y);

	double *cx = (*env)->GetDoubleArrayElements(env, jx, NULL);
	double *cy = (*env)->GetDoubleArrayElements(env, jy, NULL);

	sciErr = createNamedComplexMatrixOfDouble(pvApiCtx, name, Rows, Cols, cx, cy);
	if(sciErr.iErr)
	{
		fprintf(stderr,"%s", getErrorMessage(sciErr));
		fprintf(stderr,"Error in JNI_setMatrixOfComplex.\n");
		(*env)->ReleaseDoubleArrayElements(env, jx, cx, 0);
		(*env)->ReleaseDoubleArrayElements(env, jy, cy, 0);
		return 1; 
	}

	(*env)->ReleaseDoubleArrayElements(env, jx, cx, 0);
	(*env)->ReleaseDoubleArrayElements(env, jy, cy, 0);
	return 0;
}
/*--------------------------------------------------------------------------*/
static int JNI_setMatrixOfString(JNIEnv *env, jobject obj_this, jclass class_this, int Rows, int Cols, char *name)
{
	SciErr sciErr;
	int i = 0;
	char **pStrings = NULL;
	jfieldID id_x = (*env)->GetFieldID(env, class_this, "x", "[Ljava/lang/String;" );
	jobjectArray jx = (*env)->GetObjectField(env, obj_this, id_x);

	pStrings = (char **)MALLOC(sizeof(char*) * (Rows * Cols));
	if (pStrings == NULL)
	{
		fprintf(stderr,"Error in JNI_setMatrixOfString (1).\n");
		return 1;
	}

	for (i = 0; i < (Rows * Cols); i++)
	{
		jstring jelement = (jstring)(*env)->GetObjectArrayElement(env, jx, i);
		const char *element = (*env)->GetStringUTFChars(env, jelement, NULL);
		pStrings[i] = strdup(element);
		(*env)->ReleaseStringUTFChars(env, jelement,  element);
	}

	sciErr = createNamedMatrixOfString(pvApiCtx, name, Rows, Cols, pStrings);
	if(sciErr.iErr)
	{
		fprintf(stderr,"%s", getErrorMessage(sciErr));
		freeArrayOfString(pStrings, Rows * Cols);
		fprintf(stderr,"Error in JNI_setMatrixOfString (2).\n");
		return 1;
	}

	freeArrayOfString(pStrings, Rows * Cols);
	return 0;
}
/*--------------------------------------------------------------------------*/
static int JNI_setMatrixOfBoolean(JNIEnv *env, jobject obj_this, jclass class_this, int Rows, int Cols, char *name)
{
	SciErr sciErr;
	jfieldID id_x = (*env)->GetFieldID(env, class_this, "x", "[Z");
	jbooleanArray jx = (*env)->GetObjectField(env, obj_this, id_x);
	jboolean *cx = (*env)->GetBooleanArrayElements(env, jx, NULL);

	int *CX = (int*)MALLOC(sizeof(int)*(Rows * Cols));
	int i = 0;

	if (CX == NULL)
	{
		fprintf(stderr,"Error in JNI_setMatrixOfBoolean (1).\n");
		(*env)->ReleaseBooleanArrayElements(env, jx, (jboolean*)cx, 0);
		return 1; 
	}

	for( i = 0; i < Rows * Cols; i++)
	{
		CX[i] = (int)cx[i];
	}

	sciErr = createNamedMatrixOfBoolean(pvApiCtx, name, Rows, Cols, CX);
	if(sciErr.iErr)
	{
		fprintf(stderr,"%s", getErrorMessage(sciErr));
		fprintf(stderr,"Error in JNI_setMatrixOfBoolean (2).\n");
		(*env)->ReleaseBooleanArrayElements(env, jx, (jboolean*)cx, 0);
		return 1; 
	}

	if (CX) { FREE(CX); CX = NULL; }
	(*env)->ReleaseBooleanArrayElements(env, jx, (jboolean*)cx, 0);
	return 0;
}
/*--------------------------------------------------------------------------*/

