#include <assert.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>

#include <sys/select.h>

#include "scim-bridge-agent-client-peer-protected.h"
#include "scim-bridge-agent-imcontext.h"
#include "scim-bridge-agent-keyevent-utility.h"
#include "scim-bridge-agent-messenger.h"
#include "scim-bridge-attribute.h"
#include "scim-bridge-exception.h"
#include "scim-bridge-keyevent.h"
#include "scim-bridge-message.h"
#include "scim-bridge-messenger.h"
#include "scim-bridge-output.h"

using namespace scim;

static void *run_thread (void *arg);

/* Implementations */
ScimBridgeAgentMessenger::ScimBridgeAgentMessenger (ScimBridgeAgentClientPeerProtected &new_client_peer):
thread_running (false), id (-1), client_peer (new_client_peer)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER | SCIM_BRIDGE_DEBUG_AGENT, 3, "Initializing the agent messenger...");
}


ScimBridgeAgentMessenger::~ScimBridgeAgentMessenger ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER | SCIM_BRIDGE_DEBUG_AGENT, 3, "Finalizing the agent messenger...");

    ScimBridgeException except;
    scim_bridge_exception_initialize (&except);
    scim_bridge_finalize_messenger (&except, id);
    scim_bridge_exception_finalize (&except);

    while (thread_running) {
        usleep (100);
    }
}


void ScimBridgeAgentMessenger::open_connection (int input_fd, int output_fd) throw (ScimBridgeAgentException)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER | SCIM_BRIDGE_DEBUG_AGENT, 3, "Connecting sockets to the messenger...");

    assert (input_fd >= 0 && output_fd >= 0);

    ScimBridgeException except;
    scim_bridge_exception_initialize (&except);
    if (scim_bridge_initialize_messenger (&except, &id, input_fd, output_fd)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }
    scim_bridge_exception_finalize (&except);

    thread_running = true;
    if (pthread_create (&thread, NULL, run_thread, reinterpret_cast<void*> (this)) || pthread_detach (thread)) {
        ScimBridgeAgentException exception (String ("Cannot start the agent messenger thread: ") + strerror (errno));

        thread_running = false;
        scim_bridge_exception_initialize (&except);
        scim_bridge_finalize_messenger (&except, id);
        scim_bridge_exception_finalize (&except);

        throw exception;
    }

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER | SCIM_BRIDGE_DEBUG_AGENT, 3, "Connection is ready");
}


void ScimBridgeAgentMessenger::connection_closed ()
{
    thread_running = false;
    client_peer.connection_closed ();
}


bool ScimBridgeAgentMessenger::is_thread_running ()
{
    return thread_running;
}


ScimBridgeMessengerID ScimBridgeAgentMessenger::get_id ()
{
    return id;
}


/* Thread body */
void *run_thread (void *arg)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER | SCIM_BRIDGE_DEBUG_AGENT, 3, "run_receiver...");

    ScimBridgeAgentMessenger *messenger = reinterpret_cast <ScimBridgeAgentMessenger*> (arg);

    try {
        while (messenger->is_thread_running ()) {
            ScimBridgeException except;
            scim_bridge_exception_initialize (&except);

            scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER | SCIM_BRIDGE_DEBUG_AGENT, 3, "The messenger waiting...");
            if (scim_bridge_messenger_open_input (&except, messenger->get_id ())) {
                scim_bridge_exception_finalize (&except);
                break;
            }

            scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER | SCIM_BRIDGE_DEBUG_AGENT, 3, "A message is received");

            ScimBridgeMessageCode code;
            if (scim_bridge_messenger_read_input (&except, messenger->get_id (), &code, sizeof (code))) {
                ScimBridgeAgentException exception = ScimBridgeAgentException (except);
                scim_bridge_exception_finalize (&except);
                throw exception;
            }
            scim_bridge_exception_finalize (&except);

            switch (code) {
                case SCIM_BRIDGE_MESSAGE_ALLOC_IMCONTEXT:
                    messenger->received_message_alloc_imcontext ();
                    break;
                case SCIM_BRIDGE_MESSAGE_FREE_IMCONTEXT:
                    messenger->received_message_free_imcontext ();
                    break;
                case SCIM_BRIDGE_MESSAGE_RESET_IMCONTEXT:
                    messenger->received_message_reset_imcontext ();
                    break;
                case SCIM_BRIDGE_MESSAGE_FOCUS_CHANGED:
                    messenger->received_message_focus_changed ();
                    break;
                case SCIM_BRIDGE_MESSAGE_KEYEVENT_OCCURED:
                    messenger->received_message_keyevent_occured ();
                    break;
                case SCIM_BRIDGE_MESSAGE_CURSOR_LOCATION_CHANGED:
                    messenger->received_message_cursor_location_changed ();
                    break;
                default:
                    messenger->received_message_unknown (code);
                    break;
            }
        }

    } catch (ScimBridgeAgentException ex) {
        scim_bridge_perrorln ("A messenger caused an error: %s", ex.what ());
    }

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER | SCIM_BRIDGE_DEBUG_AGENT, 3, "The messenger closed");

    messenger->connection_closed ();

    /* The following return is a dammy.*/
    return NULL;
}


/* Received messages */
void ScimBridgeAgentMessenger::received_message_unknown (ScimBridgeMessageCode code) throw (ScimBridgeAgentException)
{
    scim_bridge_perrorln ("Received 'unkown message' (%d)", code);

    ScimBridgeException except;
    scim_bridge_exception_initialize (&except);
    if (scim_bridge_messenger_close_input (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }
    scim_bridge_exception_finalize (&except);
}


void ScimBridgeAgentMessenger::received_message_alloc_imcontext () throw (ScimBridgeAgentException)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER | SCIM_BRIDGE_DEBUG_AGENT, 3, "Received 'alloc imcontext' message");

    ScimBridgeException except;
    scim_bridge_exception_initialize (&except);

    ScimBridgeIMContextID opponent_id;
    if (scim_bridge_messenger_read_input (&except, id, &opponent_id, sizeof (opponent_id))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    ScimBridgeIMContextID ic_id = -1;

    ScimBridgeAgentIMContext *ic = client_peer.alloc_imcontext ();
    if (ic != NULL) {
        ic->set_opponent_id (opponent_id);
        ic_id = ic->get_id ();
    }

    if (scim_bridge_messenger_write_input (&except, id, &ic_id, sizeof (ic_id))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    if (scim_bridge_messenger_close_input (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }
    scim_bridge_exception_finalize (&except);
}


void ScimBridgeAgentMessenger::received_message_free_imcontext () throw (ScimBridgeAgentException)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER | SCIM_BRIDGE_DEBUG_AGENT, 3, "Received 'free imcontext' message");

    ScimBridgeException except;
    scim_bridge_exception_initialize (&except);

    ScimBridgeIMContextID ic_id;
    if (scim_bridge_messenger_read_input (&except, id, &ic_id, sizeof (ic_id))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    if (scim_bridge_messenger_close_input (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }
    scim_bridge_exception_finalize (&except);

    ScimBridgeAgentIMContext *ic = ScimBridgeAgentIMContext::find (ic_id);
    if (ic != NULL) client_peer.free_imcontext (*ic);
}


void ScimBridgeAgentMessenger::received_message_reset_imcontext () throw (ScimBridgeAgentException)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER | SCIM_BRIDGE_DEBUG_AGENT, 3, "Received 'reset imcontext' message");

    ScimBridgeException except;
    scim_bridge_exception_initialize (&except);

    ScimBridgeIMContextID ic_id;
    if (scim_bridge_messenger_read_input (&except, id, &ic_id, sizeof (ic_id))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    if (scim_bridge_messenger_close_input (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }
    scim_bridge_exception_finalize (&except);

    ScimBridgeAgentIMContext *ic = ScimBridgeAgentIMContext::find (ic_id);
    if (ic != NULL) client_peer.reset_imcontext (*ic);
}


void ScimBridgeAgentMessenger::received_message_focus_changed () throw (ScimBridgeAgentException)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER | SCIM_BRIDGE_DEBUG_AGENT, 3, "Received 'focus changed' message");

    ScimBridgeException except;
    scim_bridge_exception_initialize (&except);

    ScimBridgeIMContextID ic_id;
    if (scim_bridge_messenger_read_input (&except, id, &ic_id, sizeof (ic_id))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    int focus_in;
    if (scim_bridge_messenger_read_input (&except, id, &focus_in, sizeof (focus_in))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    if (scim_bridge_messenger_close_input (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }
    scim_bridge_exception_finalize (&except);

    ScimBridgeAgentIMContext *ic = ScimBridgeAgentIMContext::find (ic_id);
    if (ic != NULL) client_peer.focus_changed (*ic, focus_in);
}


void ScimBridgeAgentMessenger::received_message_keyevent_occured () throw (ScimBridgeAgentException)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER | SCIM_BRIDGE_DEBUG_AGENT, 3, "Received 'keyevent occured' message");

    ScimBridgeException except;
    scim_bridge_exception_initialize (&except);

    ScimBridgeIMContextID ic_id;
    if (scim_bridge_messenger_read_input (&except, id, &ic_id, sizeof (ic_id))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    ScimBridgeKeyEvent keyevent;
    if (scim_bridge_messenger_read_input (&except, id, &keyevent, sizeof (keyevent))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    KeyEvent scim_keyevent = scim_bridge_keyevent_bridge_to_scim (keyevent, client_peer.get_keyboard_layout ());

    int consumed = 0;

    ScimBridgeAgentIMContext *ic = ScimBridgeAgentIMContext::find (ic_id);
    if (ic != NULL) consumed = client_peer.keyevent_occured (*ic, scim_keyevent);

    if (scim_bridge_messenger_write_input (&except, id, &consumed, sizeof (consumed))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    if (scim_bridge_messenger_close_input (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }
    scim_bridge_exception_finalize (&except);
}


void ScimBridgeAgentMessenger::received_message_cursor_location_changed () throw (ScimBridgeAgentException)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER | SCIM_BRIDGE_DEBUG_AGENT, 3, "Received 'cursor location changed' message");

    ScimBridgeException except;
    scim_bridge_exception_initialize (&except);

    ScimBridgeIMContextID ic_id;
    if (scim_bridge_messenger_read_input (&except, id, &ic_id, sizeof (ic_id))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    int cursor_x;
    if (scim_bridge_messenger_read_input (&except, id, &cursor_x, sizeof (cursor_x))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    int cursor_y;
    if (scim_bridge_messenger_read_input (&except, id, &cursor_y, sizeof (cursor_y))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    if (scim_bridge_messenger_close_input (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }
    scim_bridge_exception_finalize (&except);

    ScimBridgeAgentIMContext *ic = ScimBridgeAgentIMContext::find (ic_id);
    if (ic != NULL) client_peer.cursor_location_changed (*ic, cursor_x, cursor_y);
}


/* Remote function */
void ScimBridgeAgentMessenger::call_commit (const ScimBridgeAgentIMContext &ic) throw (ScimBridgeAgentException)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER | SCIM_BRIDGE_DEBUG_AGENT, 3, "Sending 'commit' message...");

    ScimBridgeException except;
    scim_bridge_exception_initialize (&except);

    if (scim_bridge_messenger_open_output (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    const ScimBridgeMessageCode code = SCIM_BRIDGE_MESSAGE_COMMIT;
    if (scim_bridge_messenger_write_output (&except, id, &code, sizeof (code))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    ScimBridgeIMContextID opponent_id = ic.get_opponent_id ();
    if (scim_bridge_messenger_write_output (&except, id, &opponent_id, sizeof (opponent_id))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    if (scim_bridge_messenger_close_output (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }
    scim_bridge_exception_finalize (&except);
}


void ScimBridgeAgentMessenger::call_set_preedit_string (const ScimBridgeAgentIMContext &ic, const WideString &string) throw (ScimBridgeAgentException)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER | SCIM_BRIDGE_DEBUG_AGENT, 3, "Sending 'set preedit string' message...");
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER | SCIM_BRIDGE_DEBUG_AGENT, 1, "preedit: %s", string.c_str ());

    ScimBridgeException except;
    scim_bridge_exception_initialize (&except);

    if (scim_bridge_messenger_open_output (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    const ScimBridgeMessageCode code = SCIM_BRIDGE_MESSAGE_SET_PREEDIT_STRING;
    if (scim_bridge_messenger_write_output (&except, id, &code, sizeof (code))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    ScimBridgeIMContextID opponent_id = ic.get_opponent_id ();
    if (scim_bridge_messenger_write_output (&except, id, &opponent_id, sizeof (opponent_id))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    const size_t wstr_len = string.length ();
    if (scim_bridge_messenger_write_output (&except, id, &wstr_len, sizeof (wstr_len))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    const ucs4_t *wstr = string.c_str ();
    if (scim_bridge_messenger_write_output (&except, id, wstr, sizeof (ucs4_t) * (wstr_len + 1))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    if (scim_bridge_messenger_close_output (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }
    scim_bridge_exception_finalize (&except);
}


void ScimBridgeAgentMessenger::call_set_preedit_attributes (const ScimBridgeAgentIMContext &ic, const AttributeList &attributes) throw (ScimBridgeAgentException)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER | SCIM_BRIDGE_DEBUG_AGENT, 3, "Sending 'set preedit atributes' message...");

    ScimBridgeException except;
    scim_bridge_exception_initialize (&except);

    if (scim_bridge_messenger_open_output (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    const ScimBridgeMessageCode code = SCIM_BRIDGE_MESSAGE_SET_PREEDIT_ATTRIBUTES;
    if (scim_bridge_messenger_write_output (&except, id, &code, sizeof (code))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    ScimBridgeIMContextID opponent_id = ic.get_opponent_id ();
    if (scim_bridge_messenger_write_output (&except, id, &opponent_id, sizeof (opponent_id))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    const size_t attr_count = attributes.size ();
    if (scim_bridge_messenger_write_output (&except, id, &attr_count, sizeof (attr_count))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    ScimBridgeAttribute attrs [attr_count];
    int j = 0;
    for (AttributeList::const_iterator i = attributes.begin (); i != attributes.end (); ++i) {
        const Attribute &attribute = *i;
        attrs[j].begin = attribute.get_start ();
        attrs[j].end = attribute.get_end ();
        if (attribute.get_type () == SCIM_ATTR_DECORATE) {
            attrs[j].type = SCIM_BRIDGE_ATTRIBUTE_DECORATE;
            switch (attribute.get_value ()) {
                case SCIM_ATTR_DECORATE_UNDERLINE:
                    attrs[j].value = SCIM_BRIDGE_ATTRIBUTE_DECORATE_UNDERLINE;
                    break;
                case SCIM_ATTR_DECORATE_REVERSE:
                    attrs[j].value = SCIM_BRIDGE_ATTRIBUTE_DECORATE_REVERSE;
                    break;
                case SCIM_ATTR_DECORATE_HIGHLIGHT:
                default:
                    attrs[j].value = SCIM_BRIDGE_ATTRIBUTE_DECORATE_HIGHLIGHT;
                    break;
            }
        } else if (attribute.get_type () == SCIM_ATTR_FOREGROUND) {
            attrs[j].type = SCIM_BRIDGE_ATTRIBUTE_FOREGROUND;
            const unsigned int red = SCIM_RGB_COLOR_RED (attribute.get_value ());
            const unsigned int green = SCIM_RGB_COLOR_GREEN (attribute.get_value ());
            const unsigned int blue = SCIM_RGB_COLOR_BLUE (attribute.get_value ());
            attrs[j].value = scim_bridge_attribute_get_color (red, green, blue);
        } else if (attribute.get_type () == SCIM_ATTR_BACKGROUND) {
            attrs[j].type = SCIM_BRIDGE_ATTRIBUTE_FOREGROUND;
            const unsigned int red = SCIM_RGB_COLOR_RED (attribute.get_value ());
            const unsigned int green = SCIM_RGB_COLOR_GREEN (attribute.get_value ());
            const unsigned int blue = SCIM_RGB_COLOR_BLUE (attribute.get_value ());
            attrs[j].value = scim_bridge_attribute_get_color (red, green, blue);
        } else {
            attrs[j].type = SCIM_BRIDGE_ATTRIBUTE_NONE;
        }
        ++j;
    }

    if (scim_bridge_messenger_write_output (&except, id, attrs, sizeof (ScimBridgeAttribute) * attr_count)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    if (scim_bridge_messenger_close_output (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }
    scim_bridge_exception_finalize (&except);

}


void ScimBridgeAgentMessenger::call_set_preedit_cursor_position (const ScimBridgeAgentIMContext &ic, int cursor_position) throw (ScimBridgeAgentException)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER | SCIM_BRIDGE_DEBUG_AGENT, 3, "Sending 'set preedit cursor position' message...");

    ScimBridgeException except;
    scim_bridge_exception_initialize (&except);

    if (scim_bridge_messenger_open_output (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    const ScimBridgeMessageCode code = SCIM_BRIDGE_MESSAGE_SET_PREEDIT_CURSOR_POSITION;
    if (scim_bridge_messenger_write_output (&except, id, &code, sizeof (code))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    ScimBridgeIMContextID opponent_id = ic.get_opponent_id ();
    if (scim_bridge_messenger_write_output (&except, id, &opponent_id, sizeof (opponent_id))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    if (scim_bridge_messenger_write_output (&except, id, &cursor_position, sizeof (cursor_position))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    if (scim_bridge_messenger_close_output (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }
    scim_bridge_exception_finalize (&except);
}


void ScimBridgeAgentMessenger::call_set_preedit_shown (const ScimBridgeAgentIMContext &ic, bool shown) throw (ScimBridgeAgentException)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER | SCIM_BRIDGE_DEBUG_AGENT, 3, "Sending 'set preedit shown' message...");

    ScimBridgeException except;
    scim_bridge_exception_initialize (&except);

    if (scim_bridge_messenger_open_output (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    const ScimBridgeMessageCode code = SCIM_BRIDGE_MESSAGE_SET_PREEDIT_SHOWN;
    if (scim_bridge_messenger_write_output (&except, id, &code, sizeof (code))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    ScimBridgeIMContextID opponent_id = ic.get_opponent_id ();
    if (scim_bridge_messenger_write_output (&except, id, &opponent_id, sizeof (opponent_id))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    int value = shown;
    if (scim_bridge_messenger_write_output (&except, id, &value, sizeof (value))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    if (scim_bridge_messenger_close_output (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }
    scim_bridge_exception_finalize (&except);
}


void ScimBridgeAgentMessenger::call_update_preedit (const ScimBridgeAgentIMContext &ic) throw (ScimBridgeAgentException)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER | SCIM_BRIDGE_DEBUG_AGENT, 3, "Sending 'update preedit' message...");

    ScimBridgeException except;
    scim_bridge_exception_initialize (&except);

    if (scim_bridge_messenger_open_output (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    const ScimBridgeMessageCode code = SCIM_BRIDGE_MESSAGE_UPDATE_PREEDIT;
    if (scim_bridge_messenger_write_output (&except, id, &code, sizeof (code))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    ScimBridgeIMContextID opponent_id = ic.get_opponent_id ();
    if (scim_bridge_messenger_write_output (&except, id, &opponent_id, sizeof (opponent_id))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    if (scim_bridge_messenger_close_output (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }
    scim_bridge_exception_finalize (&except);
}


void ScimBridgeAgentMessenger::call_forward_keyevent (const ScimBridgeAgentIMContext &ic, const KeyEvent &keyevent) throw (ScimBridgeAgentException)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER | SCIM_BRIDGE_DEBUG_AGENT, 3, "Sending 'forward keyevent' message...");

    ScimBridgeException except;
    scim_bridge_exception_initialize (&except);

    if (scim_bridge_messenger_open_output (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    const ScimBridgeMessageCode code = SCIM_BRIDGE_MESSAGE_FORWARD_KEYEVENT;
    if (scim_bridge_messenger_write_output (&except, id, &code, sizeof (code))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    ScimBridgeIMContextID opponent_id = ic.get_opponent_id ();
    if (scim_bridge_messenger_write_output (&except, id, &opponent_id, sizeof (opponent_id))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    ScimBridgeKeyEvent bridge_keyevent = scim_bridge_keyevent_scim_to_bridge (keyevent);
    if (scim_bridge_messenger_write_output (&except, id, &bridge_keyevent, sizeof (bridge_keyevent))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    if (scim_bridge_messenger_close_output (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }
    scim_bridge_exception_finalize (&except);
}


void ScimBridgeAgentMessenger::call_beep (const ScimBridgeAgentIMContext &ic) throw (ScimBridgeAgentException)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER | SCIM_BRIDGE_DEBUG_AGENT, 3, "Sending 'beep' message...");

    ScimBridgeException except;
    scim_bridge_exception_initialize (&except);

    if (scim_bridge_messenger_open_output (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    const ScimBridgeMessageCode code = SCIM_BRIDGE_MESSAGE_BEEP;
    if (scim_bridge_messenger_write_output (&except, id, &code, sizeof (code))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    ScimBridgeIMContextID opponent_id = ic.get_opponent_id ();
    if (scim_bridge_messenger_write_output (&except, id, &opponent_id, sizeof (opponent_id))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    if (scim_bridge_messenger_close_output (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }
    scim_bridge_exception_finalize (&except);
}


bool ScimBridgeAgentMessenger::call_get_surrounding_string (const ScimBridgeAgentIMContext &ic, WideString &surrounding, size_t max_length, int &cursor_pos) throw (ScimBridgeAgentException)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER | SCIM_BRIDGE_DEBUG_AGENT, 3, "Sending 'get surrounding string' message...");

    ScimBridgeException except;
    scim_bridge_exception_initialize (&except);

    if (scim_bridge_messenger_open_output (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    const ScimBridgeMessageCode code = SCIM_BRIDGE_MESSAGE_GET_SURROUNDING_STRING;
    if (scim_bridge_messenger_write_output (&except, id, &code, sizeof (code))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    ScimBridgeIMContextID opponent_id = ic.get_opponent_id ();
    if (scim_bridge_messenger_write_output (&except, id, &opponent_id, sizeof (opponent_id))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    if (scim_bridge_messenger_write_output (&except, id, &max_length, sizeof (max_length))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    size_t wstr_len;
    if (scim_bridge_messenger_read_output (&except, id, &wstr_len, sizeof (wstr_len))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    ucs4_t wstr[wstr_len + 1];
    if (scim_bridge_messenger_read_output (&except, id, wstr, sizeof (wstr))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    surrounding = wstr;

    if (scim_bridge_messenger_read_output (&except, id, &cursor_pos, sizeof (cursor_pos))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    if (scim_bridge_messenger_close_output (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }
    scim_bridge_exception_finalize (&except);

    return wstr_len != 0;
}


bool ScimBridgeAgentMessenger::call_delete_surrounding_string (const ScimBridgeAgentIMContext &ic, size_t offset, size_t length) throw (ScimBridgeAgentException)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER | SCIM_BRIDGE_DEBUG_AGENT, 3, "Sending 'delete surrounding string' message...");

    ScimBridgeException except;
    scim_bridge_exception_initialize (&except);

    if (scim_bridge_messenger_open_output (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    const ScimBridgeMessageCode code = SCIM_BRIDGE_MESSAGE_DELETE_SURROUNDING_STRING;
    if (scim_bridge_messenger_write_output (&except, id, &code, sizeof (code))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    ScimBridgeIMContextID opponent_id = ic.get_opponent_id ();
    if (scim_bridge_messenger_write_output (&except, id, &opponent_id, sizeof (opponent_id))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    if (scim_bridge_messenger_write_output (&except, id, &offset, sizeof (offset))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    if (scim_bridge_messenger_write_output (&except, id, &length, sizeof (length))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    int retval;
    if (scim_bridge_messenger_read_output (&except, id, &retval, sizeof (retval))) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }

    if (scim_bridge_messenger_close_output (&except, id)) {
        ScimBridgeAgentException exception = ScimBridgeAgentException (except);
        scim_bridge_exception_finalize (&except);
        throw exception;
    }
    scim_bridge_exception_finalize (&except);

    return retval;
}
