#include <alloca.h>
#include <assert.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <gtk/gtk.h>
#include <gtk/gtkimcontext.h>

#include "scim-bridge-attribute.h"
#include "scim-bridge-client-imcontext-gtk.h"
#include "scim-bridge-client-kernel.h"
#include "scim-bridge-client-keyevent-utility-gtk.h"
#include "scim-bridge-exception.h"
#include "scim-bridge-imcontext-common.h"
#include "scim-bridge-output.h"

/* Typedef */
struct _ScimBridgeClientIMContextImpl
{
    /* extends ScimBridgeIMContext */
    /* Don't touch the first element */
    ScimBridgeIMContextParent parent;

    ScimBridgeClientIMContext *owner;

    GdkWindow *client_window;

    int preedit_shown;
    int preedit_cursor_position;

    int cursor_x;
    int cursor_y;
    int window_x;
    int window_y;

    char *preedit_string;
    size_t preedit_string_length;
    size_t preedit_string_capacity;

    char *commit_string;
    size_t commit_string_capacity;

    ScimBridgeAttribute *preedit_attributes;
    size_t preedit_attribute_count;
};

/* Private data */
static GType scim_bridge_client_imcontext_type = 0;
static GObjectClass *imcontext_root_klass = NULL;

static gboolean initialized = FALSE;

/* Variables for scim */
static ScimBridgeClientIMContextImpl *scim_focused_imcontext = NULL;

/* Private variables for SCIM */
static GdkColor color_normal_background;
static GdkColor color_normal_text;
static GdkColor color_active_background;
static GdkColor color_active_text;

/* Helper functions */
static void gtk_im_slave_commit_cb (GtkIMContext *fallback_context, const char *str, ScimBridgeClientIMContext *context);

static void initialize ();
static void finalize ();
static void attach_imcontext_impl (ScimBridgeClientIMContext *ic);
static void detach_imcontext_impl (ScimBridgeClientIMContext *ic);

static void display_exception (ScimBridgeException *exception);

static gboolean do_update_preedit (gpointer data);
static gboolean do_commit (gpointer data);
static gboolean do_shutdown (gpointer data);

/* Class functions */
static void scim_bridge_client_imcontext_class_initialize (ScimBridgeClientIMContextClass *klass, gpointer *klass_data);
static void scim_bridge_client_imcontext_initialize (ScimBridgeClientIMContext *context, ScimBridgeClientIMContextClass *klass);
static void scim_bridge_client_imcontext_finalize (GObject *object);

static gboolean scim_bridge_client_imcontext_filter_keypress (GtkIMContext *context, GdkEventKey *event);
static void scim_bridge_client_imcontext_reset (GtkIMContext *context);
static void scim_bridge_client_imcontext_get_preedit_string (GtkIMContext *context, gchar **str, PangoAttrList **attrs, gint *cursor_pos);
static void scim_bridge_client_imcontext_set_use_preedit (GtkIMContext *context, gboolean use_preedit);

static void scim_bridge_client_imcontext_set_client_window (GtkIMContext *context, GdkWindow *window);
static void scim_bridge_client_imcontext_focus_in (GtkIMContext *context);
static void scim_bridge_client_imcontext_focus_out (GtkIMContext *context);
static void scim_bridge_client_imcontext_set_cursor_location (GtkIMContext *context, GdkRectangle *area);

/* Kernel implementations */
void scim_bridge_client_kernel_impl_cleanup ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_IMCONTEXT, 3, "scim_bridge_client_kernel_impl_cleanup");

    ScimBridgeIMContext *iter;
    for (iter = scim_bridge_client_kernel_get_first_imcontext (); iter != NULL; iter = scim_bridge_client_kernel_get_next_imcontext (iter)) {
        ScimBridgeClientIMContextImpl *ic_impl = (ScimBridgeClientIMContextImpl*) iter;
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_IMCONTEXT, 1, "Freeing imcontext: %d", ic_impl->parent.id);
        detach_imcontext_impl (ic_impl->owner);
    }

    scim_focused_imcontext = NULL;

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_IMCONTEXT, 3, "Cleanup done...");
}


void scim_bridge_client_kernel_impl_finalized ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_IMCONTEXT, 3, "scim_bridge_client_kernel_impl_finalized");
    initialized = FALSE;
}


void scim_bridge_client_kernel_impl_commit (ScimBridgeIMContext *ic)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 2, "scim_bridge_client_kernel_commit");

    if (!initialized) return;

    ScimBridgeClientIMContextImpl *ic_impl = (ScimBridgeClientIMContextImpl*) ic;
    if (ic_impl->commit_string_capacity < ic_impl->preedit_string_length) {
        ic_impl->commit_string = realloc (ic_impl->commit_string, sizeof (char) * (ic_impl->preedit_string_length + 1));
        ic_impl->commit_string_capacity = ic_impl->preedit_string_length;
    }
    strcpy (ic_impl->commit_string, ic_impl->preedit_string);
    g_idle_add (do_commit, (gpointer) ic_impl->owner);
}


void scim_bridge_client_kernel_impl_set_preedit_string (ScimBridgeIMContext *ic, ucs4_t *wstr, size_t wstr_len)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 2, "scim_bridge_client_kernel_set_preedit_string");

    ScimBridgeClientIMContextImpl *ic_impl = (ScimBridgeClientIMContextImpl*) ic;
    if (ic_impl) {
        const size_t str_buflen = scim_bridge_string_strbuflen (wstr);
        if (ic_impl->preedit_string == NULL || ic_impl->preedit_string_capacity < str_buflen) {
            ic_impl->preedit_string = realloc (ic_impl->preedit_string, sizeof (char) * (str_buflen + 1));
            ic_impl->preedit_string_capacity = str_buflen;
        }

        ScimBridgeException except;
        scim_bridge_exception_initialize (&except);
        if (scim_bridge_string_wcstombs (&except, ic_impl->preedit_string, wstr, ic_impl->preedit_string_capacity, &ic_impl->preedit_string_length)) {
            scim_bridge_perrorln ("Cannot convert preedit wide string into utf8: %s", scim_bridge_exception_get_message (&except));
        }
        scim_bridge_exception_finalize (&except);

        free (wstr);
    }
}


void scim_bridge_client_kernel_impl_set_preedit_attributes (ScimBridgeIMContext *ic, ScimBridgeAttribute *attrs, size_t attr_count)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 2, "scim_bridge_client_kernel_set_preedit_attributes");

    ScimBridgeClientIMContextImpl *ic_impl = (ScimBridgeClientIMContextImpl*) ic;
    if (ic_impl) {
        free (ic_impl->preedit_attributes);
        ic_impl->preedit_attributes = attrs;
        ic_impl->preedit_attribute_count = attr_count;
    }
}


void scim_bridge_client_kernel_impl_set_preedit_cursor_position (ScimBridgeIMContext *ic, int cursor_position)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 2, "scim_bridge_client_kernel_set_preedit_cursor_position");

    ScimBridgeClientIMContextImpl *ic_impl = (ScimBridgeClientIMContextImpl*) ic;
    if (ic_impl) {
        ic_impl->preedit_cursor_position = cursor_position;
    }
}


void scim_bridge_client_kernel_impl_set_preedit_shown (ScimBridgeIMContext *ic, int shown)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 2, "scim_bridge_client_kernel_set_preedit_shown");

    ScimBridgeClientIMContextImpl *ic_impl = (ScimBridgeClientIMContextImpl*) ic;
    if (ic_impl) {
        ic_impl->preedit_shown = shown;
    }
}


void scim_bridge_client_kernel_impl_update_preedit (ScimBridgeIMContext *ic)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 2, "scim_bridge_client_kernel_impl_update_preedit");

    if (!initialized) return;

    ScimBridgeClientIMContextImpl *ic_impl = (ScimBridgeClientIMContextImpl*) ic;
    g_idle_add (do_update_preedit, (gpointer) ic_impl->owner);
}


void scim_bridge_client_kernel_impl_forward_keyevent (ScimBridgeIMContext *ic, const ScimBridgeKeyEvent *keyevent)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 2, "scim_bridge_client_kernel_forward_keyevent");

    if (!initialized) return;

    ScimBridgeClientIMContextImpl *ic_impl = (ScimBridgeClientIMContextImpl*) ic;
    GdkEventKey gdkevent = scim_bridge_keyevent_bridge_to_gdk (ic_impl->client_window, keyevent);

    if (ic_impl->owner && ic_impl->owner->slave) {
        if (!gtk_im_context_filter_keypress (ic_impl->owner->slave, &gdkevent)) {
            gdk_event_put ((GdkEvent *) &gdkevent);
        }
    }
}


void scim_bridge_client_kernel_impl_beep (ScimBridgeIMContext *ic)
{
    ScimBridgeClientIMContextImpl *ic_impl = (ScimBridgeClientIMContextImpl*) ic;
    if (ic_impl) gdk_beep ();
}


void scim_bridge_client_kernel_impl_get_surrounding_string (ScimBridgeIMContext *ic, ucs4_t *wstr, const size_t max_length, size_t *fetch_len, int *cursor_pos)
{
    ScimBridgeClientIMContextImpl *ic_impl = (ScimBridgeClientIMContextImpl*) ic;

    *fetch_len = 0;
    if (ic_impl) {
        char *str;
        int cur_pos_in_utf8;
        if (gtk_im_context_get_surrounding (GTK_IM_CONTEXT (ic_impl->owner), &str, &cur_pos_in_utf8)) {
            ScimBridgeException except;
            scim_bridge_exception_initialize (&except);

            size_t wstr_len;
            if (scim_bridge_string_mbstowcs (&except, wstr, str, max_length, &wstr_len)) {
                scim_bridge_perrorln ("Cannot convert the surrounding text into UCS4: %s", except.message);
                scim_bridge_exception_finalize (&except);

                g_free (str);
                return;
            }

            char *sub_str = str + cur_pos_in_utf8;
            ucs4_t *buf = alloca (sizeof (ucs4_t) * max_length);
            size_t buf_len;
            if (scim_bridge_string_mbstowcs (&except, buf, sub_str, (size_t) max_length, &buf_len)) {
                scim_bridge_perrorln ("Cannot convert cursor position into UCS4 order: %s", except.message);
                scim_bridge_exception_finalize (&except);

                g_free (str);
                return;
            }

            g_free (str);
            scim_bridge_exception_finalize (&except);

            *fetch_len = wstr_len;
            *cursor_pos = *fetch_len - buf_len;
        }
    }
}


void scim_bridge_client_kernel_impl_delete_surrounding_string (ScimBridgeIMContext *ic, size_t offset, size_t length, int *retval)
{
    ScimBridgeClientIMContextImpl *ic_impl = (ScimBridgeClientIMContextImpl*) ic;
    if (ic_impl) {
        *retval = gtk_im_context_delete_surrounding (GTK_IM_CONTEXT (ic_impl->owner), offset, length);
    } else {
        *retval = 0;
    }
}


void scim_bridge_client_kernel_impl_exception_occured (ScimBridgeException *except)
{
    display_exception (except);

    g_idle_add (do_shutdown, NULL);
}


/* Helper function implmentations  */
void initialize ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_IMCONTEXT, 3, "initialize");

    /* Init colors.*/
    gdk_color_parse ("gray92", &color_normal_background);
    gdk_color_parse ("black", &color_normal_text);
    gdk_color_parse ("light blue", &color_active_background);
    gdk_color_parse ("black", &color_active_text);

    scim_focused_imcontext = NULL;

    ScimBridgeException except;
    scim_bridge_exception_initialize (&except);
    if (scim_bridge_client_initialize_kernel (&except)) {
        display_exception (&except);
        scim_bridge_exception_finalize (&except);

        return;
    }
    scim_bridge_exception_finalize (&except);

    initialized = TRUE;
}


void finalize ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_IMCONTEXT, 3, "finalize");

    if (initialized) {
        ScimBridgeException except;
        scim_bridge_exception_initialize (&except);
        scim_bridge_client_finalize_kernel (&except);
        scim_bridge_exception_finalize (&except);
    }
}


void attach_imcontext_impl (ScimBridgeClientIMContext *ic)
{
    ScimBridgeClientIMContextImpl *ic_impl = malloc (sizeof (ScimBridgeClientIMContextImpl));

    ic_impl->client_window = NULL;

    ic_impl->preedit_shown = 0;
    ic_impl->preedit_string = NULL;
    ic_impl->preedit_string_length = 0;
    ic_impl->preedit_string_capacity = 0;
    ic_impl->preedit_attributes = NULL;
    ic_impl->preedit_attribute_count = 0;

    ic_impl->commit_string = NULL;
    ic_impl->commit_string_capacity = 0;

    ic_impl->cursor_x = 0;
    ic_impl->cursor_y = 0;
    ic_impl->window_x = 0;
    ic_impl->window_y = 0;

    ic_impl->owner = ic;
    ic->impl = ic_impl;
}


void detach_imcontext_impl (ScimBridgeClientIMContext *ic)
{
    ScimBridgeClientIMContextImpl *ic_impl = ic->impl;

    if (scim_focused_imcontext == ic_impl) scim_focused_imcontext = NULL;

    g_idle_remove_by_data ((gpointer) ic);
    
    if (ic_impl->client_window) g_object_unref (ic_impl->client_window);

    free (ic_impl->preedit_string);
    free (ic_impl->preedit_attributes);

    free (ic_impl->commit_string);

    free (ic_impl);

    ic->impl = NULL;
}


void gtk_im_slave_commit_cb (GtkIMContext *fallback_context, const char *str, ScimBridgeClientIMContext *context)
{
    g_return_if_fail (str);
    g_signal_emit_by_name (context, "commit", str);
}


void display_exception (ScimBridgeException *except)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_IMCONTEXT, 3, "scim_bridge_client_kernel_impl_exception_occured");
    scim_bridge_perrorln ("%s: %s", scim_bridge_exception_get_message (except), scim_bridge_exception_get_strerror (except));
}


gboolean do_update_preedit (gpointer data)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 1, "do_update_preedit");

    if (initialized) {
        ScimBridgeClientIMContext *ic = (ScimBridgeClientIMContext*) data;
        g_signal_emit_by_name (ic, "preedit_changed");
    }

    return FALSE;
}


gboolean do_commit (gpointer data)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 1, "do_commit");

    if (initialized) {
        ScimBridgeClientIMContext *ic = (ScimBridgeClientIMContext*) data;
        ScimBridgeClientIMContextImpl *ic_impl = ic->impl;
        g_signal_emit_by_name (ic, "commit", ic_impl->commit_string);
    }

    return FALSE;
}


gboolean do_shutdown (gpointer data)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 1, "do_shutdown");

    finalize ();
    return FALSE;
}


/* Implementations */
void scim_bridge_client_imcontext_class_initialize (ScimBridgeClientIMContextClass *klass, gpointer *klass_data)
{
    GtkIMContextClass *imcontext_klass = GTK_IM_CONTEXT_CLASS (klass);
    GObjectClass *gobject_klass = G_OBJECT_CLASS (klass);

    imcontext_root_klass = (GObjectClass *) g_type_class_peek_parent (klass);

    imcontext_klass->set_client_window = scim_bridge_client_imcontext_set_client_window;
    imcontext_klass->filter_keypress = scim_bridge_client_imcontext_filter_keypress;
    imcontext_klass->reset = scim_bridge_client_imcontext_reset;
    imcontext_klass->get_preedit_string = scim_bridge_client_imcontext_get_preedit_string;
    imcontext_klass->focus_in  = scim_bridge_client_imcontext_focus_in;
    imcontext_klass->focus_out = scim_bridge_client_imcontext_focus_out;
    imcontext_klass->set_cursor_location = scim_bridge_client_imcontext_set_cursor_location;
    imcontext_klass->set_use_preedit = scim_bridge_client_imcontext_set_use_preedit;
    gobject_klass->finalize = scim_bridge_client_imcontext_finalize;
}


GType scim_bridge_client_imcontext_get_type ()
{
    return scim_bridge_client_imcontext_type;
}


void scim_bridge_client_imcontext_register_type (GTypeModule *type_module)
{
    static const GTypeInfo scim_bridge_client_imcontext_ckass_info = {
        sizeof (ScimBridgeClientIMContextClass),
        /* no base class initializer */
        NULL,
        /* no base class finalizer */
        NULL,
        /* class initializer */
        (GClassInitFunc) scim_bridge_client_imcontext_class_initialize,
        /* no class finalizer */
        NULL,
        /* no class data */
        NULL,
        sizeof (ScimBridgeClientIMContext),
        0,
        /* object initizlier */
        (GtkObjectInitFunc) scim_bridge_client_imcontext_initialize,
    };

    if (!scim_bridge_client_imcontext_type)
        scim_bridge_client_imcontext_type = g_type_module_register_type (type_module, GTK_TYPE_IM_CONTEXT, "ScimBridgeClientIMContext", &scim_bridge_client_imcontext_ckass_info, 0);

    if (!initialized) initialize ();
}


void scim_bridge_client_imcontext_shutdown ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_IMCONTEXT, 3, "scim_bridge_client_imcontext_shutdown");

    if (!initialized) return;

    ScimBridgeException except;
    scim_bridge_exception_initialize (&except);
    scim_bridge_client_trigger_kernel_finalizer (&except);
    scim_bridge_exception_finalize (&except);

    while (initialized) {
        usleep (10);
    }
}


GtkIMContext *scim_bridge_client_imcontext_new ()
{
    ScimBridgeClientIMContext *retval = SCIM_BRIDGE_CLIENT_IMCONTEXT (g_object_new (GTK_TYPE_SCIM_CLIENT_IMCONTEXT, NULL));
    return GTK_IM_CONTEXT (retval);
}


void scim_bridge_client_imcontext_initialize (ScimBridgeClientIMContext *context, ScimBridgeClientIMContextClass *klass)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_IMCONTEXT, 3, "scim_bridge_client_imcontext_initialize");

    /* slave exists for using gtk+'s table based input method */
    context->slave = gtk_im_context_simple_new ();
    g_signal_connect (G_OBJECT (context->slave), "commit", G_CALLBACK (gtk_im_slave_commit_cb), context);

    attach_imcontext_impl (context);

    ScimBridgeException except;
    scim_bridge_exception_initialize (&except);
    if (scim_bridge_client_kernel_alloc_imcontext (&except, (ScimBridgeIMContext*) context->impl)) {
        display_exception (&except);
        scim_bridge_exception_finalize (&except);
        detach_imcontext_impl (context);
    } else {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_IMCONTEXT, 1, "imcontext: id = %d opponent = %d", context->impl->parent.id, context->impl->parent.opponent_id);

        scim_focused_imcontext = NULL;
    }
}


void scim_bridge_client_imcontext_finalize (GObject *object)
{
    /* FIXME I think this function contains many bugs */
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_IMCONTEXT, 3, "scim_bridge_client_imcontext_finalize");

    ScimBridgeClientIMContext *ic = SCIM_BRIDGE_CLIENT_IMCONTEXT (object);
    ScimBridgeClientIMContextImpl *ic_impl = ic->impl;
    if (ic_impl) {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_IMCONTEXT, 3, "id = %d", ic_impl->parent.id);
    } else {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_IMCONTEXT, 3, "id = NULL");
    }
    
    g_signal_handlers_disconnect_by_func (ic->slave, gtk_im_slave_commit_cb, ic);
    g_object_unref (ic->slave);

    if (ic_impl) {
        ScimBridgeException except;
        scim_bridge_exception_initialize (&except);
        if (scim_bridge_client_kernel_free_imcontext (&except, (ScimBridgeIMContext*) ic_impl)) display_exception (&except);
        scim_bridge_exception_finalize (&except);

        detach_imcontext_impl (ic);
    }

    imcontext_root_klass->finalize (object);
}


gboolean scim_bridge_client_imcontext_filter_keypress (GtkIMContext *context, GdkEventKey *event)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_IMCONTEXT, 3, "scim_bridge_client_imcontext_filter_keypress");

    ScimBridgeClientIMContext *ic = SCIM_BRIDGE_CLIENT_IMCONTEXT (context);

    if (ic) {
        if (initialized && ic->impl) {
            ScimBridgeClientIMContextImpl *ic_impl = ic->impl;

            if (ic_impl != scim_focused_imcontext) scim_bridge_client_imcontext_focus_in (GTK_IM_CONTEXT (ic_impl->owner));

            if (ic_impl->client_window) {
                int new_window_x;
                int new_window_y;
                gdk_window_get_origin (ic_impl->client_window, &new_window_x, &new_window_y);

                if (ic_impl->window_x != new_window_x || ic_impl->window_y != new_window_y) {
                    ic_impl->window_x = new_window_x;
                    ic_impl->window_y = new_window_y;

                    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_IMCONTEXT, 1,
                        "The cursor location is changed: x = %d + %d\ty = %d + %d",
                        ic_impl->window_x, ic_impl->cursor_x, ic_impl->window_y, ic_impl->cursor_y);

                    ScimBridgeException except;
                    scim_bridge_exception_initialize (&except);
                    if (scim_bridge_client_kernel_cursor_location_changed (&except, (ScimBridgeIMContext*) ic_impl, ic_impl->window_x + ic_impl->cursor_x, ic_impl->window_y + ic_impl->cursor_y)) display_exception (&except);
                    scim_bridge_exception_finalize (&except);
                }
            }

            ScimBridgeKeyEvent bridge_keyevent = scim_bridge_keyevent_gdk_to_bridge (ic_impl->client_window, event);

            int consumed = 0;

            ScimBridgeException except;
            scim_bridge_exception_initialize (&except);
            if (scim_bridge_client_kernel_keyevent_occured (&except, (ScimBridgeIMContext*) ic_impl, &bridge_keyevent, &consumed)) {
                display_exception (&except);
                scim_bridge_exception_finalize (&except);
                return FALSE;
            }
            scim_bridge_exception_finalize (&except);

            if (consumed) {
                scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_IMCONTEXT, 2, "Key event has been consumed...");
                return TRUE;
            }
        }

        if (ic->slave) return gtk_im_context_filter_keypress (ic->slave, event);
    } else {
        return FALSE;
    }
}


void scim_bridge_client_imcontext_reset (GtkIMContext *context)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_IMCONTEXT, 3, "scim_bridge_client_imcontext_reset");

    if (!initialized) return;

    ScimBridgeClientIMContext *ic = SCIM_BRIDGE_CLIENT_IMCONTEXT (context);

    if (ic && ic->impl) {
        ScimBridgeClientIMContextImpl *ic_impl = ic->impl;

        ScimBridgeException except;
        scim_bridge_exception_initialize (&except);
        if (scim_bridge_client_kernel_reset_imcontext (&except, (ScimBridgeIMContext*) ic_impl)) display_exception (&except);
        scim_bridge_exception_finalize (&except);
    }
}


void scim_bridge_client_imcontext_get_preedit_string (GtkIMContext *context, gchar **str, PangoAttrList **pango_attrs, gint *cursor_pos)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_IMCONTEXT, 3, "scim_bridge_client_imcontext_get_preedit_string");

    ScimBridgeClientIMContext *ic = SCIM_BRIDGE_CLIENT_IMCONTEXT (context);

    if (ic && ic->impl && ic->impl->preedit_shown && ic->impl->preedit_string) {
        ScimBridgeClientIMContextImpl *ic_impl = ic->impl;

        if (str) *str = g_strdup (ic_impl->preedit_string);

        if (cursor_pos) *cursor_pos = ic_impl->preedit_cursor_position;

        if (pango_attrs) {
            *pango_attrs = pango_attr_list_new ();

            gboolean underline_exists = FALSE;

            int i;
            for (i = 0; i < ic_impl->preedit_attribute_count; ++i) {
                ScimBridgeAttribute *attr = &ic_impl->preedit_attributes[i];
                const int begin_pos = attr->begin;
                const int end_pos = attr->end;

                if (begin_pos < end_pos && 0 <= begin_pos && end_pos < ic_impl->preedit_string_length) {
                    const int start_index = g_utf8_offset_to_pointer (ic_impl->preedit_string, begin_pos) - ic_impl->preedit_string;
                    const int end_index = g_utf8_offset_to_pointer (ic_impl->preedit_string, end_pos) - ic_impl->preedit_string;

                    if (attr->type == SCIM_BRIDGE_ATTRIBUTE_DECORATE) {
                        if (attr->value == SCIM_BRIDGE_ATTRIBUTE_DECORATE_UNDERLINE) {
                            PangoAttribute *pango_attr = pango_attr_underline_new (PANGO_UNDERLINE_SINGLE);
                            pango_attr->start_index = start_index;
                            pango_attr->end_index = end_index;
                            pango_attr_list_insert (*pango_attrs, pango_attr);
                            underline_exists = TRUE;
                        } else if (attr->value == SCIM_BRIDGE_ATTRIBUTE_DECORATE_REVERSE) {
                            PangoAttribute *pango_attr0 = pango_attr_foreground_new (color_normal_background.red, color_normal_background.green, color_normal_background.blue);
                            pango_attr0->start_index = start_index;
                            pango_attr0->end_index = end_index;
                            pango_attr_list_insert (*pango_attrs, pango_attr0);

                            PangoAttribute *pango_attr1 = pango_attr_background_new (color_normal_text.red, color_normal_text.green, color_normal_text.blue);
                            pango_attr1->start_index = start_index;
                            pango_attr1->end_index = end_index;
                            pango_attr_list_insert (*pango_attrs, pango_attr1);
                        } else if (attr->value == SCIM_BRIDGE_ATTRIBUTE_DECORATE_HIGHLIGHT) {
                            PangoAttribute *pango_attr0 = pango_attr_foreground_new (color_active_text.red, color_active_text.green, color_active_text.blue);
                            pango_attr0->start_index = start_index;
                            pango_attr0->end_index = end_index;
                            pango_attr_list_insert (*pango_attrs, pango_attr0);

                            PangoAttribute *pango_attr1 = pango_attr_background_new (color_active_background.red, color_active_background.green, color_active_background.blue);
                            pango_attr1->start_index = start_index;
                            pango_attr1->end_index = end_index;
                            pango_attr_list_insert (*pango_attrs, pango_attr1);
                        } else {
                            scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 3, "Unknown preedit decoration!");
                        }
                    } else if (attr->type == SCIM_BRIDGE_ATTRIBUTE_FOREGROUND) {
                        const ScimBridgeAttributeValue color = attr->value;
                        const unsigned int red = scim_bridge_attribute_get_red (color) * 256;
                        const unsigned int green = scim_bridge_attribute_get_green (color) * 256;
                        const unsigned int blue = scim_bridge_attribute_get_blue (color) * 256;

                        PangoAttribute *pango_attr = pango_attr_foreground_new (red, green, blue);
                        pango_attr->start_index = start_index;
                        pango_attr->end_index = end_index;
                        pango_attr_list_insert (*pango_attrs, pango_attr);
                    } else if (attr->type == SCIM_BRIDGE_ATTRIBUTE_BACKGROUND) {
                        const ScimBridgeAttributeValue color = attr->value;
                        const unsigned int red = scim_bridge_attribute_get_red (color) * 256;
                        const unsigned int green = scim_bridge_attribute_get_green (color) * 256;
                        const unsigned int blue = scim_bridge_attribute_get_blue (color) * 256;

                        PangoAttribute *pango_attr = pango_attr_background_new (red, green, blue);
                        pango_attr->start_index = start_index;
                        pango_attr->end_index = end_index;
                        pango_attr_list_insert (*pango_attrs, pango_attr);
                    }

                }

            }
            /* If there is no underline at all, then draw underline under the whole preedit string.*/
            if (!underline_exists) {
                PangoAttribute *pango_attr = pango_attr_underline_new (PANGO_UNDERLINE_SINGLE);
                pango_attr->start_index = 0;
                pango_attr->end_index = ic_impl->preedit_string_length;
                pango_attr_list_insert (*pango_attrs, pango_attr);
            }
        }
    } else {
        if (str) *str = g_strdup ("");
        if (cursor_pos) *cursor_pos = 0;
        if (pango_attrs) *pango_attrs = pango_attr_list_new ();
    }
}


void scim_bridge_client_imcontext_focus_in (GtkIMContext *context)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_IMCONTEXT, 3,"scim_bridge_client_imcontext_focus_in");

    if (!initialized) return;

    if (scim_focused_imcontext) scim_bridge_client_imcontext_focus_out (GTK_IM_CONTEXT (scim_focused_imcontext->owner));

    ScimBridgeClientIMContext *ic = SCIM_BRIDGE_CLIENT_IMCONTEXT (context);

    if (ic && ic->impl) {
        scim_focused_imcontext = ic->impl;

        ScimBridgeException except;
        scim_bridge_exception_initialize (&except);
        if (scim_bridge_client_kernel_focus_changed (&except, (ScimBridgeIMContext*) scim_focused_imcontext, 1)) {
            display_exception (&except);
        }
        scim_bridge_exception_finalize (&except);
    }
}


void scim_bridge_client_imcontext_focus_out (GtkIMContext *context)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_IMCONTEXT, 3, "scim_bridge_client_imcontext_focus_out");

    g_idle_remove_by_data ((gpointer) context);

    if (!initialized) return;

    ScimBridgeClientIMContext *ic = SCIM_BRIDGE_CLIENT_IMCONTEXT (context);

    if (ic && ic->impl && scim_focused_imcontext == ic->impl) {
        ScimBridgeException except;
        scim_bridge_exception_initialize (&except);
        if (scim_bridge_client_kernel_focus_changed (&except, (ScimBridgeIMContext*) scim_focused_imcontext, 0)) display_exception (&except);
        scim_bridge_exception_finalize (&except);

        scim_focused_imcontext = NULL;
    }
}


void scim_bridge_client_imcontext_set_client_window (GtkIMContext *context, GdkWindow *new_window)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_IMCONTEXT, 3, "scim_bridge_client_imcontext_set_client_window");

    ScimBridgeClientIMContext *ic = SCIM_BRIDGE_CLIENT_IMCONTEXT (context);

    if (ic && ic->impl) {
        ScimBridgeClientIMContextImpl *ic_impl = ic->impl;

        if (ic_impl->client_window) g_object_unref (ic_impl->client_window);
        if (new_window) g_object_ref (new_window);

        ic_impl->client_window = new_window;
    }
}


void scim_bridge_client_imcontext_set_cursor_location (GtkIMContext *context, GdkRectangle *area)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_IMCONTEXT, 3, "scim_bridge_client_imcontext_set_cursor_location");
    ScimBridgeClientIMContext *ic = SCIM_BRIDGE_CLIENT_IMCONTEXT (context);

    if (ic && ic->impl) {
        ScimBridgeClientIMContextImpl *ic_impl = ic->impl;

        const int new_cursor_x = area->x + area->width;
        const int new_cursor_y = area->y + area->height + 8;

        if (ic_impl->client_window) {
            int new_window_x;
            int new_window_y;
            gdk_window_get_origin (ic_impl->client_window, &new_window_x, &new_window_y);

            if (ic_impl->window_x + ic_impl->cursor_x != new_window_x + new_cursor_x || ic_impl->window_y + ic_impl->cursor_y != new_cursor_y + new_window_y) {
                ic_impl->window_x = new_window_x;
                ic_impl->window_y = new_window_y;
                ic_impl->cursor_x = new_cursor_x;
                ic_impl->cursor_y = new_cursor_y;

                scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_IMCONTEXT, 1, "The cursor location is changed: x = %d + %d\ty = %d + %d",
                    ic_impl->window_x, ic_impl->cursor_x, ic_impl->window_y, ic_impl->cursor_y);

                ScimBridgeException except;
                scim_bridge_exception_initialize (&except);
                scim_bridge_client_kernel_cursor_location_changed (&except, (ScimBridgeIMContext*) ic_impl, ic_impl->window_x + ic_impl->cursor_x, ic_impl->window_y + ic_impl->cursor_y);
                scim_bridge_exception_finalize (&except);
            }
        }
    }
}


void scim_bridge_client_imcontext_set_use_preedit (GtkIMContext *context, gboolean use_preedit)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_IMCONTEXT, 3, "scim_bridge_client_imcontext_set_use_preedit");

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_IMCONTEXT, 9, "FIXME Not supported!");
}
