#include <alloca.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/types.h>

#include "scim-bridge-environment.h"
#include "scim-bridge-output.h"

/* Private variables */
static const char SOCKET_DIR[] = "/tmp";
static const char CLIENT_TO_AGENT_SOCKET_NAME[] = "scim-bridge-socket0";
static const char AGENT_TO_CLIENT_SOCKET_NAME[] = "scim-bridge-socket1";

static const char LOCKFILE_DIR[] = "/tmp";
static const char LOCKFILE_NAME[] = "scim-bridge-lockfile";

static int display_number = -1;
static int screen_number = -1;

static char *display_name = NULL;

/* Private function */
static void init_display_name ()
{
    if (display_name != NULL) return;

    const char *env_display = getenv ("DISPLAY");
    const char *env_display_number = getenv (SCIM_BRIDGE_ENV_NAME_DISPLAY_NUMBER);
    const char *env_screen_number = getenv (SCIM_BRIDGE_ENV_NAME_SCREEN_NUMBER);

    if (env_display_number != NULL || env_screen_number != NULL) {
        display_number = atoi (env_display_number);
        screen_number = atoi (env_screen_number);
    }

    if ((display_number < 0 || screen_number < 0) && env_display != NULL) {
        int has_colon = 0;
        int has_dot = 0;
        int error = 0;
        const char *c;
        for (c = env_display; !error && *c != '\0'; ++c) {
            if (*c == ':') {
                if (has_colon) {
                    break;
                } else {
                    has_colon = 1;
                    display_number = 0;
                    screen_number = 0;
                }
            } else if (has_colon) {
                switch (*c) {
                    case '0':
                    case '1':
                    case '2':
                    case '3':
                    case '4':
                    case '5':
                    case '6':
                    case '7':
                    case '8':
                    case '9':
                        if (!has_dot) display_number = display_number * 10 + atoi (c);
                        else screen_number = screen_number * 10 + atoi (c);
                        break;
                    case '.':
                        if (!has_dot) {
                            has_dot = 1;
                            break;
                        }
                        /* else no break */
                    default:
                        display_number = -1;
                        screen_number = -1;
                        error = 1;
                        break;
                }
            }
        }
    }

    if (display_number < 0 || screen_number < 0) {
        scim_bridge_perrorln ("Cannot get the display");
        abort ();
    }

    display_name = malloc (sizeof (char) * (1 + (display_number / 10) + 1 + 1 + (screen_number / 10) + 1 + 2)) + sizeof (char);
    sprintf (display_name, ":%d.%d", display_number, screen_number);
}


/* Implementations */
const char *scim_bridge_environment_get_lockfile_path ()
{
    static char *lockfile_path = NULL;

    if (lockfile_path == NULL) {
        init_display_name ();

        const uid_t uid = getuid ();
        const size_t str_len = strlen (LOCKFILE_DIR) + 1 + strlen (LOCKFILE_NAME)
            + 1 + ((size_t) (display_number / 10) + 1) + 1 + ((size_t) (screen_number / 10) + 1) + 1 + ((size_t) (uid / 10) + 1);

        lockfile_path = malloc (sizeof (char) * (str_len + 2)) + sizeof (char);
        sprintf (lockfile_path, "%s/%s-%d-%d-%d", LOCKFILE_DIR, LOCKFILE_NAME, display_number,screen_number, uid);
    }

    return lockfile_path;
}


const char *scim_bridge_environment_get_agent_to_client_socket_path ()
{
    static char *socket_path = NULL;

    if (socket_path == NULL) {
        init_display_name ();

        const uid_t uid = getuid ();
        const size_t str_len = strlen (SOCKET_DIR) + 1 + strlen (AGENT_TO_CLIENT_SOCKET_NAME)
            + 1 + ((size_t) (display_number / 10) + 1) + 1 + ((size_t) (screen_number / 10) + 1) + 1 + ((size_t) (uid / 10) + 1);

        socket_path = malloc (sizeof (char) * (str_len + 2)) + sizeof (char);
        sprintf (socket_path, "%s/%s-%d-%d-%d", SOCKET_DIR, AGENT_TO_CLIENT_SOCKET_NAME, display_number, screen_number, uid);
    }

    return socket_path;
}


const char *scim_bridge_environment_get_client_to_agent_socket_path ()
{
    static char *socket_path = NULL;

    if (socket_path == NULL) {
        init_display_name ();

        const uid_t uid = getuid ();
        const size_t str_len = strlen (SOCKET_DIR) + 1 + strlen (CLIENT_TO_AGENT_SOCKET_NAME)
            + 1 + ((size_t) (display_number / 10) + 1) + 1 + ((size_t) (screen_number / 10) + 1) + 1 + ((size_t) (uid / 10) + 1);

        socket_path = malloc (sizeof (char) * (str_len + 2)) + sizeof (char);
        sprintf (socket_path, "%s/%s-%d-%d-%d", SOCKET_DIR, CLIENT_TO_AGENT_SOCKET_NAME, display_number, screen_number, uid);
    }

    return socket_path;
}


const char *scim_bridge_environment_get_display_name ()
{
    init_display_name ();
    return display_name;
}


unsigned int scim_bridge_environment_get_display_number ()
{
    init_display_name ();
    return display_number;
}


unsigned int scim_bridge_environment_get_screen_number ()
{
    init_display_name ();
    return screen_number;
}


ScimBridgeDebugLevel scim_bridge_environment_get_debug_level ()
{
    static ScimBridgeDebugLevel debug_level = -1;

    if (debug_level == -1) {
        const char *env_debug_level = getenv (SCIM_BRIDGE_ENV_NAME_DEBUG_LEVEL);
        if (env_debug_level != NULL) debug_level = atoi (env_debug_level);
        if (debug_level == -1) debug_level = 0;
        if (debug_level > 9) debug_level = 9;
    }

    return debug_level;
}


ScimBridgeDebugFlag scim_bridge_environment_get_debug_flags ()
{
    static ScimBridgeDebugFlag debug_flags;
    static int first_run = 1;

    if (first_run) {
        first_run = 0;
        const char *env_debug_flags = getenv (SCIM_BRIDGE_ENV_NAME_DEBUG_FLAGS);
        if (env_debug_flags == NULL) {
            debug_flags = SCIM_BRIDGE_DEBUG_NONE;
            return debug_flags;
        }

        char *flags_str = alloca (sizeof (char) * (strlen (env_debug_flags) + 1));
        strcpy (flags_str, env_debug_flags);

        char *save_ptr;
        char *token = strtok_r (flags_str, " \t", &save_ptr);
        while (token != NULL) {
            if (strcasecmp (token, SCIM_BRIDGE_ENV_VALUE_DEBUG_FLAGS_NONE) == 0) {
                debug_flags = SCIM_BRIDGE_DEBUG_NONE;
            } else if (strcasecmp (token, SCIM_BRIDGE_ENV_VALUE_DEBUG_FLAGS_ALL) == 0) {
                debug_flags = SCIM_BRIDGE_DEBUG_ALL;
            } else if (strcasecmp (token, SCIM_BRIDGE_ENV_VALUE_DEBUG_FLAGS_MESSENGER) == 0) {
                debug_flags |= SCIM_BRIDGE_DEBUG_MESSENGER;
            } else if (strcasecmp (token, SCIM_BRIDGE_ENV_VALUE_DEBUG_FLAGS_IMCONTEXT) == 0) {
                debug_flags |= SCIM_BRIDGE_DEBUG_IMCONTEXT;
            } else if (strcasecmp (token, SCIM_BRIDGE_ENV_VALUE_DEBUG_FLAGS_CLIENT) == 0) {
                debug_flags |= SCIM_BRIDGE_DEBUG_CLIENT;
            } else if (strcasecmp (token, SCIM_BRIDGE_ENV_VALUE_DEBUG_FLAGS_AGENT) == 0) {
                debug_flags |= SCIM_BRIDGE_DEBUG_AGENT;
            } else if (strcasecmp (token, SCIM_BRIDGE_ENV_VALUE_DEBUG_FLAGS_SCIM) == 0) {
                debug_flags |= SCIM_BRIDGE_DEBUG_SCIM;
            }

            token = strtok_r (NULL, " \t", &save_ptr);
        }
    }

    return debug_flags;
}
