/*
 * SCIM Bridge
 *
 * Copyright (c) 2006 Ryo Dairiki <ryo-dairiki@users.sourceforge.net>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.*
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
 * GNU Lesser General Public License for more details.*
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 */

#include <errno.h>
#include <unistd.h>

#include <sys/select.h>
#include <sys/socket.h>

#include <list>

#define Uses_SCIM_ATTRIBUTES
#define Uses_SCIM_EVENT
#include <scim.h>

#include "scim-bridge-agent-client-listener.h"
#include "scim-bridge-agent-imcontext.h"
#include "scim-bridge-agent-output.h"
#include "scim-bridge-string.h"
#include "scim-bridge-messenger.h"

#include "event/scim-bridge-agent-alloc-imcontext-event.h"
#include "event/scim-bridge-agent-beep-event.h"
#include "event/scim-bridge-agent-client-closed-event.h"
#include "event/scim-bridge-agent-client-opened-event.h"
#include "event/scim-bridge-agent-commit-event.h"
#include "event/scim-bridge-agent-cursor-location-changed-event.h"
#include "event/scim-bridge-agent-filter-key-event.h"
#include "event/scim-bridge-agent-focus-changed-event.h"
#include "event/scim-bridge-agent-forward-key-event.h"
#include "event/scim-bridge-agent-free-imcontext-event.h"
#include "event/scim-bridge-agent-delete-surrounding-text-event.h"
#include "event/scim-bridge-agent-get-surrounding-text-event.h"
#include "event/scim-bridge-agent-imcontext-allocated-event.h"
#include "event/scim-bridge-agent-imcontext-freed-event.h"
#include "event/scim-bridge-agent-key-handled-event.h"
#include "event/scim-bridge-agent-set-preedit-enabled-event.h"
#include "event/scim-bridge-agent-set-preedit-shown-event.h"
#include "event/scim-bridge-agent-set-preedit-string-event.h"
#include "event/scim-bridge-agent-set-preedit-attributes-event.h"
#include "event/scim-bridge-agent-set-preedit-cursor-position-event.h"
#include "event/scim-bridge-agent-update-preedit-event.h"
#include "event/scim-bridge-agent-surrounding-text-gotten-event.h"
#include "event/scim-bridge-agent-surrounding-text-deleted-event.h"
#include "event/scim-bridge-agent-reset-imcontext-event.h"

using std::list;

using namespace scim;

/* Static variables */
KeyboardLayout scim_keyboard_layout;

/* Class Definition */
class ScimBridgeAgentClientListenerImpl: public ScimBridgeAgentClientListener
{

    public:

        ScimBridgeAgentClientListenerImpl (int new_socket_fd);
        ~ScimBridgeAgentClientListenerImpl ();

        size_t get_imcontext_count ();

        /* The following methods are semi-public */
        bool is_active ();
        void connection_lost ();

        void process_message (ScimBridgeMessage *message);

        ScimBridgeMessage *process_event ();

        ScimBridgeMessenger *get_messenger ();

    protected:

        void do_close_event_client ();

    private:

        pthread_t socket_writer_thread;
        pthread_t socket_reader_thread;

        ScimBridgeMessenger *messenger;

        list<ScimBridgeAgentIMContext*> imcontexts;

        /* Event Handlers */
        ScimBridgeMessage *received_imcontext_allocated_event (const ScimBridgeAgentIMContextAllocatedEvent *event);
        ScimBridgeMessage *received_imcontext_freed_event (const ScimBridgeAgentIMContextFreedEvent *event);
        ScimBridgeMessage *received_set_preedit_shown_event (const ScimBridgeAgentSetPreeditShownEvent *event);
        ScimBridgeMessage *received_set_preedit_cursor_position_event (const ScimBridgeAgentSetPreeditCursorPositionEvent* event);
        ScimBridgeMessage *received_set_preedit_string_event (const ScimBridgeAgentSetPreeditStringEvent* event);
        ScimBridgeMessage *received_set_preedit_attributes_event (const ScimBridgeAgentSetPreeditAttributesEvent* event);
        ScimBridgeMessage *received_update_preedit_event (const ScimBridgeAgentUpdatePreeditEvent* event);
        ScimBridgeMessage *received_commit_event (const ScimBridgeAgentCommitEvent* event);
        ScimBridgeMessage *received_delete_surrounding_text_event (const ScimBridgeAgentDeleteSurroundingTextEvent* event);
        ScimBridgeMessage *received_get_surrounding_text_event (const ScimBridgeAgentGetSurroundingTextEvent* event);
        ScimBridgeMessage *received_key_handled_event (const ScimBridgeAgentKeyHandledEvent *event);
        ScimBridgeMessage *received_forward_key_event (const ScimBridgeAgentForwardKeyEvent *event);
        ScimBridgeMessage *received_beep_event (const ScimBridgeAgentBeepEvent *event);

};

/* Helper functions */
static void *run_socket_reader (void *arg)
{
    ScimBridgeAgentClientListenerImpl *client_listener = static_cast<ScimBridgeAgentClientListenerImpl*> (arg);

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 7, "Client reader is ready.");

    while (client_listener->is_active ()) {
        ScimBridgeMessage *message;
        if (scim_bridge_messenger_receive (client_listener->get_messenger (), &message)) {
            if (client_listener->is_active ()) client_listener->connection_lost ();
            scim_bridge_free_message (message);
            break;
        } else {
            if (client_listener->is_active ()) client_listener->process_message (message);
            scim_bridge_free_message (message);
        }
    }

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 4, "Client reader has been gone.");
    return NULL;
}


static void *run_socket_writer (void *arg)
{
    ScimBridgeAgentClientListenerImpl *client_listener = static_cast<ScimBridgeAgentClientListenerImpl*> (arg);

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 7, "Client writer is ready.");

    while (client_listener->is_active ()) {
        ScimBridgeMessage *message = client_listener->process_event ();
        if (message == NULL) continue;
        if (scim_bridge_messenger_send (client_listener->get_messenger (), message)) {
            scim_bridge_free_message (message);
            break;
        } else {
            scim_bridge_free_message (message);
        }
    }

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 4, "Client writer has been gone.");
    return NULL;
}


/* Implementations */
ScimBridgeAgentClientListener *ScimBridgeAgentClientListener::alloc (int new_socket_fd)
{
    return new ScimBridgeAgentClientListenerImpl (new_socket_fd);
}


KeyboardLayout ScimBridgeAgentClientListener::get_keyboard_layout ()
{
    return scim_keyboard_layout;
}


void ScimBridgeAgentClientListener::set_keyboard_layout (KeyboardLayout new_layout)
{
    scim_keyboard_layout = new_layout;
}


ScimBridgeAgentClientListenerImpl::ScimBridgeAgentClientListenerImpl (int socket_fd)
{
    messenger = scim_bridge_alloc_messenger (socket_fd);

    if (pthread_create (&socket_writer_thread, NULL, run_socket_writer, this) || pthread_create (&socket_reader_thread, NULL, run_socket_reader, this)) {
        scim_bridge_psyslog (SYSLOG_CRITICAL, "Cannot invoke message IO threads: %s", strerror (errno));
        abort ();
    }
}


ScimBridgeAgentClientListenerImpl::~ScimBridgeAgentClientListenerImpl ()
{
    scim_bridge_free_messenger (messenger);

    for (list<ScimBridgeAgentIMContext*>::iterator i = imcontexts.begin (); i != imcontexts.end (); ++i) {
        ScimBridgeAgentIMContext* imcontext = *i;
        imcontext->free_partially ();
        delete imcontext;
    }
    imcontexts.clear ();
}


bool ScimBridgeAgentClientListenerImpl::is_active ()
{
    return get_event_client_id () != -2 && !scim_bridge_messenger_is_closed (messenger);
}


ScimBridgeMessenger *ScimBridgeAgentClientListenerImpl::get_messenger ()
{
    return messenger;
}


void ScimBridgeAgentClientListenerImpl::process_message (ScimBridgeMessage *message)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 6, "process_message");

    ScimBridgeAgentEvent *event = NULL;

    const char *message_header = scim_bridge_message_get_header (message);
    if (strcmp (message_header, SCIM_BRIDGE_MESSAGE_ALLOC_IMCONTEXT) == 0) {
        event = new ScimBridgeAgentAllocIMContextEvent (this);
    } else if (strcmp (message_header, SCIM_BRIDGE_MESSAGE_FREE_IMCONTEXT) == 0) {
        const char *imcontext_id_str = scim_bridge_message_get_argument (message, 0);
        unsigned int imcontext_id;
        if (scim_bridge_string_to_uint (&imcontext_id, imcontext_id_str)) {
            scim_bridge_psyslogln (SYSLOG_ERROR, "Invalid message: free_imcontext (%s)", imcontext_id_str);
        } else {
            event = new ScimBridgeAgentFreeIMContextEvent (imcontext_id, this);
        }
    } else if (strcmp (message_header, SCIM_BRIDGE_MESSAGE_FOCUS_CHANGED) == 0) {
        const char *imcontext_id_str = scim_bridge_message_get_argument (message, 0);
        const char *focus_in_str = scim_bridge_message_get_argument (message, 1);
        unsigned int imcontext_id;
        boolean focus_in;
        if (scim_bridge_string_to_uint (&imcontext_id, imcontext_id_str) || scim_bridge_string_to_boolean (&focus_in, focus_in_str)) {
            scim_bridge_psyslogln (SYSLOG_ERROR, "Invalid message: focus_changed (%s, %s)", imcontext_id_str, focus_in_str);
        } else {
            event = new ScimBridgeAgentFocusChangedEvent (imcontext_id, focus_in, this);
        }
    } else if (strcmp (message_header, SCIM_BRIDGE_MESSAGE_RESET_IMCONTEXT) == 0) {
        const char *imcontext_id_str = scim_bridge_message_get_argument (message, 0);

        unsigned int imcontext_id;
        if (scim_bridge_string_to_uint (&imcontext_id, imcontext_id_str)) {
            scim_bridge_psyslogln (SYSLOG_ERROR, "Invalid message: reset_changed (%s)", imcontext_id_str);
        } else {
            event = new ScimBridgeAgentResetIMContextEvent (imcontext_id, this);
        }
    } else if (strcmp (message_header, SCIM_BRIDGE_MESSAGE_KEY_EVENT_OCCURED) == 0) {
        const char *imcontext_id_str = scim_bridge_message_get_argument (message, 0);
        const char *key_code_str = scim_bridge_message_get_argument (message, 1);
        const char *key_pressed_str = scim_bridge_message_get_argument (message, 2);
        unsigned int imcontext_id;
        unsigned int key_code;
        boolean key_pressed;
        if (scim_bridge_string_to_uint (&imcontext_id, imcontext_id_str) || scim_bridge_string_to_uint (&key_code, key_code_str)
            || scim_bridge_string_to_boolean (&key_pressed, key_pressed_str)) {
            scim_bridge_perrorln ("Invalid message: key_event_occured (%s, %s, %s,...)", imcontext_id_str, key_code_str, key_pressed_str);

            KeyEvent key_event (SCIM_KEY_NullKey, scim_keyboard_layout, SCIM_KEY_NullMask);
            event = new ScimBridgeAgentFilterKeyEvent (imcontext_id, key_event, this);
        } else {
            unsigned int modifiers;
            if (key_pressed) {
                modifiers = SCIM_KEY_NullMask;
            } else {
                modifiers = SCIM_KEY_ReleaseMask;
            }

            for (size_t j = 3; j < scim_bridge_message_get_argument_count (message); ++j) {
                const char *modifier_str = scim_bridge_message_get_argument (message, j);

                if (strcmp (modifier_str, SCIM_BRIDGE_MESSAGE_SHIFT) == 0) {
                    modifiers |= SCIM_KEY_ShiftMask;
                } else if (strcmp (modifier_str, SCIM_BRIDGE_MESSAGE_CONTROL) == 0) {
                    modifiers |= SCIM_KEY_ControlMask;
                } else if (strcmp (modifier_str, SCIM_BRIDGE_MESSAGE_ALT) == 0) {
                    modifiers |= SCIM_KEY_AltMask;
                } else if (strcmp (modifier_str, SCIM_BRIDGE_MESSAGE_META) == 0) {
                    modifiers |= SCIM_KEY_MetaMask;
                } else if (strcmp (modifier_str, SCIM_BRIDGE_MESSAGE_SUPER) == 0) {
                    modifiers |= SCIM_KEY_SuperMask;
                } else if (strcmp (modifier_str, SCIM_BRIDGE_MESSAGE_HYPER) == 0) {
                    modifiers |= SCIM_KEY_HyperMask;
                } else if (strcmp (modifier_str, SCIM_BRIDGE_MESSAGE_CAPS_LOCK) == 0) {
                    modifiers |= SCIM_KEY_CapsLockMask;
                } else if (strcmp (modifier_str, SCIM_BRIDGE_MESSAGE_NUM_LOCK) == 0) {
                    modifiers |= SCIM_KEY_NumLockMask;
                } else {
                    scim_bridge_perrorln ("Unknown modifier: %s", modifier_str);
                }
            }

            KeyEvent key_event (key_code, modifiers, scim_keyboard_layout);
            event = new ScimBridgeAgentFilterKeyEvent (imcontext_id, key_event, this);
        }
    } else if (strcmp (message_header, SCIM_BRIDGE_MESSAGE_CURSOR_LOCATION_CHANGED) == 0) {
        const char *imcontext_id_str = scim_bridge_message_get_argument (message, 0);
        const char *cursor_x_str = scim_bridge_message_get_argument (message, 1);
        const char *cursor_y_str = scim_bridge_message_get_argument (message, 2);
        unsigned int imcontext_id;
        int cursor_x;
        int cursor_y;
        if (scim_bridge_string_to_uint (&imcontext_id, imcontext_id_str) || scim_bridge_string_to_int (&cursor_x, cursor_x_str)
            || scim_bridge_string_to_int (&cursor_y, cursor_y_str)) {
            scim_bridge_perrorln ("Invalid message: cursor_location_changed (%s, %s, %s,...)", imcontext_id_str, cursor_x_str, cursor_y_str);
        } else {
            event = new ScimBridgeAgentCursorLocationChangedEvent (imcontext_id, cursor_x, cursor_y, this);
        }
    } else if (strcmp (message_header, SCIM_BRIDGE_MESSAGE_SURROUNDING_TEXT_GOTTEN) == 0) {
        const char *response_id_str = scim_bridge_message_get_argument (message, 0);
        const char *succeeded_str = scim_bridge_message_get_argument (message, 1);
        const char *cursor_position_str = scim_bridge_message_get_argument (message, 2);

        boolean succeeded;
        unsigned int response_id;
        unsigned int cursor_position;
        if (scim_bridge_string_to_uint (&response_id, response_id_str) || scim_bridge_string_to_boolean (&succeeded, succeeded_str) || (cursor_position_str != NULL && scim_bridge_string_to_uint (&cursor_position, cursor_position_str))) {
            scim_bridge_psyslogln (SYSLOG_ERROR, "Invalid message: surrounding_text_gotten (%s, %s,...)", response_id_str, succeeded_str);
        } else {

            if (succeeded) {
                wchar *wstr;
                scim_bridge_string_to_wstring (&wstr, scim_bridge_message_get_argument (message, 3));
                push_response (new ScimBridgeAgentSurroundingTextGottenEvent (response_id, WideString (wstr), cursor_position, succeeded));
                free (wstr);
            } else {
                push_response (new ScimBridgeAgentSurroundingTextGottenEvent (response_id, L"", 0, succeeded));
            }
        }
    } else if (strcmp (message_header, SCIM_BRIDGE_MESSAGE_SURROUNDING_TEXT_DELETED) == 0) {
        const char *response_id_str = scim_bridge_message_get_argument (message, 0);
        const char *succeed_str = scim_bridge_message_get_argument (message, 1);
        const char *need_reset_str = scim_bridge_message_get_argument (message, 2);

        boolean succeed;
        boolean need_reset;
        unsigned int response_id;
        if (scim_bridge_string_to_uint (&response_id, response_id_str) || scim_bridge_string_to_boolean (&succeed, succeed_str) || scim_bridge_string_to_boolean (&need_reset, need_reset_str)) {
            scim_bridge_psyslogln (SYSLOG_ERROR, "Invalid message: surrounding_text_deleted (%s, %s, %s)", response_id_str, succeed_str, need_reset_str);
        } else {
            push_response (new ScimBridgeAgentSurroundingTextDeletedEvent (response_id, succeed, need_reset));
        }
    } else if (strcmp (message_header, SCIM_BRIDGE_MESSAGE_SET_PREEDIT_ENABLED) == 0) {
        const char *imcontext_id_str = scim_bridge_message_get_argument (message, 0);
        const char *enabled_str = scim_bridge_message_get_argument (message, 1);

        boolean enabled;
        unsigned int imcontext_id;
        if (scim_bridge_string_to_uint (&imcontext_id, imcontext_id_str) || scim_bridge_string_to_boolean (&enabled, enabled_str)) {
            scim_bridge_psyslogln (SYSLOG_ERROR, "Invalid message: set_preedit_enabled (%s, %s)", imcontext_id_str, enabled_str);
        } else {
            event = new ScimBridgeAgentSetPreeditEnabledEvent (imcontext_id, enabled, this);
        }
    } else {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 7, "Unknown message: %s", message_header);
    }

    if (event != NULL) get_event_server ()->push_event (event);
}


ScimBridgeMessage *ScimBridgeAgentClientListenerImpl::process_event ()
{
    const ScimBridgeAgentEvent *event = poll_event ();
    ScimBridgeMessage *message = NULL;
    if (event != NULL) {
        switch (event->get_code ()) {
            case IMCONTEXT_ALLOCATED_EVENT:
                message = received_imcontext_allocated_event (static_cast<const ScimBridgeAgentIMContextAllocatedEvent*> (event));
                break;
            case IMCONTEXT_FREED_EVENT:
                message = received_imcontext_freed_event (static_cast<const ScimBridgeAgentIMContextFreedEvent*> (event));
                break;
            case KEY_HANDLED_EVENT:
                message = received_key_handled_event (static_cast<const ScimBridgeAgentKeyHandledEvent*> (event));
                break;
            case SET_PREEDIT_SHOWN_EVENT:
                message = received_set_preedit_shown_event (static_cast<const ScimBridgeAgentSetPreeditShownEvent*> (event));
                break;
            case SET_PREEDIT_CURSOR_POSITION_EVENT:
                message = received_set_preedit_cursor_position_event (static_cast<const ScimBridgeAgentSetPreeditCursorPositionEvent*> (event));
                break;
            case SET_PREEDIT_STRING_EVENT:
                message = received_set_preedit_string_event (static_cast<const ScimBridgeAgentSetPreeditStringEvent*> (event));
                break;
            case SET_PREEDIT_ATTRIBUTES_EVENT:
                message = received_set_preedit_attributes_event (static_cast<const ScimBridgeAgentSetPreeditAttributesEvent*> (event));
                break;
            case UPDATE_PREEDIT_EVENT:
                message = received_update_preedit_event (static_cast<const ScimBridgeAgentUpdatePreeditEvent*> (event));
                break;
            case COMMIT_EVENT:
                message = received_commit_event (static_cast<const ScimBridgeAgentCommitEvent*> (event));
                break;
            case DELETE_SURROUNDING_TEXT_EVENT:
                message = received_delete_surrounding_text_event (static_cast<const ScimBridgeAgentDeleteSurroundingTextEvent*> (event));
                break;
            case GET_SURROUNDING_TEXT_EVENT:
                message = received_get_surrounding_text_event (static_cast<const ScimBridgeAgentGetSurroundingTextEvent*> (event));
                break;
            case BEEP_EVENT:
                message = received_beep_event (static_cast<const ScimBridgeAgentBeepEvent*> (event));
                break;
            case FORWARD_KEY_EVENT:
                message = received_forward_key_event (static_cast<const ScimBridgeAgentForwardKeyEvent*> (event));
                break;
            default:
                break;
        }
    }

    delete event;
    return message;
}


void ScimBridgeAgentClientListenerImpl::connection_lost ()
{
    get_event_server ()->push_event (new ScimBridgeAgentClientClosedEvent (this));
}


void ScimBridgeAgentClientListenerImpl::do_close_event_client ()
{
    scim_bridge_messenger_close (messenger);

    pthread_kill (socket_writer_thread, SIGNOTIFY);
    pthread_join (socket_writer_thread, NULL);

    pthread_kill (socket_reader_thread, SIGNOTIFY);
    pthread_join (socket_reader_thread, NULL);
}


/* Event Handlers */
ScimBridgeMessage *ScimBridgeAgentClientListenerImpl::received_imcontext_allocated_event (const ScimBridgeAgentIMContextAllocatedEvent *event)
{
    imcontexts.push_back (ScimBridgeAgentIMContext::find (event->get_imcontext_id ()));

    ScimBridgeMessage *message = scim_bridge_alloc_message (SCIM_BRIDGE_MESSAGE_IMCONTEXT_ALLOCATED, 1);

    char *imcontext_id_str;
    scim_bridge_string_from_uint (&imcontext_id_str, event->get_imcontext_id ());
    scim_bridge_message_set_argument (message, 0, imcontext_id_str);
    free (imcontext_id_str);

    return message;
}


ScimBridgeMessage *ScimBridgeAgentClientListenerImpl::received_imcontext_freed_event (const ScimBridgeAgentIMContextFreedEvent *event)
{
    for (list<ScimBridgeAgentIMContext*>::iterator i = imcontexts.begin (); i != imcontexts.end (); ++i) {
        ScimBridgeAgentIMContext *imcontext = *i;
        if (imcontext->get_id () == event->get_imcontext_id ()) {
            imcontexts.erase (i);
            delete imcontext;
            break;
        }
    }

    ScimBridgeMessage *message = scim_bridge_alloc_message (SCIM_BRIDGE_MESSAGE_IMCONTEXT_FREED, 1);

    char *imcontext_id_str;
    scim_bridge_string_from_uint (&imcontext_id_str, event->get_imcontext_id ());
    scim_bridge_message_set_argument (message, 0, imcontext_id_str);
    free (imcontext_id_str);

    return message;
}


ScimBridgeMessage *ScimBridgeAgentClientListenerImpl::received_set_preedit_shown_event (const ScimBridgeAgentSetPreeditShownEvent *event)
{
    ScimBridgeMessage *message = scim_bridge_alloc_message (SCIM_BRIDGE_MESSAGE_SET_PREEDIT_SHOWN, 2);

    char *imcontext_id_str;
    scim_bridge_string_from_uint (&imcontext_id_str, event->get_imcontext_id ());
    scim_bridge_message_set_argument (message, 0, imcontext_id_str);
    free (imcontext_id_str);

    char *shown_str;
    scim_bridge_string_from_boolean (&shown_str, event->is_shown ());
    scim_bridge_message_set_argument (message, 1, shown_str);
    free (shown_str);

    return message;
}


ScimBridgeMessage *ScimBridgeAgentClientListenerImpl::received_set_preedit_cursor_position_event (const ScimBridgeAgentSetPreeditCursorPositionEvent* event)
{
    ScimBridgeMessage *message = scim_bridge_alloc_message (SCIM_BRIDGE_MESSAGE_SET_PREEDIT_CURSOR_POSITION, 2);

    char *imcontext_id_str;
    scim_bridge_string_from_uint (&imcontext_id_str, event->get_imcontext_id ());
    scim_bridge_message_set_argument (message, 0, imcontext_id_str);
    free (imcontext_id_str);

    char *cursor_position_str;
    scim_bridge_string_from_uint (&cursor_position_str, event->get_cursor_position ());
    scim_bridge_message_set_argument (message, 1, cursor_position_str);
    free (cursor_position_str);

    return message;
}


ScimBridgeMessage *ScimBridgeAgentClientListenerImpl::received_set_preedit_string_event (const ScimBridgeAgentSetPreeditStringEvent* event)
{
    ScimBridgeMessage *message = scim_bridge_alloc_message (SCIM_BRIDGE_MESSAGE_SET_PREEDIT_STRING, 2);

    char *imcontext_id_str;
    scim_bridge_string_from_uint (&imcontext_id_str, event->get_imcontext_id ());
    scim_bridge_message_set_argument (message, 0, imcontext_id_str);
    free (imcontext_id_str);

    char *str;
    scim_bridge_wstring_to_string (&str, event->get_preedit_string ().c_str ());
    scim_bridge_message_set_argument (message, 1, str);
    free (str);

    return message;
}


ScimBridgeMessage *ScimBridgeAgentClientListenerImpl::received_set_preedit_attributes_event (const ScimBridgeAgentSetPreeditAttributesEvent* event)
{
    AttributeList *attributes = &event->get_preedit_attributes ();
    ScimBridgeMessage *message = scim_bridge_alloc_message (SCIM_BRIDGE_MESSAGE_SET_PREEDIT_ATTRIBUTES, attributes->size () * 4 + 1);

    char *imcontext_id_str;
    scim_bridge_string_from_uint (&imcontext_id_str, event->get_imcontext_id ());
    scim_bridge_message_set_argument (message, 0, imcontext_id_str);
    free (imcontext_id_str);

    int arg_index = 1;
    for (AttributeList::iterator i = attributes->begin (); i != attributes->end (); ++i) {
        Attribute &attribute = *i;

        char *begin_str;
        char *end_str;
        scim_bridge_string_from_uint (&begin_str, attribute.get_start ());
        scim_bridge_string_from_uint (&end_str, attribute.get_end ());

        const char *type_str;
        const char *value_str;

        if (attribute.get_type () == SCIM_ATTR_DECORATE) {
            type_str = SCIM_BRIDGE_MESSAGE_DECORATE;
            switch (attribute.get_value ()) {
                case SCIM_ATTR_DECORATE_UNDERLINE:
                    value_str = SCIM_BRIDGE_MESSAGE_UNDERLINE;
                    break;
                case SCIM_ATTR_DECORATE_REVERSE:
                    value_str = SCIM_BRIDGE_MESSAGE_REVERSE;
                    break;
                case SCIM_ATTR_DECORATE_HIGHLIGHT:
                    value_str = SCIM_BRIDGE_MESSAGE_HIGHLIGHT;
                    break;
                default:
                    type_str = SCIM_BRIDGE_MESSAGE_NONE;
                    value_str = SCIM_BRIDGE_MESSAGE_NONE;
            }
        } else if (attribute.get_type () == SCIM_ATTR_FOREGROUND || attribute.get_type () == SCIM_ATTR_BACKGROUND) {
            if (attribute.get_type () == SCIM_ATTR_FOREGROUND) {
                type_str = SCIM_BRIDGE_MESSAGE_FOREGROUND;
            } else {
                type_str = SCIM_BRIDGE_MESSAGE_BACKGROUND;
            }
            char *tmp_str = static_cast<char*> (alloca (sizeof (char) * (strlen (SCIM_BRIDGE_MESSAGE_COLOR) + 7)));
            value_str = tmp_str;
            strcpy (tmp_str, SCIM_BRIDGE_MESSAGE_COLOR);
            char *color_str = tmp_str + sizeof (char) * strlen (SCIM_BRIDGE_MESSAGE_COLOR);
            color_str[6] = '\0';
            const char chars[] = {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};
            for (unsigned int k = 0; k < 6; ++k) {
                const unsigned int digit_index = (attribute.get_value () >> (k * 4)) % 0x10;
                color_str[k] = chars[digit_index];
            }
        } else {
            type_str = SCIM_BRIDGE_MESSAGE_NONE;
            value_str = SCIM_BRIDGE_MESSAGE_NONE;
        }

        scim_bridge_message_set_argument (message, arg_index + 0, begin_str);
        scim_bridge_message_set_argument (message, arg_index + 1, end_str);
        scim_bridge_message_set_argument (message, arg_index + 2, type_str);
        scim_bridge_message_set_argument (message, arg_index + 3, value_str);

        free (begin_str);
        free (end_str);

        arg_index += 4;
    }

    return message;
}


ScimBridgeMessage *ScimBridgeAgentClientListenerImpl::received_update_preedit_event (const ScimBridgeAgentUpdatePreeditEvent* event)
{
    ScimBridgeMessage *message = scim_bridge_alloc_message (SCIM_BRIDGE_MESSAGE_UPDATE_PREEDIT, 1);

    char *imcontext_id_str;
    scim_bridge_string_from_uint (&imcontext_id_str, event->get_imcontext_id ());
    scim_bridge_message_set_argument (message, 0, imcontext_id_str);
    free (imcontext_id_str);

    return message;
}


ScimBridgeMessage *ScimBridgeAgentClientListenerImpl::received_commit_event (const ScimBridgeAgentCommitEvent* event)
{
    ScimBridgeMessage *message = scim_bridge_alloc_message (SCIM_BRIDGE_MESSAGE_COMMIT, 2);

    char *imcontext_id_str;
    scim_bridge_string_from_uint (&imcontext_id_str, event->get_imcontext_id ());
    scim_bridge_message_set_argument (message, 0, imcontext_id_str);
    free (imcontext_id_str);

    char *str;
    scim_bridge_wstring_to_string (&str, event->get_commit_string ().c_str ());
    scim_bridge_message_set_argument (message, 1, str);
    free (str);

    return message;
}


ScimBridgeMessage *ScimBridgeAgentClientListenerImpl::received_delete_surrounding_text_event (const ScimBridgeAgentDeleteSurroundingTextEvent* event)
{
    ScimBridgeMessage *message = scim_bridge_alloc_message (SCIM_BRIDGE_MESSAGE_DELETE_SURROUNDING_TEXT, 4);

    char *response_id_str;
    scim_bridge_string_from_int (&response_id_str, event->get_response_id ());
    scim_bridge_message_set_argument (message, 0, response_id_str);
    free (response_id_str);

    char *imcontext_id_str;
    scim_bridge_string_from_uint (&imcontext_id_str, event->get_imcontext_id ());
    scim_bridge_message_set_argument (message, 1, imcontext_id_str);
    free (imcontext_id_str);

    char *offset_str;
    scim_bridge_string_from_uint (&offset_str, event->get_offset ());
    scim_bridge_message_set_argument (message, 2, offset_str);
    free (offset_str);

    char *length_str;
    scim_bridge_string_from_uint (&length_str, event->get_length ());
    scim_bridge_message_set_argument (message, 3, length_str);
    free (length_str);

    return message;
}


ScimBridgeMessage *ScimBridgeAgentClientListenerImpl::received_get_surrounding_text_event (const ScimBridgeAgentGetSurroundingTextEvent* event)
{
    ScimBridgeMessage *message = scim_bridge_alloc_message (SCIM_BRIDGE_MESSAGE_GET_SURROUNDING_TEXT, 4);

    char *response_id_str;
    scim_bridge_string_from_int (&response_id_str, event->get_response_id ());
    scim_bridge_message_set_argument (message, 0, response_id_str);
    free (response_id_str);

    char *imcontext_id_str;
    scim_bridge_string_from_uint (&imcontext_id_str, event->get_imcontext_id ());
    scim_bridge_message_set_argument (message, 1, imcontext_id_str);
    free (imcontext_id_str);

    char *max_before_str;
    scim_bridge_string_from_uint (&max_before_str, event->get_max_before ());
    scim_bridge_message_set_argument (message, 2, max_before_str);
    free (max_before_str);

    char *max_after_str;
    scim_bridge_string_from_uint (&max_after_str, event->get_max_after ());
    scim_bridge_message_set_argument (message, 3, max_after_str);
    free (max_after_str);

    return message;
}


ScimBridgeMessage *ScimBridgeAgentClientListenerImpl::received_key_handled_event (const ScimBridgeAgentKeyHandledEvent *event)
{
    ScimBridgeMessage *message = scim_bridge_alloc_message (SCIM_BRIDGE_MESSAGE_KEY_EVENT_HANDLED, 1);

    char *consumed_str;
    scim_bridge_string_from_boolean (&consumed_str, event->is_consumed ());
    scim_bridge_message_set_argument (message, 0, consumed_str);
    free (consumed_str);

    return message;
}


ScimBridgeMessage *ScimBridgeAgentClientListenerImpl::received_beep_event (const ScimBridgeAgentBeepEvent *event)
{
    ScimBridgeMessage *message = scim_bridge_alloc_message (SCIM_BRIDGE_MESSAGE_BEEP, 1);

    char *imcontext_id_str;
    scim_bridge_string_from_int (&imcontext_id_str, event->get_imcontext_id ());
    scim_bridge_message_set_argument (message, 0, imcontext_id_str);

    free (imcontext_id_str);

    return message;
}


ScimBridgeMessage *ScimBridgeAgentClientListenerImpl::received_forward_key_event (const ScimBridgeAgentForwardKeyEvent *event)
{
    const KeyEvent &key_event = event->get_key_event ();

    size_t mod_count = 0;
    if (key_event.is_shift_down ()) ++mod_count;
    if (key_event.is_control_down ()) ++mod_count;
    if (key_event.is_alt_down ()) ++mod_count;
    if (key_event.is_meta_down ()) ++mod_count;
    if (key_event.is_hyper_down ()) ++mod_count;
    if (key_event.is_super_down ()) ++mod_count;
    if (key_event.is_caps_lock_down ()) ++mod_count;
    if (key_event.is_num_lock_down ()) ++mod_count;

    char *imcontext_id_str;
    scim_bridge_string_from_int (&imcontext_id_str, event->get_imcontext_id ());

    char* pressed_str;
    scim_bridge_string_from_boolean (&pressed_str, key_event.is_key_press ());

    char *key_code_str;
    scim_bridge_string_from_uint (&key_code_str, key_event.code);

    ScimBridgeMessage *message = scim_bridge_alloc_message (SCIM_BRIDGE_MESSAGE_FORWARD_KEY_EVENT, mod_count + 3);

    scim_bridge_message_set_argument (message, 0, imcontext_id_str);
    scim_bridge_message_set_argument (message, 1, key_code_str);
    scim_bridge_message_set_argument (message, 2, pressed_str);

    size_t mod_index = 0;
    if (key_event.is_shift_down ()) {
        scim_bridge_message_set_argument (message, mod_index, SCIM_BRIDGE_MESSAGE_SHIFT);
        ++mod_index;
    }
    if (key_event.is_control_down ()) {
        scim_bridge_message_set_argument (message, mod_index, SCIM_BRIDGE_MESSAGE_CONTROL);
        ++mod_index;
    }
    if (key_event.is_alt_down ()) {
        scim_bridge_message_set_argument (message, mod_index, SCIM_BRIDGE_MESSAGE_ALT);
        ++mod_index;
    }
    if (key_event.is_meta_down ()) {
        scim_bridge_message_set_argument (message, mod_index, SCIM_BRIDGE_MESSAGE_META);
        ++mod_index;
    }
    if (key_event.is_hyper_down ()) {
        scim_bridge_message_set_argument (message, mod_index, SCIM_BRIDGE_MESSAGE_HYPER);
        ++mod_index;
    }
    if (key_event.is_super_down ()) {
        scim_bridge_message_set_argument (message, mod_index, SCIM_BRIDGE_MESSAGE_SUPER);
        ++mod_index;
    }
    if (key_event.is_caps_lock_down ()) {
        scim_bridge_message_set_argument (message, mod_index, SCIM_BRIDGE_MESSAGE_CAPS_LOCK);
        ++mod_index;
    }
    if (key_event.is_num_lock_down ()) {
        scim_bridge_message_set_argument (message, mod_index, SCIM_BRIDGE_MESSAGE_NUM_LOCK);
        ++mod_index;
    }

    free (imcontext_id_str);
    free (key_code_str);
    free (pressed_str);

    return message;
}
