/*
 * SCIM Bridge
 *
 * Copyright (c) 2006 Ryo Dairiki <ryo-dairiki@users.sourceforge.net>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.*
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
 * GNU Lesser General Public License for more details.*
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 */

#include <assert.h>

#include "scim-bridge-agent-event.h"
#include "scim-bridge-agent-event-client.h"

using std::list;

/* Implementations */
ScimBridgeAgentEventClient::ScimBridgeAgentEventClient ():
event_server (NULL), event_client_id (-1)
{
    pthread_mutex_init (&mutex, NULL);
    sem_init (&semaphore, 0, 0);
}


ScimBridgeAgentEventClient::~ScimBridgeAgentEventClient ()
{
    for (list<const ScimBridgeAgentEvent*>::iterator i = events.begin (); i != events.end (); ++i) {
        const ScimBridgeAgentEvent *event = *i;
        delete event;
    }

    for (list<const ScimBridgeAgentEvent*>::iterator i = responses.begin (); i != responses.end (); ++i) {
        const ScimBridgeAgentEvent *response = *i;
        delete response;
    }

    pthread_mutex_destroy (&mutex);
    sem_post (&semaphore);
    sem_destroy (&semaphore);
}


void ScimBridgeAgentEventClient::close_event_client ()
{
    if (event_client_id < 0) return;

    pthread_mutex_lock (&mutex);
    event_server = NULL;
    event_client_id = -2;
    sem_post (&semaphore);
    pthread_mutex_unlock (&mutex);
    do_close_event_client ();
}


void ScimBridgeAgentEventClient::push_event (const ScimBridgeAgentEvent *event)
{
    if (event_client_id < 0 || event == NULL) return;

    pthread_mutex_lock (&mutex);
    events.push_back (event);
    int sem_value;
    sem_getvalue (&semaphore, &sem_value);
    if (sem_value == 0) sem_post (&semaphore);
    pthread_mutex_unlock (&mutex);
}


const ScimBridgeAgentEvent *ScimBridgeAgentEventClient::poll_response ()
{
    if (event_client_id < 0 || event_server == NULL) return NULL;

    pthread_mutex_lock (&mutex);
    if (responses.empty ()) {
        pthread_mutex_unlock (&mutex);
        return NULL;
    } else {
        const ScimBridgeAgentEvent *response = responses.front ();
        responses.pop_front ();
        pthread_mutex_unlock (&mutex);
        return response;
    }
}


const ScimBridgeAgentEvent *ScimBridgeAgentEventClient::poll_event ()
{
    pthread_mutex_lock (&mutex);
    while (event_client_id >= 0) {
        if (events.empty ()) {
            int sem_value;
            do {
                sem_trywait (&semaphore);
                sem_getvalue (&semaphore, &sem_value);
            } while (sem_value > 0);
            pthread_mutex_unlock (&mutex);
            sem_wait (&semaphore);
            pthread_mutex_lock (&mutex);
        } else {
            const ScimBridgeAgentEvent *event = events.front ();
            events.pop_front ();
            pthread_mutex_unlock (&mutex);
            return event;
        }
    }

    pthread_mutex_unlock (&mutex);
    return NULL;
}


void ScimBridgeAgentEventClient::push_response (const ScimBridgeAgentEvent *event)
{
    if (event_client_id < 0 || event == NULL) return;

    pthread_mutex_lock (&mutex);
    responses.push_back (event);
    pthread_mutex_unlock (&mutex);
}


ScimBridgeAgentEventServer *ScimBridgeAgentEventClient::get_event_server ()
{
    return event_server;
}


void ScimBridgeAgentEventClient::set_event_server (ScimBridgeAgentEventServer *server)
{
    event_server = server;
}


scim_bridge_agent_event_client_id_t ScimBridgeAgentEventClient::get_event_client_id () const
{
    return event_client_id;
}


void ScimBridgeAgentEventClient::set_event_client_id (scim_bridge_agent_event_client_id_t id)
{
    event_client_id = id;
}
