/*
 * SCIM Bridge
 *
 * Copyright (c) 2006 Ryo Dairiki <ryo-dairiki@users.sourceforge.net>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.*
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
 * GNU Lesser General Public License for more details.*
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 */

/**
 * @file
 * @author Ryo Dairiki <ryo-dairiki@users.sourceforge.net>
 * @brief This header describes about key events.
 */
#ifndef SCIM_BRIDGE_KEY_EVENT_H_
#define SCIM_BRIDGE_KEY_EVENT_H_

#include "scim-bridge.h"

/**
 * This is the type of key codes.
 */
typedef unsigned int scim_bridge_key_code_t;

/**
 * This is the type of modifiers.
 */
typedef unsigned int scim_bridge_key_modifier_t;

/**
 * This is the data type of KeyEvent.
 */
typedef struct _ScimBridgeKeyEvent ScimBridgeKeyEvent;

/* All the key_codes are exactly the same as the scim key codes */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_NullKey = 0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_VoidSymbol = 0xFFFFFF;

/* function keys */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_BackSpace = 0xFF08;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Tab = 0xFF09;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Linefeed = 0xFF0A;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Clear = 0xFF0B;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Return = 0xFF0D;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pause = 0xFF13;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Scroll_Lock = 0xFF14;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Sys_Req = 0xFF15;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Escape = 0xFF1B;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Delete = 0xFFFF;

/* International & multi-key character composition */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Multi_key = 0xFF20;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Codeinput = 0xFF37;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_SingleCandidate = 0xFF3C;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_MultipleCandidate = 0xFF3D;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_PreviousCandidate = 0xFF3E;

/* Japanese keyboard support */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Kanji = 0xFF21;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Muhenkan = 0xFF22;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Henkan_Mode = 0xFF23;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Henkan = 0xFF23;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Romaji = 0xFF24;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hiragana = 0xFF25;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Katakana = 0xFF26;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hiragana_Katakana = 0xFF27;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Zenkaku = 0xFF28;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hankaku = 0xFF29;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Zenkaku_Hankaku = 0xFF2A;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Touroku = 0xFF2B;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Massyo = 0xFF2C;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Kana_Lock = 0xFF2D;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Kana_Shift = 0xFF2E;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Eisu_Shift = 0xFF2F;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Eisu_toggle = 0xFF30;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Kanji_Bangou = 0xFF37;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Zen_Koho = 0xFF3D;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Mae_Koho = 0xFF3E;

/* Cursor control & motion */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Home = 0xFF50;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Left = 0xFF51;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Up = 0xFF52;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Right = 0xFF53;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Down = 0xFF54;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Prior = 0xFF55;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Page_Up = 0xFF55;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Next = 0xFF56;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Page_Down = 0xFF56;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_End = 0xFF57;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Begin = 0xFF58;

/* Misc Functions */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Select = 0xFF60;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Print = 0xFF61;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Execute = 0xFF62;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Insert = 0xFF63;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Undo = 0xFF65;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Redo = 0xFF66;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Menu = 0xFF67;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Find = 0xFF68;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cancel = 0xFF69;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Help = 0xFF6A;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Break = 0xFF6B;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Mode_switch = 0xFF7E;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Num_Lock = 0xFF7F;

/* Keypad */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_Space = 0xFF80;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_Tab = 0xFF89;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_Enter = 0xFF8D;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_F1 = 0xFF91;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_F2 = 0xFF92;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_F3 = 0xFF93;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_F4 = 0xFF94;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_Home = 0xFF95;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_Left = 0xFF96;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_Up = 0xFF97;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_Right = 0xFF98;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_Down = 0xFF99;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_Prior = 0xFF9A;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_Page_Up = 0xFF9A;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_Next = 0xFF9B;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_Page_Down = 0xFF9B;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_End = 0xFF9C;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_Begin = 0xFF9D;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_Insert = 0xFF9E;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_Delete = 0xFF9F;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_Equal = 0xFFBD;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_Multiply = 0xFFAA;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_Add = 0xFFAB;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_Separator = 0xFFAC;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_Subtract = 0xFFAD;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_Decimal = 0xFFAE;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_Divide = 0xFFAF;

static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_0 = 0xFFB0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_1 = 0xFFB1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_2 = 0xFFB2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_3 = 0xFFB3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_4 = 0xFFB4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_5 = 0xFFB5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_6 = 0xFFB6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_7 = 0xFFB7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_8 = 0xFFB8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_KP_9 = 0xFFB9;

/* Auxilliary Functions */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F1 = 0xFFBE;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F2 = 0xFFBF;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F3 = 0xFFC0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F4 = 0xFFC1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F5 = 0xFFC2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F6 = 0xFFC3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F7 = 0xFFC4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F8 = 0xFFC5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F9 = 0xFFC6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F10 = 0xFFC7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F11 = 0xFFC8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F12 = 0xFFC9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F13 = 0xFFCA;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F14 = 0xFFCB;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F15 = 0xFFCC;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F16 = 0xFFCD;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F17 = 0xFFCE;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F18 = 0xFFCF;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F19 = 0xFFD0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F20 = 0xFFD1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F21 = 0xFFD2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F22 = 0xFFD3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F23 = 0xFFD4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F24 = 0xFFD5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F25 = 0xFFD6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F26 = 0xFFD7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F27 = 0xFFD8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F28 = 0xFFD9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F29 = 0xFFDA;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F30 = 0xFFDB;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F31 = 0xFFDC;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F32 = 0xFFDD;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F33 = 0xFFDE;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F34 = 0xFFDF;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F35 = 0xFFE0;

/* modifier keys */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Shift_L = 0xFFE1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Shift_R = 0xFFE2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Control_L = 0xFFE3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Control_R = 0xFFE4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Caps_Lock = 0xFFE5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Shift_Lock = 0xFFE6;

static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Meta_L = 0xFFE7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Meta_R = 0xFFE8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Alt_L = 0xFFE9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Alt_R = 0xFFEA;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Super_L = 0xFFEB;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Super_R = 0xFFEC;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hyper_L = 0xFFED;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hyper_R = 0xFFEE;

/*
 * ISO 9995 Function and Modifier Keys
 * Byte 3 = 0xFE
 */

static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Lock = 0xFE01;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Level2_Latch = 0xFE02;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Level3_Shift = 0xFE03;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Level3_Latch = 0xFE04;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Level3_Lock = 0xFE05;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Group_Shift = 0xFF7E;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Group_Latch = 0xFE06;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Group_Lock = 0xFE07;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Next_Group = 0xFE08;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Next_Group_Lock = 0xFE09;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Prev_Group = 0xFE0A;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Prev_Group_Lock = 0xFE0B;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_First_Group = 0xFE0C;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_First_Group_Lock = 0xFE0D;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Last_Group = 0xFE0E;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Last_Group_Lock = 0xFE0F;

static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Left_Tab = 0xFE20;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Move_Line_Up = 0xFE21;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Move_Line_Down = 0xFE22;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Partial_Line_Up = 0xFE23;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Partial_Line_Down = 0xFE24;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Partial_Space_Left = 0xFE25;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Partial_Space_Right = 0xFE26;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Set_Margin_Left = 0xFE27;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Set_Margin_Right = 0xFE28;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Release_Margin_Left = 0xFE29;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Release_Margin_Right = 0xFE2A;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Release_Both_Margins = 0xFE2B;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Fast_Cursor_Left = 0xFE2C;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Fast_Cursor_Right = 0xFE2D;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Fast_Cursor_Up = 0xFE2E;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Fast_Cursor_Down = 0xFE2F;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Continuous_Underline = 0xFE30;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Discontinuous_Underline = 0xFE31;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Emphasize = 0xFE32;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Center_Object = 0xFE33;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ISO_Enter = 0xFE34;

static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_dead_grave = 0xFE50;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_dead_acute = 0xFE51;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_dead_circumflex = 0xFE52;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_dead_tilde = 0xFE53;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_dead_macron = 0xFE54;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_dead_breve = 0xFE55;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_dead_abovedot = 0xFE56;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_dead_diaeresis = 0xFE57;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_dead_abovering = 0xFE58;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_dead_doubleacute = 0xFE59;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_dead_caron = 0xFE5A;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_dead_cedilla = 0xFE5B;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_dead_ogonek = 0xFE5C;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_dead_iota = 0xFE5D;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_dead_voiced_sound = 0xFE5E;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_dead_semivoiced_sound = 0xFE5F;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_dead_belowdot = 0xFE60;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_dead_hook = 0xFE61;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_dead_horn = 0xFE62;

static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_First_Virtual_Screen = 0xFED0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Prev_Virtual_Screen = 0xFED1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Next_Virtual_Screen = 0xFED2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Last_Virtual_Screen = 0xFED4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Terminate_Server = 0xFED5;

static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_AccessX_Enable = 0xFE70;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_AccessX_Feedback_Enable = 0xFE71;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_RepeatKeys_Enable = 0xFE72;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_SlowKeys_Enable = 0xFE73;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_BounceKeys_Enable = 0xFE74;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_StickyKeys_Enable = 0xFE75;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_MouseKeys_Enable = 0xFE76;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_MouseKeys_Accel_Enable = 0xFE77;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Overlay1_Enable = 0xFE78;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Overlay2_Enable = 0xFE79;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_AudibleBell_Enable = 0xFE7A;

static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_Left = 0xFEE0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_Right = 0xFEE1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_Up = 0xFEE2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_Down = 0xFEE3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_UpLeft = 0xFEE4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_UpRight = 0xFEE5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_DownLeft = 0xFEE6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_DownRight = 0xFEE7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_Button_Dflt = 0xFEE8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_Button1 = 0xFEE9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_Button2 = 0xFEEA;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_Button3 = 0xFEEB;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_Button4 = 0xFEEC;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_Button5 = 0xFEED;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_DblClick_Dflt = 0xFEEE;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_DblClick1 = 0xFEEF;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_DblClick2 = 0xFEF0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_DblClick3 = 0xFEF1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_DblClick4 = 0xFEF2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_DblClick5 = 0xFEF3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_Drag_Dflt = 0xFEF4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_Drag1 = 0xFEF5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_Drag2 = 0xFEF6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_Drag3 = 0xFEF7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_Drag4 = 0xFEF8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_Drag5 = 0xFEFD;

static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_EnableKeys = 0xFEF9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_Accelerate = 0xFEFA;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_DfltBtnNext = 0xFEFB;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pointer_DfltBtnPrev = 0xFEFC;

/*
 * 3270 Terminal Keys
 * Byte 3 = 0xFD
 */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_Duplicate = 0xFD01;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_FieldMark = 0xFD02;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_Right2 = 0xFD03;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_Left2 = 0xFD04;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_BackTab = 0xFD05;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_EraseEOF = 0xFD06;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_EraseInput = 0xFD07;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_Reset = 0xFD08;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_Quit = 0xFD09;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_PA1 = 0xFD0A;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_PA2 = 0xFD0B;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_PA3 = 0xFD0C;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_Test = 0xFD0D;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_Attn = 0xFD0E;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_CursorBlink = 0xFD0F;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_AltCursor = 0xFD10;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_KeyClick = 0xFD11;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_Jump = 0xFD12;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_Ident = 0xFD13;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_Rule = 0xFD14;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_Copy = 0xFD15;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_Play = 0xFD16;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_Setup = 0xFD17;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_Record = 0xFD18;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_ChangeScreen = 0xFD19;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_DeleteWord = 0xFD1A;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_ExSelect = 0xFD1B;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_CursorSelect = 0xFD1C;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_PrintScreen = 0xFD1D;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3270_Enter = 0xFD1E;

/* Latin 1 */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_space = 0x020;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_exclam = 0x021;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_quotedbl = 0x022;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_numbersign = 0x023;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_dollar = 0x024;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_percent = 0x025;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ampersand = 0x026;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_apostrophe = 0x027;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_quoteright = 0x027;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_parenleft = 0x028;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_parenright = 0x029;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_asterisk = 0x02a;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_plus = 0x02b;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_comma = 0x02c;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_minus = 0x02d;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_period = 0x02e;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_slash = 0x02f;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_0 = 0x030;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_1 = 0x031;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_2 = 0x032;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_3 = 0x033;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_4 = 0x034;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_5 = 0x035;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_6 = 0x036;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_7 = 0x037;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_8 = 0x038;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_9 = 0x039;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_colon = 0x03a;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_semicolon = 0x03b;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_less = 0x03c;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_equal = 0x03d;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_greater = 0x03e;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_question = 0x03f;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_at = 0x040;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_A = 0x041;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_B = 0x042;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_C = 0x043;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_D = 0x044;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_E = 0x045;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_F = 0x046;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_G = 0x047;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_H = 0x048;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_I = 0x049;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_J = 0x04a;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_K = 0x04b;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_L = 0x04c;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_M = 0x04d;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_N = 0x04e;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_O = 0x04f;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_P = 0x050;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Q = 0x051;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_R = 0x052;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_S = 0x053;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_T = 0x054;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_U = 0x055;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_V = 0x056;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_W = 0x057;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_X = 0x058;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Y = 0x059;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Z = 0x05a;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_bracketleft = 0x05b;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_backslash = 0x05c;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_bracketright = 0x05d;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_asciicircum = 0x05e;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_underscore = 0x05f;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_grave = 0x060;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_a = 0x061;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_b = 0x062;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_c = 0x063;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_d = 0x064;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_e = 0x065;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_f = 0x066;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_g = 0x067;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_h = 0x068;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_i = 0x069;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_j = 0x06a;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_k = 0x06b;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_l = 0x06c;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_m = 0x06d;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_n = 0x06e;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_o = 0x06f;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_p = 0x070;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_q = 0x071;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_r = 0x072;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_s = 0x073;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_t = 0x074;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_u = 0x075;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_v = 0x076;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_w = 0x077;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_x = 0x078;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_y = 0x079;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_z = 0x07a;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_braceleft = 0x07b;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_bar = 0x07c;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_braceright = 0x07d;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_asciitilde = 0x07e;

static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_nobreakspace = 0x0a0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_exclamdown = 0x0a1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_cent = 0x0a2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_sterling = 0x0a3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_currency = 0x0a4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_yen = 0x0a5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_brokenbar = 0x0a6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_section = 0x0a7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_diaeresis = 0x0a8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_copyright = 0x0a9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ordfeminine = 0x0aa;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_guillemotleft = 0x0ab;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_notsign = 0x0ac;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hyphen = 0x0ad;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_registered = 0x0ae;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_macron = 0x0af;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_degree = 0x0b0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_plusminus = 0x0b1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_twosuperior = 0x0b2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_threesuperior = 0x0b3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_acute = 0x0b4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_mu = 0x0b5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_paragraph = 0x0b6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_periodcentered = 0x0b7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_cedilla = 0x0b8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_onesuperior = 0x0b9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_masculine = 0x0ba;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_guillemotright = 0x0bb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_onequarter = 0x0bc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_onehalf = 0x0bd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_threequarters = 0x0be;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_questiondown = 0x0bf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Agrave = 0x0c0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Aacute = 0x0c1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Acircumflex = 0x0c2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Atilde = 0x0c3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Adiaeresis = 0x0c4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Aring = 0x0c5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_AE = 0x0c6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ccedilla = 0x0c7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Egrave = 0x0c8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Eacute = 0x0c9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ecircumflex = 0x0ca;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ediaeresis = 0x0cb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Igrave = 0x0cc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Iacute = 0x0cd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Icircumflex = 0x0ce;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Idiaeresis = 0x0cf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ETH = 0x0d0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Eth = 0x0d0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ntilde = 0x0d1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ograve = 0x0d2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Oacute = 0x0d3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ocircumflex = 0x0d4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Otilde = 0x0d5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Odiaeresis = 0x0d6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_multiply = 0x0d7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ooblique = 0x0d8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Oslash = 0x0d8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ugrave = 0x0d9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Uacute = 0x0da;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ucircumflex = 0x0db;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Udiaeresis = 0x0dc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Yacute = 0x0dd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_THORN = 0x0de;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thorn = 0x0de;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ssharp = 0x0df;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_agrave = 0x0e0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_aacute = 0x0e1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_acircumflex = 0x0e2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_atilde = 0x0e3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_adiaeresis = 0x0e4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_aring = 0x0e5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ae = 0x0e6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ccedilla = 0x0e7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_egrave = 0x0e8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_eacute = 0x0e9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ecircumflex = 0x0ea;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ediaeresis = 0x0eb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_igrave = 0x0ec;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_iacute = 0x0ed;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_icircumflex = 0x0ee;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_idiaeresis = 0x0ef;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_eth = 0x0f0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ntilde = 0x0f1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ograve = 0x0f2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_oacute = 0x0f3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ocircumflex = 0x0f4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_otilde = 0x0f5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_odiaeresis = 0x0f6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_division = 0x0f7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_oslash = 0x0f8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ooblique = 0x0f8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ugrave = 0x0f9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_uacute = 0x0fa;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ucircumflex = 0x0fb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_udiaeresis = 0x0fc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_yacute = 0x0fd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_thorn = 0x0fe;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ydiaeresis = 0x0ff;

/*
 *   Latin 2
 *   Byte 3 = 1
 */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Aogonek = 0x1a1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_breve = 0x1a2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Lstroke = 0x1a3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Lcaron = 0x1a5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Sacute = 0x1a6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Scaron = 0x1a9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Scedilla = 0x1aa;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Tcaron = 0x1ab;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Zacute = 0x1ac;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Zcaron = 0x1ae;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Zabovedot = 0x1af;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_aogonek = 0x1b1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ogonek = 0x1b2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_lstroke = 0x1b3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_lcaron = 0x1b5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_sacute = 0x1b6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_caron = 0x1b7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_scaron = 0x1b9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_scedilla = 0x1ba;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_tcaron = 0x1bb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_zacute = 0x1bc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_doubleacute = 0x1bd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_zcaron = 0x1be;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_zabovedot = 0x1bf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Racute = 0x1c0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Abreve = 0x1c3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Lacute = 0x1c5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cacute = 0x1c6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ccaron = 0x1c8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Eogonek = 0x1ca;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ecaron = 0x1cc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Dcaron = 0x1cf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Dstroke = 0x1d0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Nacute = 0x1d1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ncaron = 0x1d2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Odoubleacute = 0x1d5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Rcaron = 0x1d8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Uring = 0x1d9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Udoubleacute = 0x1db;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Tcedilla = 0x1de;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_racute = 0x1e0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_abreve = 0x1e3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_lacute = 0x1e5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_cacute = 0x1e6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ccaron = 0x1e8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_eogonek = 0x1ea;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ecaron = 0x1ec;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_dcaron = 0x1ef;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_dstroke = 0x1f0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_nacute = 0x1f1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ncaron = 0x1f2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_odoubleacute = 0x1f5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_udoubleacute = 0x1fb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_rcaron = 0x1f8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_uring = 0x1f9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_tcedilla = 0x1fe;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_abovedot = 0x1ff;

/*
 *   Latin 3
 *   Byte 3 = 2
 */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hstroke = 0x2a1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hcircumflex = 0x2a6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Iabovedot = 0x2a9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Gbreve = 0x2ab;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Jcircumflex = 0x2ac;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hstroke = 0x2b1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hcircumflex = 0x2b6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_idotless = 0x2b9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_gbreve = 0x2bb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_jcircumflex = 0x2bc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cabovedot = 0x2c5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ccircumflex = 0x2c6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Gabovedot = 0x2d5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Gcircumflex = 0x2d8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ubreve = 0x2dd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Scircumflex = 0x2de;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_cabovedot = 0x2e5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ccircumflex = 0x2e6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_gabovedot = 0x2f5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_gcircumflex = 0x2f8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ubreve = 0x2fd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_scircumflex = 0x2fe;

/*
 *   Latin 4
 *   Byte 3 = 3
 */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kra = 0x3a2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kappa = 0x3a2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Rcedilla = 0x3a3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Itilde = 0x3a5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Lcedilla = 0x3a6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Emacron = 0x3aa;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Gcedilla = 0x3ab;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Tslash = 0x3ac;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_rcedilla = 0x3b3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_itilde = 0x3b5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_lcedilla = 0x3b6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_emacron = 0x3ba;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_gcedilla = 0x3bb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_tslash = 0x3bc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ENG = 0x3bd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_eng = 0x3bf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Amacron = 0x3c0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Iogonek = 0x3c7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Eabovedot = 0x3cc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Imacron = 0x3cf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ncedilla = 0x3d1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Omacron = 0x3d2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Kcedilla = 0x3d3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Uogonek = 0x3d9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Utilde = 0x3dd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Umacron = 0x3de;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_amacron = 0x3e0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_iogonek = 0x3e7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_eabovedot = 0x3ec;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_imacron = 0x3ef;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ncedilla = 0x3f1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_omacron = 0x3f2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kcedilla = 0x3f3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_uogonek = 0x3f9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_utilde = 0x3fd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_umacron = 0x3fe;

/*
 * Latin-8
 * Byte 3 = 18
 */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Babovedot = 0x12a1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_babovedot = 0x12a2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Dabovedot = 0x12a6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Wgrave = 0x12a8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Wacute = 0x12aa;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_dabovedot = 0x12ab;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ygrave = 0x12ac;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Fabovedot = 0x12b0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_fabovedot = 0x12b1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Mabovedot = 0x12b4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_mabovedot = 0x12b5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Pabovedot = 0x12b7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_wgrave = 0x12b8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_pabovedot = 0x12b9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_wacute = 0x12ba;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Sabovedot = 0x12bb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ygrave = 0x12bc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Wdiaeresis = 0x12bd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_wdiaeresis = 0x12be;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_sabovedot = 0x12bf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Wcircumflex = 0x12d0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Tabovedot = 0x12d7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ycircumflex = 0x12de;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_wcircumflex = 0x12f0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_tabovedot = 0x12f7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ycircumflex = 0x12fe;

/*
 * Latin-9 (a.k.a.Latin-0)
 * Byte 3 = 19
 */

static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_OE = 0x13bc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_oe = 0x13bd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ydiaeresis = 0x13be;

/*
 * Katakana
 * Byte 3 = 4
 */

static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_overline = 0x47e;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_fullstop = 0x4a1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_openingbracket = 0x4a2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_closingbracket = 0x4a3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_comma = 0x4a4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_conjunctive = 0x4a5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_middledot = 0x4a5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_WO = 0x4a6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_a = 0x4a7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_i = 0x4a8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_u = 0x4a9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_e = 0x4aa;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_o = 0x4ab;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_ya = 0x4ac;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_yu = 0x4ad;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_yo = 0x4ae;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_tsu = 0x4af;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_tu = 0x4af;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_prolongedsound = 0x4b0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_A = 0x4b1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_I = 0x4b2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_U = 0x4b3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_E = 0x4b4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_O = 0x4b5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_KA = 0x4b6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_KI = 0x4b7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_KU = 0x4b8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_KE = 0x4b9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_KO = 0x4ba;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_SA = 0x4bb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_SHI = 0x4bc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_SU = 0x4bd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_SE = 0x4be;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_SO = 0x4bf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_TA = 0x4c0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_CHI = 0x4c1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_TI = 0x4c1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_TSU = 0x4c2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_TU = 0x4c2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_TE = 0x4c3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_TO = 0x4c4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_NA = 0x4c5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_NI = 0x4c6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_NU = 0x4c7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_NE = 0x4c8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_NO = 0x4c9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_HA = 0x4ca;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_HI = 0x4cb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_FU = 0x4cc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_HU = 0x4cc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_HE = 0x4cd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_HO = 0x4ce;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_MA = 0x4cf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_MI = 0x4d0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_MU = 0x4d1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_ME = 0x4d2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_MO = 0x4d3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_YA = 0x4d4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_YU = 0x4d5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_YO = 0x4d6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_RA = 0x4d7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_RI = 0x4d8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_RU = 0x4d9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_RE = 0x4da;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_RO = 0x4db;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_WA = 0x4dc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_N = 0x4dd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_voicedsound = 0x4de;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_semivoicedsound = 0x4df;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_kana_switch = 0xFF7E;

/*
 *  Arabic
 *  Byte 3 = 5
 */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Farsi_0 = 0x590;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Farsi_1 = 0x591;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Farsi_2 = 0x592;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Farsi_3 = 0x593;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Farsi_4 = 0x594;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Farsi_5 = 0x595;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Farsi_6 = 0x596;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Farsi_7 = 0x597;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Farsi_8 = 0x598;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Farsi_9 = 0x599;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_percent = 0x5a5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_superscript_alef = 0x5a6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_tteh = 0x5a7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_peh = 0x5a8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_tcheh = 0x5a9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_ddal = 0x5aa;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_rreh = 0x5ab;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_comma = 0x5ac;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_fullstop = 0x5ae;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_0 = 0x5b0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_1 = 0x5b1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_2 = 0x5b2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_3 = 0x5b3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_4 = 0x5b4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_5 = 0x5b5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_6 = 0x5b6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_7 = 0x5b7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_8 = 0x5b8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_9 = 0x5b9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_semicolon = 0x5bb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_question_mark = 0x5bf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_hamza = 0x5c1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_maddaonalef = 0x5c2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_hamzaonalef = 0x5c3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_hamzaonwaw = 0x5c4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_hamzaunderalef = 0x5c5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_hamzaonyeh = 0x5c6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_alef = 0x5c7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_beh = 0x5c8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_tehmarbuta = 0x5c9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_teh = 0x5ca;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_theh = 0x5cb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_jeem = 0x5cc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_hah = 0x5cd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_khah = 0x5ce;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_dal = 0x5cf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_thal = 0x5d0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_ra = 0x5d1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_zain = 0x5d2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_seen = 0x5d3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_sheen = 0x5d4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_sad = 0x5d5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_dad = 0x5d6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_tah = 0x5d7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_zah = 0x5d8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_ain = 0x5d9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_ghain = 0x5da;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_tatweel = 0x5e0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_feh = 0x5e1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_qaf = 0x5e2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_kaf = 0x5e3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_lam = 0x5e4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_meem = 0x5e5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_noon = 0x5e6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_ha = 0x5e7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_heh = 0x5e7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_waw = 0x5e8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_alefmaksura = 0x5e9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_yeh = 0x5ea;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_fathatan = 0x5eb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_dammatan = 0x5ec;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_kasratan = 0x5ed;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_fatha = 0x5ee;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_damma = 0x5ef;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_kasra = 0x5f0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_shadda = 0x5f1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_sukun = 0x5f2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_madda_above = 0x5f3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_hamza_above = 0x5f4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_hamza_below = 0x5f5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_jeh = 0x5f6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_veh = 0x5f7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_keheh = 0x5f8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_gaf = 0x5f9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_noon_ghunna = 0x5fa;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_heh_doachashmee = 0x5fb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Farsi_yeh = 0x5fc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_farsi_yeh = 0x5fc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_yeh_baree = 0x5fd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_heh_goal = 0x5fe;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Arabic_switch = 0xFF7E;

/*
 * Cyrillic
 * Byte 3 = 6
 */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_GHE_bar = 0x680;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_ghe_bar = 0x690;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_ZHE_descender = 0x681;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_zhe_descender = 0x691;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_KA_descender = 0x682;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_ka_descender = 0x692;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_KA_vertstroke = 0x683;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_ka_vertstroke = 0x693;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_EN_descender = 0x684;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_en_descender = 0x694;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_U_straight = 0x685;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_u_straight = 0x695;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_U_straight_bar = 0x686;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_u_straight_bar = 0x696;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_HA_descender = 0x687;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_ha_descender = 0x697;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_CHE_descender = 0x688;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_che_descender = 0x698;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_CHE_vertstroke = 0x689;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_che_vertstroke = 0x699;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_SHHA = 0x68a;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_shha = 0x69a;

static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_SCHWA = 0x68c;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_schwa = 0x69c;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_I_macron = 0x68d;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_i_macron = 0x69d;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_O_bar = 0x68e;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_o_bar = 0x69e;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_U_macron = 0x68f;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_u_macron = 0x69f;

static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Serbian_dje = 0x6a1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Macedonia_gje = 0x6a2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_io = 0x6a3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ukrainian_ie = 0x6a4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ukranian_je = 0x6a4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Macedonia_dse = 0x6a5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ukrainian_i = 0x6a6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ukranian_i = 0x6a6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ukrainian_yi = 0x6a7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ukranian_yi = 0x6a7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_je = 0x6a8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Serbian_je = 0x6a8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_lje = 0x6a9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Serbian_lje = 0x6a9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_nje = 0x6aa;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Serbian_nje = 0x6aa;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Serbian_tshe = 0x6ab;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Macedonia_kje = 0x6ac;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ukrainian_ghe_with_upturn = 0x6ad;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Byelorussian_shortu = 0x6ae;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_dzhe = 0x6af;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Serbian_dze = 0x6af;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_numerosign = 0x6b0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Serbian_DJE = 0x6b1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Macedonia_GJE = 0x6b2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_IO = 0x6b3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ukrainian_IE = 0x6b4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ukranian_JE = 0x6b4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Macedonia_DSE = 0x6b5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ukrainian_I = 0x6b6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ukranian_I = 0x6b6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ukrainian_YI = 0x6b7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ukranian_YI = 0x6b7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_JE = 0x6b8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Serbian_JE = 0x6b8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_LJE = 0x6b9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Serbian_LJE = 0x6b9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_NJE = 0x6ba;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Serbian_NJE = 0x6ba;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Serbian_TSHE = 0x6bb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Macedonia_KJE = 0x6bc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ukrainian_GHE_WITH_UPTURN = 0x6bd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Byelorussian_SHORTU = 0x6be;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_DZHE = 0x6bf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Serbian_DZE = 0x6bf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_yu = 0x6c0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_a = 0x6c1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_be = 0x6c2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_tse = 0x6c3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_de = 0x6c4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_ie = 0x6c5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_ef = 0x6c6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_ghe = 0x6c7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_ha = 0x6c8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_i = 0x6c9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_shorti = 0x6ca;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_ka = 0x6cb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_el = 0x6cc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_em = 0x6cd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_en = 0x6ce;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_o = 0x6cf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_pe = 0x6d0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_ya = 0x6d1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_er = 0x6d2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_es = 0x6d3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_te = 0x6d4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_u = 0x6d5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_zhe = 0x6d6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_ve = 0x6d7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_softsign = 0x6d8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_yeru = 0x6d9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_ze = 0x6da;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_sha = 0x6db;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_e = 0x6dc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_shcha = 0x6dd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_che = 0x6de;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_hardsign = 0x6df;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_YU = 0x6e0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_A = 0x6e1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_BE = 0x6e2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_TSE = 0x6e3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_DE = 0x6e4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_IE = 0x6e5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_EF = 0x6e6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_GHE = 0x6e7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_HA = 0x6e8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_I = 0x6e9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_SHORTI = 0x6ea;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_KA = 0x6eb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_EL = 0x6ec;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_EM = 0x6ed;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_EN = 0x6ee;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_O = 0x6ef;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_PE = 0x6f0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_YA = 0x6f1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_ER = 0x6f2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_ES = 0x6f3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_TE = 0x6f4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_U = 0x6f5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_ZHE = 0x6f6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_VE = 0x6f7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_SOFTSIGN = 0x6f8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_YERU = 0x6f9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_ZE = 0x6fa;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_SHA = 0x6fb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_E = 0x6fc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_SHCHA = 0x6fd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_CHE = 0x6fe;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Cyrillic_HARDSIGN = 0x6ff;

/*
 * Greek
 * Byte 3 = 7
 */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_ALPHAaccent = 0x7a1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_EPSILONaccent = 0x7a2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_ETAaccent = 0x7a3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_IOTAaccent = 0x7a4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_IOTAdieresis = 0x7a5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_IOTAdiaeresis = 0x7a5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_OMICRONaccent = 0x7a7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_UPSILONaccent = 0x7a8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_UPSILONdieresis = 0x7a9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_OMEGAaccent = 0x7ab;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_accentdieresis = 0x7ae;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_horizbar = 0x7af;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_alphaaccent = 0x7b1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_epsilonaccent = 0x7b2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_etaaccent = 0x7b3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_iotaaccent = 0x7b4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_iotadieresis = 0x7b5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_iotaaccentdieresis = 0x7b6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_omicronaccent = 0x7b7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_upsilonaccent = 0x7b8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_upsilondieresis = 0x7b9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_upsilonaccentdieresis = 0x7ba;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_omegaaccent = 0x7bb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_ALPHA = 0x7c1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_BETA = 0x7c2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_GAMMA = 0x7c3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_DELTA = 0x7c4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_EPSILON = 0x7c5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_ZETA = 0x7c6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_ETA = 0x7c7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_THETA = 0x7c8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_IOTA = 0x7c9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_KAPPA = 0x7ca;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_LAMDA = 0x7cb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_LAMBDA = 0x7cb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_MU = 0x7cc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_NU = 0x7cd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_XI = 0x7ce;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_OMICRON = 0x7cf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_PI = 0x7d0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_RHO = 0x7d1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_SIGMA = 0x7d2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_TAU = 0x7d4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_UPSILON = 0x7d5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_PHI = 0x7d6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_CHI = 0x7d7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_PSI = 0x7d8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_OMEGA = 0x7d9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_alpha = 0x7e1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_beta = 0x7e2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_gamma = 0x7e3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_delta = 0x7e4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_epsilon = 0x7e5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_zeta = 0x7e6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_eta = 0x7e7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_theta = 0x7e8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_iota = 0x7e9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_kappa = 0x7ea;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_lamda = 0x7eb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_lambda = 0x7eb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_mu = 0x7ec;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_nu = 0x7ed;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_xi = 0x7ee;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_omicron = 0x7ef;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_pi = 0x7f0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_rho = 0x7f1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_sigma = 0x7f2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_finalsmallsigma = 0x7f3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_tau = 0x7f4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_upsilon = 0x7f5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_phi = 0x7f6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_chi = 0x7f7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_psi = 0x7f8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_omega = 0x7f9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Greek_switch = 0xFF7E;

/*
 * Technical
 * Byte 3 = 8
 */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_leftradical = 0x8a1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_topleftradical = 0x8a2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_horizconnector = 0x8a3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_topintegral = 0x8a4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_botintegral = 0x8a5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_vertconnector = 0x8a6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_topleftsqbracket = 0x8a7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_botleftsqbracket = 0x8a8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_toprightsqbracket = 0x8a9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_botrightsqbracket = 0x8aa;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_topleftparens = 0x8ab;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_botleftparens = 0x8ac;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_toprightparens = 0x8ad;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_botrightparens = 0x8ae;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_leftmiddlecurlybrace = 0x8af;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_rightmiddlecurlybrace = 0x8b0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_topleftsummation = 0x8b1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_botleftsummation = 0x8b2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_topvertsummationconnector = 0x8b3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_botvertsummationconnector = 0x8b4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_toprightsummation = 0x8b5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_botrightsummation = 0x8b6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_rightmiddlesummation = 0x8b7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_lessthanequal = 0x8bc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_notequal = 0x8bd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_greaterthanequal = 0x8be;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_integral = 0x8bf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_therefore = 0x8c0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_variation = 0x8c1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_infinity = 0x8c2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_nabla = 0x8c5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_approximate = 0x8c8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_similarequal = 0x8c9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ifonlyif = 0x8cd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_implies = 0x8ce;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_identical = 0x8cf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_radical = 0x8d6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_includedin = 0x8da;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_includes = 0x8db;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_intersection = 0x8dc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_union = 0x8dd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_logicaland = 0x8de;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_logicalor = 0x8df;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_partialderivative = 0x8ef;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_function = 0x8f6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_leftarrow = 0x8fb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_uparrow = 0x8fc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_rightarrow = 0x8fd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_downarrow = 0x8fe;

/*
 * Special
 * Byte 3 = 9
 */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_blank = 0x9df;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_soliddiamond = 0x9e0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_checkerboard = 0x9e1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ht = 0x9e2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ff = 0x9e3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_cr = 0x9e4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_lf = 0x9e5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_nl = 0x9e8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_vt = 0x9e9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_lowrightcorner = 0x9ea;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_uprightcorner = 0x9eb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_upleftcorner = 0x9ec;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_lowleftcorner = 0x9ed;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_crossinglines = 0x9ee;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_horizlinescan1 = 0x9ef;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_horizlinescan3 = 0x9f0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_horizlinescan5 = 0x9f1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_horizlinescan7 = 0x9f2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_horizlinescan9 = 0x9f3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_leftt = 0x9f4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_rightt = 0x9f5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_bott = 0x9f6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_topt = 0x9f7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_vertbar = 0x9f8;

/*
 * Publishing
 * Byte 3 = a
 */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_emspace = 0xaa1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_enspace = 0xaa2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_em3space = 0xaa3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_em4space = 0xaa4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_digitspace = 0xaa5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_punctspace = 0xaa6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_thinspace = 0xaa7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hairspace = 0xaa8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_emdash = 0xaa9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_endash = 0xaaa;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_signifblank = 0xaac;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ellipsis = 0xaae;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_doubbaselinedot = 0xaaf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_onethird = 0xab0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_twothirds = 0xab1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_onefifth = 0xab2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_twofifths = 0xab3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_threefifths = 0xab4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_fourfifths = 0xab5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_onesixth = 0xab6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_fivesixths = 0xab7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_careof = 0xab8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_figdash = 0xabb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_leftanglebracket = 0xabc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_decimalpoint = 0xabd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_rightanglebracket = 0xabe;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_marker = 0xabf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_oneeighth = 0xac3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_threeeighths = 0xac4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_fiveeighths = 0xac5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_seveneighths = 0xac6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_trademark = 0xac9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_signaturemark = 0xaca;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_trademarkincircle = 0xacb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_leftopentriangle = 0xacc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_rightopentriangle = 0xacd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_emopencircle = 0xace;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_emopenrectangle = 0xacf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_leftsinglequotemark = 0xad0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_rightsinglequotemark = 0xad1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_leftdoublequotemark = 0xad2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_rightdoublequotemark = 0xad3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_prescription = 0xad4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_minutes = 0xad6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_seconds = 0xad7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_latincross = 0xad9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hexagram = 0xada;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_filledrectbullet = 0xadb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_filledlefttribullet = 0xadc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_filledrighttribullet = 0xadd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_emfilledcircle = 0xade;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_emfilledrect = 0xadf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_enopencircbullet = 0xae0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_enopensquarebullet = 0xae1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_openrectbullet = 0xae2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_opentribulletup = 0xae3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_opentribulletdown = 0xae4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_openstar = 0xae5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_enfilledcircbullet = 0xae6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_enfilledsqbullet = 0xae7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_filledtribulletup = 0xae8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_filledtribulletdown = 0xae9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_leftpointer = 0xaea;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_rightpointer = 0xaeb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_club = 0xaec;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_diamond = 0xaed;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_heart = 0xaee;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_maltesecross = 0xaf0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_dagger = 0xaf1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_doubledagger = 0xaf2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_checkmark = 0xaf3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ballotcross = 0xaf4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_musicalsharp = 0xaf5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_musicalflat = 0xaf6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_malesymbol = 0xaf7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_femalesymbol = 0xaf8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_telephone = 0xaf9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_telephonerecorder = 0xafa;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_phonographcopyright = 0xafb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_caret = 0xafc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_singlelowquotemark = 0xafd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_doublelowquotemark = 0xafe;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_cursor = 0xaff;

/*
 * APL
 * Byte 3 = b
 */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_leftcaret = 0xba3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_rightcaret = 0xba6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_downcaret = 0xba8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_upcaret = 0xba9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_overbar = 0xbc0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_downtack = 0xbc2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_upshoe = 0xbc3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_downstile = 0xbc4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_underbar = 0xbc6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_jot = 0xbca;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_quad = 0xbcc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_uptack = 0xbce;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_circle = 0xbcf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_upstile = 0xbd3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_downshoe = 0xbd6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_rightshoe = 0xbd8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_leftshoe = 0xbda;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_lefttack = 0xbdc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_righttack = 0xbfc;

/*
 * Hebrew
 * Byte 3 = c
 */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_doublelowline = 0xcdf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_aleph = 0xce0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_bet = 0xce1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_beth = 0xce1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_gimel = 0xce2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_gimmel = 0xce2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_dalet = 0xce3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_daleth = 0xce3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_he = 0xce4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_waw = 0xce5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_zain = 0xce6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_zayin = 0xce6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_chet = 0xce7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_het = 0xce7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_tet = 0xce8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_teth = 0xce8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_yod = 0xce9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_finalkaph = 0xcea;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_kaph = 0xceb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_lamed = 0xcec;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_finalmem = 0xced;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_mem = 0xcee;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_finalnun = 0xcef;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_nun = 0xcf0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_samech = 0xcf1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_samekh = 0xcf1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_ayin = 0xcf2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_finalpe = 0xcf3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_pe = 0xcf4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_finalzade = 0xcf5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_finalzadi = 0xcf5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_zade = 0xcf6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_zadi = 0xcf6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_qoph = 0xcf7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_kuf = 0xcf7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_resh = 0xcf8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_shin = 0xcf9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_taw = 0xcfa;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_hebrew_taf = 0xcfa;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hebrew_switch = 0xFF7E;

/*
 * Thai
 * Byte 3 = d
 */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_kokai = 0xda1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_khokhai = 0xda2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_khokhuat = 0xda3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_khokhwai = 0xda4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_khokhon = 0xda5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_khorakhang = 0xda6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_ngongu = 0xda7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_chochan = 0xda8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_choching = 0xda9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_chochang = 0xdaa;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_soso = 0xdab;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_chochoe = 0xdac;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_yoying = 0xdad;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_dochada = 0xdae;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_topatak = 0xdaf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_thothan = 0xdb0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_thonangmontho = 0xdb1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_thophuthao = 0xdb2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_nonen = 0xdb3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_dodek = 0xdb4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_totao = 0xdb5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_thothung = 0xdb6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_thothahan = 0xdb7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_thothong = 0xdb8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_nonu = 0xdb9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_bobaimai = 0xdba;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_popla = 0xdbb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_phophung = 0xdbc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_fofa = 0xdbd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_phophan = 0xdbe;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_fofan = 0xdbf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_phosamphao = 0xdc0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_moma = 0xdc1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_yoyak = 0xdc2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_rorua = 0xdc3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_ru = 0xdc4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_loling = 0xdc5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_lu = 0xdc6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_wowaen = 0xdc7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_sosala = 0xdc8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_sorusi = 0xdc9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_sosua = 0xdca;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_hohip = 0xdcb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_lochula = 0xdcc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_oang = 0xdcd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_honokhuk = 0xdce;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_paiyannoi = 0xdcf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_saraa = 0xdd0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_maihanakat = 0xdd1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_saraaa = 0xdd2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_saraam = 0xdd3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_sarai = 0xdd4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_saraii = 0xdd5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_saraue = 0xdd6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_sarauee = 0xdd7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_sarau = 0xdd8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_sarauu = 0xdd9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_phinthu = 0xdda;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_maihanakat_maitho = 0xdde;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_baht = 0xddf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_sarae = 0xde0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_saraae = 0xde1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_sarao = 0xde2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_saraaimaimuan = 0xde3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_saraaimaimalai = 0xde4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_lakkhangyao = 0xde5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_maiyamok = 0xde6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_maitaikhu = 0xde7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_maiek = 0xde8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_maitho = 0xde9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_maitri = 0xdea;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_maichattawa = 0xdeb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_thanthakhat = 0xdec;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_nikhahit = 0xded;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_leksun = 0xdf0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_leknung = 0xdf1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_leksong = 0xdf2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_leksam = 0xdf3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_leksi = 0xdf4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_lekha = 0xdf5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_lekhok = 0xdf6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_lekchet = 0xdf7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_lekpaet = 0xdf8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Thai_lekkao = 0xdf9;

/*
 *   Korean
 *   Byte 3 = e
 */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul = 0xff31;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_Start = 0xff32;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_End = 0xff33;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_Hanja = 0xff34;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_Jamo = 0xff35;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_Romaja = 0xff36;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_Codeinput = 0xff37;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_Jeonja = 0xff38;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_Banja = 0xff39;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_PreHanja = 0xff3a;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_PostHanja = 0xff3b;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_SingleCandidate = 0xff3c;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_MultipleCandidate = 0xff3d;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_PreviousCandidate = 0xff3e;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_Special = 0xff3f;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_switch = 0xFF7E;

/* Hangul Consonant Characters */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_Kiyeog = 0xea1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_SsangKiyeog = 0xea2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_KiyeogSios = 0xea3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_Nieun = 0xea4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_NieunJieuj = 0xea5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_NieunHieuh = 0xea6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_Dikeud = 0xea7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_SsangDikeud = 0xea8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_Rieul = 0xea9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_RieulKiyeog = 0xeaa;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_RieulMieum = 0xeab;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_RieulPieub = 0xeac;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_RieulSios = 0xead;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_RieulTieut = 0xeae;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_RieulPhieuf = 0xeaf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_RieulHieuh = 0xeb0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_Mieum = 0xeb1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_Pieub = 0xeb2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_SsangPieub = 0xeb3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_PieubSios = 0xeb4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_Sios = 0xeb5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_SsangSios = 0xeb6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_Ieung = 0xeb7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_Jieuj = 0xeb8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_SsangJieuj = 0xeb9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_Cieuc = 0xeba;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_Khieuq = 0xebb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_Tieut = 0xebc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_Phieuf = 0xebd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_Hieuh = 0xebe;

/* Hangul Vowel Characters */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_A = 0xebf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_AE = 0xec0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_YA = 0xec1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_YAE = 0xec2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_EO = 0xec3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_E = 0xec4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_YEO = 0xec5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_YE = 0xec6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_O = 0xec7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_WA = 0xec8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_WAE = 0xec9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_OE = 0xeca;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_YO = 0xecb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_U = 0xecc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_WEO = 0xecd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_WE = 0xece;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_WI = 0xecf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_YU = 0xed0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_EU = 0xed1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_YI = 0xed2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_I = 0xed3;

/* Hangul syllable-final (JongSeong) Characters */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_Kiyeog = 0xed4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_SsangKiyeog = 0xed5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_KiyeogSios = 0xed6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_Nieun = 0xed7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_NieunJieuj = 0xed8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_NieunHieuh = 0xed9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_Dikeud = 0xeda;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_Rieul = 0xedb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_RieulKiyeog = 0xedc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_RieulMieum = 0xedd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_RieulPieub = 0xede;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_RieulSios = 0xedf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_RieulTieut = 0xee0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_RieulPhieuf = 0xee1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_RieulHieuh = 0xee2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_Mieum = 0xee3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_Pieub = 0xee4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_PieubSios = 0xee5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_Sios = 0xee6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_SsangSios = 0xee7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_Ieung = 0xee8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_Jieuj = 0xee9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_Cieuc = 0xeea;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_Khieuq = 0xeeb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_Tieut = 0xeec;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_Phieuf = 0xeed;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_Hieuh = 0xeee;

/* Ancient Hangul Consonant Characters */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_RieulYeorinHieuh = 0xeef;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_SunkyeongeumMieum = 0xef0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_SunkyeongeumPieub = 0xef1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_PanSios = 0xef2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_KkogjiDalrinIeung = 0xef3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_SunkyeongeumPhieuf = 0xef4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_YeorinHieuh = 0xef5;

/* Ancient Hangul Vowel Characters */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_AraeA = 0xef6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_AraeAE = 0xef7;

/* Ancient Hangul syllable-final (JongSeong) Characters */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_PanSios = 0xef8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_KkogjiDalrinIeung = 0xef9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Hangul_J_YeorinHieuh = 0xefa;

/* Korean currency symbol */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Korean_Won = 0xeff;

/*
 *   Armenian
 *   Byte 3 = 0x14
 */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_eternity = 0x14a1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_ligature_ew = 0x14a2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_full_stop = 0x14a3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_verjaket = 0x14a3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_parenright = 0x14a4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_parenleft = 0x14a5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_guillemotright = 0x14a6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_guillemotleft = 0x14a7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_em_dash = 0x14a8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_dot = 0x14a9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_mijaket = 0x14a9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_separation_mark = 0x14aa;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_but = 0x14aa;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_comma = 0x14ab;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_en_dash = 0x14ac;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_hyphen = 0x14ad;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_yentamna = 0x14ad;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_ellipsis = 0x14ae;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_exclam = 0x14af;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_amanak = 0x14af;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_accent = 0x14b0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_shesht = 0x14b0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_question = 0x14b1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_paruyk = 0x14b1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_AYB = 0x14b2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_ayb = 0x14b3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_BEN = 0x14b4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_ben = 0x14b5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_GIM = 0x14b6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_gim = 0x14b7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_DA = 0x14b8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_da = 0x14b9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_YECH = 0x14ba;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_yech = 0x14bb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_ZA = 0x14bc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_za = 0x14bd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_E = 0x14be;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_e = 0x14bf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_AT = 0x14c0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_at = 0x14c1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_TO = 0x14c2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_to = 0x14c3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_ZHE = 0x14c4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_zhe = 0x14c5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_INI = 0x14c6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_ini = 0x14c7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_LYUN = 0x14c8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_lyun = 0x14c9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_KHE = 0x14ca;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_khe = 0x14cb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_TSA = 0x14cc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_tsa = 0x14cd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_KEN = 0x14ce;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_ken = 0x14cf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_HO = 0x14d0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_ho = 0x14d1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_DZA = 0x14d2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_dza = 0x14d3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_GHAT = 0x14d4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_ghat = 0x14d5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_TCHE = 0x14d6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_tche = 0x14d7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_MEN = 0x14d8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_men = 0x14d9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_HI = 0x14da;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_hi = 0x14db;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_NU = 0x14dc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_nu = 0x14dd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_SHA = 0x14de;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_sha = 0x14df;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_VO = 0x14e0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_vo = 0x14e1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_CHA = 0x14e2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_cha = 0x14e3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_PE = 0x14e4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_pe = 0x14e5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_JE = 0x14e6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_je = 0x14e7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_RA = 0x14e8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_ra = 0x14e9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_SE = 0x14ea;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_se = 0x14eb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_VEV = 0x14ec;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_vev = 0x14ed;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_TYUN = 0x14ee;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_tyun = 0x14ef;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_RE = 0x14f0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_re = 0x14f1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_TSO = 0x14f2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_tso = 0x14f3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_VYUN = 0x14f4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_vyun = 0x14f5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_PYUR = 0x14f6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_pyur = 0x14f7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_KE = 0x14f8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_ke = 0x14f9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_O = 0x14fa;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_o = 0x14fb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_FE = 0x14fc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_fe = 0x14fd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_apostrophe = 0x14fe;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Armenian_section_sign = 0x14ff;

/*
 * Georgian
 * Byte 3 = 0x15
 */

static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_an = 0x15d0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_ban = 0x15d1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_gan = 0x15d2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_don = 0x15d3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_en = 0x15d4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_vin = 0x15d5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_zen = 0x15d6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_tan = 0x15d7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_in = 0x15d8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_kan = 0x15d9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_las = 0x15da;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_man = 0x15db;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_nar = 0x15dc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_on = 0x15dd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_par = 0x15de;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_zhar = 0x15df;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_rae = 0x15e0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_san = 0x15e1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_tar = 0x15e2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_un = 0x15e3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_phar = 0x15e4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_khar = 0x15e5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_ghan = 0x15e6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_qar = 0x15e7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_shin = 0x15e8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_chin = 0x15e9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_can = 0x15ea;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_jil = 0x15eb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_cil = 0x15ec;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_char = 0x15ed;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_xan = 0x15ee;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_jhan = 0x15ef;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_hae = 0x15f0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_he = 0x15f1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_hie = 0x15f2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_we = 0x15f3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_har = 0x15f4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_hoe = 0x15f5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Georgian_fi = 0x15f6;

/*
 * Azeri (and other Turkic or Caucasian languages of ex-USSR)
 * Byte 3 = 0x16
 */

/* latin */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ccedillaabovedot = 0x16a2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Xabovedot = 0x16a3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Qabovedot = 0x16a5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ibreve = 0x16a6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_IE = 0x16a7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_UO = 0x16a8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Zstroke = 0x16a9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Gcaron = 0x16aa;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Obarred = 0x16af;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ccedillaabovedot = 0x16b2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_xabovedot = 0x16b3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ocaron = 0x16b4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_qabovedot = 0x16b5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ibreve = 0x16b6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ie = 0x16b7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_uo = 0x16b8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_zstroke = 0x16b9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_gcaron = 0x16ba;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ocaron = 0x16bd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_obarred = 0x16bf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_SCHWA = 0x16c6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_schwa = 0x16f6;
/* those are not really Caucasus; but I put them here for now */
/* For Inupiak */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Lbelowdot = 0x16d1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Lstrokebelowdot = 0x16d2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_lbelowdot = 0x16e1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_lstrokebelowdot = 0x16e2;
/* For Guarani */
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Gtilde = 0x16d3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_gtilde = 0x16e3;

/*
 * Vietnamese
 * Byte 3 = 0x1e
 */

static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Abelowdot = 0x1ea0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_abelowdot = 0x1ea1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ahook = 0x1ea2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ahook = 0x1ea3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Acircumflexacute = 0x1ea4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_acircumflexacute = 0x1ea5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Acircumflexgrave = 0x1ea6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_acircumflexgrave = 0x1ea7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Acircumflexhook = 0x1ea8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_acircumflexhook = 0x1ea9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Acircumflextilde = 0x1eaa;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_acircumflextilde = 0x1eab;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Acircumflexbelowdot = 0x1eac;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_acircumflexbelowdot = 0x1ead;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Abreveacute = 0x1eae;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_abreveacute = 0x1eaf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Abrevegrave = 0x1eb0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_abrevegrave = 0x1eb1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Abrevehook = 0x1eb2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_abrevehook = 0x1eb3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Abrevetilde = 0x1eb4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_abrevetilde = 0x1eb5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Abrevebelowdot = 0x1eb6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_abrevebelowdot = 0x1eb7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ebelowdot = 0x1eb8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ebelowdot = 0x1eb9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ehook = 0x1eba;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ehook = 0x1ebb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Etilde = 0x1ebc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_etilde = 0x1ebd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ecircumflexacute = 0x1ebe;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ecircumflexacute = 0x1ebf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ecircumflexgrave = 0x1ec0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ecircumflexgrave = 0x1ec1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ecircumflexhook = 0x1ec2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ecircumflexhook = 0x1ec3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ecircumflextilde = 0x1ec4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ecircumflextilde = 0x1ec5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ecircumflexbelowdot = 0x1ec6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ecircumflexbelowdot = 0x1ec7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ihook = 0x1ec8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ihook = 0x1ec9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ibelowdot = 0x1eca;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ibelowdot = 0x1ecb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Obelowdot = 0x1ecc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_obelowdot = 0x1ecd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ohook = 0x1ece;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ohook = 0x1ecf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ocircumflexacute = 0x1ed0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ocircumflexacute = 0x1ed1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ocircumflexgrave = 0x1ed2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ocircumflexgrave = 0x1ed3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ocircumflexhook = 0x1ed4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ocircumflexhook = 0x1ed5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ocircumflextilde = 0x1ed6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ocircumflextilde = 0x1ed7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ocircumflexbelowdot = 0x1ed8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ocircumflexbelowdot = 0x1ed9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ohornacute = 0x1eda;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ohornacute = 0x1edb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ohorngrave = 0x1edc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ohorngrave = 0x1edd;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ohornhook = 0x1ede;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ohornhook = 0x1edf;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ohorntilde = 0x1ee0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ohorntilde = 0x1ee1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ohornbelowdot = 0x1ee2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ohornbelowdot = 0x1ee3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ubelowdot = 0x1ee4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ubelowdot = 0x1ee5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Uhook = 0x1ee6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_uhook = 0x1ee7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Uhornacute = 0x1ee8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_uhornacute = 0x1ee9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Uhorngrave = 0x1eea;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_uhorngrave = 0x1eeb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Uhornhook = 0x1eec;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_uhornhook = 0x1eed;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Uhorntilde = 0x1eee;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_uhorntilde = 0x1eef;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Uhornbelowdot = 0x1ef0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_uhornbelowdot = 0x1ef1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ybelowdot = 0x1ef4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ybelowdot = 0x1ef5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Yhook = 0x1ef6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_yhook = 0x1ef7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ytilde = 0x1ef8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ytilde = 0x1ef9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Ohorn = 0x1efa;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ohorn = 0x1efb;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_Uhorn = 0x1efc;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_uhorn = 0x1efd;

static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_combining_tilde = 0x1e9f;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_combining_grave = 0x1ef2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_combining_acute = 0x1ef3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_combining_hook = 0x1efe;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_combining_belowdot = 0x1eff;

static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_EcuSign = 0x20a0;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_ColonSign = 0x20a1;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_CruzeiroSign = 0x20a2;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_FFrancSign = 0x20a3;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_LiraSign = 0x20a4;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_MillSign = 0x20a5;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_NairaSign = 0x20a6;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_PesetaSign = 0x20a7;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_RupeeSign = 0x20a8;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_WonSign = 0x20a9;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_NewSheqelSign = 0x20aa;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_DongSign = 0x20ab;
static const scim_bridge_key_code_t SCIM_BRIDGE_KEY_CODE_EuroSign = 0x20ac;

#ifdef __cplusplus
extern "C"
{
#endif

    /**
     * Allocate a new key event.
     *
     * @return The new key event.
     */
    ScimBridgeKeyEvent *scim_bridge_alloc_key_event ();

    /**
     * Free a key event.
     *
     * @param key_event The key event to free.
     * @note Do not use free (). It will cause memleaks.
     */
    void scim_bridge_free_key_event (ScimBridgeKeyEvent *key_event);

    /**
     * Get the key code of an event.
     *
     * @param key_event The key event.
     * @return The key code of the event.
     */
    scim_bridge_key_code_t scim_bridge_key_event_get_code (const ScimBridgeKeyEvent *key_event);

    /**
     * Set the key code of an event.
     *
     * @param key_event The key event.
     * @param key_code The key code.
     */
    void scim_bridge_key_event_set_code (ScimBridgeKeyEvent *key_event, scim_bridge_key_code_t key_code);

    /**
     * Get the key state of an event.
     *
     * @param key_event The key event.
     * @return True if pressed, otherwise it returns False.
     */
    boolean scim_bridge_key_event_is_pressed (const ScimBridgeKeyEvent *key_event);

    /**
     * Set the key state of an event.
     *
     * @param key_event The key event.
     * @param pressed Give True if pressed, otherwise give False.
     */
    void scim_bridge_key_event_set_pressed (ScimBridgeKeyEvent *key_event, boolean pressed);

    /**
     * Clear all the modifier flags of an event.
     *
     * @param key_event The key event.
     */
    void scim_bridge_key_event_clear_modifiers (ScimBridgeKeyEvent *key_event);

    /**
     * Get the modifier flags of an event.
     *
     * @param key_event The key event.
     * @return The modifier flags of the event.
     */
    scim_bridge_key_modifier_t scim_bridge_key_event_get_modifiers (const ScimBridgeKeyEvent *key_event);

    /**
     * Set the modifier flags of an event.
     *
     * @param key_event The key event.
     * @param modifiers The modifier flags of the event.
     */
    void scim_bridge_key_event_set_modifiers (ScimBridgeKeyEvent *key_event, scim_bridge_key_modifier_t modifiers);

    /**
     * Get the state of the shift key of an event.
     *
     * @param key_event The key event.
     * @return True when the shift key is down, otherwise it returns False.
     */
    boolean scim_bridge_key_event_is_shift_down (const ScimBridgeKeyEvent *key_event);

    /**
     * Set the state of the shift key of an event.
     *
     * @param key_event The key event.
     * @param down Give True when the shift key is down, otherwise give False.
     */
    void scim_bridge_key_event_set_shift_down (ScimBridgeKeyEvent *key_event, boolean down);

    /**
     * Get the state of the caps lock key of an event.
     *
     * @param key_event The key event.
     * @return True when the caps lock key is down, otherwise it returns False.
     */
    boolean scim_bridge_key_event_is_caps_lock_down (const ScimBridgeKeyEvent *key_event);

    /**
     * Set the state of the caps lock key of an event.
     *
     * @param key_event The key event.
     * @param down Give True when the caps lock key is down, otherwise give False.
     */
    void scim_bridge_key_event_set_caps_lock_down (ScimBridgeKeyEvent *key_event, boolean down);

    /**
     * Get the control state of an event.
     *
     * @param key_event The key event.
     * @return True when the control key is down, otherwise it returns False.
     */
    boolean scim_bridge_key_event_is_control_down (const ScimBridgeKeyEvent *key_event);

    /**
     * Set the state of the control key of an event.
     *
     * @param key_event The key event.
     * @param down Give True when the control is down, otherwise give False.
     */
    void scim_bridge_key_event_set_control_down (ScimBridgeKeyEvent *key_event, boolean down);

    /**
     * Get the state of the alt key of an event.
     *
     * @param key_event The key event.
     * @return True when the alt key is down, otherwise it returns False.
     */
    boolean scim_bridge_key_event_is_alt_down (const ScimBridgeKeyEvent *key_event);

    /**
     * Set the state of the alt key of an event.
     *
     * @param key_event The key event.
     * @param down Give True when the alt key is down, otherwise give False.
     */
    void scim_bridge_key_event_set_alt_down (ScimBridgeKeyEvent *key_event, boolean down);

    /**
     * Get the state of the meta key of an event.
     *
     * @param key_event The key event.
     * @return True when the meta key is down, otherwise it returns False.
     */
    boolean scim_bridge_key_event_is_meta_down (const ScimBridgeKeyEvent *key_event);

    /**
     * Set the state of the meta key of an event.
     *
     * @param key_event The key event.
     * @param down Give True when the meta key is down, otherwise give False.
     */
    void scim_bridge_key_event_set_meta_down (ScimBridgeKeyEvent *key_event, boolean down);

    /**
     * Get the state of the super key of an event.
     *
     * @param key_event The key event.
     * @return True when the super key is down, otherwise it returns False.
     */
    boolean scim_bridge_key_event_is_super_down (const ScimBridgeKeyEvent *key_event);

    /**
     * Set the state of the super key of an event.
     *
     * @param key_event The key event.
     * @param down Give True when the super key is down, otherwise give False.
     */
    void scim_bridge_key_event_set_super_down (ScimBridgeKeyEvent *key_event, boolean down);

    /**
     * Get the state of the hyper key of an event.
     *
     * @param key_event The key event.
     * @return True when the hyper key is down, otherwise it returns False.
     */
    boolean scim_bridge_key_event_is_hyper_down (const ScimBridgeKeyEvent *key_event);

    /**
     * Set the state of the huper key of an event.
     *
     * @param key_event The key event.
     * @param down Give True when the hyper key is down, otherwise give False.
     */
    void scim_bridge_key_event_set_hyper_down (ScimBridgeKeyEvent *key_event, boolean down);

    /**
     * Get the state of the num lock key of an event.
     *
     * @param key_event The key event.
     * @return True when the num lock key is down, otherwise it returns False.
     */
    boolean scim_bridge_key_event_is_num_lock_down (const ScimBridgeKeyEvent *key_event);

    /**
     * Set the state of the num lock key of an event.
     *
     * @param key_event The key event.
     * @param down Give True when the num lock key is down, otherwise give False.
     */
    void scim_bridge_key_event_set_num_lock_down (ScimBridgeKeyEvent *key_event, boolean down);

#ifdef __cplusplus
}
#endif
#endif                                            /*SCIM_BRIDGE_KEY_EVENT_H_*/
