/*
 * Smart Common Input Method
 * 
 * Copyright (c) 2004 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2003 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim.cpp,v 1.31.2.1 2004/05/01 13:23:37 suzhe Exp $
 *
 */

#define Uses_SCIM_FRONTEND_MODULE
#define Uses_SCIM_SERVER_MODULE
#define Uses_SCIM_BACKEND
#define Uses_SCIM_CONFIG_PATH
#define Uses_C_LOCALE
#include "scim_private.h"
#include <scim.h>
#include <sys/types.h>
#include <unistd.h>
#include <signal.h>

using namespace scim;
using std::cout;
using std::cerr;
using std::endl;

FrontEndModule *frontend_module = 0;
ConfigModule   *config_module = 0;
ServerModule   *server_modules = 0;

void signalhandler(int sig)
{
    delete frontend_module;
    delete [] server_modules;

    /* FIXME: delete config_module will cause segfault when using gconf module.
    delete config_module; 
    */

    cerr << "SCIM successfully exited.\n";

    exit (0);
}

void make_daemon ()
{
#if HAVE_DAEMON
    if (daemon (0, 0) == -1)
        cerr << "Error to make SCIM into a daemon!\n";

    return;
#else        
    pid_t id;
 
    id = fork ();
    if (id == -1) {
        cerr << "Error to make SCIM into a daemon!\n";
        return;
    } else if (id > 0) {
        exit (0);
    }

    id = fork ();
    if (id == -1) {
        cerr << "Error to make SCIM into a daemon!\n";
        return;
    } else if (id > 0) {
        exit (0);
    }

    return;
#endif
}

int main (int argc, char *argv [])
{
    ServerFactoryPointer server;
    ConfigPointer        config;
    BackEndPointer       backend;

    std::vector<String>  frontend_list;
    std::vector<String>  server_list;
    std::vector<String>  config_list;
    std::vector<String>  exclude_server_list;
    std::vector<String>  disabled_server_factories;

    String def_frontend;
    String def_config;
    String debug_output;
    String utf8_locales;

    std::vector<String> debug_mask_list;

    uint32 verbose_level = 0;

    int i;
    int all_factories_count = 0;
    int module_factories_count = 0;

    bool daemon = false;

    //Display version info
    cout << "Smart Common Input Method " << SCIM_VERSION << "\n\n";

    //get modules list
    scim_get_frontend_module_list (frontend_list);
    scim_get_server_module_list (server_list);
    scim_get_config_module_list (config_list);

    //Add a dummy config module, it's not really a module!
    config_list.push_back ("dummy");

    //Use x11 FrontEnd as default if available.
    if (frontend_list.size ()) {
        def_frontend = String ("x11");
        if (std::find (frontend_list.begin (),
                       frontend_list.end (),
                       def_frontend) == frontend_list.end ())
            def_frontend = frontend_list [0];
    }

    //Use simple Config module as default if available.
    if (config_list.size ()) {
        def_config = String ("simple");
        if (std::find (config_list.begin (),
                       config_list.end (),
                       def_config) == config_list.end ())
            def_config = config_list [0];
    }

    //parse command options
    i = 0;
    while (i<argc) {
        if (++i >= argc) break;

        if (String ("-l") == argv [i] ||
            String ("--list") == argv [i]) {
            std::vector<String>::iterator it;

            cout << endl;
            cout << "Available FrontEnd module:\n";
            for (it = frontend_list.begin (); it != frontend_list.end (); it++)
                cout << "    " << *it << endl;

            cout << endl;
            cout << "Available Config module:\n";
            for (it = config_list.begin (); it != config_list.end (); it++)
                cout << "    " << *it << endl;

            cout << endl;
            cout << "Available Server module:\n";
            for (it = server_list.begin (); it != server_list.end (); it++)
                cout << "    " << *it << endl;

            return 0;
        }

        if (String ("-f") == argv [i] ||
            String ("--frontend") == argv [i]) {
            if (++i >= argc) {
                cerr << "No argument for option " << argv [i-1] << endl;
                return -1;
            }
            def_frontend = argv [i];
            continue;
        }

        if (String ("-c") == argv [i] ||
            String ("--config") == argv [i]) {
            if (++i >= argc) {
                cerr << "No argument for option " << argv [i-1] << endl;
                return -1;
            }
            def_config = argv [i];
            continue;
        }

        if (String ("-h") == argv [i] ||
            String ("--help") == argv [i]) {
            cout << "Usage: " << argv [0] << " [option]...\n\n"
                 << "The options are: \n"
                 << "  -l, --list              List all of available modules.\n"
                 << "  -f, --frontend name     Use specified FrontEnd module.\n"
                 << "  -c, --config name       Use specified Config module.\n"
                 << "  -s, --servers name      Load specified set of servers.\n"
                 << "  -ns,--no-servers name   Do not load those set of servers.\n"
                 << "  -d, --daemon            Run " << argv [0] << " as a daemon.\n"
#if ENABLE_DEBUG
                 << "  -v, --verbose           Enable debug info, multiple -v for more.\n"
                 << "  -o, --output file       Output debug information into file.\n"
                 << "  -m, --mask mask         Set debug output mask, the valid values are:\n"
                 << "                          all,main,config,server,backend,frontend,module,\n"
                 << "                          utility,iconv,lookuptable,socket.\n"
                 << "                          multiple masks can be separated by comma.\n"
#endif
                 << "  -h, --help              Show this help message.\n";
            return 0;
        }

        if (String ("-d") == argv [i] ||
            String ("--daemon") == argv [i]) {
            daemon = true;
            continue;
        }

        if (String ("-s") == argv [i] ||
            String ("--server") == argv [i]) {
            if (++i >= argc) {
                cerr << "No argument for option " << argv [i-1] << endl;
                return -1;
            }
            scim_split_string_list (server_list, String (argv [i]), ',');
            continue;
        }

        if (String ("-ns") == argv [i] ||
            String ("--no-servers") == argv [i]) {
            if (++i >= argc) {
                cerr << "No argument for option " << argv [i-1] << endl;
                return -1;
            }
            scim_split_string_list (exclude_server_list, String (argv [i]), ',');
            continue;
        }

        if (String ("-v") == argv [i] ||
            String ("--verbose") == argv [i]) {
            verbose_level += 1;
            continue;
        }

        if (String ("-m") == argv [i] ||
            String ("--mask") == argv [i]) {
            if (++i >= argc) {
                cerr << "No argument for option " << argv [i-1] << endl;
                return -1;
            }
            scim_split_string_list (debug_mask_list, String (argv [i]), ',');
            continue;
        }

        if (String ("-o") == argv [i] ||
            String ("--output") == argv [i]) {
            if (++i >= argc) {
                cerr << "No argument for option " << argv [i-1] << endl;
                return -1;
            }
            debug_output = argv [i];
            continue;
        }

        if (String ("--") == argv [i])
            break;
    } //End of command line parsing.

    DebugOutput::set_output (debug_output);
    DebugOutput::set_verbose_level (verbose_level);

    if (debug_mask_list.size ()) {
        DebugOutput::disable_debug (SCIM_DEBUG_AllMask);
        for (i=0; i<debug_mask_list.size (); i++)
            DebugOutput::enable_debug_by_name (debug_mask_list [i]);
    }

    if (!def_frontend.length ()) {
        cerr << "No FrontEnd module is available!\n";
        return -1;
    }

    if (!def_config.length ()) {
        cerr << "No Config module is available!\n";
        return -1;
    }

    if (!server_list.size ()) {
        cerr << "No Server module is available or specified!\n";
        return -1;
    }

    // Try to load config module
    try {
        cout << "Loading " << def_config << " Config module ...\n";
        if (def_config != "dummy") {
            //load config module
            config_module = new ConfigModule (def_config);

            if (!config_module->valid ()) {
                cerr << "Can not load " << def_config << " Config module. Using dummy module instead.\n";
                delete config_module;
                config_module = 0;
            }

        }

        if (config_module) {
            config = config_module->create_config ("scim");
        } else {
            config = new DummyConfig ("scim");
        }

        if (!config) {
            cerr << "Can not create Config Object!\n";
            return -1;
        }
    } catch (const std::exception & err) {
        cerr << err.what () << "\n";
        return -1;
    }

    // create backend
    try {
        cout << "Creating backend ...\n";
        backend = new CommonBackEnd ();
    } catch (const std::exception & err) {
        cerr << err.what () << "\n";
        return -1;      
    }

    // Get disabled factories list.
    config->read (SCIM_CONFIG_DISABLED_SERVER_FACTORIES, &disabled_server_factories);

    // Set the default supported locales.
    config->read (SCIM_CONFIG_SUPPORTED_UNICODE_LOCALES, &utf8_locales);
    cast_dynamic <CommonBackEnd, BackEndBase> (backend)->set_supported_unicode_locales (utf8_locales);

    // Try to load all server modules
    try {
        server_modules = new ServerModule [server_list.size ()];
    } catch (const std::exception & err) {
        cerr << err.what () << "\n";
        return -1;
    }

    //load Server modules
    for (i=0; i<server_list.size (); i++) {
        //Skip exculded server module.
        if (std::find (exclude_server_list.begin (),
                       exclude_server_list.end (),
                       server_list [i]) != exclude_server_list.end ())
            continue;

        cout << "Loading Server module: " << server_list [i] << " ...\n";

        module_factories_count = 0;

        if (server_modules [i].load (server_list[i], config) &&
            server_modules [i].valid ()) {
            for (uint32 j=0; j < server_modules [i].number_of_servers (); j++) {
                cout << "    Loading Server Factory " << j << " ... ";

                // Try to load a Server Factory.
                try {
                    server = server_modules [i].create_factory (j);
                } catch (...) {
                    server.reset ();
                }

                if (!server.null ()) {
                    // Check if it's disabled.
                    if (std::find (disabled_server_factories.begin (),
                                   disabled_server_factories.end (),
                                   server->get_uuid ()) == disabled_server_factories.end () &&
                        cast_dynamic <CommonBackEnd, BackEndBase> (backend)->add_server_factory (server)) {

                        all_factories_count ++;
                        module_factories_count ++;
                        cout << ": OK\n";
                    } else {
                        cout << ": Disabled\n";
                        server.reset ();
                    }
                } else {
                    cout << ": Failed\n";
                }
            }
            if (module_factories_count) {
                cout << server_list [i] << " Server module is successfully loaded.\n";
            } else {
                cout << "No Server Factory loaded from " << server_list [i]
                     << " Server module!\n";
                server_modules [i].unload ();
            }
        } else {
            cerr << "Failed to load " << server_list [i] << " Server module.\n";
        }
    }

    if (!all_factories_count) {
        server = new DummyServerFactory ();
        cast_dynamic <CommonBackEnd, BackEndBase> (backend)->add_server_factory (server);
    }

    server.reset ();

    //load FrontEnd module
    try {
        cout << "Loading " << def_frontend << " FrontEnd module ...\n";
        frontend_module = new FrontEndModule (def_frontend, backend, config, argc, argv);
    } catch (const std::exception & err) {
        cerr << err.what () << "\n";
        frontend_module = 0;
    }

    if (!frontend_module || !frontend_module->valid ()) {
        cerr << "Failed to load " << def_frontend << " FrontEnd module.\n";
        return -1;
    }

    //reset backend pointer, in order to destroy backend automatically.
    backend.reset ();

    //reset config pointer, in order to destroy config automatically.
    config.reset ();

    signal(SIGQUIT, signalhandler);
    signal(SIGTERM, signalhandler);
    signal(SIGINT,  signalhandler);
    signal(SIGHUP,  signalhandler);

    if (daemon) {
        cout << "Starting SCIM as daemon ..." << endl;
        make_daemon ();
    } else {
        cout << "Starting SCIM ..." << endl;
    }

    try {
        frontend_module->run ();
    } catch (const std::exception & err) {
        cerr << err.what () << endl;
        return -1;
    }

    return 0;
}

/*
vi:ts=4:ai:nowrap:expandtab
*/
