/** @file scim_attributes.h
 *  @brief Definition of scim::Attribute and scim::AttributeList
 *
 *  Provide class scim::Attribute to control the
 *  drawing effect of strings.
 */

/*
 * Smart Common Input Method
 * 
 * Copyright (c) 2004 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2003 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_attributes.h,v 1.10 2004/02/06 07:53:15 suzhe Exp $
 */


#ifndef __SCIM_ATTRIBUTES_H
#define __SCIM_ATTRIBUTES_H

namespace scim {

/**
 * @addtogroup Helper
 *
 * The helper classes and functions, including Attribute, IConvert, LookupTable etc.
 *
 * @{
 */

/**
 * @brief Enum values of the valid attributes.
 */
enum AttributeType
{
    SCIM_ATTR_NONE,         ///< No attribute
    SCIM_ATTR_UNDERLINE,    ///< Draw a line under the text
    SCIM_ATTR_HIGHLIGHT,    ///< Draw the text in highlighted color
    SCIM_ATTR_REVERSE       ///< Draw the text in reverse color mode
};

/**
 * @brief Class to store the string attributes.
 *
 * The string attributes control the effect of the string
 * drawn by FrontEnds. There are currently four valid effects.
 * Please refer to scim::AttributeType for effect details.
 *
 * @sa scim::AttributeType scim::FrontEndBase
 */
class Attribute
{
    unsigned int  m_start;
    unsigned int  m_length;

    AttributeType m_type;

public:
    /**
     * @brief Constructor
     * @param start - the start position in the string of this attribute.
     * @param length - the length of this attribute, the range is [start,start+length).
     * @param type - the type of this attribute.
     */
    Attribute (unsigned int  start = 0,
               unsigned int  length = 0,
               AttributeType type  = SCIM_ATTR_NONE) :
        m_start (start), m_length (length), m_type (type)
        { }

    /**
     * @brief Get the type of this attribute.
     * @return The type of this attribute.
     */
    AttributeType get_type () const { return m_type; }

    /**
     * @brief Get the start position of this attribute.
     * @return The start position of this attribute in the string.
     */
    unsigned int get_start () const { return m_start; }

    /**
     * @brief Get the length of this attribute.
     * @return The length of this attribute in the string.
     */
    unsigned int get_length () const { return m_length; }

    /**
     * @brief Get the end position of this attribute.
     * @return The end position of this attribute.
     */
    unsigned int get_end () const { return m_start + m_length; }

    /**
     * @brief Set the type of this attribute.
     * @param type - the new attribute type to be set.
     */
    void set_type (AttributeType type) { m_type = type; }

    /**
     * @brief Set the start position of this attribute.
     * @param start - the new start position in the string.
     */
    void set_start (unsigned int start) {
        m_start = start;
    }

    /**
     * @brief Set the length of this attribute.
     * @param length - the new length of this attribute.
     */
    void set_length (unsigned int length) {
        m_length = length;
    }
};

/**
 * @typedef typedef std::vector<Attribute> AttributeList
 * @brief The container to store a set of Attribute objects.
 *
 * You should use the STL container methods to manipulate its objects.
 */
typedef std::vector<Attribute> AttributeList;

/** @} */

} // namespace scim

#endif //__SCIM_ATTRIBUTES_H

/*
vi:ts=4:nowrap:ai:expandtab
*/
